function recon_dti(path, fn_analyze_base)
% Reconstructs SEMS file
% Usage:
%       recon_sems(path)
% Parameters:
%       path            Directory containing the 'fid' file
%       fn_analyze_base Base of the analyze filename where to save the 
%                       output. For instance if you pass in 'file.img'
%                       then the output will be a series of images
%                       file_b0.img
%                       file_1000#1.img
%                       file_1000#2.img (and so on)

% Get the dimensions of the image
np = read_procpar(path,'np') / 2;
nv = read_procpar(path,'nv');
ns = read_procpar(path,'ns');

% Get the FOV specification
lro = read_procpar(path,'lro');
lpe = read_procpar(path,'lpe');

% Get the slice positions
pss = read_procpar(path,'pss');

% Get the number of DTI components
ad = read_procpar(path,'arraydim');
nc = ad / nv;

% Get the DT parameters
bvalue = read_procpar(path,'bvalue');

% Compute the voxel size in mm
vox(1) = 10 * lro / np;
vox(2) = 10 * lpe / nv;
vox(3) = 10 * (pss(end) - pss(1)) / (ns - 1);
vox(4) = 1;
fprintf('Voxel Size: (%f, %f, %f)\n',vox(1:3));

% Read the FID file
[RE,IM,NP,NB,NT,HDR]=read_fid(path);
fprintf('Read FID: NP = %d, NB = %d, NT = %d\n',NP,NB,NT);

% Check the outputs
if (NT ~= ns) || (NB ~= nv * nc) || (NP ~= np)
    error('PROCPAR parameters do not match FID file');
end

% Create a series of filenames for output
[fn_path fn_name fn_ext fn_ver] = fileparts(fn_analyze_base);
fn_out = fullfile(fn_path,[fn_name '_b0.img']);
for i = 2:nc
    fn_next = sprintf('%s_%04d#%d.img',fn_name,round(bvalue(i)),i-1);
    fn_out = strvcat(fn_out, fullfile(fn_path, fn_next));
end

% Create the output image header
Vi.dim = [np nv ns 64];
Vi.mat = diag(vox);
Vi.pinfo = [1;0;0];

% Loop over the DTI components
for c = 1:nc

    % Allocate data for output
    Y = zeros(np,nv,ns);

    % Loop over the slices
    for i = 1:ns
        
        % The column index for this slice
        idx = i+(c-1)*ns:nc*ns:nc*ns*nv;
    
        % Get the slice of the K-space
        kslice = complex(RE(:,idx),IM(:,idx));

        % Compute the FFT
        islice = fftshift(fft2(fftshift(kslice)));

        % Store the slice in Y
        Y(:,:,i) = abs(islice);

    end
    
    % Save the volume
    Vi.fname = deblank(fn_out(c,:));
    fprintf('Writing %s\n', Vi.fname);
    Vo = spm_create_vol(Vi);
    spm_write_vol(Vo, Y);
end

% Write diffusion directions into a file
dir(:,1) = read_procpar(path,'dro');
dir(:,2) = read_procpar(path,'dpe');
dir(:,3) = read_procpar(path,'dsl');

fn_dti = fullfile(fn_path,[fn_name '_dir.txt']);
fd = fopen(fn_dti,'w');
fprintf(fd,'%f %f %f\n',dir');
fclose(fd);

% Also, write the scheme file for CAMINO
t = read_procpar(path, 'tDELTA') - read_procpar(path, 'tdelta') / 3;
bval = read_procpar(path, 'bvalue');
qvec = dir .* (sqrt(bval / t) * ones(1,3))


fn_scheme = fullfile(fn_path,[fn_name '_scheme.txt']);
fs = fopen(fn_scheme, 'w');
fprintf(fs, '%f\n%d\n', t, length(bval));
for c = 1:nc
    fprintf(fs,'%f\n%f\n%f\n', qvec(c,:));
end
fclose(fs);

% Finally, write a manifest file for CAMINO
fn_manifest = fullfile(fn_path,[fn_name '_manifest.txt']);
fm = fopen(fn_manifest, 'w');
for c = 1:nc
    [p2 f2 e2 v2] = fileparts(deblank(fn_out(c,:)));
    fprintf(fm, '%s.hdr\n', f2); 
end
fclose(fm)
