#!/usr/bin/env python
###########################################################################
# @package test
# @brief Script that runs all four synthetic test cases
# It takes as input 
# @param installDir the installation directory
# It outputs 
# bin/ libexec/ share/ doc/ within install dir
#
# @author Harini Eavani
#
# @link: https://www.cbica.upenn.edu/sbia/software/
#
# @author: sbia-software@uphs.upenn.edu
##########################################################################

import numpy as  np
from draw_moe_plot import draw_moe_plot
import pandas as pd
from MOEUtils import execCmd, fileExists, cryandexit
from SV_fuzzy import OutputVals

import os, sys

## iterate through four test cases
N=[1,2,3,2]
testcases = ['Case'+str(i+1) for i in range(4)]
for e,testcase in enumerate(testcases):
    ## set up input and output command line arguments
    print('Running MOE on test '+str(testcase))
    inData='test/'+testcase+'/'+testcase+'.csv'
    prefix=testcase+'_results'
    outDir='test/'+testcase+'/'
    cmdArray=['./moe','-d',inData,'-p',prefix,'-H','label','-l','AD','-o',outDir,'-n',str(N[e]),'-c',str(1000),'-f',str(10)]
    ## call executable 
    execCmd(cmdArray,verbose=0,simulate=False)
    
    ## check if output file is created
    outputfile = outDir+'/'+prefix+'_allOptResults.npz'
    if not fileExists(outputfile):
        cryandexit("Error in running MOE, ", outputfile," not found")
        
    ## read input data
    mydata = pd.read_csv(inData,header=0,quotechar='"',sep=',',na_values = ['NA', '-', '.', ''],index_col=0)
    y = mydata['label'].as_matrix()
    mydata = mydata.drop('label', 1)
    X = mydata.as_matrix().astype('float64') 

    ## read output data
    out_data = np.load(outputfile)      
    out_vals = OutputVals()
    out_vals.hyperplanes = out_data['hyperplanes']
    out_vals.intercepts = out_data['intercepts']
    out_vals.memberships = out_data['memberships']
    
    ## now draw the plot and save
    outImage = outDir+'/'+prefix+'_MOEplot.png'
    draw_moe_plot(X,y,out_vals,outImage)
            

    ## run MOE for K=1, ie., linear SVM
    print('Running linear-SVM on test '+str(testcase))
    prefix=testcase+'_linearSVM_results'
    cmdArray=['./moe','-d',inData,'-p',prefix,'-H','label','-l','AD','-o',outDir,'-n',str(1),'-c',str(1000),'-f',str(10)]
    ## call executable 
    execCmd(cmdArray,verbose=0,simulate=False)
    
    out_linearSVM_file = outDir+'/'+prefix+'_allOptResults.npz'
    if not fileExists(out_linearSVM_file):
        cryandexit("Error in running MOE, ", out_linearSVM_file," not found")

    ## read linear svm accuracy
    out_linearSVM_data = np.load(out_linearSVM_file)
        
    ## check cross-validated accuracy
    if out_data['cvBest']>= out_linearSVM_data['cvBest']:
        print('Ten fold cross-validated accuracy for MOE >= Linear SVM, '+testcase+' passed!\n')
    else:
        print('Ten fold cross-validated accuracy for MOE < Linear SVM, '+testcase+' failed!\n')     
    
    
    
