/*!
 * \file   SbiaOdvbaCreateIndex.cpp
 * \brief  Command line tool which creates the index file (template).
 * \author Tianhao Zhang, revised by Andreas Schuh
 * \date   01/04/2011
 *
 * $Revision: 143 $
 * $Id: SbiaOdvbaCreateIndex.cpp 143 2011-02-08 18:17:30Z schuha@UPHS.PENNHEALTH.PRV $
 *
 * <b>Last changed:</b>
 * $Author: schuha@UPHS.PENNHEALTH.PRV $
 * $Date: 2011-02-08 13:17:30 -0500 (Tue, 08 Feb 2011) $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For copyright information see Copyright section of project
 * ReadMe.html in the project's root directory.
 *
 * Contact: sbia-software@uphs.upenn.edu
 */


#include <getopt.h>
#include <stdio.h>

#include "SbiaOdvbaUtilities.h"


SBIA_ODVBA_NAMESPACE_USE;


//////////////////////////////////////////////////////////////////////////////
// usage / help
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Extract program name from first command line argument.
 *
 * \param argv Command line arguments.
 *
 * \return Program name without relative or absolute path to directory where
 *         the executable is located. Do not try to free the memory of the
 *         returned pointer! It just points to the first character of the
 *         program name within argv [0].        
 **/
const char *getProgName (char *argv[])
{
	const char *progName = argv [0];

	if (progName && *progName)
	{
		// go to last character of string
		while (*progName) ++progName;
		--progName;
		// rewind until beginning of string or
		// '/' (UNIX) or '\\' (WIN) is encountered
		while (progName != argv [0])
		{
			--progName;
			if (*progName == '/' || *progName == '\\')
			{
				++progName;
				break;
			}
		}
	}

	return progName;
}

/*!
 * \brief Prints version information.
 *
 * \param progName Name of program.
 */
void version (const char *progName)
{
	printf ("%s %s", progName, SBIA_ODVBA_VERSION);
#if SBIA_ODVBA_REVISION_VALID
	printf (" (Rev. %d)", SBIA_ODVBA_REVISION);
#endif // SBIA_ODVBA_REVISION_VALID
	printf ("\n");
}

/*!
 * \brief Prints usage information / help.
 *
 * \param progName Name of program.
 */
void usage (const char *progName)
{
	version (progName);
	printf ("\n");
	printf ("Description:\n");
	printf ("  This program determines the indices of voxels in the set of input images\n");
	printf ("  whose average voxel value exceeds a certain threshold. Hence, it determines\n");
	printf ("  the indices of voxels relevant to the group analysis and outputs them to a\n");
	printf ("  text file which is used as input to the main program which performs the\n");
	printf ("  group analysis.\n");
	printf ("  \n");
	printf ("  T. Zhang and C. Davatzikos; ODVBA: Optimally-Dicriminative ");
	printf ("  Voxel-Based Analysis\n");
	printf ("\n");
	printf ("Usage:\n");
	printf ("  %s [options] <subjects.txt>\n", progName);
	printf ("\n");
	printf ("Required options:\n");
	printf ("  <subjects.txt> : The subjets list which specifies the input data given as NIfTI-1 images.\n");
	printf ("\n");
	printf ("Options:\n");
	printf ("  [-o --out <filename>] : Filename of the output index file.\n");
	printf ("                          The default value is 'index.txt'.\n");
	printf ("  [-h --help]           : Print help and exit.\n");
	printf ("  [-u --usage]          : Print usage information and exit.\n");
	printf ("  [-V --version]        : Print version information and exit.\n");
	printf ("  [-v --verbose]        : Enable verbose messages. Can be specified multiple times\n");
	printf ("                          to increase the verbosity.\n");
	printf ("\n");
	printf ("Example:\n");
	printf ("  %s subjects.txt -o index.txt\n", progName);
}

//////////////////////////////////////////////////////////////////////////////
// main
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Main.
 *
 * \param [in] argc Number of command line arguments.
 * \param [in] argv Command line arguments.
 */
int main (int argc, char *argv[])
{
	const char *progName = getProgName (argv);
	bool        ok       = true;

	/**
	 * \struct long_options
	 * \brief  Structure of long program options.	 
	 */
	static struct option long_options [] =
	{
		{"out",     required_argument, NULL, 'o'},
		{"usage",   no_argument,       NULL, 'u'},
		{"help",    no_argument,       NULL, 'h'},
		{"version", no_argument,       NULL, 'V'},
		{"Version", no_argument,       NULL, 'V'},
		{"verbose", no_argument,       NULL, 'v'}
	}; // struct long_options

	int c      = -1;
	int optidx = 0;

	// default options
	int         verbosity   = 0;           // verbosity of messages
	const char *sublistFile = NULL;        // subjects list file
	const char *indexFile   = "index.txt"; // index file

	while ((c = getopt_long (argc, argv, "o:uhVv", long_options, &optidx)) != -1)
	{
		switch (c)
		{
		case 'o':
			indexFile = optarg;
			break;

		case 'u':
			// fall-through intended
		case 'h':
			usage (progName);
			exit (EXIT_SUCCESS);

		case 'V':
			version (progName);
			exit (EXIT_SUCCESS);

		case 'v':
			++ verbosity;
			break;

		case '?':
			// getopt_long already printed an error message
			usage (progName);
			exit (EXIT_FAILURE);
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1)
	{
		usage (progName);
		exit (EXIT_FAILURE);
	}

	sublistFile = argv [0];

	// read input data
	CvMat *data = NULL;

	printf ("Parsing subject list and reading image data\n");
	fflush (stdout);

	ok = ((data = readData  (sublistFile)) != NULL);

	// create index
	CvMat *index = NULL;

	if (ok)
	{
		printf ("Generating index\n");
		fflush (stdout);

		ok = (index = createIndex (data)) != NULL;
	}

	// write index
	if (ok)
	{
		printf ("Writing index to file '%s'\n", indexFile);
		fflush (stdout);

		ok = writeCvMat (indexFile, index, "%.0f");
	}

	// clean up
	if (index)
	{
		cvReleaseMat (&index);
		index = NULL;
	}

	if (data)
	{
		cvReleaseMat (&data);
		data = NULL;
	}

	exit (ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
