/*!
 * \file   SbiaOdvbaUtilities.h
 * \brief  Common utility functions used by the ODVBA implementations.
 * \author Tianhao Zhang, revised by Andreas Schuh
 * \date   12/9/2010
 *
 * $Revision: 173 $
 * $Id: SbiaOdvbaUtilities.h 173 2011-02-15 14:38:34Z schuha@UPHS.PENNHEALTH.PRV $
 *
 * <b>Last changed:</b>
 * $Author: schuha@UPHS.PENNHEALTH.PRV $
 * $Date: 2011-02-15 09:38:34 -0500 (Tue, 15 Feb 2011) $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For copyright information see Copyright section of project
 * ReadMe.html in the project's root directory.
 *
 * Contact: sbia-software@uphs.upenn.edu
 */

#pragma once
#ifndef SbiaOdvbaUtilities_h
#define SbiaOdvbaUtilities_h


#include <cv.h>
#include <nifti1_io.h>

#include "SbiaOdvbaConfig.h" // Keep as last include statement
                             // (document reason(s) for exceptions)!


SBIA_ODVBA_NAMESPACE_BEGIN


//////////////////////////////////////////////////////////////////////////////
// IO
//////////////////////////////////////////////////////////////////////////////

// ---------------------------------------------------------------------------
// read input datasets
// ---------------------------------------------------------------------------

/*!
 * \brief Read image data of subjects.
 *
 * Read data according to the subject list in which the following information
 * is given: number of subjects in each group, directory of the data,
 * the name of each subject.
 *
 * \param [in]  filename Path of to subject list file.
 * \param [out] n1       Number of subjects belonging to group 1 (optional).
 * \param [out] hdr1     Header of first NIfTI-1 input image (optional).
 *
 * \return Read data of all subjects, where the data of each subject is
 *         stored in a single column of the matrix.
 *         The data has to be released by the caller using cvReleaseMat().
 */
CvMat *readData (const char     *filename,
                 int            *n1   = NULL,
                 nifti_1_header *hdr1 = NULL);

// ---------------------------------------------------------------------------
// read/write OpenCV matrix
// ---------------------------------------------------------------------------

/*!
 * \brief Read OpenCV matrix from file.
 *
 * \param [in] filename Path to file.
 * \param [in] hdr      Whether the file has a header containing matrix dimensions.
 *
 * \return Read OpenCV matrix or NULL if an error occurred.
 *         The returned matrix has to be released using cvReleaseMat ().
 */
CvMat *readCvMat (const char *filename, bool hdr = false);

/*!
 * \brief Write OpenCV matrix to file.
 *
 * \param [in] filename Output filename.
 * \param [in] matrix   Matrix.
 * \param [in] fmt      Format string used with fprintf () to output single element.
 * \param [in] hdr      Whether the matrix dimension should be saved in a header.
 *
 * \return Whether the output file was written successfully.
 */
bool writeCvMat (const char *filename, const CvMat *matrix, const char *fmt = "%e", bool hdr = false);

/*!
 * \brief Write OpenCV matrix as raw image.
 *
 * \param [in] filename Output filename.
 * \param [in] image    Image matrix. Can be either a 1D or 2D matrix.
 *
 * \return Whether the output file was written successfully.
 */
bool writeRawImage (const char *filename, const CvMat *image);

/*!
 * \brief Write OpenCV matrix as NIfTI-1 image.
 *
 * \param [in] filename Output filename (prefix).
 * \param [in] hdr      NIfTI-1 header.
 * \param [in] image    Image matrix. Can be either a 1D or 2D matrix.
 *                      The values of the matrix elements are written consecutively
 *                      do the output image, where the dimensions of the output
 *                      image are specified by the NIfTI-1 header. Hence,
 *                      the number of rows and columns of the input matrix are
 *                      ignored.
 *
 * \return Whether the output image was written successfully.
 */
bool writeNiftiImage (const char *filename, const nifti_1_header &hdr, const CvMat *image);

//////////////////////////////////////////////////////////////////////////////
// initialization
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Create index, i.e., the union of indices of non-zero voxels of the
 *        data of all subjects.
 *
 * Only voxels whose index is listed in the index data structure are considered
 * during the voxel-based analysis. The index can be considered as template
 * mask of the input volumes.
 *
 * \param [in] data Volume data of all subjects, where the each column
 *                  corresponds to one subject.
 *
 * \return Index data structure or NULL on error (i.e., memory allocation failed).
 *         The returned matrix has to be released by caller using cvReleaseMat ().
 */
CvMat *createIndex (const CvMat *data);

/*!
 * \brief Create neighborhoods.
 *
 * This function first selects randomly numNI voxels of all non-zero voxels.
 * Then, numVox voxels are randomly drawn from the voxels which lie within a
 * neighborhood of size (2 * sizNIx + 1) * (2 * sizNIy + 1) * (2 * sizNIz + 1)
 * centered at each of the voxels selected. If the random selection contained
 * invalid voxels, i.e., ones with voxel value zero, the corresponding entry
 * in the output matrix is set to the invalid index zero.
 *
 * By selecting random neighborhoods, the number of the involved learning
 * sets can be reduced and therefore improve the performance of the
 * group analysis.
 *
 * \param [in] index  The index of non-zero voxels.
 * \param [in] dimx   The x dimension of the image.
 * \param [in] dimy   The y dimension of the image.
 * \param [in] dimz   The z dimension of the image.
 * \param [in] sizNIx Radius of each neighborhood along x dimension (excluding center voxel).
 * \param [in] sizNIy Radius of each neighborhood along y dimension (excluding center voxel).
 * \param [in] sizNIz Radius of each neighborhood along z dimension (excluding center voxel).
 * \param [in] numNI  Number of neighborhoods.
 * \param [in] numVox Number of voxels randomly selected from each neighborhood.
 *
 * \return Neighborhood data structure or NULL on error (e.g., memory allocation failed).
 *         The returned matrix has to be released by caller using cvReleaseMat ().
 */
CvMat *createNI (const CvMat *index,
                 const int dimx,   const int dimy,   const int dimz,
				 const int sizNIx, const int sizNIy, const int sizNIz,
                 const int numNI,  const int numVox);

/*!
 * \brief Create data matrix X from original input and index.
 *
 * \see readData()
 * \see createIndex()
 * \see readIndex()
 *
 * \param [in] data  Input data of all subjects.
 * \param [in] index Index, see createIndex().
 *
 * \return Created data matrix X or NULL on failure.
 *         The returned memory has to be released using cvReleaseMat ().
 */
CvMat *createX (CvMat *data, CvMat *index);

//////////////////////////////////////////////////////////////////////////////
// random numbers / permutations
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Generates a vector whose elments are random integers.
 *
 * \param [in] n    Number of random integers.
 * \param [in] seed Seed for random number generator. If a negative value is
 *                  given as argument, the current timer value is used instead.
 *
 * \return Array of n random integers.
 *         The returned memory has to be released by the caller using delete [].
 */
int *random (int n, int seed = -1);

/*!
 * \brief Generates a random permutation.
 *
 * \param [in] n    Upper bound of range (excluded).
 * \param [in] seed Seed for random number generator. If a negative value is
 *                  given as argument, the current timer value is used instead.
 *
 * \return Permutation of integers from 1 to n.
 *         The returned memory has to be released by the caller using delete [].
 */
int *randperm (int n, int seed = -1);


SBIA_ODVBA_NAMESPACE_END


#endif // SbiaOdvbaUtilities_h
