/*!
 * \file  CreateNI.cpp
 * \brief Command line tool which creates the neighborhood file.
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */

#include <getopt.h>

#include "SbiaOdvbaUtilities.h"

#include "MainAux.h"


using namespace sbia::odvba;


//////////////////////////////////////////////////////////////////////////////
// usage / help
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Prints usage information / help.
 *
 * \param progName Name of program.
 */
void usage (const char *progName)
{
	version (progName);
	printf ("\n");
	printf ("Description:\n");
	printf ("  This program randomly selects neighorhoods from all the ones\n");
	printf ("  in order to reduce the computational cost of the ODVBA method.\n");
	printf ("  The selected neighborhoods are used to form the learning sets on\n");
	printf ("  which the discrimative coefficients are calulated.\n");
	printf ("\n");
	printf ("  T. Zhang and C. Davatzikos; ODVBA: Optimally-Dicriminative\n");
	printf ("  Voxel-Based Analysis\n");
	printf ("\n");
	printf ("Usage:\n");
	printf ("  %s [options] <subjects.txt> [<index.txt>]\n", progName);
	printf ("\n");
	printf ("Required options:\n");
	printf ("  <subjects.txt>      : The subjects list which specifies the input data given as NIfTI-1 images.\n");
	printf ("  [-s --sizeNI <real>]: Size of neighborhood in mm.\n");
	printf ("                        Required and used only if <NI.txt> file not provided.\n");
	printf ("  [-n --numNI <int>]  : Number of neighborhoods.\n");
	printf ("                        At most, the neighborhood around each non-zero voxel can be\n");
	printf ("                        considered, i.e., numNI <= m, where m is the number of non-zero voxels.\n");
	printf ("  [-e --numVox <int>] : Number of voxel in each neighborhood.\n");
	printf ("\n");
	printf ("Options:\n");
	printf ("  <index.txt>           : Location of non-zero voxel in the volumes.\n");
	printf ("  [-o --out <filename>] : Filename of the neighborhood file.\n");
	printf ("                          The default value is 'NI.txt'.\n");
	printf ("  [-h --help]           : Print help and exit.\n");
	printf ("  [-u --usage]          : Print usage information and exit.\n");
	printf ("  [-V --version]        : Print version information and exit.\n");
	printf ("  [-v --verbose]        : Enable verbose messages. Can be specified multiple times\n");
	printf ("                          to increase the verbosity.\n");
	printf ("\n");
	printf ("Example:\n");
	printf (" %s subjects.txt index.txt -n 10000 -e 400 -s 15\n", progName);
}

//////////////////////////////////////////////////////////////////////////////
// main
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Main function of program.
 *
 * \param [in] argc Number of command line arguments.
 * \param [in] argv Command line arguments.
 *
 * \return Exit status of program.
 *
 * \retval EXIT_SUCCESS on success.
 * \retval EXIT_FAILURE on failure.
 */
int main (int argc, char **argv)
{
	const char *progName = getProgName (argv);
	bool        ok       = true;

	static struct option long_options [] =
	{
		{"out",     required_argument, NULL, 'o'},
		{"numNI",   required_argument, NULL, 'n'},
		{"numVox",  required_argument, NULL, 'e'},
		{"sizeNI",  required_argument, NULL, 's'},
		{"usage",   no_argument,       NULL, 'u'},
		{"help",    no_argument,       NULL, 'h'},
		{"version", no_argument,       NULL, 'V'},
		{"Version", no_argument,       NULL, 'V'},
		{"verbose", no_argument,       NULL, 'v'},
        {0, 0, 0, 0}
	}; // struct long_options

	int c      = -1;
	int optidx = 0;

	// default options
	int         verbosity   = 0;        // verbosity of messages
	const char *sublistFile = NULL;     // input subjects list
	const char *indexFile   = NULL;     // input index file
	const char *NIFile      = "NI.txt"; // output neighborhood file
	int         numNI       = 0;        // number of neighborhoods
	double      sizNI       = 0;        // size of each neighborhood in mm
	int         numVox      = 0;        // number of voxels sampled from each neighborhood

	while ((c = getopt_long (argc, argv, "n:e:s:o:uhVv", long_options, &optidx)) != -1)
	{
		switch (c)
		{
		case 'o':
			NIFile = optarg;
			break;

		case 'n':
			numNI = atoi (optarg);
			break;

		case 'e':
			numVox = atoi (optarg);
			break;

		case 's':
			sizNI = atof (optarg);
			break;

		case 'u':
			// fall-through intended
		case 'h':
			usage (progName);
			exit (EXIT_SUCCESS);

		case 'V':
			version (progName);
			exit (EXIT_SUCCESS);

		case 'v':
			++ verbosity;
			break;

		case '?':
			// getopt_long already printed an error message
			usage (progName);
			exit (EXIT_FAILURE);
		}
	}


	argc -= optind;
	argv += optind;

	if (numNI == 0 || numVox == 0 || sizNI <= 0 || argc < 1 || argc > 2)
	{
		usage (progName);
		exit (EXIT_FAILURE);
	}

	sublistFile = argv [0];
	if (argc > 1 ) indexFile = argv [1];

	// read database
	CvMat         *data  = NULL; // image data of the n subjects
	CvMat         *index = NULL; // indices of "non-zero" voxels (template)
	int            n1    = 0;    // number of subjects belonging to group 1
	nifti_1_header hdr;          // image header of first input image

	printf ("Parsing subject list and reading image data\n");
	fflush (stdout);

	ok = ((data = readData  (sublistFile, &n1, &hdr)) != NULL);

	if (ok)
	{
		if (indexFile)
		{
			printf ("Reading index\n");
			fflush (stdout);

			ok = ((index = readCvMat (indexFile)) != NULL);
		}
		else
		{
			printf ("Generating index\n");
			fflush (stdout);

			ok = ((index = createIndex (data)) != NULL);
		}
	}

	// generating neighborhoods
	CvMat *NI = NULL;

	if (ok)
	{
		printf ("Generating neighborhoods\n");
		fflush (stdout);

		// note: one half voxel is subtracted because createNI accounts for it
		int sizNIx = static_cast <int> (round (sizNI / hdr.pixdim [1] - 0.5)); // size of neighborhoods along x dimension
		int sizNIy = static_cast <int> (round (sizNI / hdr.pixdim [2] - 0.5)); // size of neighborhoods along y dimension
		int sizNIz = static_cast <int> (round (sizNI / hdr.pixdim [3] - 0.5)); // size of neighborhoods along z dimension

		int xyz_units = XYZT_TO_SPACE (hdr.xyzt_units);

		if (xyz_units == NIFTI_UNITS_MICRON)
		{
			sizNIx /= 1000;
			sizNIy /= 1000;
			sizNIz /= 1000;
		}
		else if (xyz_units == NIFTI_UNITS_METER)
		{
			sizNIx *= 1000;
			sizNIy *= 1000;
			sizNIz *= 1000;
		}
		
		if (sizNIx <= 0) sizNIx = 1;
		if (sizNIy <= 0) sizNIy = 1;
		if (sizNIz <= 0) sizNIz = 1;

		ok = ((NI = createNI (index, hdr.dim[1], hdr.dim[2], hdr.dim[3], sizNIx, sizNIy, sizNIz, numNI, numVox)) != NULL);
	}

	// write index
	if (ok)
	{
		printf ("Writing neighborhood(s) to file '%s'\n", NIFile);
		fflush (stdout);

		int fieldWidth = 0;
		int maxValue   = index->rows;

		while (maxValue > 0)
		{
			++ fieldWidth;
			maxValue = maxValue / 10;
		}

		char fmt [24];		
		sprintf (fmt, "%%%d.0f", fieldWidth);

		ok = writeCvMat (NIFile, NI, fmt);
	}

	// clean up
	if (NI)
	{
		cvReleaseMat (&NI);
		NI = NULL;
	}

	if (index)
	{
		cvReleaseMat (&index);
		index = NULL;
	}

	if (data)
	{
		cvReleaseMat (&data);
		data = NULL;
	}

	exit (ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
