#! /bin/bash

##############################################################################
# \file  OdvbaMPIOnSGE.sh
# \brief Bash script used to submit SGE job for parallel execution of OdvbaMPI.
#
# The variable NSLOTS used by this script is set by SGE to match the number
# of slots requested using the -pe option. A specification of the hosts
# is not required. For testing purposes, however, set the variable NSLOTS to
# the number of slots and the variable MYMACHINES to the complete option needed
# to specify the machines for mpiexec manually prior to executing this script,
# as in (BASH syntax)
#
#  MYMACHINES="-machinefile ~/mymachines" NSLOTS=42 OdvbaMPIOnSGE.sh ...
#
# For copyright information please see Copyright.txt in the root
# directory of the project.
#
# Contact: SBIA Group <sbia-software@uphs.upenn.edu>
##############################################################################

##############################################################################
# SGE
##############################################################################

# SGE will read options that are treated by the shell as comments. The SGE
# parameters must begin with the characters "#$", followed by the option.
# An option given as "##$" is hence disabled.
#
# Environment:
#$ -S /bin/bash # shell used under SGE
#$ -cwd         # run command in current working directory
#
# Save the output:
##$ -o OdvbaMPI.$JOB_ID.log # stdout output file (specify on command line)
##$ -e OdvbaMPI.$JOB_ID.log # stderr output file (specify on command line)
#
# Email notifications:
##$ -M email@uphs.upenn.edu # Email address      (specify on command line)
##$ -m b # Send mail at the beginning of the job (specify on command line)
##$ -m e # Send mail at the end of the job       (specify on command line)
##$ -m a # Send mail in case the job is aborted  (specify on command line)
#
# Parallel environment:
#$ -pe openmpi 6 # Default number of CPUs requested. Give this option with a
#                # different number on the command line in order to overwrite
#                # the default.

##############################################################################
# variables
##############################################################################

progName=${0##*/} # name of this script

version_major=@PROJECT_VERSION_MAJOR@  # major version number
version_minor=@PROJECT_VERSION_MINOR@  # minor version number
version_patch=@PROJECT_VERSION_PATCH@  # version patch number

# version string
version="$version_major.$version_minor.$version_patch"

verbosity=0                           # verbosity of output messages
bin="@PROJECT_BIN_DIR@"               # executable output path (build or install tree)
prefix="@PROJECT_OUTPUT_NAME_PREFIX@" # prefix used for executable output names
odvba="${prefix}OdvbaMPI"             # name of ODVBA exectuable
args=""                               # arguments for ODVBA executable

##############################################################################
# functions
##############################################################################

# ////////////////////////////////////////////////////////////////////////////
# version / usage / help
# ////////////////////////////////////////////////////////////////////////////

# \brief Prints version information
version ()
{
	echo "$progName $version [using `$bin/$odvba --version`]"
}

# \brief Prints usage information
usage ()
{
	version
	echo
	echo "Description:"
	echo "  Script used to run $odvba as parallel job on SGE."
	echo "  For a description of $odvba, see below."
	echo
	echo "Usage:"
	echo "  qsub [qsub options] $progName [options of $odvba]"
	echo
	echo "Common qsub options:"
	echo "  [-pe openmpi <int>]    : Specify number of CPUs to be requested."
	echo "  [-M me@uphs.upenn.edu] : Email address for email notifications."
	echo "                           Enable also notifications on begin, end, and abortion"
    echo "                           by specifying the options -b, -e, and -a, respectively."
	echo
	echo "Example:"
	echo "  qsub -pe openmpi 10 $progName [options of $odvba]"
	echo
	echo
	$bin/$odvba -u
}

# \brief Prints help.
help ()
{
	usage
}

# ////////////////////////////////////////////////////////////////////////////
# options
# ////////////////////////////////////////////////////////////////////////////

options ()
{
	for arg in $@
	do
		case "$arg" in
			-u|--usage)
				usage
				exit 0
				;;
			-h|--help)
				help
				exit 0
				;;
			-V|--version)
				version
				exit 0
				;;
		esac
	done
}

##############################################################################
# main
##############################################################################

# ----------------------------------------------------------------------------
# load required modules
# ----------------------------------------------------------------------------

. /usr/share/Modules/init/bash

if [ $? != 0 ]
then
	echo "Failure to load /usr/share/Modules/init/bash"
	exit 1
fi

module load openmpi/1.3.3

if [ $? != 0 ]
then
	echo "Failed to load module openmpi/1.3.3"
	exit 1
fi

# ----------------------------------------------------------------------------
# parse options
# ----------------------------------------------------------------------------

# parse options to overwrite usage, help, and version information
# \note This has to be done after the Open MPI module was loaded!
options $*

# ----------------------------------------------------------------------------
# default output to enable identification of job in output files
# ----------------------------------------------------------------------------

echo "Command: $progName"
echo "Arguments: $*"
echo -e "Executing in: \c"; pwd
echo -e "Executing on: \c"; hostname
echo -e "Executed at:  \c"; date
echo "----- STDOUT from $progName below this line -----"

echo "Command: $progName" 1>&2
echo "Arguments: $*" 1>&2
(echo -e "Executing in: \c"; pwd) 1>&2
(echo -e "Executing on: \c"; hostname) 1>&2
(echo -e "Executed at:  \c"; date) 1>&2
echo "----- STDERR from $progName below this line -----" 1>&2

# ----------------------------------------------------------------------------
# run ODVBA
# ----------------------------------------------------------------------------

# \note $OPENMPI is set by the "openmpi/1.3.3" module
# \note $NSLOTS is set by SGE to match the number of slots requested
# \note $MYMACHINES is only used for script testing purposes
# \note The option "--mca orte_forward_job_control 1" enables jobs to be
#       suspended and restarted by SGE depending on the load on the server.
echo "Executing $bin/$odvba"
mpiexec --prefix $OPENMPI -x PATH="$PATH" -x LD_LIBRARY_PATH="$LD_LIBRARY_PATH" -np $NSLOTS $MYMACHINES --mca orte_forward_job_control 1 $bin/$odvba $*
