/**
 * @file  utilities.cxx
 * @brief Common utility functions used by the ODVBA implementations.
 *
 * Copyright (c) 2010-2012 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */

#include <fstream>
#include <algorithm> // sort()
#include <string>

#include <sbia/basis/path.h>

#include "utilities.h"


// acceptable in .cxx file
using namespace std;
using namespace sbia::basis;


namespace sbia
{

namespace odvba
{


// ===========================================================================
// IO
// ===========================================================================

// ---------------------------------------------------------------------------
// read input datasets
// ---------------------------------------------------------------------------

/**
 * @brief Validate NIfTI image attributes.
 *
 * @see read_data()
 *
 * @param [in] nim Input NIfTI image.
 * @param [in] ref Reference NIfTI image, e.g., first input image.
 *
 * @returns Whether or not the input image nim is valid.
 */
bool check_image(const nifti_image* const nim, const nifti_image* const ref)
{
    if (!nim) return false;

    // check image attributes independent of reference image
    //
    // Note: Example images generated by resampleImage.py at SBIA had
    //       ndim == 3 even though they should have nim->ndim == 4?
    //       Anyways, the check of nx, ny, nz, and nt should be sufficient.
    if (/*nim->ndim != 4 ||*/nim->nx <= 1 || nim->ny <= 1 || nim->nz <= 1 || nim->nt != 1) return false;

    if (nim->datatype != DT_FLOAT && nim->datatype != DT_SIGNED_SHORT) return false;

    // no reference image given, so image is ok
    if (!ref) return true;

    // compare to reference image
    if (nim->nx != ref->nx || nim->ny != ref->ny || nim->nz != ref->nz) return false;

    // TODO Add more checks, e.g., orientation, pixel dimensions...

    return true;
}

// ---------------------------------------------------------------------------
CvMat* read_data(const char* filename, int* n1, nifti_1_header* hdr1)
{
    char str[1024]; // temporary memory to read strings from file

    // open subject list file
    FILE* fp_list = fopen(filename, "r");

    if (!fp_list) {
        fprintf(stderr, "Failed to open subject list '%s'\n", filename);
        return NULL;
    }

    // parse header
    int    nSubjects1 = 0; // number of subjects in group 1
    int    nSubjects2 = 0; // number of subjects in group 2
    string data_dir;       // default data root directory

    if (fscanf(fp_list, "%d %d", &nSubjects1, &nSubjects2) != 2 || nSubjects1 < 0 || nSubjects2 < 0) {
        fprintf(stderr, "Failed to parse subject list '%s': number of subjects\n", filename);
        return NULL;
    }

    if (fscanf(fp_list, "%s", &str) != 1) {
        fprintf(stderr, "Failed to parse subject list '%s': data root\n", filename);
        return NULL;
    }

    data_dir = to_absolute_path(get_file_directory(filename).c_str(), str);

    // process input images
    string       data_file;  // path to data file (NIfTI-1 header file)
    nifti_image* nim1;       // image attributes of first input image
                             // used to check whether other input images
                             // have identical attributes

    // read header information of first input image
    if (fscanf(fp_list, "%s", &str) != 1) {
        fprintf(stderr, "Failed to parse subject list '%s': image filename\n", filename);
        return NULL;
    }

    data_file = to_absolute_path(data_dir.c_str(), str);
    nim1     = nifti_image_read(data_file.c_str(), 0);

    if (!nim1) {
        fprintf(stderr, "Failed to read NIfTI-1 header of file '%s'\n", data_file.c_str());
        return NULL;
    }

    if (!check_image (nim1, NULL)) {
        fprintf(stderr, "Invalid or unsupported NIfTI-1 image file '%s'. Only DT_SIGNED_SHORT and DT_FLOAT supported.\n", data_file.c_str());
        nifti_image_free(nim1);
        return NULL;
    }

    // allocate matrix for output data
    CvMat* data = cvCreateMat(nim1->nx * nim1->ny * nim1->nz, nSubjects1 + nSubjects2, CV_32FC1);

    if (!data) {
        fprintf(stderr, "Failed to allocate memory\n");
        nifti_image_free(nim1);
        return NULL;
    }

    // read data of each subject
    bool         ok  = true; // whether everything is fine yet
    nifti_image* nim = NULL; // current input NIfTI image

    for (int i = 0; ok && i < data->cols; i++) {
        // header of first input image was already read
        if (i == 0) {
            nim = nim1;
        // read image header
        } else {
            if (fscanf(fp_list, "%s", &str) != 1) {
                fprintf(stderr, "Failed to parse subject list '%s': image filename\n", filename);
                ok = false;
                break;
            }

            data_file = data_dir + '/' + str;

            // read NIfTI-1 image header
            nim = nifti_image_read(data_file.c_str(), 0);
    
            if (!nim) {
                fprintf(stderr, "Failed to open image file '%s'\n", data_file.c_str());
                ok = false;
                continue;
            }

            // verify image attributes
            if (!check_image(nim, nim1)) {
                fprintf (stderr, "Attributes of image '%s' differ from attributes of first image\n", data_file.c_str());
                ok = false;
                continue;
            }
        }

        // read image data (optionally compressed)
        znzFile fp_data = nifti_image_open(data_file.c_str(), "rb", &nim);

        if (znz_isnull(fp_data)) {
            fprintf(stderr, "Failed to open data file of image '%s'\n", data_file.c_str());
            ok = false;
        }

        // get image offset: a negative offset means to figure from end of file
        int ioff = 0;

        if (ok) {
            if (nim->iname_offset < 0) {
                if (nifti_is_gzfile(nim->iname)) {
                    fprintf(stderr, "Invalid offset for compressed data file of image '%s'\n", data_file.c_str());
                    ok = false;
                } else {
                    int ii = nifti_get_filesize(nim->iname);

                    if (ii <= 0) {
                        fprintf(stderr, "Data file of image '%s' is empty\n", data_file.c_str());
                        ok = false;
                    } else {
                        int ntot = static_cast<int>(nifti_get_volsize(nim));
                        ioff = ((ii > ntot) ? (ii - ntot) : 0);
                    }
                }
            } else {
                ioff = nim->iname_offset;
            }
        }

        // seek to the appropriate read position
        if (ok && znzseek(fp_data, static_cast<long>(ioff), SEEK_SET) < 0) {
            fprintf(stderr, "Failed to seek to offset %u of data file of image '%s'\n",
                    static_cast<unsigned int>(ioff), data_file.c_str());
            ok = false;
        }

        // read image data
        if (nim->datatype == DT_SIGNED_SHORT) {
            short tmp;

            for (int k = 0; ok && k < data->rows; k++) {
                if (nifti_read_buffer(fp_data, &tmp, sizeof(short), nim) == sizeof(short)) {
                    cvSetReal2D(data, k, i, static_cast<double>(tmp));
                } else {
                    fprintf (stderr, "Failed to read data of image '%s'\n", data_file.c_str());
                    ok = false;
                }
            }
        } else if (nim->datatype == DT_FLOAT) {
            float tmp;

            for (int k = 0; ok && k < data->rows; k++) {
                if (nifti_read_buffer(fp_data, &tmp, sizeof(float), nim) == sizeof(float)) {
                    cvSetReal2D(data, k, i, static_cast<double>(tmp));
                } else {
                    fprintf(stderr, "Failed to read data of image '%s'\n", data_file.c_str());
                    ok = false;
                }
            }
        } else {
            fprintf(stderr, "Image '%s' has unsupported voxel type. Only DT_SIGNED_SHORT and DT_FLOAT are supported.\n", data_file.c_str());
            ok = false;
        }

        // clean up
        if (!znz_isnull(fp_data)) znzclose(fp_data);

        if (nim && nim != nim1) {
            nifti_image_free(nim);
            nim = NULL;
        }
    }

    fclose(fp_list);

    // normalize data
    if (ok) norm_data(data, 100);

    // other return values
    if (n1)   *n1   = nSubjects1;
    if (hdr1) *hdr1 = nifti_convert_nim2nhdr(nim1);

    // clean up
    nifti_image_free(nim1);

    if (!ok) {
        cvReleaseMat(&data);
        data = NULL;
    }

    return data;
}

// ---------------------------------------------------------------------------
void norm_data(CvMat* data, const int meanValue)
{
    // calculate normalization factor
    double norm = meanValue / cvAvg(data).val[0];

    // normalize data in-place
    for (int i = 0; i < data->rows ; i++) {
        for (int j = 0; j < data->cols; j++) {
            cvSetReal2D(data, i, j, norm * cvGetReal2D(data, i, j));
        }
    }
}

// ---------------------------------------------------------------------------
// read/write OpenCV matrix
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
CvMat* read_matrix(const char* filename, bool hdr)
{
    bool ok   = true;
    int  rows = 0;
    int  cols = 0;

    // open file
    FILE* fp = fopen(filename, "r");

    if (fp == NULL) {
        fprintf(stderr, "Failed to open file '%s'\n", filename);
        return NULL;
    }

    // parse header to extract matrix dimensions (if applicable)
    if (hdr) {
        if (fscanf(fp, "%d %d", &rows, &cols) != 2 || rows < 0 || cols < 0) {
            fprintf(stderr, "Failed to parse header of file '%s': matrix dimensions\n", filename);
            ok = false;
        }
    // otherwise scan file first to determine matrix dimensions
    } else {
        int  c;
        int  cnt  = 0;
        bool empty = true; // flag used to detect empty lines 
        bool skip  = true; // flag used to skip whitespaces
                           // (init with true to allow whitespaces before first entry)

        while (ok && (c = fgetc(fp)) != EOF) {
            if (c == '\n') {
                if (empty) {
                    // skip empty line
                } else {
                    rows++;
                    cnt++;

                    if (rows > 1) {
                        if (cnt != cols) {
                            fprintf(stderr, "Number of columns in row %d differs from first row\n", rows);
                            ok = false;
                        }
                    } else {
                        cols = cnt;
                    }

                    cnt   = 0;
                    empty = true;
                    skip  = true;
                }
            } else if (c == '\r') {
                // ignore it, '\n' will be next...
            } else if (c == ' ' || c == '\t') {
                if (!skip) {
                    cnt++;
                    skip = true;
                }
            } else {
                empty = false;
                skip  = false;
            }
        }

        if (ferror(fp) || rows == 0 || cols == 0) {
            fprintf(stderr, "Invalid input file '%s'\n", filename);
            ok = false;
        }

        fclose(fp);
        fp = NULL;
    }

    // allocate matrix
    CvMat* matrix = NULL;
    
    if (ok) {
        if ((matrix = cvCreateMat(rows, cols, CV_32FC1)) == NULL) {
            fprintf(stderr, "Failed to allocate memory\n");
            ok = false;
        }
    }

    // re-open file (if necessary)
    if (ok && fp == NULL) {
        ok = ((fp = fopen(filename, "r")) != NULL);
    }

    // read matrix elements
    float tmp;

    for (int i = 0; ok && i < rows; i++) {
        for (int j = 0; ok && j < cols; j++) {
            if (fscanf(fp, "%f", &tmp) == 1) {
                cvSetReal2D(matrix, i, j, static_cast<double>(tmp));
            } else {
                ok = false;
            }
        }
    }

    if (!ok) {
        fprintf(stderr, "Failed to read data from file '%s'\n", filename);
        cvReleaseMat(&matrix);
        matrix = NULL;
    }

    // close file
    if (fp) {
        fclose(fp);
        fp = NULL;
    }

    return matrix;
}

// ---------------------------------------------------------------------------
bool write_matrix(const char* filename, const CvMat* matrix, const char* fmt, bool hdr)
{
    // open file
    FILE* fp = fopen(filename, "w");

    if (fp == NULL) {
        fprintf(stderr, "Failed to open file '%s' for writing\n", filename);
        return false;
    }

    // write header
    if (hdr) fprintf(fp, "%d %d\n", matrix->rows, matrix->cols);

    // write matrix elements
    for (int i = 0; i < matrix->rows; i++) {
        for (int j = 0; j < matrix->cols; j++) {
            if (j > 0) fprintf(fp, " ");
            fprintf(fp, fmt, cvGetReal2D(matrix, i, j));
        }
        fprintf(fp, "\n");
    }

    // close file
    fclose(fp);

    return true;
}

// ---------------------------------------------------------------------------
// write OpenCV matrix as image
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
bool write_raw_image(const char* filename, const CvMat* image)
{
    // open file
    FILE *fp = fopen(filename, "wb");

    if (fp == NULL) {
        fprintf(stderr, "Failed to open file '%s'\n", filename);
        return false;
    }

    // write matrix elements as raw image data
    float tmp;

    if (image->cols > 0) {
        for (int j = 0; j < image->cols; j++) {
            for (int i = 0; i < image->rows; i++) {
                tmp = static_cast<float>(cvGetReal2D(image, i, j));
                fwrite (&tmp, sizeof(float), 1, fp);
            }
        }
    } else {
        for (int i = 0; i < image->rows; i++) {
            tmp = static_cast<float>(cvGetReal1D(image, i));
            fwrite(&tmp, sizeof(float), 1, fp);
        }
    }

    fclose(fp);
    return true;
}

// ---------------------------------------------------------------------------
bool write_nifti_image(const char* filename, const nifti_1_header& hdrIn, const CvMat* image)
{
    const int dim[8] = {4, hdrIn.dim[1], hdrIn.dim[2], hdrIn.dim[3], 1, 0, 0, 0};

    // check consistency of matrix dimensions and image dimensions
    if (dim[1] * dim[2] * dim[3] < image->rows * image->cols) {
        fprintf(stderr, "Invalid output image dimensions\n");
        return false;
    }

    // create new image header
    nifti_1_header* hdr = nifti_make_new_header(dim, NIFTI_TYPE_FLOAT32);

    if (!hdr) {
        fprintf(stderr, "Failed to initialize image header\n");
        return false;
    }

    // default header values
    // (write hdr/img files by default; may be overwritten according to prefix filename extension)
    strcpy(hdr->magic, "ni1");

    // copy selected header elements
    hdr->dim_info    = hdrIn.dim_info;
    hdr->intent_p1   = hdrIn.intent_p1;
    hdr->intent_p2   = hdrIn.intent_p2;
    hdr->intent_p3   = hdrIn.intent_p3;
    hdr->intent_code = hdrIn.intent_code;
    hdr->pixdim[1]   = hdrIn.pixdim[1];
    hdr->pixdim[2]   = hdrIn.pixdim[2];
    hdr->pixdim[3]   = hdrIn.pixdim[3];
    hdr->xyzt_units  = hdrIn.xyzt_units;
    hdr->cal_min     = 0;
    hdr->cal_max     = 1;
    memcpy(hdr->descrip, hdrIn.descrip, 80);
    hdr->qform_code  = hdrIn.qform_code;
    hdr->sform_code  = hdrIn.sform_code;
    hdr->quatern_b   = hdrIn.quatern_b;
    hdr->quatern_c   = hdrIn.quatern_c;
    hdr->quatern_d   = hdrIn.quatern_d;
    hdr->qoffset_x   = hdrIn.qoffset_x;
    hdr->qoffset_y   = hdrIn.qoffset_y;
    hdr->qoffset_z   = hdrIn.qoffset_z;
    memcpy(hdr->srow_x, hdrIn.srow_x, 4);
    memcpy(hdr->srow_y, hdrIn.srow_y, 4);
    memcpy(hdr->srow_z, hdrIn.srow_z, 4);
    memcpy(hdr->intent_name, hdrIn.intent_name, 16);

    // convert NIfTI-1 header to nifti_image structure
    nifti_image* nim = nifti_convert_nhdr2nim(*hdr, filename);

    free (hdr);
    hdr = NULL;

    if (!nim) {
        fprintf(stderr, "Failed to create NIfTI-1 image structure\n");
        return false;
    }

    // write image header
    znzFile fp_data = nifti_image_write_hdr_img(nim, 2, "wb");

    // write image data
    if (znz_isnull(fp_data)) {
        fprintf(stderr, "Failed to write image data\n");
        nifti_image_free(nim);
        return false;
    }

    float tmp; // temporary voxel data

    if (image->cols > 0) {
        for (int j = 0; j < image->cols; j++) {
            for (int i = 0; i < image->rows; i++) {
                tmp = static_cast<float>(cvGetReal2D(image, i, j));
                nifti_write_buffer(fp_data, &tmp, sizeof(float));
            }
        }
    } else {
        for (int i = 0; i < image->rows; i++) {
            tmp = static_cast<float>(cvGetReal1D(image, i));
            nifti_write_buffer(fp_data, &tmp, sizeof(float));
        }
    }

    nifti_image_free(nim);
    znzclose(fp_data);

    return true;
}

// ===========================================================================
// initialization
// ===========================================================================

// ---------------------------------------------------------------------------
CvMat* create_index(const CvMat* data)
{
    CvMat* index1 = cvCreateMat(data->rows, 1, CV_32FC1);

    if (!index1) return NULL;

    cvZero(index1);

    int count = 0;

    for (int i = 0; i < data->rows; i++) {
        double me = 0;
        for (int j = 0; j < data->cols; j++) {
            me = me + cvGetReal2D (data, i, j);
        }
        me = me / data->cols;

        if (me > 40) {
            cvSetReal1D(index1, i, static_cast<double>(i + 1));
            count++;
        }
    }

    if (count == 0) {
        fprintf(stderr, "No \"non-zero\" voxels found\n");
        return NULL;
    }

    int idx = 0;

    CvMat* index2 = cvCreateMat(count, 1, CV_32FC1);

    if (!index2) {
        cvReleaseMat(&index1);
        return NULL;
    }

    for (int i = 0; i < data->rows; i++) {
        double ele = cvGetReal1D(index1, i);

        if (ele > 0) {
            cvSetReal1D(index2, idx, ele);
            idx++;
        }
    }

    cvReleaseMat(&index1);
    index1 = NULL;

    return index2;
}

// ---------------------------------------------------------------------------
CvMat* create_ni(const CvMat* index,
                 const int dim_x, const int dim_y, const int dim_z,
                 const int ni_size_x, const int ni_size_y, const int ni_size_z,
                 const int ni_num, const int vox_num)
{
    const int m = index->rows; // number of non-zero voxels

    // check arguments
    if (ni_size_x < 1 || ni_size_y < 1 || ni_size_z < 1) {
        fprintf(stderr, "create_ni(): invalid argument for parameter 'ni_size_*'\n");
        return NULL;
    }

    if (ni_num < 1 || ni_num > m) {
        fprintf(stderr, "create_ni(): invalid argument for parameter 'ni_num' (0 < ni_num <= %d)\n", m);
        return NULL;
    }

    if (vox_num < 1 || vox_num > ((2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1))) {
        fprintf(stderr, "create_ni(): invalid argument for parameter 'vox_num' (0 < vox_num < %d)\n",
                (2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1));
        return NULL;
    }

    // allocate memory
    CvMat* NI       = cvCreateMat(vox_num, ni_num, CV_32FC1); // neighborhood(s)
    CvMat* samples  = cvCreateMat(m,       1,      CV_32FC1); // indices of centering voxels (only first ni_num used)
    CvMat* x_vector = cvCreateMat(m,       1,      CV_32FC1); // maps voxel index to x coordinate of voxel
    CvMat* y_vector = cvCreateMat(m,       1,      CV_32FC1); // maps voxel index to y coordinate of voxel
    CvMat* z_vector = cvCreateMat(m,       1,      CV_32FC1); // maps voxel index to z coordinate of voxel

    if (!NI || !samples || !x_vector || !y_vector || !z_vector) {
        fprintf(stderr, "Failed to allocated required memory\n");

        if (x_vector) cvReleaseMat(&x_vector);
        if (y_vector) cvReleaseMat(&y_vector);
        if (z_vector) cvReleaseMat(&z_vector);
        if (samples)  cvReleaseMat(&samples);
        if (NI)       cvReleaseMat(&NI);

        return NULL;
    }

    // create mapping from voxel indices to (x, y, z) coordinates
    // index [i] => (x_vector [i], y_vector [i], z_vector [i])
    for (int i = 0; i < m; i++) {
        double idx = cvGetReal1D(index, i);

        double z_temp = ceil((idx / dim_x) / dim_y);
        double y_temp = ceil((idx - (z_temp - 1) * dim_x * dim_y) / dim_x);
        double x_temp = idx - (z_temp - 1) * dim_x * dim_y - (y_temp - 1) * dim_x;

        cvSetReal1D(x_vector, i, x_temp);
        cvSetReal1D(y_vector, i, y_temp);  
        cvSetReal1D(z_vector, i, z_temp);
    }

    // randomly select ni_num voxels out of the m non-zero voxels
    {
        int* ra = randperm(m);

        for (int i = 0; i < ni_num; i++) {
            cvSetReal1D(samples, i, static_cast <double> (ra [i] - 1));
        }

        delete [] ra;
        ra = NULL;
    }

    // for each neighborhood, randomly select vox_num voxels
    int ni1[(2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1)];
    int ni2[vox_num];

    for (int i = 0; i < ni_num; ++ i) {
        // map index of neighborhood center to (x, y, z)
        int r  = static_cast<int>(cvGetReal1D(samples,  i));
        int xi = static_cast<int>(cvGetReal1D(x_vector, r));
        int yi = static_cast<int>(cvGetReal1D(y_vector, r));
        int zi = static_cast<int>(cvGetReal1D(z_vector, r));

        // determine boundaries of neighborhood
        int minx = xi - ni_size_x;
        if (minx < 1) minx = 1;

        int maxx = xi + ni_size_x;
        if (maxx > dim_x) maxx = dim_x;

        int miny = yi - ni_size_y;
        if (miny < 1) miny = 1;

        int maxy = yi + ni_size_y;
        if (maxy > dim_y) maxy = dim_y;

        int minz = zi - ni_size_z;
        if (minz < 1) minz = 1;

        int maxz = zi + ni_size_z;
        if (maxz > dim_z) maxz = dim_z;

        // determine indices of all neighboring non-zero voxels
        int current_num_ni = 0;

        for (int j = 0; j < m; j++) {
            int xj = static_cast<int>(cvGetReal1D(x_vector, j));
            int yj = static_cast<int>(cvGetReal1D(y_vector, j));
            int zj = static_cast<int>(cvGetReal1D(z_vector, j));

            if ((xj < maxx + 1) && (xj > minx - 1)
                    && (yj < maxy + 1) && (yj > miny - 1)
                    && (zj < maxz + 1) && (zj > minz - 1)) {
                ni1[current_num_ni] = j + 1;
                current_num_ni++;
            }
        }

        while (current_num_ni < vox_num) {
            ni1[current_num_ni] = 0; // note: zero indicates an invalid index
            current_num_ni++;
        }

        // randomly select vox_num voxels of these neighboring voxels
        {
            int* ra = randperm(current_num_ni);
            for (int ss = 0; ss < vox_num; ss++) {
                ni2[ss] = ni1[ra[ss] - 1];
            }

            delete [] ra;
            ra = NULL;
        }

        // set first entry to center of neighborhood
        int flag = 0;

        for (int tt = 0; tt < vox_num; tt++) {
            if (ni2[tt] == r + 1) {
                int tmp = ni2[0];
                ni2[0]  = ni2[tt];
                ni2[tt] = tmp;

                flag = 1;
                break;
            }
        }

        if (flag == 0) ni2[0] = r + 1;

        // copy neighborhood to column i of NI
        for (int ss = 0; ss < vox_num; ss++) {
            cvSetReal2D(NI, ss, i, ni2[ss]);
        }
    }

    // clean up
    cvReleaseMat(&x_vector);
    x_vector = NULL;

    cvReleaseMat(&y_vector);
    y_vector = NULL;

    cvReleaseMat(&z_vector);
    z_vector = NULL;

    cvReleaseMat(&samples);
    samples = NULL;

    return NI;
}

// ---------------------------------------------------------------------------
CvMat* create_x(CvMat* data, CvMat* index)
{
    const unsigned int n = data->cols;  // number of subjects
    const unsigned int m = index->rows; // number of voxels per subject (template)

    CvMat* X = NULL;

    try {
        X = cvCreateMat(m, n, CV_32FC1);
    } catch (...) {
        fprintf (stderr, "Failed to allocate memory for matrix X\n");
    }

    if (!X) return NULL;

    bool ok = true;
    int  idx;

    for (int i = 0; ok && i < index->rows; i++) {
        for (int j = 0; ok && j < data->cols; j++) {
            idx = static_cast<int>(cvGetReal1D(index, i)) - 1;
            if (0 <= idx && idx < data->rows) {
                cvSetReal2D(X, i, j, cvGetReal2D(data, idx, j));
            } else {
                ok = false;
            }   
        }
    }

    if (!ok) {
        fprintf(stderr, "Invalid index '%d'\n", idx);
        cvReleaseMat(&X);
        X = NULL;
    }

    return X;
}

// ===========================================================================
// random numbers / generation of permutations
// ===========================================================================

// ---------------------------------------------------------------------------
int* random(int n)
{
    int* d = new int[n];
    for (int i = 0; i < n; i++) d[i] = rand();
    return d;
}

// ---------------------------------------------------------------------------
int *randperm (int n)
{
    int* d     = random(n);
    int* rep_d = new int[n];
    for (int i = 0; i < n; i++) rep_d[i] = d[i];

    sort(d, d + n);

    int* ind_sort = new int[n];
    for (int i = 0; i < n; i++) {
        for (int j = 0; j < n; j++) {
            if (rep_d[i] == d[j]) ind_sort[j] = i + 1;
        }
    }

    delete [] d;
    delete [] rep_d;

    return ind_sort;
}


} // namespace odvba

} // namespace sbia
