/**
 * @file  odvba-ni.cxx
 * @brief Command line tool which creates the neighborhood file.
 *
 * Copyright (c) 2010-2012 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */

#include <iostream>

#include <odvba/basis.h>
#include <odvba/odvba.h>
#include <odvba/utilities.h>


// acceptable .cxx file
using namespace std;
using namespace basis;
using namespace odvba;


// ===========================================================================
// main
// ===========================================================================

// ---------------------------------------------------------------------------
int main(int argc, char* argv[])
{
    bool ok = true;

    // -----------------------------------------------------------------------
    // command-line argument constraints
    PositiveValueConstraint<double> positive_double_constraint("<double>");
    PositiveValueConstraint<int>    positive_int_constraint("<int>");
    ExistingFileConstraint          subjects_list_constraint("<subjects_list>");
    ExistingFileConstraint          existing_file_constraint;

    // -----------------------------------------------------------------------
    // command-line arguments
    PositionalArg sublist_file("subjects_list",
            "Text file which lists the NIfTI-1 image data files of the input subjects.",
            true, "", &subjects_list_constraint);

    StringArg index_file("", "index",
            "File generated by odvba-index command with information on the"
            " location of non-zero voxels in the volumes.",
            false, "", &existing_file_constraint);

    StringArg ni_file("", "ni",
            "Name of the output neighborhood indices file.",
            false, "ni.txt", "<file>");

    DoubleArg ni_size("s", "ni-size",
            "Size of neighborhood in mm. Required and used only if neighborhood"
            " indices file not provided.",
            true, 0, &positive_double_constraint);

    IntArg ni_num("n", "ni-num",
            "Number of neighborhoods. At most, the neighborhood around each"
            " non-zero voxel can be considered, i.e., n <= m, where m is the"
            " number of non-zero voxels.",
            true, 0, &positive_int_constraint);

    IntArg vox_num("e", "ni-voxels",
            "Number of voxels in each neighborhood.",
            true, 0, &positive_int_constraint);

    SwitchArg srand_time("", "srand-time",
            "Use the execution time to initialize the pseudo-random number"
            " generator. If this option is not given, a fixed seed is used to"
            " initialize it which ensures identical results whenever this"
            " program is run on the same machine with the same input images"
            " and parameter settings. Otherwise, the results may differ between"
            " executions.",
            false);

    // -----------------------------------------------------------------------
    // parse command-line
    try {
        CmdLine cmd(// program identification
                    "odvba-ni", PROJECT,
                    // program description
                    "This program randomly selects neighorhoods from all the"
                    " ones in order to reduce the computational cost of the"
                    " ODVBA method. The selected neighborhoods are used to form"
                    " the learning sets on which the discrimative coefficients"
                    " are calulated.",
                    // example usage
                    "EXECNAME --ni_num 10000 --ni_voxels 400 --ni_size 15 subjects.txt --index index.txt --ni ni.txt",
                    // version and copyright information
                    RELEASE, COPYRIGHT,
                    // license information
                    "See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.",
                    // contact
                    "SBIA Group <sbia-software at uphs.upenn.edu>");

        // The constructor of the CmdLine class has already added the standard
        // arguments --help, --helpshort, --helpxml, --helpman, and --version.

        cmd.add(index_file);
        cmd.add(ni_file);
        cmd.add(ni_size);
        cmd.add(ni_num);
        cmd.add(vox_num);
        cmd.add(srand_time);
        cmd.add(sublist_file);

        cmd.parse(argc, argv);
    } catch (CmdLineException& e) {
        // invalid command-line specification
        cerr << e.error() << endl;
        exit(1);
    }

    // -----------------------------------------------------------------------
    // initialize random number generator
    if (srand_time.getValue()) {
        srand(static_cast<unsigned int>(time(NULL)));
    } else {
        srand(1);
    }

    // -----------------------------------------------------------------------
    // read database
    CvMat         *data  = NULL; // image data of the n subjects
    CvMat         *index = NULL; // indices of "non-zero" voxels (template)
    int            n1    = 0;    // number of subjects belonging to group 1
    nifti_1_header hdr;          // image header of first input image

    // read image data
    cout << "Parsing subject list and reading image data" << endl;
    cout.flush();
    ok = ((data = read_data(sublist_file.getValue().c_str(), &n1, &hdr)) != NULL);
    if (!ok) cerr << "Failed to read image data" << endl;

    // read/create index
    if (ok) {
        if (index_file.isSet()) {
            cout << "Reading index" << endl;
            cout.flush();
            ok = ((index = read_matrix(index_file.getValue().c_str())) != NULL);
            if (!ok) cerr << "Failed to read index" << endl;
        } else {
            cout << "Generating index" << endl;
            cout.flush();
            ok = ((index = create_index(data)) != NULL);
            if (!ok) cerr << "Failed to generate index" << endl;
        }
    }

    // -----------------------------------------------------------------------
    // generate neighborhoods
    CvMat *ni = NULL;

    if (ok) {
        cout << "Generating neighborhoods" << endl;
        cout.flush();
        // Note: One half voxel is subtracted because createni accounts for it.
        int ni_sizex = static_cast<int>(round(ni_size.getValue() / hdr.pixdim[1] - 0.5));
        int ni_sizey = static_cast<int>(round(ni_size.getValue() / hdr.pixdim[2] - 0.5));
        int ni_sizez = static_cast<int>(round(ni_size.getValue() / hdr.pixdim[3] - 0.5));
        // unlikely not in mm, but this check will not hurt
        int xyz_units = XYZT_TO_SPACE(hdr.xyzt_units);
        if (xyz_units == NIFTI_UNITS_MICRON) {
            ni_sizex /= 1000;
            ni_sizey /= 1000;
            ni_sizez /= 1000;
        } else if (xyz_units == NIFTI_UNITS_METER) {
            ni_sizex *= 1000;
            ni_sizey *= 1000;
            ni_sizez *= 1000;
        }
        // enforce minimum of one voxel neighborhood in each direction
        if (ni_sizex <= 0) ni_sizex = 1;
        if (ni_sizey <= 0) ni_sizey = 1;
        if (ni_sizez <= 0) ni_sizez = 1;
        // create neighborhoods
        ok = ((ni = create_ni(index,
                              hdr.dim[1], hdr.dim[2], hdr.dim[3],
                              ni_sizex, ni_sizey, ni_sizez,
                              ni_num.getValue(),
                              vox_num.getValue())) != NULL);
    }

    // -----------------------------------------------------------------------
    // write neighborhoods
    if (ok) {
        cout << "Writing neighborhood(s) to file " << ni_file.getValue() << endl;
        cout.flush();
        int field_width = 0;
        int max_value   = index->rows;
        while (max_value > 0) {
            field_width++;
            max_value = max_value / 10;
        }
        char fmt[24];      
        sprintf(fmt, "%%%d.0f", field_width);
        ok = write_matrix(ni_file.getValue().c_str(), ni, fmt);
    }

    // -----------------------------------------------------------------------
    // clean up
    if (ni) {
        cvReleaseMat(&ni);
        ni = NULL;
    }
    if (index) {
        cvReleaseMat(&index);
        index = NULL;
    }
    if (data) {
        cvReleaseMat(&data);
        data = NULL;
    }

    exit(ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
