/**
 * @file  odvba-lib.cxx
 * @brief Implements Optimally-Discriminative Voxel-Based Analysis (ODVBA).
 *
 * Copyright (c) 2010-2012 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */

#include <time.h> // clock(), time()

#include <odvba/odvba.h>
#include <odvba/utilities.h>


// acceptable in .cxx file
namespace ublas = boost::numeric::ublas;
namespace atlas = boost::numeric::bindings::atlas;


namespace odvba {


// ===========================================================================
// data types
// ===========================================================================

// ---------------------------------------------------------------------------
// Database
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
Database::Database()
:
    n1   (0),
    X    (NULL),
    index(NULL),
    NI   (NULL)
{
}

// ---------------------------------------------------------------------------
Database::~Database()
{
}

// ---------------------------------------------------------------------------
void Database::release()
{
    if (X) {
        cvReleaseMat(&X);
        X = NULL;
    }
    if (index) {
        cvReleaseMat(&index);
        index = NULL;
    }

    if (NI) {
        cvReleaseMat(&NI);
        NI = NULL;
    }
}

// ---------------------------------------------------------------------------
// NDPData
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
NDPData::NDPData(const int m)
:
    A_rep    (NULL),
    Eigvector(NULL),
    Eigvalue (NULL),
    A        (NULL)
{
    if (m > 0) {
        A_rep     = cvCreateMat(m, m, CV_32FC1);
        Eigvector = cvCreateMat(m, m, CV_32FC1);
        Eigvalue  = cvCreateMat(1, m, CV_32FC1);
        A         = cvCreateMat(m, m, CV_32FC1);
    }
}

// ---------------------------------------------------------------------------
NDPData::~NDPData()
{
    if (A_rep)     cvReleaseMat(&A_rep);
    if (Eigvector) cvReleaseMat(&Eigvector);
    if (Eigvalue)  cvReleaseMat(&Eigvalue);
    if (A)         cvReleaseMat(&A);
}

// ---------------------------------------------------------------------------
// CompMapData
// ---------------------------------------------------------------------------

// ---------------------------------------------------------------------------
CompMapData::CompMapData(const int m,
                         const int n,
                         const int N,
                         const int k)
:
    delta(NULL),
    ndp  ((m > 0 && n > 0 && N > 0 && k > 0) ? m : 0),
    W    (NULL)
{
    if (m > 0 && n > 0 && N > 0 && k > 0) {
        delta = cvCreateMat(N, 1, CV_32FC1);
        W     = cvCreateMat(k, N, CV_32FC1);
    }
}

// ---------------------------------------------------------------------------
CompMapData::~CompMapData()
{
    if (delta) cvReleaseMat(&delta);
    if (W)     cvReleaseMat(&W);
}

// ===========================================================================
// initialization
// ===========================================================================

// ---------------------------------------------------------------------------
CvMat* create_index(const CvMat* data)
{
    CvMat* index1 = cvCreateMat(data->rows, 1, CV_32FC1);

    if (!index1) return NULL;

    cvZero(index1);

    int count = 0;

    for (int i = 0; i < data->rows; i++) {
        double me = 0;
        for (int j = 0; j < data->cols; j++) {
            me = me + cvGetReal2D (data, i, j);
        }
        me = me / data->cols;

        if (me > 40) {
            cvSetReal1D(index1, i, static_cast<double>(i + 1));
            count++;
        }
    }

    if (count == 0) {
        fprintf(stderr, "No \"non-zero\" voxels found\n");
        return NULL;
    }

    int idx = 0;

    CvMat* index2 = cvCreateMat(count, 1, CV_32FC1);

    if (!index2) {
        cvReleaseMat(&index1);
        return NULL;
    }

    for (int i = 0; i < data->rows; i++) {
        double ele = cvGetReal1D(index1, i);

        if (ele > 0) {
            cvSetReal1D(index2, idx, ele);
            idx++;
        }
    }

    cvReleaseMat(&index1);
    index1 = NULL;

    return index2;
}

// ---------------------------------------------------------------------------
CvMat* create_ni(const CvMat* index,
                 const int dim_x, const int dim_y, const int dim_z,
                 const int ni_size_x, const int ni_size_y, const int ni_size_z,
                 const int ni_num, const int vox_num)
{
    const int m = index->rows; // number of non-zero voxels

    // check arguments
    if (ni_size_x < 1 || ni_size_y < 1 || ni_size_z < 1) {
        fprintf(stderr, "create_ni(): invalid argument for parameter 'ni_size_*'\n");
        return NULL;
    }

    if (ni_num < 1 || ni_num > m) {
        fprintf(stderr, "create_ni(): invalid argument for parameter 'ni_num' (0 < ni_num <= %d)\n", m);
        return NULL;
    }

    if (vox_num < 1 || vox_num > ((2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1))) {
        fprintf(stderr, "create_ni(): invalid argument for parameter 'vox_num' (0 < vox_num < %d)\n",
                (2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1));
        return NULL;
    }

    // allocate memory
    CvMat* NI       = cvCreateMat(vox_num, ni_num, CV_32FC1); // neighborhood(s)
    CvMat* samples  = cvCreateMat(m,       1,      CV_32FC1); // indices of centering voxels (only first ni_num used)
    CvMat* x_vector = cvCreateMat(m,       1,      CV_32FC1); // maps voxel index to x coordinate of voxel
    CvMat* y_vector = cvCreateMat(m,       1,      CV_32FC1); // maps voxel index to y coordinate of voxel
    CvMat* z_vector = cvCreateMat(m,       1,      CV_32FC1); // maps voxel index to z coordinate of voxel

    if (!NI || !samples || !x_vector || !y_vector || !z_vector) {
        fprintf(stderr, "Failed to allocated required memory\n");

        if (x_vector) cvReleaseMat(&x_vector);
        if (y_vector) cvReleaseMat(&y_vector);
        if (z_vector) cvReleaseMat(&z_vector);
        if (samples)  cvReleaseMat(&samples);
        if (NI)       cvReleaseMat(&NI);

        return NULL;
    }

    // create mapping from voxel indices to (x, y, z) coordinates
    // index [i] => (x_vector [i], y_vector [i], z_vector [i])
    for (int i = 0; i < m; i++) {
        double idx = cvGetReal1D(index, i);

        double z_temp = ceil((idx / dim_x) / dim_y);
        double y_temp = ceil((idx - (z_temp - 1) * dim_x * dim_y) / dim_x);
        double x_temp = idx - (z_temp - 1) * dim_x * dim_y - (y_temp - 1) * dim_x;

        cvSetReal1D(x_vector, i, x_temp);
        cvSetReal1D(y_vector, i, y_temp);  
        cvSetReal1D(z_vector, i, z_temp);
    }

    // randomly select ni_num voxels out of the m non-zero voxels
    {
        int* ra = randperm(m);

        for (int i = 0; i < ni_num; i++) {
            cvSetReal1D(samples, i, static_cast <double> (ra [i] - 1));
        }

        delete [] ra;
        ra = NULL;
    }

    // for each neighborhood, randomly select vox_num voxels
    int ni1[(2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1)];
    int ni2[vox_num];

    for (int i = 0; i < ni_num; ++ i) {
        // map index of neighborhood center to (x, y, z)
        int r  = static_cast<int>(cvGetReal1D(samples,  i));
        int xi = static_cast<int>(cvGetReal1D(x_vector, r));
        int yi = static_cast<int>(cvGetReal1D(y_vector, r));
        int zi = static_cast<int>(cvGetReal1D(z_vector, r));

        // determine boundaries of neighborhood
        int minx = xi - ni_size_x;
        if (minx < 1) minx = 1;

        int maxx = xi + ni_size_x;
        if (maxx > dim_x) maxx = dim_x;

        int miny = yi - ni_size_y;
        if (miny < 1) miny = 1;

        int maxy = yi + ni_size_y;
        if (maxy > dim_y) maxy = dim_y;

        int minz = zi - ni_size_z;
        if (minz < 1) minz = 1;

        int maxz = zi + ni_size_z;
        if (maxz > dim_z) maxz = dim_z;

        // determine indices of all neighboring non-zero voxels
        int current_num_ni = 0;

        for (int j = 0; j < m; j++) {
            int xj = static_cast<int>(cvGetReal1D(x_vector, j));
            int yj = static_cast<int>(cvGetReal1D(y_vector, j));
            int zj = static_cast<int>(cvGetReal1D(z_vector, j));

            if ((xj < maxx + 1) && (xj > minx - 1)
                    && (yj < maxy + 1) && (yj > miny - 1)
                    && (zj < maxz + 1) && (zj > minz - 1)) {
                ni1[current_num_ni] = j + 1;
                current_num_ni++;
            }
        }

        while (current_num_ni < vox_num) {
            ni1[current_num_ni] = 0; // note: zero indicates an invalid index
            current_num_ni++;
        }

        // randomly select vox_num voxels of these neighboring voxels
        {
            int* ra = randperm(current_num_ni);
            for (int ss = 0; ss < vox_num; ss++) {
                ni2[ss] = ni1[ra[ss] - 1];
            }

            delete [] ra;
            ra = NULL;
        }

        // set first entry to center of neighborhood
        int flag = 0;

        for (int tt = 0; tt < vox_num; tt++) {
            if (ni2[tt] == r + 1) {
                int tmp = ni2[0];
                ni2[0]  = ni2[tt];
                ni2[tt] = tmp;

                flag = 1;
                break;
            }
        }

        if (flag == 0) ni2[0] = r + 1;

        // copy neighborhood to column i of NI
        for (int ss = 0; ss < vox_num; ss++) {
            cvSetReal2D(NI, ss, i, ni2[ss]);
        }
    }

    // clean up
    cvReleaseMat(&x_vector);
    x_vector = NULL;

    cvReleaseMat(&y_vector);
    y_vector = NULL;

    cvReleaseMat(&z_vector);
    z_vector = NULL;

    cvReleaseMat(&samples);
    samples = NULL;

    return NI;
}

// ---------------------------------------------------------------------------
CvMat* create_x(CvMat* data, CvMat* index)
{
    const unsigned int n = data->cols;  // number of subjects
    const unsigned int m = index->rows; // number of voxels per subject (template)

    CvMat* X = NULL;

    try {
        X = cvCreateMat(m, n, CV_32FC1);
    } catch (...) {
        fprintf (stderr, "Failed to allocate memory for matrix X\n");
    }

    if (!X) return NULL;

    bool ok = true;
    int  idx;

    for (int i = 0; ok && i < index->rows; i++) {
        for (int j = 0; ok && j < data->cols; j++) {
            idx = static_cast<int>(cvGetReal1D(index, i)) - 1;
            if (0 <= idx && idx < data->rows) {
                cvSetReal2D(X, i, j, cvGetReal2D(data, idx, j));
            } else {
                ok = false;
            }   
        }
    }

    if (!ok) {
        fprintf(stderr, "Invalid index '%d'\n", idx);
        cvReleaseMat(&X);
        X = NULL;
    }

    return X;
}

// ===========================================================================
// non-negative discriminative projection (NDP)
// ===========================================================================

// ---------------------------------------------------------------------------
ublas::vector<double> ndp(const ublas::matrix<double>& theta,
                          const int                    n1,
                          const double                 gamma,
                          NDPData&                     tmp)
{
    const int m = theta.size1(); // number of voxels in data (per subject)
    const int n = theta.size2(); // number of subjects

    const int n2 = n - n1; // number of subjects belonging to group 2

    // Note: The following local variables are pre-allocated by
    //       instantiating NDPData. This is done to avoid costly
    //       (re-)allocations of these local variables.
    ublas::matrix<double>& M          = tmp.M;
    ublas::matrix<double>& MT         = tmp.MT;
    ublas::vector<double>& e          = tmp.e;
    ublas::vector<double>& intraMean1 = tmp.intraMean1;
    ublas::matrix<double>& intraCov1  = tmp.intraCov1;
    ublas::vector<double>& intraMean2 = tmp.intraMean2;
    ublas::matrix<double>& intraCov2  = tmp.intraCov2;
    CvMat*&                A_rep      = tmp.A_rep;
    CvMat*&                Eigvector  = tmp.Eigvector;
    CvMat*&                Eigvalue   = tmp.Eigvalue;
    CvMat*&                A          = tmp.A;
    ublas::matrix<double>& Ap         = tmp.Ap;
    ublas::matrix<double>& An         = tmp.An;
    ublas::vector<double>& a          = tmp.a;
    ublas::vector<double>& c          = tmp.c;
    ublas::vector<double>& w          = tmp.w;

    // compute mean/covariance of voxel values belonging to group 1
    M         .resize(m,  n1);
    MT        .resize(n1, m);
    e         .resize(n1);
    intraMean1.resize(m);
    intraCov1 .resize(m, m);

    for (int i = 0; i < n1; i++) {
        ublas::matrix_column< ublas::matrix<double> >(M, i)
                = ublas::matrix_column< const ublas::matrix<double> >(theta, i);
    }

    for (int i = 0; i < n1; i++) {
        e(i) = 1.0 / n1;
    }

    atlas::gemv(M, e, intraMean1);

    for (int i = 0; i < static_cast<int>(M.size1 ()); i++) {
        for (int j = 0; j < static_cast<int>(M.size2 ()); j++) {
            M (i, j) -= intraMean1(i);
        }
    }

    MT = ublas::trans(M);
    atlas::gemm(M, MT, intraCov1);

    // compute mean/covariance of voxel values belonging to group 2
    M         .resize(m,  n2);
    MT        .resize(n2, m);
    e         .resize(n2);
    intraMean2.resize(m);
    intraCov2 .resize(m, m);

    for (int i = 0; i < n2; i++) {
        ublas::matrix_column<ublas::matrix <double> >(M, i)
                = ublas::matrix_column<const ublas::matrix <double> >(theta, n1 + i);
    }

    for (int i = 0; i < n2; i++) {
        e(i) = 1.0 / n2;
    }

    atlas::gemv(M, e, intraMean2);

    for (int i = 0; i < static_cast<int>(M.size1 ()); i++) {
        for (int j = 0; j < static_cast<int>(M.size2 ()); j++) {
            M(i, j) -= intraMean2(i);
        }
    }

    MT = ublas::trans(M);
    atlas::gemm(M, MT, intraCov2);

    // compute intra/within-class compactness S_W (and store in intraCov1)
    for (int i = 0; i < m; i++) {
        for (int j = 0; j < m; j++) {
            intraCov1(i, j) += intraCov2(i, j);
        }
    }

    // compute inter/between-class compactness S_B (and store in intraCov2, alias meanCov)
    ublas::matrix<double>& meanCov = intraCov2;
    meanCov = ublas::zero_matrix<double>(m, m); // necessary as atlas::ger adds result of
                                                // outer vector product to initial values!

#if 1
    ublas::vector<double> extraMean(m);

    e.resize(n);
    for (int i = 0; i < n; i++) {
        e(i) = 1.0 / n;
    }

    atlas::gemv(theta, e, extraMean);

    intraMean1 = intraMean2 - extraMean;
    atlas::ger(intraMean1, intraMean1, meanCov);

    // compute (gamma * S_W - S_B) part of A defined in MICCAI paper
    if (!A_rep || A_rep->rows != m || A_rep->cols != m) {
        if (A_rep) cvReleaseMat(&A_rep);
        A_rep = cvCreateMat(m, m, CV_32FC1);
    }

    for (int i = 0; i < m; i++) {
        for (int j = 0; j < m; j++) {
            cvSetReal2D(A_rep, i, j, gamma * intraCov1(i, j) - 2.0 * meanCov(i, j));
        }
    }
#else
    intraMean1 = intraMean1 - intraMean2;
    atlas::ger(intraMean1, intraMean1, meanCov);

    // compute (gamma * S_W - S_B) part of A defined in MICCAI paper
    if (!A_rep || A_rep->rows != m || A_rep->cols != m) {
        if (A_rep) cvReleaseMat(&A_rep);
        A_rep = cvCreateMat(m, m, CV_32FC1);
    }

    for (int i = 0; i < m; i++) {
        for (int j = 0; j < m; j++) {
            cvSetReal2D(A_rep, i, j, gamma * intraCov1(i, j) - meanCov(i, j));
        }
    }
#endif

    // compute positive definite matrix A defined in the MICCAI paper
    const double eigPrecision = 0.1;    // precision of eigenvalue computation
    const double tauSqrt      = 1.0e-6; // regularization parameter << 1

    if (!A || A->rows != m || A->cols != m) {
        if (A) cvReleaseMat(&A);
        A = cvCreateMat(m, m, CV_32FC1);
    }

    if (!Eigvector || Eigvector->rows != m || Eigvector->cols != m) {
        if (Eigvector) cvReleaseMat(&Eigvector);
        Eigvector = cvCreateMat(m, m, CV_32FC1);
    }

    if (!Eigvalue || Eigvalue->rows != 1 || Eigvalue->cols != m) {
        if (Eigvalue) cvReleaseMat(&Eigvalue);
        Eigvalue = cvCreateMat(1, m, CV_32FC1);
    }

    cvCopy(A_rep, A); // copy needed because cvEigenVV modifies the input matrix!

    cvEigenVV(A_rep, Eigvector, Eigvalue, eigPrecision);
    double minEigvalue = 0.0, dump = 0.0;
    cvMinMaxLoc(Eigvalue, &minEigvalue, &dump);
    minEigvalue = fabs(minEigvalue);

    for (int i = 0; i < m; i++) {
        cvSetReal2D(A, i, i, cvGetReal2D(A, i, i) + minEigvalue + tauSqrt);
    }

    // apply iterative Non-negative Quadratic Programming (NQP) approach to
    // minimize the objective function defined in the MICCAI paper given
    // the positive definite matrix A computed above
    const int    numIter = 15;     // number of iterations
    const double b       = - 0.99;

    a .resize(m);
    c .resize(m);
    w .resize(m);
    Ap.resize(m, m);
    An.resize(m, m);

    for (int i = 0; i < m; i++) {
        w(i) = 1;
        for (int j = 0; j < m; j++) {
            const double A_element = cvGetReal2D(A, i, j);
            if (A_element > 0) {
                Ap(i, j) = A_element;
                An(i, j) = 0.0;
            } else {
                Ap(i, j) = 0.0;
                An(i, j) = - A_element;
            }
        }
    }

    for (int i = 0; i < numIter; i++) {
        atlas::gemv(Ap, w, a);
        atlas::gemv(An, w, c);

        for (int j = 0; j < m; j++) {
            a(j) = a(j) + 1.0e-17;
//          a(j) = (b(j) + sqrt(16.0 * a(j) * c(j) + pow(b(j), 2))) / (4.0 * a(j));
            a(j) = (b    + sqrt(16.0 * a(j) * c(j) + pow(b   , 2))) / (4.0 * a(j));
        }

        w = ublas::element_prod(w, a);
    }

    return w;
}

// ===========================================================================
// determination of each voxel's statistic
// ===========================================================================

// ---------------------------------------------------------------------------
void comp_map(const Database& data, const Options& opt, CvMat* map, CompMapData& tmp)
{
    const int n = data.X ->cols; // number of subjects
    const int m = data.X ->rows; // number of voxels per subject
    const int N = data.NI->cols; // number of neighborhoods
    const int k = data.NI->rows; // number of samples per neighborhood

    const int n1 = data.n1;

    // verify parameters
    if (!map || map->rows != m || map->cols != 1) {
        fprintf (stderr, "comp_map(): invalid argument for parameter 'map'\n");
    }

    double sum1;        // used to compute sums which iterate over subjects belonging to group 1
    double sum2;        // used to compute sums which iterate over subjects belonging to group 2
    double meanWeight1; // mean weight computed considering subjects of group 1 only
    double meanWeight2; // mean weight computed considering subjects of group 2 only
    double denom;       // used to calculate denominator of discriminative degree
    double value;       // double value, e.g., used for calculation of discriminative degree

    // Note: The following local variables are pre-allocated by
    //       instantiating CompMapData. This is done to avoid costly
    //       (re-)allocations of these local variables.
    ublas::matrix<double>& theta  = tmp.theta;
    ublas::matrix<double>& thetaT = tmp.thetaT;
    ublas::vector<double>& weight = tmp.weight;
    ublas::vector<double>& w      = tmp.ndp.w;
    CvMat*&                delta  = tmp.delta;
    CvMat*&                W      = tmp.W;

    if (!delta || delta->rows != N || delta->cols != 1) {
        if (delta) cvReleaseMat(&delta);
        delta = cvCreateMat(N, 1, CV_32FC1);
    }

    if (!W || W->rows != k || W->cols != N) {
        if (W) cvReleaseMat(&W);
        W = cvCreateMat(k, N, CV_32FC1);
    }

    // reset result
    cvZero(map);

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // 1. Determine statistics of each voxel and each neighborhood
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    theta .resize(k, n);
    thetaT.resize(n, k);
    w     .resize(k);
    weight.resize(n);

    // for each neighborhood/for each voxel in template ...
    for (int i = 0; i < N; i++) {
        // ignore empty neighborhoods (only center, i.e., first entry is valid)
        if (cvGetReal2D(data.NI, 1, i) <= 0) continue;

        // construct learning set centered at current voxel of template
        for (int r = 0; r < k; r++) {
            int idx = static_cast<int>(cvGetReal2D(data.NI, r, i)) - 1;

            if (idx >= 0) {
                for (int j = 0; j < n; j++) {
                    theta(r, j) = cvGetReal2D(data.X, idx, j);
                }
            } else {
                for (int j = 0; j < n; j++) {
                    theta(r, j) = 0;
                }
            }
        }

        // calculate weights for samples in neighborhood using NDP, i.e.,
        // the projection vector. Therefore, check whether learning set
        // is suitable for NDP
        sum1 = 0;
        for (int j = 0; j < data.n1; j++) {
            for (int r = 0; r < k; r++) {
                sum1 += theta(r, j);
            }
        }

        sum2 = 0;
        for (int j = data.n1; j < n; j++) {
            for (int r = 0; r < k; r++) {
                sum2 += theta(r, j);
            }
        }

        if (((sum1/n1) > (sum2/(n-n1))) || (sum1 == 0)) {
            w      = ublas::zero_vector<double>(k);
            weight = ublas::zero_vector<double>(n);
        } else {
            // note: w = tmp.ndp.w
            /*w =*/ ndp(theta, n1, opt.gamma, tmp.ndp);
            thetaT = ublas::trans(theta);
            atlas::gemv(thetaT, w, weight);
        }

        sum1 = 0;
        sum2 = 0;
        for (int j = 0;  j < n1; j++) sum1 += weight(j);
        for (int j = n1; j < n;  j++) sum2 += weight(j);

        meanWeight1 = sum1 / n1;
        meanWeight2 = sum2 / (n - n1);

        if (meanWeight1 > meanWeight2) w = ublas::zero_vector<double>(k);
        for (int r = 0; r < k; r++) cvSetReal2D(W, r, i, w(r));

        // calculate discriminating degree
        // (cf. Eq. (11) in MICCAI paper)
        denom = 0;
        for (int j = 0; j < n1; j++) {
            denom += pow(fabs(weight(j) - meanWeight1), 2);
        }
        for (int j = n1; j < n; j++) {
            denom += pow(fabs(weight(j) - meanWeight2), 2);
        }
        denom = sqrt(denom);
        
        value = ((denom > 0) ? ((fabs(meanWeight1 - meanWeight2) / denom) * sqrt(n - 2)) : 0);
        value = pow(value, opt.phi);

        cvSetReal1D(delta, i, value);
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // 2. Get final statistics of voxels
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    // calculate map of voxel statistics by combining the results obtained
    // for each neighborhood (cf. Eq. (12) in MICCAI paper)
    for (int i = 0; i < N; i++) {
        for (int j = 0; j < k; j++) {
            const int idx = static_cast<int>(cvGetReal2D(data.NI, j, i)) - 1;
            if (idx >= 0) {
                cvSetReal1D(map, idx, cvGetReal1D(map, idx)
                        + cvGetReal2D(W, j, i) * cvGetReal1D(delta, i));
            }
        }
    }
}

// ===========================================================================
// group analysis using permutation tests 
// ===========================================================================

// ---------------------------------------------------------------------------
CvMat* perform_analysis (const Database& data,
                         const Options&  opt,
                         bool            initial,
                         int             verbose,
                         const char*     msgPrefix)
{
    time_t startTime;

    const int m = data.X->rows; // number of voxels per subject
    const int n = data.X->cols; // number of subjects
    const int p = ((opt.nPerm > 0) ? opt.nPerm : (opt.perms ? opt.perms->rows : 0));

    if (p == 0 && !initial) {
        fprintf(stderr, "%sperform_analysis(): invalid number of tests\n", msgPrefix);
        return NULL;
    }

    if (opt.perms && opt.perms->cols != n) {
        fprintf(stderr, "%sperform_analysis(): invalid permutations matrix\n", msgPrefix);
        return NULL;
    }

    // initialize
    if (verbose > 1) {
        startTime = clock();
        printf("%sInitializing data structures\n", msgPrefix);
        fflush(stdout);
    }

    CompMapData tmp(m, n);
    Database    permutedData;

    permutedData.n1    = data.n1;
    permutedData.index = data.index;
    permutedData.NI    = data.NI;
    permutedData.X     = cvCreateMat(m, n, CV_32FC1);

    CvMat* map  = cvCreateMat(m, 1, CV_32FC1);
    CvMat* maps = cvCreateMat((initial ? (1 + p) : p), m, CV_32FC1);

    if (!permutedData.X || !map || !maps) {
        fprintf(stderr, "%sFailed to initialize data structures\n", msgPrefix);

        if (maps)           cvReleaseMat(&maps);
        if (map)            cvReleaseMat(&map);
        if (permutedData.X) cvReleaseMat(&permutedData.X);

        return NULL;
    }

    if (verbose > 1) {
        double duration = static_cast<double>(clock() - startTime) / CLOCKS_PER_SEC;
        printf("%sData structures initialized in %.2f sec\n", msgPrefix, duration);
        fflush(stdout);
    }

    // initialize random number generator
    srand(opt.seed);

    // compute initial results
    if (initial) {
        if (verbose > 0) {
            startTime = clock();
            printf("%sPerforming initial analysis\n", msgPrefix);
            fflush(stdout);
        }

        comp_map(data, opt, map, tmp);

        for (int i = 0; i < m; i++) {
            cvSetReal2D(maps, 0, i, cvGetReal1D(map, i));
        }

        if (verbose > 0) {
            double duration = static_cast<double>(clock() - startTime) / CLOCKS_PER_SEC;
            printf("%sPerformed initial analysis in %.2f sec\n", msgPrefix, duration);
            fflush(stdout);
        }
    }

    // perform permutation tests
    for (int i = 1; i <= p; i++) {
        if (verbose > 0) {
            startTime = clock();

            if (i > 3) {
                printf("%sPerforming %dth permutation test\n", msgPrefix, i);
            } else if (i == 1) {
                printf("%sPerforming 1st permutation test\n", msgPrefix);
            } else if (i == 2) {
                printf("%sPerforming 2nd permutation test\n", msgPrefix);
            } else if (i == 3) {
                printf("%sPerforming 3rd permutation test\n", msgPrefix);
            }
            fflush(stdout);
        }

        // permute data randomly
        int* permutation = NULL;

        if (opt.nPerm == 0) {
            permutation = new int[n];

            for (int j = 0; j < n; j++) {
                permutation[j] = static_cast<int>(cvGetReal2D(opt.perms, i - 1, j));
            }
        } else {
            permutation = randperm(n);
            if (opt.perms) {
                for (int j = 0; j < n; j++) {
                    cvSetReal2D(opt.perms, i - 1, j, static_cast<float>(permutation[j]));
                }
            }
        }

        for (int j = 0; j < m; j++) {
            for (int k = 0; k < n; k++) {
                cvSetReal2D(permutedData.X, j, permutation[k] - 1, cvGetReal2D(data.X, j, k));
            }
        }

        delete [] permutation;

        // perform analysis on permuted data
        comp_map(permutedData, opt, map, tmp);

        // copy results
        for (int j = 0; j < m; j++) {
            cvSetReal2D(maps, (initial ? i : (i - 1)), j, cvGetReal1D(map, j));
        }

        if (verbose > 0) {
            double duration = static_cast<double>(clock() - startTime) / CLOCKS_PER_SEC;
            if (i > 3) {
                printf("%sPerformed %dth permutation test in %.2f sec\n", msgPrefix, i, duration);
            } else if (i == 1) {
                printf("%sPerformed 1st permutation test in %.2f sec\n", msgPrefix, duration);
            } else if (i == 2) {
                printf("%sPerformed 2nd permutation test in %.2f sec\n", msgPrefix, duration);
            } else if (i == 3) {
                printf("%sPerformed 3rd permutation test in %.2f sec\n", msgPrefix, duration);
            }
            fflush(stdout);
        }
    }

    cvReleaseMat(&permutedData.X);

    return maps;
}

// ===========================================================================
// p-value / p-image
// ===========================================================================

// ---------------------------------------------------------------------------
CvMat* get_p_value(CvMat* maps)
{
    CvMat* pValue = cvCreateMat(maps->cols, 1, CV_32FC1);

    if (!pValue) {
        fprintf(stderr, "get_p_value(): failed to allocate memory\n");
        return NULL;
    }

    cvZero(pValue);

    for (int i = 1; i < maps->rows; i++) {
        for (int j = 0; j < maps->cols; j++) {
            double initial  = cvGetReal2D(maps, 0, j);         
            double permuted = cvGetReal2D(maps, i, j);

            if (permuted >= initial) {
                cvSetReal1D(pValue, j, cvGetReal1D(pValue, j) + 1);
            }
        }
    }

    return pValue;
}

// ---------------------------------------------------------------------------
CvMat* get_p_image(CvMat* maps, CvMat* index, int nVoxels, CvMat* pValueIn)
{
    CvMat* pValue = pValueIn;

    if (nVoxels < maps->cols) {
        fprintf(stderr, "get_p_image(): invalid number of image voxels\n");
        return NULL;
    }
    
    if (pValueIn && ((pValueIn->rows != index->rows) || (pValueIn->cols != index->cols))) {
        fprintf(stderr, "get_p_image(): index and p-value matrices must have same dimensions\n");
        return NULL;
    }

    if (maps->rows == 0) {
        fprintf(stderr, "get_p_image(): no group analysis results given\n");
        return NULL;
    }

    if (!pValue) {
        pValue = get_p_value(maps);
        if (!pValue) {
            fprintf(stderr, "get_p_image(): failed to compute p-values\n");
            return NULL;
        }
    }

    CvMat* pImage = cvCreateMat(nVoxels, 1, CV_32FC1);

    for (int i = 0; i < pImage->rows; i++) {
        cvSetReal1D (pImage, i, 1);
    }

    for (int i = 0; i < pValue->rows; i++) {
        int idx = static_cast<int>(cvGetReal1D(index, i)) - 1;
        cvSetReal1D(pImage, idx, cvGetReal1D(pValue, i) / maps->rows);
    }

    if (pValue && pValue != pValueIn) {
        cvReleaseMat(&pValue);
        pValue = NULL;
    }

    return pImage;
}


} // namespace odvba
