/*!
 * \file   SbiaOdvbaAlgorithm.cpp
 * \brief  Implements Optimally-Discriminative Voxel-Based Analysis (ODVBA).
 * \author Tianhao Zhang, revised by Andreas Schuh
 * \date   12/9/2010
 *
 * $Revision: 143 $
 * $Id: SbiaOdvbaAlgorithm.cpp 143 2011-02-08 18:17:30Z schuha@UPHS.PENNHEALTH.PRV $
 *
 * <b>Last changed:</b>
 * $Author: schuha@UPHS.PENNHEALTH.PRV $
 * $Date: 2011-02-08 13:17:30 -0500 (Tue, 08 Feb 2011) $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For copyright information see Copyright section of project
 * ReadMe.html in the project's root directory.
 *
 * Contact: sbia-software@uphs.upenn.edu
 */


#include "SbiaOdvbaAlgorithm.h"
#include "SbiaOdvbaUtilities.h"


namespace ublas = boost::numeric::ublas;
namespace atlas = boost::numeric::bindings::atlas;


SBIA_ODVBA_NAMESPACE_BEGIN;


//////////////////////////////////////////////////////////////////////////////
// Data types
//////////////////////////////////////////////////////////////////////////////

// ===========================================================================
// Structure: Database
// ===========================================================================

/****************************************************************************/
Database
::Database ()
:
	n1    (0),
	X     (NULL),
	index (NULL),
	NI    (NULL)
{
}

/****************************************************************************/
Database
::~Database ()
{
}

/****************************************************************************/
void
Database
::release ()
{
	if (X)
	{
		cvReleaseMat (&X);
		X = NULL;
	}

	if (index)
	{
		cvReleaseMat (&index);
		index = NULL;
	}

	if (NI)
	{
		cvReleaseMat (&NI);
		NI = NULL;
	}
}

// ===========================================================================
// Structure: NDPData
// ===========================================================================

/****************************************************************************/
NDPData
::NDPData (const int m)
:
	A_rep     (NULL),
	A         (NULL),
	Eigvector (NULL),
	Eigvalue  (NULL)
{
	if (m > 0)
	{
		A_rep     = cvCreateMat (m, m, CV_32FC1);
		Eigvector = cvCreateMat (m, m, CV_32FC1);
		Eigvalue  = cvCreateMat (1, m, CV_32FC1);
		A         = cvCreateMat (m, m, CV_32FC1);
	}
}

/****************************************************************************/
NDPData
::~NDPData ()
{
	if (A_rep)     cvReleaseMat (&A_rep);
	if (Eigvector) cvReleaseMat (&Eigvector);
	if (Eigvalue)  cvReleaseMat (&Eigvalue);
	if (A)         cvReleaseMat (&A);
}

// ===========================================================================
// Structure: CompMapData
// ===========================================================================

/****************************************************************************/
CompMapData
::CompMapData (const int m,
               const int n,
               const int N,
               const int k)
:
	delta (NULL),
	ndp   ((m > 0 && n > 0 && N > 0 && k > 0) ? m : 0),
	W     (NULL)
{
	if (m > 0 && n > 0 && N > 0 && k > 0)
	{
		delta = cvCreateMat (N, 1, CV_32FC1);
		W     = cvCreateMat (k, N, CV_32FC1);
	}
}

/****************************************************************************/
CompMapData
::~CompMapData ()
{
	if (delta) cvReleaseMat (&delta);
	if (W)     cvReleaseMat (&W);
}

//////////////////////////////////////////////////////////////////////////////
// Non-negative Discriminative Projection (NDP)
//////////////////////////////////////////////////////////////////////////////

/****************************************************************************/
ublas::vector<double> NDP (const ublas::matrix<double> &theta,
                           const int                    n1,
                           const double                 gamma,
                           NDPData                     &tmp)
{
	const int m = theta.size1(); // number of voxels in data (per subject)
	const int n = theta.size2(); // number of subjects

	const int n2 = n - n1; // number of subjects belonging to group 2

	// note: The following local variables are pre-allocated by
	//       instantiating NDPData. This is done to avoid costly
	//       (re-)allocations of these local variables.
	ublas::matrix<double> &M          = tmp.M;
	ublas::matrix<double> &MT         = tmp.MT;
	ublas::vector<double> &e          = tmp.e;
	ublas::vector<double> &intraMean1 = tmp.intraMean1;
	ublas::matrix<double> &intraCov1  = tmp.intraCov1;
	ublas::vector<double> &intraMean2 = tmp.intraMean2;
	ublas::matrix<double> &intraCov2  = tmp.intraCov2;
	CvMat                *&A_rep      = tmp.A_rep;
	CvMat                *&Eigvector  = tmp.Eigvector;
	CvMat                *&Eigvalue   = tmp.Eigvalue;
	CvMat                *&A          = tmp.A;
	ublas::matrix<double> &Ap         = tmp.Ap;
	ublas::matrix<double> &An         = tmp.An;
	ublas::vector<double> &a          = tmp.a;
	ublas::vector<double> &c          = tmp.c;
	ublas::vector<double> &w          = tmp.w;

	// compute mean/covariance of voxel values belonging to group 1
	M         .resize (m,  n1);
	MT        .resize (n1, m);
	e         .resize (n1);
	intraMean1.resize (m);
	intraCov1 .resize (m, m);

	for (int i = 0; i < n1; ++ i)
	{
		ublas::matrix_column< ublas::matrix<double> > (M, i) = ublas::matrix_column< const ublas::matrix<double> > (theta, i);
	}

	for (int i = 0; i < n1; ++ i)
	{
		e (i) = 1.0 / n1;
	}

	atlas::gemv (M, e, intraMean1);

	for (int i = 0; i < M.size1 (); ++ i)
	{
		for (int j = 0; j < M.size2 (); ++ j)
		{
			M (i, j) -= intraMean1 (i);
		}
	}

	MT = ublas::trans (M);
	atlas::gemm (M, MT, intraCov1);

	// compute mean/covariance of voxel values belonging to group 2
	M         .resize (m,  n2);
	MT        .resize (n2, m);
	e         .resize (n2);
	intraMean2.resize (m);
	intraCov2 .resize (m, m);

	for (int i = 0; i < n2; ++ i)
	{
		ublas::matrix_column< ublas::matrix<double> > (M, i) = ublas::matrix_column< const ublas::matrix<double> > (theta, n1 + i);
	}

	for (int i = 0; i < n2; ++ i)
	{
		e (i) = 1.0 / n2;
	}

	atlas::gemv (M, e, intraMean2);

	for (int i = 0; i < M.size1 (); ++ i)
	{
		for (int j = 0; j < M.size2 (); ++ j)
		{
			M (i, j) -= intraMean2 (i);
		}
	}

	MT = ublas::trans (M);
	atlas::gemm (M, MT, intraCov2);

	// compute intra/within-class compactness S_W (and store in intraCov1)
	for (int i = 0; i < m; ++ i)
	{
		for (int j = 0; j < m; ++ j)
		{
			intraCov1 (i, j) += intraCov2 (i, j);
		}
	}

	// compute inter/between-class compactness S_B (and store in intraCov2, alias meanCov)
	ublas::matrix<double> &meanCov = intraCov2;
	meanCov = ublas::zero_matrix<double> (m, m); // necessary as atlas::ger adds result of
	                                             // outer vector product to initial values!

#if 1
	ublas::vector<double> extraMean (m);

	e.resize (n);

	for (int i = 0; i < n; ++ i)
	{
		e (i) = 1.0 / n;
	}

	atlas::gemv (theta, e, extraMean);
	
	intraMean1 = intraMean2 - extraMean;
	atlas::ger (intraMean1, intraMean1, meanCov);

	// compute (gamma * S_W - S_B) part of A defined in MICCAI paper
	if (!A_rep || A_rep->rows != m || A_rep->cols != m)
	{
		if (A_rep) cvReleaseMat (&A_rep);
		A_rep = cvCreateMat (m, m, CV_32FC1);
	}

	for (int i = 0; i < m; ++ i)
	{
		for (int j = 0; j < m; ++ j)
		{
			cvSetReal2D (A_rep, i, j, gamma * intraCov1 (i, j) - 2.0 * meanCov (i, j));
		}
	}
#else
	intraMean1 = intraMean1 - intraMean2;
	atlas::ger (intraMean1, intraMean1, meanCov);

	// compute (gamma * S_W - S_B) part of A defined in MICCAI paper
	if (!A_rep || A_rep->rows != m || A_rep->cols != m)
	{
		if (A_rep) cvReleaseMat (&A_rep);
		A_rep = cvCreateMat (m, m, CV_32FC1);
	}

	for (int i = 0; i < m; ++ i)
	{
		for (int j = 0; j < m; ++ j)
		{
			cvSetReal2D (A_rep, i, j, gamma * intraCov1 (i, j) - meanCov (i, j));
		}
	}
#endif

	// compute positive definite matrix A defined in the MICCAI paper
	const double eigPrecision = 0.1;    // precision of eigenvalue computation
	const double tauSqrt      = 1.0e-6; // regularization parameter << 1

	if (!A || A->rows != m || A->cols != m)
	{
		if (A) cvReleaseMat (&A);
		A = cvCreateMat (m, m, CV_32FC1);
	}

	if (!Eigvector || Eigvector->rows != m || Eigvector->cols != m)
	{
		if (Eigvector) cvReleaseMat (&Eigvector);
		Eigvector = cvCreateMat (m, m, CV_32FC1);
	}

	if (!Eigvalue || Eigvalue->rows != 1 || Eigvalue->cols != m)
	{
		if (Eigvalue) cvReleaseMat (&Eigvalue);
		Eigvalue = cvCreateMat (1, m, CV_32FC1);
	}

	cvCopy (A_rep, A); // copy needed because cvEigenVV modifies the input matrix!

	cvEigenVV (A_rep, Eigvector, Eigvalue, eigPrecision);
	double minEigvalue = 0.0, dump = 0.0;
	cvMinMaxLoc (Eigvalue, &minEigvalue, &dump);
	minEigvalue = fabs (minEigvalue);

	for (int i = 0; i < m; ++ i)
	{
		cvSetReal2D (A, i, i, cvGetReal2D (A, i, i) + minEigvalue + tauSqrt);
	}

	// apply iterative Non-negative Quadratic Programming (NQP) approach to
	// minimize the objective function defined in the MICCAI paper given
	// the positive definite matrix A computed above
	const int    numIter = 15;     // number of iterations
	const double b       = - 0.99;

	a .resize (m);
	c .resize (m);
	w .resize (m);
	Ap.resize (m, m);
	An.resize (m, m);

	for (int i = 0; i < m; ++ i)
	{
		w (i) = 1;

		for (int j = 0; j < m; ++ j)
		{
			const double A_element = cvGetReal2D (A, i, j);

			if (A_element > 0)
			{
				Ap (i, j) = A_element;
				An (i, j) = 0.0;
			}
			else
			{
				Ap (i, j) = 0.0;
				An (i, j) = - A_element;
			}
		}
	}

	for (int i = 0; i < numIter; ++ i)
	{
		atlas::gemv (Ap, w, a);
		atlas::gemv (An, w, c);

		for (int j = 0; j < m; ++ j)
		{
			a (j) = a (j) + 1.0e-17;
//			a (j) = (b (j) + sqrt (16.0 * a (j) * c (j) + pow (b (j), 2))) / (4.0 * a (j));
			a (j) = (b     + sqrt (16.0 * a (j) * c (j) + pow (b    , 2))) / (4.0 * a (j));
		}

		w = ublas::element_prod (w, a);
	}

	return w;
}

//////////////////////////////////////////////////////////////////////////////
// Determination of each voxel's statistic
//////////////////////////////////////////////////////////////////////////////

/****************************************************************************/
void compMap (const Database &data, const Options &opt, CvMat *map, CompMapData &tmp)
{
	const int n = data.X ->cols; // number of subjects
	const int m = data.X ->rows; // number of voxels per subject
	const int N = data.NI->cols; // number of neighborhoods
	const int k = data.NI->rows; // number of samples per neighborhood

	const int n1 = data.n1;

	// verify parameters
	if (!map || map->rows != m || map->cols != 1)
	{
		fprintf (stderr, "compMap (): invalid argument for parameter 'map'\n");
	}

	double sum1;        // used to compute sums which iterate over subjects belonging to group 1
	double sum2;        // used to compute sums which iterate over subjects belonging to group 2
	double meanWeight1; // mean weight computed considering subjects of group 1 only
	double meanWeight2; // mean weight computed considering subjects of group 2 only
	double denom;       // used to calculate denominator of discriminative degree
	double value;       // double value, e.g., used for calculation of discriminative degree

	// note: The following local variables are pre-allocated by
	//       instantiating CompMapData. This is done to avoid costly
	//       (re-)allocations of these local variables.
	ublas::matrix<double> &theta  = tmp.theta;
	ublas::matrix<double> &thetaT = tmp.thetaT;
	ublas::vector<double> &weight = tmp.weight;
	ublas::vector<double> &w      = tmp.ndp.w;
	CvMat                *&delta  = tmp.delta;
	CvMat                *&W      = tmp.W;

	if (!delta || delta->rows != N || delta->cols != 1)
	{
		if (delta) cvReleaseMat (&delta);
		delta = cvCreateMat (N, 1, CV_32FC1);
	}

	if (!W || W->rows != k || W->cols != N)
	{
		if (W) cvReleaseMat (&W);
		W = cvCreateMat (k, N, CV_32FC1);
	}

	// reset result
	cvZero (map);

	// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 1. Determine statistics of each voxel and each neighborhood
	// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	theta .resize (k, n);
	thetaT.resize (n, k);
	w     .resize (k);
	weight.resize (n);

	// for each neighborhood/for each voxel in template ...
	for (int i = 0; i < N; ++ i)
	{
		// ignore empty neighborhoods (only center, i.e., first entry is valid)
		if (cvGetReal2D (data.NI, 1, i) <= 0) continue;

		// construct learning set centered at current voxel of template
		for (int r = 0; r < k; ++ r)
		{
			int idx = static_cast<int> (cvGetReal2D (data.NI, r, i)) - 1;

			if (idx >= 0)
			{
				for (int j = 0; j < n; ++ j)
				{
					theta (r, j) = cvGetReal2D (data.X, idx, j);
				}
			}
			else
			{
				for (int j = 0; j < n; ++ j)
				{
					theta (r, j) = 0;
				}
			}
		}

		// calculate weights for samples in neighborhood using NDP, i.e.,
		// the projection vector. Therefore, check whether learning set
		// is suitable for NDP
		sum1 = 0;
		for (int j = 0; j < data.n1; ++ j)
		{
			for (int r = 0; r < k; ++ r)
			{
				sum1 += theta (r, j);
			}
		}

		sum2 = 0;
		for (int j = data.n1; j < n; ++ j)
		{
			for (int r = 0; r < k; ++ r)
			{
				sum2 += theta (r, j);
			}
		}

		if ((sum1 > sum2) || (sum1 == 0))
		{
			w      = ublas::zero_vector<double> (k);
			weight = ublas::zero_vector<double> (n);
		}
		else
		{
			// note: w = tmp.ndp.w
			/*w =*/ NDP (theta, n1, opt.gamma, tmp.ndp);
			thetaT = ublas::trans (theta);

			atlas::gemv (thetaT, w, weight);
		}

		sum1 = 0;
		for (int j = 0; j < n1; ++ j)
		{
			sum1 += weight (j);
		}

		sum2 = 0;
		for (int j = n1; j < n; ++ j)
		{
			sum2 += weight (j);
		}

		meanWeight1 = sum1 / n1;
		meanWeight2 = sum2 / (n - n1);

		if (sum1 > sum2)
		{
			w = ublas::zero_vector<double> (k);
		}

		for (int r = 0; r < k; ++ r)
		{
			cvSetReal2D (W, r, i, w (r));
		}

		// calculate discriminating degree
		// (cf. Eq. (11) in MICCAI paper)
		denom = 0;
		for (int j = 0; j < n1; ++ j)
		{
			denom += pow (fabs (weight (j) - meanWeight1), 2);
		}
		for (int j = n1; j < n; ++ j)
		{
			denom += pow (fabs (weight (j) - meanWeight2), 2);
		}
		denom = sqrt (denom);

		value = ((denom > 0) ? ((fabs (meanWeight1 - meanWeight2) / denom) * sqrt (n - 2)) : 0);
		value = pow (value, opt.phi);

		cvSetReal1D (delta, i, value);
	}

	// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// 2. Get final statistics of voxels
	// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	// calculate map of voxel statistics by combining the results obtained
	// for each neighborhood (cf. Eq. (12) in MICCAI paper)
	for (int i = 0; i < N; ++ i)
	{
		for (int j = 0; j < k; ++ j)
		{
			int idx = static_cast<int> (cvGetReal2D (data.NI, j, i)) - 1;

			if (idx >= 0)
			{
				cvSetReal1D (map, idx, cvGetReal1D (map, idx) + cvGetReal2D (W, j, i) * cvGetReal1D (delta, i));
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////
// group analysis using permutation tests 
//////////////////////////////////////////////////////////////////////////////

/****************************************************************************/
CvMat *performAnalysis (const Database &data,
                        const Options  &opt,
                        bool            initial,
                        int             verbosity,
                        const char     *msgPrefix)
{
	time_t startTime;

	const int m = data.X->rows; // number of voxels per subject
	const int n = data.X->cols; // number of subjects
	const int p = ((opt.nPerm > 0) ? opt.nPerm : (opt.perms ? opt.perms->rows : 0));

	if (p == 0 && !initial)
	{
		fprintf (stderr, "%sperformAnalysis (): invalid number of tests\n", msgPrefix);
		return NULL;
	}

	if (opt.perms && opt.perms->cols != n)
	{
		fprintf (stderr, "%sperformAnalysis (): invalid permutations matrix\n", msgPrefix);
		return NULL;
	}

	// initialize
	if (verbosity > 1)
	{
		startTime = clock ();
		printf ("%sInitializing data structures\n", msgPrefix);
		fflush (stdout);
	}

	CompMapData tmp (m, n);
	Database    permutedData;

	permutedData.n1    = data.n1;
	permutedData.index = data.index;
	permutedData.NI    = data.NI;
	permutedData.X     = cvCreateMat (m, n, CV_32FC1);

	CvMat *map  = cvCreateMat (m, 1, CV_32FC1);
	CvMat *maps = cvCreateMat ((initial ? (1 + p) : p), m, CV_32FC1);

	if (!permutedData.X || !map || !maps)
	{
		fprintf (stderr, "%sFailed to initialize data structures\n", msgPrefix);

		if (maps)           cvReleaseMat (&maps);
		if (map)            cvReleaseMat (&map);
		if (permutedData.X) cvReleaseMat (&permutedData.X);

		return NULL;
	}

	if (verbosity > 1)
	{
		double duration = static_cast <double> (clock () - startTime) / CLOCKS_PER_SEC;
		printf ("%sData structures initialized in %.2f sec\n", msgPrefix, duration);
		fflush (stdout);
	}

	// compute initial results
	if (initial)
	{
		if (verbosity > 0)
		{
			startTime = clock ();
			printf ("%sPerforming initial analysis\n", msgPrefix);
			fflush (stdout);
		}

		compMap (data, opt, map, tmp);

		for (int i = 0; i < m; ++ i)
		{
			cvSetReal2D (maps, 0, i, cvGetReal1D (map, i));
		}

		if (verbosity > 0)
		{
			double duration = static_cast <double> (clock () - startTime) / CLOCKS_PER_SEC;
			printf ("%sPerformed initial analysis in %.2f sec\n", msgPrefix, duration);
			fflush (stdout);
		}
	}

	// perform permutation tests
	for (int i = 1; i <= p; ++ i)
	{
		if (verbosity > 0)
		{
			startTime = clock ();

			if (i > 3)
			{
				printf ("%sPerforming %dth permutation test\n", msgPrefix, i);
			}
			else if (i == 1)
			{
				printf ("%sPerforming 1st permutation test\n", msgPrefix);
			}
			else if (i == 2)
			{
				printf ("%sPerforming 2nd permutation test\n", msgPrefix);
			}
			else if (i == 3)
			{
				printf ("%sPerforming 3rd permutation test\n", msgPrefix);
			}
			fflush (stdout);
		}

		// permute data randomly
		int *permutation = NULL;

		if (opt.nPerm == 0)
		{
			permutation = new int [n];

			for (int j = 0; j < n; ++ j)
			{
				permutation [j] = static_cast <int> (cvGetReal2D (opt.perms, i - 1, j));
			}
		}
		else
		{
			permutation = randperm (n);

			if (opt.perms)
			{
				for (int j = 0; j < n; ++ j)
				{
					cvSetReal2D (opt.perms, i - 1, j, static_cast <float> (permutation [j]));
				}
			}
		}

		for (int j = 0; j < m; ++ j)
		{
			for (int k = 0; k < n; ++ k)
			{
				cvSetReal2D (permutedData.X, j, permutation [k] - 1, cvGetReal2D (data.X, j, k));
			}
		}

		delete [] permutation;

		// perform analysis on permuted data
		compMap (permutedData, opt, map, tmp);

		// copy results
		for (int j = 0; j < m; ++ j)
		{
			cvSetReal2D (maps, (initial ? i : (i - 1)), j, cvGetReal1D (map, j));
		}

		if (verbosity > 0)
		{
			double duration = static_cast <double> (clock () - startTime) / CLOCKS_PER_SEC;

			if (i > 3)
			{
				printf ("%sPerformed %dth permutation test in %.2f sec\n", msgPrefix, i, duration);
			}
			else if (i == 1)
			{
				printf ("%sPerformed 1st permutation test in %.2f sec\n", msgPrefix, duration);
			}
			else if (i == 2)
			{
				printf ("%sPerformed 2nd permutation test in %.2f sec\n", msgPrefix, duration);
			}
			else if (i == 3)
			{
				printf ("%sPerformed 3rd permutation test in %.2f sec\n", msgPrefix, duration);
			}
			fflush (stdout);
		}
	}
	
	cvReleaseMat (&permutedData.X);

	return maps;
}

//////////////////////////////////////////////////////////////////////////////
// p-value / p-image
//////////////////////////////////////////////////////////////////////////////

/****************************************************************************/
CvMat *getPValue (CvMat *maps)
{
	CvMat *pValue = cvCreateMat (maps->cols, 1, CV_32FC1);

	if (!pValue)
	{
		fprintf (stderr, "getPValue (): failed to allocate memory\n");
		return NULL;
	}

	cvZero (pValue);

	for (int i = 1; i < maps->rows; ++ i)
	{
		for (int j = 0; j < maps->cols; ++ j)
		{
			double initial  = cvGetReal2D (maps, 0, j);			
			double permuted = cvGetReal2D (maps, i, j);

			if (permuted >= initial)
			{
				cvSetReal1D (pValue, j, cvGetReal1D (pValue, j) + 1);
			}
		}
	}

	return pValue;
}

/****************************************************************************/
CvMat *getPImage (CvMat *maps, CvMat *index, int nVoxels, CvMat *pValueIn)
{
	CvMat *pValue = pValueIn;

	if (nVoxels < maps->cols)
	{
		fprintf (stderr, "getPImage (): invalid number of image voxels\n");
		return NULL;
	}
	
	if (pValueIn && ((pValueIn->rows != index->rows) || (pValueIn->cols != index->cols)))
	{
		fprintf (stderr, "getPImage (): index and p-value matrices must have same dimensions\n");
		return NULL;
	}

	if (maps->rows == 0)
	{
		fprintf (stderr, "getPImage ():  no group analysis results given\n");
		return NULL;
	}

	if (!pValue)
	{
		pValue = getPValue (maps);

		if (!pValue)
		{
			fprintf (stderr, "getPImage (): failed to compute p-values\n");
			return NULL;
		}
	}

	CvMat *pImage = cvCreateMat (nVoxels, 1, CV_32FC1);

	for (int i = 0; i < pImage->rows; ++ i)
	{
		cvSetReal1D (pImage, i, 1);
	}

	for (int i = 0; i < pValue->rows; ++ i)
	{
		int idx = static_cast <int> (cvGetReal1D (index, i)) - 1;
		cvSetReal1D (pImage, idx, cvGetReal1D (pValue, i) / maps->rows);
	}

	if (pValue && pValue != pValueIn)
	{
		cvReleaseMat (&pValue);
		pValue = NULL;
	}

	return pImage;
}


SBIA_ODVBA_NAMESPACE_END
