/*!
 * \file   SbiaOdvbaCreateNI.cpp
 * \brief  Command line tool which creates the neighborhood file.
 * \author Tianhao Zhang, revised by Andreas Schuh
 * \date   01/05/2011
 *
 * $Revision: 183 $
 * $Id: SbiaOdvbaCreateNI.cpp 183 2011-02-17 18:51:38Z schuha@UPHS.PENNHEALTH.PRV $
 *
 * <b>Last changed:</b>
 * $Author: schuha@UPHS.PENNHEALTH.PRV $
 * $Date: 2011-02-17 13:51:38 -0500 (Thu, 17 Feb 2011) $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For copyright information see Copyright section of project
 * ReadMe.html in the project's root directory.
 *
 * Contact: sbia-software@uphs.upenn.edu
 */


#include <getopt.h>

#include "SbiaOdvbaUtilities.h"


SBIA_ODVBA_NAMESPACE_USE;


//////////////////////////////////////////////////////////////////////////////
// usage / help
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Extract program name from first command line argument.
 *
 * \param argv Command line arguments.
 *
 * \return Program name without relative or absolute path to directory where
 *         the executable is located. Do not try to free the memory of the
 *         returned pointer! It just points to the first character of the
 *         program name within argv [0].        
 **/
const char *getProgName (char *argv[])
{
	const char *progName = argv [0];

	if (progName && *progName)
	{
		// go to last character of string
		while (*progName) ++progName;
		--progName;
		// rewind until beginning of string or
		// '/' (UNIX) or '\\' (WIN) is encountered
		while (progName != argv [0])
		{
			--progName;
			if (*progName == '/' || *progName == '\\')
			{
				++progName;
				break;
			}
		}
	}

	return progName;
}

/*!
 * \brief Prints version information.
 *
 * \param progName Name of program.
 */
void version (const char *progName)
{
	printf ("%s %s", progName, SBIA_ODVBA_VERSION);
#if SBIA_ODVBA_REVISION_VALID
	printf (" (Rev. %d)", SBIA_ODVBA_REVISION);
#endif // SBIA_ODVBA_REVISION_VALID
	printf ("\n");
}

/*!
 * \brief Prints usage information / help.
 *
 * \param progName Name of program.
 */
void usage (const char *progName)
{
	version (progName);
	printf ("\n");
	printf ("Description:\n");
	printf ("  This program randomly selects neighorhoods from all the ones.\n");
	printf ("\n");
	printf ("  T. Zhang and C. Davatzikos; ODVBA: Optimally-Dicriminative\n");
	printf ("  Voxel-Based Analysis\n");
	printf ("\n");
	printf ("Usage:\n");
	printf ("  %s [options] <subjects.txt> [<index.txt>]\n", progName);
	printf ("\n");
	printf ("Required options:\n");
	printf ("  <subjects.txt>      : The subjects list which specifies the input data given as NIfTI-1 images.\n");
	printf ("  [-s --sizeNI <real>]: Size of neighborhood in mm.\n");
	printf ("                        Required and used only if <NI.txt> file not provided.\n");
	printf ("  [-n --numNI <int>]  : Number of neighborhoods.\n");
	printf ("                        At most, the neighborhood around each non-zero voxel can be\n");
	printf ("                        considered, i.e., numNI <= m, where m is the number of non-zero voxels.\n");
	printf ("  [-e --numVox <int>] : Number of voxel in each neighborhood.\n");
	printf ("\n");
	printf ("Options:\n");
	printf ("  <index.txt>           : Location of non-zero voxel in the volumes.\n");
	printf ("  [-o --out <filename>] : Filename of the neighborhood file.\n");
	printf ("                          The default value is 'NI.txt'.\n");
	printf ("  [-h --help]           : Print help and exit.\n");
	printf ("  [-u --usage]          : Print usage information and exit.\n");
	printf ("  [-V --version]        : Print version information and exit.\n");
	printf ("  [-v --verbose]        : Enable verbose messages. Can be specified multiple times\n");
	printf ("                          to increase the verbosity.\n");
	printf ("\n");
	printf ("Example:\n");
	printf (" %s subjects.txt index.txt -n 10000 -e 400 -s 15\n", progName);
}

//////////////////////////////////////////////////////////////////////////////
// main
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Main.
 *
 * \param [in] argc Number of command line arguments.
 * \param [in] argv Command line arguments.
 */
int main (int argc, char **argv)
{
	const char *progName = getProgName (argv);
	bool        ok       = true;

	/**
	 * \struct long_options
	 * \brief  Structure of long program options.	 
	 */
	static struct option long_options [] =
	{
		{"out",     required_argument, NULL, 'o'},
		{"numNI",   required_argument, NULL, 'n'},
		{"numVox",  required_argument, NULL, 'e'},
		{"sizeNI",  required_argument, NULL, 's'},
		{"usage",   no_argument,       NULL, 'u'},
		{"help",    no_argument,       NULL, 'h'},
		{"version", no_argument,       NULL, 'V'},
		{"Version", no_argument,       NULL, 'V'},
		{"verbose", no_argument,       NULL, 'v'}
	};

	int c      = -1;
	int optidx = 0;

	// default options
	int         verbosity   = 0;        // verbosity of messages
	const char *sublistFile = NULL;     // input subjects list
	const char *indexFile   = NULL;     // input index file
	const char *NIFile      = "NI.txt"; // output neighborhood file
	int         numNI       = 0;        // number of neighborhoods
	double      sizNI       = 0;        // size of each neighborhood in mm
	int         numVox      = 0;        // number of voxels sampled from each neighborhood

	while ((c = getopt_long (argc, argv, "n:e:s:o:uhVv", long_options, &optidx)) != -1)
	{
		switch (c)
		{
		case 'o':
			NIFile = optarg;
			break;

		case 'n':
			numNI = atoi (optarg);
			break;

		case 'e':
			numVox = atoi (optarg);
			break;

		case 's':
			sizNI = atof (optarg);
			break;

		case 'u':
			// fall-through intended
		case 'h':
			usage (progName);
			exit (EXIT_SUCCESS);

		case 'V':
			version (progName);
			exit (EXIT_SUCCESS);

		case 'v':
			++ verbosity;
			break;

		case '?':
			// getopt_long already printed an error message
			usage (progName);
			exit (EXIT_FAILURE);
		}
	}


	argc -= optind;
	argv += optind;

	if (numNI == 0 || numVox == 0 || sizNI <= 0 || argc < 1 || argc > 2)
	{
		usage (progName);
		exit (EXIT_FAILURE);
	}

	sublistFile = argv [0];
	if (argc > 1 ) indexFile = argv [1];

	// read database
	CvMat         *data  = NULL; // image data of the n subjects
	CvMat         *index = NULL; // indices of "non-zero" voxels (template)
	int            n1    = 0;    // number of subjects belonging to group 1
	nifti_1_header hdr;          // image header of first input image

	printf ("Parsing subject list and reading image data\n");
	fflush (stdout);

	ok = ((data = readData  (sublistFile, &n1, &hdr)) != NULL);

	if (ok)
	{
		if (indexFile)
		{
			printf ("Reading index\n");
			fflush (stdout);

			ok = ((index = readCvMat (indexFile)) != NULL);
		}
		else
		{
			printf ("Generating index\n");
			fflush (stdout);

			ok = ((index = createIndex (data)) != NULL);
		}
	}

	// generating neighborhoods
	CvMat *NI = NULL;

	if (ok)
	{
		printf ("Generating neighborhoods\n");
		fflush (stdout);

		int sizNIx = static_cast <int> (floor (sizNI / hdr.pixdim [1])); // size of neighborhoods along x dimension
		int sizNIy = static_cast <int> (floor (sizNI / hdr.pixdim [2])); // size of neighborhoods along y dimension
		int sizNIz = static_cast <int> (floor (sizNI / hdr.pixdim [3])); // size of neighborhoods along z dimension

		int xyz_units = XYZT_TO_SPACE (hdr.xyzt_units);

		if (xyz_units == NIFTI_UNITS_MICRON)
		{
			sizNIx /= 1000;
			sizNIy /= 1000;
			sizNIz /= 1000;
		}
		else if (xyz_units == NIFTI_UNITS_METER)
		{
			sizNIx *= 1000;
			sizNIy *= 1000;
			sizNIz *= 1000;
		}
		
		if (sizNIx <= 0) sizNIx = 1;
		if (sizNIy <= 0) sizNIy = 1;
		if (sizNIz <= 0) sizNIz = 1;

		ok = ((NI = createNI (index, hdr.dim[1], hdr.dim[2], hdr.dim[3], sizNIx, sizNIy, sizNIz, numNI, numVox)) != NULL);
	}

	// write index
	if (ok)
	{
		printf ("Writing neighborhood(s) to file '%s'\n", NIFile);
		fflush (stdout);

		int fieldWidth = 0;
		int maxValue   = index->rows;

		while (maxValue > 0)
		{
			++ fieldWidth;
			maxValue = maxValue / 10;
		}

		char fmt [24];		
		sprintf (fmt, "%%%d.0f", fieldWidth);

		ok = writeCvMat (NIFile, NI, fmt);
	}

	// clean up
	if (NI)
	{
		cvReleaseMat (&NI);
		NI = NULL;
	}

	if (index)
	{
		cvReleaseMat (&index);
		index = NULL;
	}

	if (data)
	{
		cvReleaseMat (&data);
		data = NULL;
	}

	exit (ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
