##############################################################################
#
# \file   SbiaMacros.cmake
# \brief  Definition of CMake macros commonly used at SBIA.
# \author Andreas Schuh
# \date   01/05/10
#
# $Revision: 155 $
# $Id: SbiaMacros.cmake 155 2011-02-09 16:44:55Z schuha@UPHS.PENNHEALTH.PRV $
#
# <b>Last changed</b>
# $Author: schuha@UPHS.PENNHEALTH.PRV $
# $Date: 2011-02-09 11:44:55 -0500 (Wed, 09 Feb 2011) $
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# For copyright information see Copyright section of project
# ReadMe.html in the project's root directory.
#
# Contact: sbia-software@uphs.upenn.edu
#
##############################################################################

# ============================================================================
# Subversion
# ============================================================================

# ****************************************************************************
macro (sbia_get_svn_revision DIR VARIABLE)
  if (SBIA_CMD_SVN)
    execute_process (
      COMMAND ${SBIA_CMD_SVN} info ${DIR}
      OUTPUT_VARIABLE ${VARIABLE}
      OUTPUT_STRIP_TRAILING_WHITESPACE
    )

    string (REGEX REPLACE "^(.*\n)?Revision: ([^\n]+).*" "\\2" ${VARIABLE} "${${VARIABLE}}")
  else ()
    set (${VARIABLE} "")
  endif ()
endmacro (sbia_get_svn_revision)

# ============================================================================
# targets
# ============================================================================

# ****************************************************************************
macro (sbia_add_executable NAME)
  message (STATUS "Adding executable ${NAME}...")

  # add executable target
  add_executable (${NAME} ${ARGN})

  # prefix target output name
  if (SBIA_USE_OUTPUT_NAME_PREFIX)
    set_target_properties (
      ${NAME}
      PROPERTIES
        OUTPUT_NAME "${SBIA_OUTPUT_NAME_PREFIX}${NAME}"
    )
  endif ()

  # target version information
  set_target_properties (
    ${NAME}
    PROPERTIES
      VERSION ${PROJECT_VERSION}
      SOVERSION ${PROJECT_SOVERSION}
  )
  
  # install executable
  install (
    TARGETS ${NAME}
    DESTINATION bin
  )

  message (STATUS "Adding executable ${NAME}... - done")
endmacro (sbia_add_executable)

# ****************************************************************************
macro (sbia_add_library NAME)
  message (STATUS "Adding library ${NAME}...")

  # add library target
  add_library (${NAME} ${ARGN})

  # prefix target output name
  if (SBIA_USE_OUTPUT_NAME_PREFIX)
    set_target_properties (
      ${NAME}
      PROPERTIES
        OUTPUT_NAME "${SBIA_OUTPUT_NAME_PREFIX}${NAME}"
    )
  endif ()

  # target version information
  set_target_properties (
    ${NAME}
    PROPERTIES
      VERSION   "${PROJECT_VERSION}"
      SOVERSION "${PROJECT_SOVERSION}"
  )
  
  # install headers
  # \todo Write CMake code to install header files in ROOT/include

  # install library
  install (
    TARGETS ${NAME}
    DESTINATION lib
  )

  message (STATUS "Adding library ${NAME}... - done")
endmacro (sbia_add_library)

# ****************************************************************************
macro (sbia_add_script NAME)
  message (STATUS "Adding script ${NAME}...")

  # prefix target output name
  if (SBIA_USE_OUTPUT_NAME_PREFIX)
    set (TARGET_NAME "${SBIA_OUTPUT_NAME_PREFIX}${NAME}")
  else ()
    set (TARGET_NAME "${NAME}")
  endif ()

  # configure/copy script
  set (PROJECT_EXECUTABLE_DIR "${EXECUTABLE_OUTPUT_PATH}")
  configure_file (${CMAKE_CURRENT_SOURCE_DIR}/${NAME} ${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME} @ONLY)
  set (PROJECT_EXECUTABLE_DIR "${CMAKE_INSTALL_PREFIX}/bin")
  configure_file (${CMAKE_CURRENT_SOURCE_DIR}/${NAME} ${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}.install @ONLY)

  if (UNIX)
    execute_process (COMMAND chmod +x ${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME})
  endif ()

  # install script
  install (
    PROGRAMS    ${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}.install
    RENAME      ${TARGET_NAME}
    DESTINATION bin
  )

  # clear "local" variables
  set (TARGET_NAME)
  
  message (STATUS "Adding script ${NAME}... - done")
endmacro (sbia_add_script)

# ****************************************************************************
# \macro sbia_add_doc
# \brief Adds custom target to doc target for automatic API doc generation.
#
# This macro is used to add a custom target to the common doc target which
# is used to generate documentation from input files such as in particular
# source code files.
#
# \param GENERATOR Documentation generation tool. The arguments for the
#                  different generators follow this parameter and are documented
#                  below.
#
# Generator:  Doxygen
# 
# \param DOXY_TEMPLATE Template Doxyfile.
#
# Before the sbia_add_doc macro is called, the following CMake variables have
# to be set:
#
#   DOXYGEN_PROJECT_NAME     Project name.
#   DOXYGEN_PROJECT_NUMBER   Project version number.
#   DOXYGEN_INPUT            Input directories/files.
#   DOXYGEN_OUTPUT_DIRECTORY Output directory.
#   DOXYGEN_GENERATE_HTML    Either "YES" or "NO".
#   DOXYGEN_GENERATE_LATEX   Either "YES" or "NO".
#   DOXYGEN_GENERATE_RTF     Either "YES" or "NO".
#   DOXYGEN_GENERATE_MAN     Either "YES" or "NO".
#
macro (sbia_add_doc NAME GENERATOR)
  message (STATUS "Adding documentation ${NAME}...")

  if (TARGET ${NAME})
    message (ERROR "A target named ${NAME} already exists")
  endif ()

  # --------------------------------------------------------------------------
  # generator: doxygen
  if (${GENERATOR} MATCHES "Doxygen")
    if (NOT DOXYGEN_EXECUTABLE)
	  find_package (Doxygen REQUIRED)
	endif ()

    if (${ARGC} LESS 3)
      message (ERROR "No input Doxyfile specified")
    endif ()
    
    set (DOXY_TEMPLATE ${ARGV2})

    if (EXISTS ${DOXY_TEMPLATE})
      # configure Doxyfile
      set (DOXY_FILE ${PROJECT_BINARY_DIR}/Sbia${PROJECT_NAME}.doxy)
      configure_file (${DOXY_TEMPLATE} ${DOXY_FILE} @ONLY)

      # add doxygen as target
      add_custom_target (${NAME} ${DOXYGEN_EXECUTABLE} ${DOXY_FILE})

      # cleanup on "make clean"
      set (
        ADDITIONAL_CLEAN_DIRS
          ${DOXYGEN_OUTPUT_DIRECTORY}/html
          ${DOXYGEN_OUTPUT_DIRECTORY}/latex
          ${DOXYGEN_OUTPUT_DIRECTORY}/rtf
          ${DOXYGEN_OUTPUT_DIRECTORY}/man
      )

      set_property (
        DIRECTORY APPEND PROPERTY
          ADDITIONAL_MAKE_CLEAN_FILES ${ADDITIONAL_CLEAN_DIRS}
      )

      # add doxygen as dependency to doc target
      if (NOT TARGET doc)
        add_custom_target (doc)
      endif ()

      add_dependencies (doc ${NAME})

      # install documentation
      if (${DOXYGEN_GENERATE_HTML} MATCHES "YES")
        install (
          DIRECTORY
            ${DOXYGEN_OUTPUT_DIRECTORY}/html
          DESTINATION
            doc/API
        )
      endif ()

      if (${DOXYGEN_GENERATE_LATEX} MATCHES "YES")
        install (
          DIRECTORY
            ${DOXYGEN_OUTPUT_DIRECTORY}/latex
          DESTINATION
            doc/API
        )
      endif ()

      if (${DOXYGEN_GENERATE_RTF} MATCHES "YES")
        install (
          DIRECTORY
            ${DOXYGEN_OUTPUT_DIRECTORY}/rtf
          DESTINATION
            doc/API
        )
      endif ()

      if (${DOXYGEN_GENERATE_MAN} MATCHES "YES")
        install (
          DIRECTORY
            ${DOXYGEN_OUTPUT_DIRECTORY}/man
          DESTINATION
            doc/API
        )
      endif ()

      # clear "local" variables
      set (DOXY_FILE)
      set (ADDITIONAL_CLEAN_DIRS)

    else ()
      message (ERROR "Input Doxyfile ${DOXY_TEMPLATE} does not exist")
    endif ()
  else ()
    message (ERROR "Unknown documentation generator")
  endif ()

  message (STATUS "Adding documentation ${NAME}... - done")
endmacro (sbia_add_doc)
