/**
 * @file  utilities.h
 * @brief Common utility functions used by the ODVBA implementations.
 *
 * Copyright (c) 2010-2012 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */

#pragma once
#ifndef _SBIA_ODVBA_UTILITIES_h
#define _SBIA_ODVBA_UTILITIES_h


#include <cv.h>
#include <nifti1_io.h>

#include <sbia/odvba/config.h>


namespace sbia
{

namespace odvba
{


// ===========================================================================
// IO
// ===========================================================================

// ---------------------------------------------------------------------------
// read input datasets
// ---------------------------------------------------------------------------

/**
 * @brief Read image data of subjects.
 *
 * Read data according to the subject list in which the following information
 * is given: number of subjects in each group, directory of the data,
 * the name of each subject.
 *
 * @param [in]  filename Path of to subject list file.
 * @param [out] n1       Number of subjects belonging to group 1 (optional).
 * @param [out] hdr1     Header of first NIfTI-1 input image (optional).
 *
 * @returns Read data of all subjects, where the data of each subject is
 *          stored in a single column of the matrix.
 *          The data has to be released by the caller using cvReleaseMat().
 */
CvMat* read_data(const char*     filename,
                 int*            n1   = NULL,
                 nifti_1_header* hdr1 = NULL);

// ---------------------------------------------------------------------------
// normalize the inital data
// ---------------------------------------------------------------------------

/**
 * @brief Normalize the intial input image data.
 *
 * This function normalizes the input image data such that the mean value of
 * the image data is equal to the specified desired mean value.
 *
 * @param [in,out] data      The initial input data which will be normalized.
 * @param [in]     meanValue The mean value of the normalized data.
 *
 * @ingroup PublicAPI
 */
void norm_data(CvMat* data, const int meanValue);

// ---------------------------------------------------------------------------
// read/write OpenCV matrix
// ---------------------------------------------------------------------------

/**
 * @brief Read OpenCV matrix from file.
 *
 * @param [in] filename Path to file.
 * @param [in] hdr      Whether the file has a header containing matrix dimensions.
 *
 * @returns Read OpenCV matrix or NULL if an error occurred.
 *          The returned matrix has to be released using cvReleaseMat().
 */
CvMat* read_matrix(const char* filename, bool hdr = false);

/**
 * @brief Write OpenCV matrix to file.
 *
 * @param [in] filename Output filename.
 * @param [in] matrix   Matrix.
 * @param [in] fmt      Format string used with fprintf() to output single element.
 * @param [in] hdr      Whether the matrix dimension should be saved in a header.
 *
 * @returns Whether the output file was written successfully.
 */
bool write_matrix(const char* filename, const CvMat* matrix, const char* fmt = "%e", bool hdr = false);

/**
 * @brief Write OpenCV matrix as raw image.
 *
 * @param [in] filename Output filename.
 * @param [in] image    Image matrix. Can be either a 1D or 2D matrix.
 *
 * @returns Whether the output file was written successfully.
 */
bool write_raw_image(const char* filename, const CvMat* image);

/**
 * @brief Write OpenCV matrix as NIfTI-1 image.
 *
 * @param [in] filename Output filename (prefix).
 * @param [in] hdr      NIfTI-1 header.
 * @param [in] image    Image matrix. Can be either a 1D or 2D matrix.
 *                      The values of the matrix elements are written consecutively
 *                      do the output image, where the dimensions of the output
 *                      image are specified by the NIfTI-1 header. Hence,
 *                      the number of rows and columns of the input matrix are
 *                      ignored.
 *
 * @returns Whether the output image was written successfully.
 */
bool write_nifti_image(const char* filename, const nifti_1_header& hdr, const CvMat* image);

// ===========================================================================
// random numbers / permutations
// ===========================================================================

/**
 * @brief Generate a vector whose elments are random integers.
 *
 * @attention This function does not call srand() to initialize the random
 *            number generator. It simply calls rand() n times.
 *
 * @param [in] n Number of random integers.
 *
 * @returns Array of n random integers.
 *          The returned memory has to be released by the caller using delete [].
 */
int* random(int n);

/**
 * @brief Generate a random permutation.
 *
 * @attention This function does not call srand() to initialize the random
 *            number generator. It simply calls rand() n times.
 *
 * @param [in] n Upper bound of range (excluded).
 *
 * @returns Permutation of integers from 1 to n.
 *          The returned memory has to be released by the caller using delete [].
 */
int* randperm(int n);


} // namespace odvba

} // namespace sbia


#endif // _SBIA_ODVBA_UTILITIES_H
