/**
 * @file  odvba-index.cxx
 * @brief Command line tool which creates the index file (template).
 *
 * Copyright (c) 2010-2012 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */

#include <string>
#include <iostream>

#include <odvba/basis.h>
#include <odvba/odvba.h>
#include <odvba/utilities.h>


// acceptable in .cxx file
using namespace std;
using namespace basis;
using namespace odvba;


// ===========================================================================
// main
// ===========================================================================

// ---------------------------------------------------------------------------
int main(int argc, char* argv[])
{
    bool ok = true;

    // -----------------------------------------------------------------------
    // command-line argument constraints
    ExistingFileConstraint subjects_list_constraint("<subjects_list>");

    // -----------------------------------------------------------------------
    // command-line arguments
    PositionalArg sublist_file("subjects_list",
            "Text file which lists the NIfTI-1 image data files of the input subjects.",
            true, "", &subjects_list_constraint);

    StringArg index_file("", "index",
            "Name of the output index file.",
            false, "index.txt", "<file>");

    // -----------------------------------------------------------------------
    // parse command-line
    try {
        CmdLine cmd(// program identification
                    "odvba-index", PROJECT,
                    // program description
                    "This program determines the indices of voxels in the set"
                    " of input images whose average voxel value exceeds a"
                    " certain threshold. Hence, it determines the indices of"
                    " voxels relevant to the group analysis and outputs them"
                    " to a text file. The generated file can be used as input"
                    " for the main odvba command which performs the group"
                    " analysis.",
                    // example usage
                    "EXENAME subjects.txt --index idx.txt",
                    // version and copyright information
                    RELEASE, COPYRIGHT,
                    // license information
                    "See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.",
                    // contact
                    "SBIA Group <sbia-software at uphs.upenn.edu>");

        // The constructor of the CmdLine class has already added the standard
        // arguments --help, --helpshort, --helpxml, --helpman, and --version.

        cmd.add(index_file);
        cmd.add(sublist_file);

        cmd.parse(argc, argv);
    } catch (CmdLineException& e) {
        // invalid command-line specification
        cerr << e.error() << endl;
        exit(1);
    }

    // -----------------------------------------------------------------------
    // read input data
    CvMat *data = NULL;

    cout << "Parsing subject list and reading image data" << endl;
    cout.flush();
    ok = ((data = read_data(sublist_file.getValue().c_str())) != NULL);

    // -----------------------------------------------------------------------
    // create index
    CvMat *index = NULL;

    if (ok) {
        cout << "Generating index" << endl;
        cout.flush();
        ok = (index = create_index(data)) != NULL;
    }

    // -----------------------------------------------------------------------
    // write index
    if (ok) {
        cout << "Writing index to file " << index_file.getValue() << endl;
        cout.flush();
        ok = write_matrix(index_file.getValue().c_str(), index, "%.0f");
    }

    // -----------------------------------------------------------------------
    // clean up
    if (index) {
        cvReleaseMat(&index);
        index = NULL;
    }
    if (data) {
        cvReleaseMat(&data);
        data = NULL;
    }

    exit(ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
