/**
 * @file  CAlgorithm.h
 * @brief Header for CAlgorithm.cpp: contains common utility functions used by the ODVBA implementations.
 *
http://www.cbica.upenn.edu/sbia/software/ <br>
sbia-software@uphs.upenn.edu

Copyright (c) 2015 University of Pennsylvania. All rights reserved. <br>
See COPYING file or http://www.cbica.upenn.edu/sbia/software/license.html
 */
 

#ifndef _ALGORITHM_H
#define _ALGORITHM_H

#define True  1
#define False 0

#include <string.h>
#include <stdio.h>
#include <iostream>
#include <vector>
#include <sys/stat.h>
#include <cstdlib>
# include <sys/time.h>
#include <boost/any.hpp>
#include <algorithm>
#include <sys/types.h>
#include <dirent.h>
#include <nifti1_io.h>
#include <boost/numeric/ublas/matrix.hpp> 
#include <boost/numeric/ublas/matrix_proxy.hpp>
#include <boost/numeric/ublas/vector.hpp> 
#include <boost/numeric/ublas/vector_proxy.hpp> 
#include <boost/numeric/ublas/operation.hpp> 
#include <boost/numeric/ublas/operation_sparse.hpp> 
#include <boost/numeric/ublas/matrix_sparse.hpp> 
#include <boost/numeric/ublas/vector_sparse.hpp> 
#include <boost/numeric/ublas/symmetric.hpp>
#include <boost/numeric/ublas/io.hpp> 
#include <boost/numeric/bindings/atlas/cblas3.hpp>
#include <boost/numeric/bindings/atlas/cblas2.hpp>
#include <boost/numeric/bindings/traits/ublas_matrix.hpp>
#include <boost/numeric/ublas/io.hpp>
#include <cstdio>
#include <boost/numeric/bindings/atlas/clapack.hpp>
#include <boost/numeric/bindings/lapack/lapack.h>
#include <boost/numeric/bindings/lapack/syevx.hpp>
#include <boost/numeric/bindings/traits/traits.hpp>
#include <boost/numeric/bindings/traits/detail/utils.hpp>
#include <boost/numeric/bindings/traits/detail/array.hpp>
#include <boost/numeric/bindings/traits/ublas_matrix.hpp>
#include <boost/numeric/ublas/io.hpp>
#include <boost/numeric/ublas/matrix_proxy.hpp>

namespace ublas = boost::numeric::ublas;
namespace atlas = boost::numeric::bindings::atlas; 
namespace lapack = boost::numeric::bindings::lapack;
namespace traits = boost::numeric::bindings::traits;
namespace detail = boost::numeric::bindings::traits::detail;
using namespace std; 
 
#define EUCLIDEAN	0
#define HEATKERNEL	1
#define COSINE		2
#define	SIMPLE		3
#define MINVALUE 10e-5
#define MAXVALUE 10e+10

 
/**
\struct mapResults
\brief Results related to the maps

\var ublas::matrix<double> W1 Contains generalized eigenvectors of NDP, size n*k
\var ublas::matrix<double> map1 Contains the whole brain mapping calculated by each permutation
*/
typedef struct mapResults
{
  ublas::matrix<double> W1;
	ublas::matrix<double> map1;	
}mapResults;

/**
\struct Var_matrix
\brief the variances in the matrix computation

\var int m Number of subjects
\var int n Number of voxels in each neighborhood
\var int nSmp1 Number of subjects of group #1
\var int nSmp2 Number of subjects of group #2
\var ublas::vector<double> intraMean1 Mean of of group #1.
\var ublas::vector<double> intraMean2 Mean of group #2.
\var ublas::vector<double> E1 1\the number of subjects of group #1.
\var ublas::vector<double> E2 1\the number of subjects of group #2.
\var ublas::matrix<double> intraClass1 (shifted) samples in group #1. 
\var ublas::matrix<double> transintraClass1 transposed (shifted) samples in group #1. 
\var ublas::matrix<double> intraClass2 (shifted) samples in group #1. 
\var ublas::matrix<double> transintraClass2 transposed (shifted) samples in group #2. 
\var ublas::matrix<double> eintraCov1 Covariance of of group #1.
\var ublas::matrix<double> eintraCov2 Covariance of of group #2.
\var ublas::matrix<double> meanCov1 between-class scatter matrix
\var ublas::matrix<double,ublas::column_major> AA The matrix
\var ublas::matrix<float,ublas::column_major> The matrix A caculated by lapack functions
\var ublas::matrix<double> A1 A^+ 
\var ublas::matrix<double> A2 A^-
\var ublas::vector<double> a A^+ * w
\var ublas::vector<double> b (\mu*e)^2
\var ublas::vector<double> c A^- * w
\var ublas::vector<double> ad the multiplicative update
\var ublas::vector<double> W  the weight coffients
\var ublas::vector<double> weight Projected learning set using projection direction obtained via NDP.
\var ublas::vector<double> vot the discriminative degree.
\var double A_temp the element in the matrix A.
*/
typedef struct Var_matrix
{	
	int m;
  int n;
	int nSmp1;
	int nSmp2;
	ublas::vector<double> intraMean1;
	ublas::vector<double> intraMean2;
	ublas::vector<double> a ;
	ublas::vector<double> b;
	ublas::vector<double> c ;
	ublas::vector<double> ad ;
	ublas::vector<double> W ;
	ublas::vector<double> weight ;
	ublas::vector<double> vot ;
	ublas::vector<double> E1;
	ublas::vector<double> E2;
  ublas::matrix<double> intraClass1;
  ublas::matrix<double> transintraClass1;
  ublas::matrix<double> intraClass2;
	ublas::matrix<double> transintraClass2;
	ublas::matrix<double> eintraCov1;
	ublas::matrix<double> eintraCov2;
	ublas::matrix<double> meanCov1;
	ublas::matrix<double> A1;
	ublas::matrix<double> A2;
	ublas::matrix<float,ublas::column_major> A_rep_lapack;
	ublas::matrix<double,ublas::column_major> AA;
	double A_temp;
}Var_matrix;

/**
\struct Options
\brief the options of the algorithm

\var double aa Exponent phi of discrimination degree
\var double g \Gamma
\var int nNeighbor the Number of voxel in each neighborhood
\var int nSample the Number of subjects
\var int nPerm the number of permutations
\var int nn Number of neighborhoods
\fun Options() Constructor which initializes the data structures.
\fun ~Options() Destructor
*/
typedef struct Options
{
	double aa;
	double g;
	int nNeighbor;
	int nSample;
	int nPerm;
	int nn;
	Options()
  {
    nPerm=100;
    aa=1;
	g=0.0000001;
  }
  ~Options()
	{}
}Options;

/**
\struct MapParas
\brief the intermediate parameters on the projected space via weight coiffients obtained via NDP
*/
typedef struct MapParas
{
	double sum11;
  double sum22;
	double sum1;
  double sum2;
	double sw1;
  double sw2;
	double voti;
	double sw;
	double me1;
	double me2;
	double diff;
	double delta;
}MapParas;

/**
\struct ExDatabase
\brief experimental database, and ID of samples

\var long m number of effective voxels which are involved in computation.
\var long n number of samples.
\var int n1 number of subjects of group #1.
\var int n2 number of subjects of group #2.
\var long nn Number of neighborhoods.
\var int s Size of neighborhood.
\var ublas::matrix<double> NI neighborhoods
\var ublas::matrix<double> index index of effective voxels  
\var int NE Number of voxel in each neighborhood 
\fun ExDatabase() Constructor which initializes the data structures.
\fun ~ExDatabase() Destructor
*/
typedef struct ExDatabase
{
	long m;	     
	long n;
  int n1;
  int n2;	
	long nn;	 
	int s; 
	ublas::matrix<double> NI;
	ublas::matrix<double> index;
	int NE; 
  
  //! The default constructor
	ExDatabase()
  {
    nn = 10000;
    s = 15;
    NE  = 400;
  }
  ~ExDatabase()
	{}
}ExDatabase;

/**
\class transferData
\brief transfering data including reading and writing.

\var int  _DebugFlag Flag for debugging
\var char *_file_sublist the filename of subject list.
\var char **_file_name the filenames of all the subjects.
\var int _num_group1 number of subjects of group #1.
\var int _num_group2 number of subjects of group #1.
\var int  _xdim dimension of x direction.
\var int  _ydim dimension of y direction.
\var int  _zdim dimension of z direction.
\var int  _subject_id number of groups
\var nifti_1_header hdr header file of the first subject
\var char _data_root_dir[200] directory of samples
\var ublas::matrix<double> origin_data the data
\fun transferData() Constructor
\fun ~transferData() Destructor
\fun void Debug(char *) Prints debugging messages if debugging is enabled
\fun void SetInput (char *sublist,int subject_id) initialization
\fun void GetInput() locating the subjects. 
\fun void Read_nifti_Data() reading the nifti data
\fun bool write_nifti_img(ublas::matrix<double>& X,const char* name) writing the nifti data
*/

class transferData 
{
public:
 
  int  _DebugFlag;
  transferData();
  ~transferData();
  /// Prints debugging messages if debugging is enabled
  void Debug(char *);

public:
 
  char *_file_sublist;
  char **_file_name;  
  int _num_group1;
  int _num_group2;
  int  _xdim;
  int  _ydim;
  int  _zdim;
  int  _subject_id; // for leave one out test
  nifti_1_header hdr;
  char _data_root_dir[200];
  ublas::matrix<double> origin_data;
  void ReadData();
  /// Sets input for the feature selection routine
  void Read_nifti_Data();
  bool write_nifti_img(ublas::matrix<double>& X,const char* name);// write nifti data
  void SetInput (char *sublist,int subject_id);
  void GetInput();  
  void Initialize();
};


// ===========================================================================
// non-negative discriminative projection (NDP)
// ===========================================================================

/**
 * @brief Non-negative Discriminative Projection (NDP).
 *
 * Call this function with each learning set to get the matrix W.
 * @sa Section 2.1 of the MICCAI paper.
 *
 * @param [in]  X     An input learning set constructed from the neighborhoods
 *                    centered at a particular voxel. Each column of the matrix
 *                    corresponds to a subject, where the first n1 subjects belong
 *                    to group 1 and the remaining n2 = n - n1 subjects to group 2.
 *                    The columns of the learning set further correspond to the
 *                    subvolume vectors which were constructed by randomly sampling
 *                    the voxels neighborhood within the subject's original image.
 * @param [in]  Mat   The variances and parameters involved in the matrix computation.
 * @param [in]  aa    Exponent phi of discrimination degree
 *
 * @returns The vector w which stores the coffients for the input learning set
 *          (note that the result is also stored in tmp.w).
 *
 *
 */
ublas::vector<double> NDP(ublas::matrix<double>& X, Var_matrix mat, double aa);
// ---------------------------------------------------------------------------
// read data in txt format
// ---------------------------------------------------------------------------

/**
 * @brief Read txt data
 *
 * Read data according to the subject list in which the following information
 * is given: number of subjects in each group, directory of the data,
 * the name of each subject.
 *
 * @param [in]  X     a matrix to contain the data
 * @param [in]  name  Path of to subject list file.
 *
 * @returns txt data
 */
bool readData(ublas::matrix<double>& X,const char* name);

// ---------------------------------------------------------------------------
// write data in txt format
// ---------------------------------------------------------------------------
/**
 * @brief Write matrix to file.
 *
 * @param [in] X   Matrix.
 * @param [in] filename Output filename.
 *
 * @returns Whether the output file was written successfully.
 */
bool writeData(ublas::matrix<double>& X,const char* name);	

/**
 * @brief Write  matrix as raw image.
 *
 * @param [in] image    Image matrix. 
 * @param [in] filename Output filename.
 *
 * @returns Whether the output file was written successfully.
 */
bool writeimg(ublas::matrix<double>& X,const char* name);// write data

// ===========================================================================
// initialization
// ===========================================================================

/**
 * @brief Create index, i.e., the union of indices of non-zero voxels of the
 *        data of all subjects.
 *
 * Only voxels whose index is listed in the index data structure are considered
 * during the voxel-based analysis. The index can be considered as template
 * mask of the input volumes.
 *
 * @param [in] data Volume data of all subjects, where the each column
 *                  corresponds to one subject.
 *
 * @returns Index data structure or NULL on error.
 *
 */
ublas::matrix<double> create_index(ublas::matrix<double> data);

/**
 * @brief Create neighborhoods.
 *
 * This function first selects randomly numNI voxels of all non-zero voxels.
 * Then, numVox voxels are randomly drawn from the voxels which lie within a
 * neighborhood of size (2 * ni_size_x + 1) * (2 * ni_size_y + 1) * (2 * ni_size_z + 1)
 * centered at each of the voxels selected. If the random selection contained
 * invalid voxels, i.e., ones with voxel value zero, the corresponding entry
 * in the output matrix is set to the invalid index zero.
 *
 * All the neighborhoods corresponding to all the voxels can be used to get the 
 * discriminative coefficients, but it is really computationally expensive. By 
 * selecting random neighborhoods, the number of the involved learning sets can 
 * be reduced and therefore improve the performance of the group analysis.
 *
 * @param [in] index     The index of non-zero voxels.
 * @param [in] dim_x     The x dimension of the image.
 * @param [in] dim_y     The y dimension of the image.
 * @param [in] dim_z     The z dimension of the image.
 * @param [in] ni_size_x Radius of each neighborhood along x dimension (excluding center voxel).
 * @param [in] ni_size_y Radius of each neighborhood along y dimension (excluding center voxel).
 * @param [in] ni_size_z Radius of each neighborhood along z dimension (excluding center voxel).
 * @param [in] ni_num    Number of neighborhoods.
 * @param [in] vox_num   Number of voxels randomly selected from each neighborhood.
 *
 * @returns Neighborhood data structure or NULL on error.
 *
 */
ublas::matrix<double> create_ni(ublas::matrix<double> index,
                 const int dim_x, const int dim_y, const int dim_z,
                 const int ni_size_x, const int ni_size_y, const int ni_size_z,
                 const int ni_num, const int vox_num);


/**
 * @brief Create matrix for computation from the raw data.
 *
 * @param [in] X         The matrix .
 * @param [in] data      The raw data.
 * @param [in] index     The index of non-zero voxels.

 *
 * @returns the matrix containing the non-zero voxels of the  data of all subjects.
 *
 */
void createX(ublas::matrix<double>& X,ublas::matrix<double> data,ublas::matrix<double>& index);	

// ===========================================================================
// random numbers / permutations
// ===========================================================================

/**
 * @brief Generate a vector whose elments are random integers.
 *
 * @attention This function does not call srand() to initialize the random
 *            number generator. It simply calls rand() n times.
 *
 * @param [in] n Number of random integers.
 *
 * @returns Array of n random integers.
 *          The returned memory has to be released by the caller using delete [].
 */

int* random(int n);  

/**
 * @brief Generate a random permutation.
 *
 * @attention This function does not call srand() to initialize the random
 *            number generator. It simply calls rand() n times.
 *
 * @param [in] n Upper bound of range (excluded).
 *
 * @returns Permutation of integers from 1 to n.
 *          The returned memory has to be released by the caller using delete [].
 */
int* randperm(int n);
Var_matrix init_mat(Var_matrix matt, int n,int n1, int n2, int m, int N, int type);

// ===========================================================================
// determination of each voxel's statistic
// ===========================================================================

/**
 * @brief Determine statistics of each voxel using NDP.
 *
 * @sa Section 2.2 of the MICCAI paper.
 *
 *
 * @param [in]  X        Input database including original image data,
 *                       template mask (index) and the matrix specifying the
 *                       pre-computed neighborhoods.
 * @param [in]  mapW     Results related to the maps
 * @param [in]  opt      Parameters of the ODVBA method.
 * @param [in]  mat      The variances in the matrix computation.
 * @param [in]  MRIdata  Experimental data and the related parameters.
 * @param [in]  mParas   The intermediate parameters on the projected space via weight coiffients obtained via NDP.
 * 
 * @returns the map which has each voxel's statistic
 *
 */
ublas::matrix<double> compMap(mapResults mapW, ublas::matrix<double>& X,  Options opt,  Var_matrix  mat, ExDatabase MRIdata, ublas::matrix<double> fea,  MapParas mParas);

/**
 * @brief creating the array to contain the subject list
 *
 *
 *
 * @param [in]  x     Number of subjects
 * @param [in]  y     length of the filename
 * 
 * @returns the array to contain the subject list
 *
 */
char **sub_list(long x, long y);

/**
 * @brief free the memory of the array to contain the subject list
 *
 *
 *
 * @param [in]  x     Number of subjects
 * @param [in]  y     length of the filename
 * 
 */
void free_sub_list(char  **t, long x, long y);


// ---------------------------------------------------------------------------
// normalize the inital data
// ---------------------------------------------------------------------------

/**
 * @brief Normalize the intial input image data.
 *
 * This function normalizes the input image data such that the mean value of
 * the image data is equal to the specified desired mean value.
 *
 * @param [in,out] data      The initial input data which will be normalized.
 * @param [in]     meanValue The mean value of the normalized data.
 *
 */
void norm_data(ublas::matrix<double> &data, const int meanValue);


#endif
