///////////////////////////////////////////////////////////////////////////////////////
// BrainTumorRegistration.cpp
// Developed by Dongjin Kwon
///////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2014 University of Pennsylvania. All rights reserved.
// See http://www.cbica.upenn.edu/sbia/software/license.html or COYPING file.
//
// Contact: SBIA Group <sbia-software at uphs.upenn.edu>
///////////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "MyUtils.h"
#include "Volume.h"
#include "VolRegOpt.h"
#include "VolumeBP.h"
//
#include "itkImageFunction.h"
#include "vnl/algo/vnl_qr.h"
#include "vnl/vnl_math.h"
//
#if defined(WIN32) || defined(WIN64)
#include <io.h>
#else
#define _dup dup
#define _dup2 dup2
#endif
//
#include <omp.h>
//
#include <itkImageFileReader.h>
#include "itkMultiResolutionJointSegmentationRegistration.h"
#include "itkJointSegmentationRegistrationFunction.h"


#if (!defined(WIN32) && !defined(WIN64)) || !defined(_DEBUG)
char EVALUATE_Q_PATH[1024];
char HOPSPACK_PATH[1024];
char FLIRT_PATH[1024];
char CONVERT_XFM_PATH[1024];
char SIMULATOR_PATH[1024];
char RESAMPLE_IMAGE_PATH[1024];
char RESAMPLE_DEFORMATION_FIELD_PATH[1024];
char REVERSE_DEFORMATION_FIELD_PATH[1024];
char WARP_IMAGE_PATH[1024];
char CONCATENATE_FIELD_PATH[1024];
#endif


#define PI						3.1415926535897932384626433832795
#define eps						1e-8
#define epss					1e-32
#define MaxNumberOFTumorSeeds	10
#define MaxNumberOFPoints		100

typedef vnl_matrix_fixed<double, NumberOfImageChannels, NumberOfImageChannels> VarianceType;
typedef vnl_vector_fixed<double, NumberOfImageChannels> MeanType;
typedef std::vector<VarianceType> VarianceVectorType;
typedef std::vector<MeanType> MeanVectorType;

const unsigned int ImageDimension = 3;

typedef float FixedPixelType;
typedef float MovingPixelType;
typedef itk::Vector<float, ImageDimension> VectorType;

typedef itk::Image<VectorType, ImageDimension> DeformationFieldType;
typedef itk::Image<FixedPixelType, ImageDimension> FixedImageType; 
typedef itk::Image<MovingPixelType, ImageDimension> MovingImageType;  

typedef itk::ImageFileReader<FixedImageType> FixedImageReaderType;
typedef std::vector<FixedImageReaderType::Pointer> FixedImageReaderVectorType;

typedef itk::ImageFileReader<MovingImageType> MovingImageReaderType;
typedef itk::ImageFileWriter<MovingImageType> MovingImageFileWriterType;
typedef std::vector<MovingImageReaderType::Pointer> MovingImageReaderVectorType;

typedef itk::ImageFileReader<DeformationFieldType> DeformationFieldReaderType;
typedef itk::ImageFileWriter<DeformationFieldType> DeformationFieldWriterType;

typedef itk::MultiResolutionJointSegmentationRegistration
                <FixedImageType, MovingImageType, DeformationFieldType,
                 MovingPixelType, NumberOfImageChannels, NumberOfPriorChannels>
        JointSegmentationRegistrationType;

typedef JointSegmentationRegistrationType::SegmentationRegistrationType
        JointSegmentationRegistrationFilteType;

typedef JointSegmentationRegistrationFilteType::JointSegmentationRegistrationFunctionType
        JointSegmentationRegistrationFunctionType;


BOOL TransformPoints(char* scan_image_file, char* scan_atlas_reg_image_file, char* scan_to_atlas_mat_file, double (*scan_seeds_info)[4], double (*scan_atlas_reg_seeds_info)[4], int scan_seeds_num, int check_orientation);
BOOL TransformPoints2(char* scan_image_file, char* scan_align_image_file, char* scan_atlas_reg_image_file, char* scan_align_mat_file, char* scan_to_atlas_mat_file, double (*scan_seeds_info)[4], double (*scan_atlas_reg_seeds_info)[4], int scan_seeds_num, int check_orientation);
BOOL GetPoints(char* scan_image_file, double (*scan_points_info)[4], double (*scan_points_info_out)[4], int scan_points_num, int check_orientation);
BOOL ModifyPriors(FVolume* priors, double seeds_info[MaxNumberOFTumorSeeds][4], int seeds_num);

BOOL MakeScalesFile(const char* scales_file);
BOOL MakeSimulatorInputFile(const char* simulator_input_file, double o_dx, double o_dy, double o_dz, int x, int y, int z, double dx, double dy, double dz);
BOOL MakeHOPSFile(const char* hops_file, double gxc, double gyc, double gzc, double T, const char* mask_file, const char* scan2_atlas_reg_abnor_mask_file,
	const char* scan0_atlas_reg_image_list, const char* scan0_atlas_reg_init_prior_list, const char* scan0_means_file, const char* scan0_variances_file, const char* scan0_h_hdr_file, const char* scan0_h2_hdr_file,
	const char* scan2_atlas_reg_image_list, const char* scan2_atlas_reg_posterior_list, const char* label_map_s_img_file, 
	const char* simulator_input_file, const char* scales_file, const char* out_folder, const char* tmp_folder, char* solution_file, int max_eval, BOOL optim_xyz, int num_hop_threads);

BOOL LoadMeansAndVariances(const char* means_file, const char* variances_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector);
BOOL SaveMeansAndVariances(const char* means_file, const char* variances_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector);
BOOL InitializeMeansAndVariances(const char* means_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector);
BOOL InitializeMeansAndVariancesFromPoints(double (*scan_points_info)[4], int scan_points_num, char (*image_files)[1024], MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, int check_orientation, int* b_valid_label);
BOOL UpdateMeansAndVariances(FVolume* vd, FVolume* posteriors, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, BOOL bFirst);
BOOL ComputePosteriors(FVolume* vd, FVolume* priors, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, FVolume* posteriors);
BOOL ComputePosteriors5(FVolume* vd, FVolume* priors, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, FVolume* posteriors);
BOOL GetFieldFromXFM(char* m2f_xfm_file, char* m_file, char* f_file, char* m2f_v_hdr_file);
BOOL ConcatenateAllFields(char* p0_file, char* p2_file, char* a0_hdr_file, char* h_hdr_file, char* u_hdr_file, char* a2_inv_hdr_file, char* f_hdr_file);
BOOL ConcatenateAllFields(char* p0_file, char* p2_file, char* a0_hdr_file, char* hu_hdr_file, char* a2_inv_hdr_file, char* f_hdr_file);

BOOL EstimateDeformationFieldJSR(char atlas_reg_image_files[NumberOfImageChannels][1024], char atlas_reg_prior_files[NumberOfPriorChannels][1024], 
	MeanVectorType* pMeanVector, char* h_hdr_file, char* means_file, char* variances_file, char* cost_file, int iter, int max_iter, int num_itk_threads, bool bEstimateTumorRegionOnly, float fTumorRegionThreshold = 10e-6, bool bElastic = true);
//
extern BOOL UpdateDeformationFieldSyM(FVolume* vd1, FVolume* vd2, FVolume& sc1, FVolume& sc2, FVolume& tu1, FVolume& tu2, FVolume& ab1, FVolume& ab2, 
	FVolume& h_x, FVolume& h_y, FVolume& h_z, FVolume& h_r_x, FVolume& h_r_y, FVolume& h_r_z,
	char* name, int dmode, int mmode, int m_reg_iter = 2, int m_sub_reg_iter = 3, int m_sub_reg_iter_rep = 1, bool mesh_iter = false,
	bool save_int = true, bool save_pyrd = true, float lambda_D = 1.0, float lambda_P = 0.2,
	float _gamma = -1, float _alpha_O1 = -1, float _d_O1 = -1, float _alpha_O2 = -1, float _d_O2 = -1);
//
extern BOOL SmoothField(FVolume& vx, FVolume& vy, FVolume& vz, FVolume& vx_g, FVolume& vy_g, FVolume& vz_g, float sig);
extern BOOL ReverseDeformationField(FVolume& vx, FVolume& vy, FVolume& vz, FVolume& vxr, FVolume& vyr, FVolume& vzr);
extern BOOL ReverseField(FVolume& vx, FVolume& vy, FVolume& vz, FVolume& vxr, FVolume& vyr, FVolume& vzr);

BOOL MakeProbForTumorAndEdema2(FVolume& priors_TU, FVolume& priors_ED, FVolume& priors_CSF, 
#ifdef USE_11A_PRIORS
	FVolume& priors_VT, 
#endif
	FVolume& priors_GM, FVolume& priors_WM, 
#ifdef USE_WARPED_BG
	FVolume& priors_BG,
#endif
	FVolume& tumor_density);


void version()
{
	printf("==========================================================================\n");
	printf("PORTR: Pre-Operative and post-Recurrence brain Tumor Registration\n");
#ifdef SW_VER
	printf("  Version %s\n", SW_VER);
#endif
#ifdef SW_REV
	printf("  Revision %s\n", SW_REV);
#endif
	printf("Copyright (c) 2014 University of Pennsylvania. All rights reserved.\n");
	printf("See http://www.cbica.upenn.edu/sbia/software/license.html or COPYING file.\n");
	printf("==========================================================================\n");
}
void usage()
{
	printf("\n");
	printf("Parameters:\n\n");
	printf("  --scan0_list, -sl0 <file>         Text file listing Scan 0, one per line.\n");
	printf("  --scan2_list, -sl2 <file>         Text file listing Scan 2, one per line.\n");
	printf("                                    It is recommended to use T1, T1-CE, T2, and FLAIR\n");
	printf("                                    patient scans as input.\n");
	printf("  --scan0_seed, -ss0 <file>         Tumor seed information for Scan 0.\n");
	printf("  --scan2_seed, -ss2 <file>         Tumor seed information for Scan 2.\n");
	printf("                                    The format of the seed file is:\n");
	printf("                                    \n");
	printf("                                      <n: number of seeds>\n");
	printf("                                      <x1> <y1> <z1> <d1>\n");
	printf("                                      ...\n");
	printf("                                      <xn> <yn> <zn> <dn>\n");
	printf("                                    \n");
	printf("                                    where <xi> <yi> <zi> is the RAS coordinate of ith seed and\n");
	printf("                                    <di> is the approximated diameter of ith tumor.\n");
	printf("                                    \n");
	printf("  --scan0_point, -sp0 <file>        One sample point for each class of Scan 0. This option overrides -sm0.\n");
	printf("  --scan2_point, -sp2 <file>        One sample point for each class of Scan 2. This option overrides -sm2.\n");
	printf("                                    The format of the point file is:\n");
	printf("                                    \n");
	printf("                                      <Class 1>\n");
	printf("                                      <x1> <y1> <z1>\n");
	printf("                                      ...\n");
	printf("                                      <Class n>\n");
	printf("                                      <xn> <yn> <zn>\n");
	printf("                                    \n");
	printf("                                    where <Class i> is the name of ith tissue class and\n");
	printf("                                    <xi> <yi> <zi> is the RAS coordinate of ith tissue class.\n");
	printf("                                    \n");
	printf("  --atlas_folder, -af <dir>         The directory path for the atlas.\n");
	printf("  --outputdir, -d <dir>             The directory path for output files.\n");
	printf("\n");
	printf("Optional Parameters:\n\n");
	printf("  --workingdir, -w <dir>            The directory path for intermediate files.\n");
#ifdef USE_ALIGN_SCAN
	printf("  --align_scan, -as <int>           1 if align Scan 2 to Scan 0 before registration, 0 do nothing.\n");
#endif
	printf("  --atlas_template, -at <file>      The atlas template file.\n");
	printf("  --atlas_prior, -ap <file>         The list file for atlas priors.\n");
	printf("  --atlas_mask, -am <file>          The atlas mask file.\n");
	printf("  --atlas_label, -al <file>         The atlas lable map.\n");
	printf("  --num_omp_threads, -not <int>     The number of OpemMP threads.\n");
	printf("  --num_itk_threads, -nit <int>     The number of ITK threads.\n");
	printf("  --num_hop_threads, -nht <int>     The number of HOPSPACK threads.\n");
	printf("\n");
	printf("  --verbose, -v <int>               if greater than 0, print log messages: 1 (default), 2 (detailed).\n");
	printf("  --help, -h                        Print help and exit.\n");
	printf("  --usage, -u                       Print help and exit.\n");
	printf("  --version, -V                     Print version information and exit.\n");
	printf("\n");
	printf("Obsolete Parameters:\n\n");
	printf("  --scan0_mean, -sm0 <file>         Initial mean values for Scan 0.\n");
	printf("  --scan2_mean, -sm2 <file>         Initial mean values for Scan 2.\n");
	printf("\n\n");
	//*
	printf("Example:\n\n");
	printf("  1. with 8 parameters\n\n");
	printf("  PORTR -sl0 [scan0_image_list] -sl2 [scan2_image_list] -ss0 [scan0_seed_info] -ss2 [scan2_seed_info] -sp0 [scan0_point_info] -sp2 [scan2_point_info] -af [atlas_folder] -out [out_folder]\n");
	printf("\n");
	printf("  2. with 11 parameters\n\n");
	printf("  PORTR -sl0 [scan0_image_list] -sl2 [scan2_image_list] -ss0 [scan0_seed_info] -ss2 [scan2_seed_info] -sp0 [scan0_point_info] -sp2 [scan2_point_info] -at [atlas_template_file] -ap [atlas_prior_list] -am [atlas_mask_file] -al [atlas_label_map_s] -out [out_folder]\n");
	printf("\n\n");
	//*/
}


int main(int argc, char* argv[])
{
	// intput: scan 0, scan 2, tumor center, T (time or tumor volume)
	// output: deform field 0 to 2, posteriors for scan 0 and scan 2, means and variances for scan 0 and scan 2
	char scan0_image_list[1024] = {0,};
    char scan2_image_list[1024] = {0,};
	char scan0_init_mean[1024] = {0,};
	char scan2_init_mean[1024] = {0,};
	char scan0_init_seed[1024] = {0,};
	char scan2_init_seed[1024] = {0,};
	char scan0_init_point[1024] = {0,};
	char scan2_init_point[1024] = {0,};
	char atlas_template_file[1024] = {0,};
	char atlas_prior_list[1024] = {0,};
	char atlas_mask_file[1024] = {0,};
	char atlas_label_map_s[1024] = {0,};
	char out_folder[1024] = {0,};
	char tmp_folder[1024] = {0,};
    //char deform_field_file[1024] = {0,};
	char atlas_folder[1024] = {0,};
	char out2_folder[1024] = {0,};
	char tmp2_folder[1024] = {0,};
	//
	float lambda_D = 1.0;
	float lambda_P = 0.2;
	bool save_int = false;
	bool save_pyrd = false;
	bool delete_tmp_folder = true;
	bool b_scan0_point_info = false;
	bool b_scan2_point_info = false;
	//
	int check_orientation = 0;
	//
	int b_scan0_valid_label[NumberOfPriorChannels];
	int b_scan2_valid_label[NumberOfPriorChannels];
	//
#if defined(WIN32) || defined(WIN64)
	int num_omp_threads = 3;
	int num_itk_threads = 3;
	int num_hop_threads = 3;
#else
	int num_omp_threads = 3;
	int num_itk_threads = 3;
	// consider memory limitations
	int num_hop_threads = 3;
#endif
	//
	bool scan0_seed_of_scan0 = false;
	bool scan0_reinit_means = false;
	bool align_scan = false;
	//
	bool b_run_test = false;
	int verbose = 1;

	// parse command line
	{
		int i;
		if (argc == 1) {
			printf("use option -h or --help for help\n");
			exit(EXIT_FAILURE);
		}
		if (argc == 2) {
			if        (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0) {
				version();
				usage();
				exit(EXIT_FAILURE);
			} else if (strcmp(argv[1], "-u") == 0 || strcmp(argv[1], "--usage") == 0) {
				version();
				usage();
				exit(EXIT_FAILURE);
			} else if (strcmp(argv[1], "-V") == 0 || strcmp(argv[1], "--version") == 0) {
				version();
				exit(EXIT_FAILURE);
			} else {
				printf("error: wrong arguments\n");
				printf("use option -h or --help for help\n");
				exit(EXIT_FAILURE);
			}
		}
		for (i = 1; i < argc; i++) {
			if (argv[i] == NULL || argv[i+1] == NULL) {
				printf("error: not specified argument\n");
				printf("use option -h or --help for help\n");
				exit(EXIT_FAILURE);
			}
			if        (strcmp(argv[i], "-sl0" ) == 0 || strcmp(argv[i], "--scan0_list"      ) == 0) { sprintf(scan0_image_list   , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-sl2" ) == 0 || strcmp(argv[i], "--scan2_list"      ) == 0) { sprintf(scan2_image_list   , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-sm0" ) == 0 || strcmp(argv[i], "--scan0_mean"      ) == 0) { sprintf(scan0_init_mean    , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-sm2" ) == 0 || strcmp(argv[i], "--scan2_mean"      ) == 0) { sprintf(scan2_init_mean    , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-sp0" ) == 0 || strcmp(argv[i], "--scan0_point"     ) == 0) { 
				sprintf(scan0_init_point, "%s", argv[i+1]); b_scan0_point_info = true;
				i++;
			} else if (strcmp(argv[i], "-sp2" ) == 0 || strcmp(argv[i], "--scan2_point"     ) == 0) { 
				sprintf(scan2_init_point, "%s", argv[i+1]); b_scan2_point_info = true;
				i++;
			} else if (strcmp(argv[i], "-ss0" ) == 0 || strcmp(argv[i], "--scan0_seed"      ) == 0) { sprintf(scan0_init_seed    , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-ss2" ) == 0 || strcmp(argv[i], "--scan2_seed"      ) == 0) { sprintf(scan2_init_seed    , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-af"  ) == 0 || strcmp(argv[i], "--atlas_folder"    ) == 0) { sprintf(atlas_folder       , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-at"  ) == 0 || strcmp(argv[i], "--atlas_template"  ) == 0) { sprintf(atlas_template_file, "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-ap"  ) == 0 || strcmp(argv[i], "--atlas_prior_list") == 0) { sprintf(atlas_prior_list   , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-am"  ) == 0 || strcmp(argv[i], "--atlas_mask"      ) == 0) { sprintf(atlas_mask_file    , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-al"  ) == 0 || strcmp(argv[i], "--atlas_label_map" ) == 0) { sprintf(atlas_label_map_s  , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-d"   ) == 0 || strcmp(argv[i], "--outputdir"       ) == 0) { sprintf(out_folder         , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-w"   ) == 0 || strcmp(argv[i], "--workingdir"      ) == 0) { sprintf(tmp_folder         , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-d2"  ) == 0 || strcmp(argv[i], "--outputdir2"      ) == 0) { sprintf(out2_folder        , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-w2"  ) == 0 || strcmp(argv[i], "--workingdir2"     ) == 0) { sprintf(tmp2_folder        , "%s", argv[i+1]); i++;
			} else if (strcmp(argv[i], "-lD"  ) == 0 || strcmp(argv[i], "--lambda_D"        ) == 0) { 
				lambda_D = (float)atof(argv[i+1]);
				i++;
			} else if (strcmp(argv[i], "-lP"  ) == 0 || strcmp(argv[i], "--lambda_P"        ) == 0) { 
				lambda_P = (float)atof(argv[i+1]);
				i++;
			} else if (strcmp(argv[i], "-si"  ) == 0 || strcmp(argv[i], "--save_int"        ) == 0) { 
				if ((int)atoi(argv[i+1]) == 0) {
					save_int = false;
				} else {
					save_int = true;
				}
				i++;
			} else if (strcmp(argv[i], "-sp"  ) == 0 || strcmp(argv[i], "--save_pyrd"       ) == 0) {
				if ((int)atoi(argv[i+1]) == 0) {
					save_pyrd = false;
				} else {
					save_pyrd = true;
				}
				i++;
			} else if (strcmp(argv[i], "-ss0" ) == 0 || strcmp(argv[i], "--scan0_seed_of_scan0") == 0) { 
				if ((int)atoi(argv[i+1]) == 0) {
					scan0_seed_of_scan0 = false;
				} else {
					scan0_seed_of_scan0 = true;
				}
				i++;
			} else if (strcmp(argv[i], "-srm" ) == 0 || strcmp(argv[i], "--scan0_reinit_means") == 0) { 
				if ((int)atoi(argv[i+1]) == 0) {
					scan0_reinit_means = false;
				} else {
					scan0_reinit_means = true;
				}
				i++;
#ifdef USE_ALIGN_SCAN
			} else if (strcmp(argv[i], "-as" ) == 0 || strcmp(argv[i], "--align_scan") == 0) { 
				if ((int)atoi(argv[i+1]) == 0) {
					align_scan = false;
				} else {
					align_scan = true;
				}
				i++;
#endif
			} else if (strcmp(argv[i], "-rt" ) == 0 || strcmp(argv[i], "--run_test") == 0) { 
				if ((int)atoi(argv[i+1]) == 0) {
					b_run_test = false;
				} else {
					b_run_test = true;
				}
				i++;
			} else if (strcmp(argv[i], "-not" ) == 0 || strcmp(argv[i], "--num_omp_threads" ) == 0) {
				num_omp_threads = atoi(argv[i+1]);
				i++;
			} else if (strcmp(argv[i], "-nit" ) == 0 || strcmp(argv[i], "--num_itk_threads" ) == 0) {
				num_itk_threads = atoi(argv[i+1]);
				i++;
			} else if (strcmp(argv[i], "-nht" ) == 0 || strcmp(argv[i], "--num_hop_threads" ) == 0) {
				num_hop_threads = atoi(argv[i+1]);
				i++;
			} else if (strcmp(argv[i], "-v"   ) == 0 || strcmp(argv[i], "--verbose"         ) == 0) { verbose = atoi(argv[i+1]); i++;
			} else {
				printf("error: %s is not recognized\n", argv[i]);
				printf("use option -h or --help for help\n");
				exit(EXIT_FAILURE);
			}
		}
		//
		if (atlas_folder[0] != 0) {
			char atlas_folder_full[1024] = {0,};
			//
			if (strrchr(atlas_folder, '/') == NULL && strrchr(atlas_folder, '\\') == NULL) {
				char sPath[1024];
				if (!GetModulePath(sPath)) {
					printf("GetModulePath failed\n");
					exit(EXIT_FAILURE);
				}
				str_divide_footer(sPath, sPath, NULL, DIR_SEP_C);
				str_divide_footer(sPath, sPath, NULL, DIR_SEP_C);
				//
				sprintf(atlas_folder_full, "%s%s%s", sPath, DIR_SEP, atlas_folder);
			} else {
				strcpy(atlas_folder_full, atlas_folder);
			}
			//
			sprintf(atlas_template_file, "%s%sjakob_stripped_with_cere_lps_256256128.nii.gz", atlas_folder_full, DIR_SEP);
			if (!IsFileExist(atlas_template_file)) {
				sprintf(atlas_template_file, "%s%sJHU_1m_lps_T1.nii.gz", atlas_folder_full, DIR_SEP);
				if (!IsFileExist(atlas_template_file)) {
					printf("error: couldn't find atlas_template_file\n");
					exit(EXIT_FAILURE);
				}
			}
			//printf("atlas_template_file found: %s\n", atlas_template_file);
			sprintf(atlas_prior_list, "%s%satlas_priors11.lst", atlas_folder_full, DIR_SEP);
			sprintf(atlas_mask_file, "%s%smask_256256128.nii.gz", atlas_folder_full, DIR_SEP);
			sprintf(atlas_label_map_s, "%s%slabel_646464.img", atlas_folder_full, DIR_SEP);
		}
		if (atlas_folder[0] == 0 && (atlas_template_file[0] == 0 || atlas_prior_list[0] == 0 || atlas_mask_file[0] == 0 || atlas_label_map_s[0] == 0)) {
			char sPath[1024];
			if (!GetModulePath(sPath)) {
				printf("GetModulePath failed\n");
				exit(EXIT_FAILURE);
			}
			str_divide_footer(sPath, sPath, NULL, DIR_SEP_C);
			str_divide_footer(sPath, sPath, NULL, DIR_SEP_C);
			//
			sprintf(atlas_template_file, "%s%s%s%sjakob_stripped_with_cere_lps_256256128.nii.gz", sPath, DIR_SEP, "atlas9", DIR_SEP);
			if (!IsFileExist(atlas_template_file)) {
				printf("error: couldn't find atlas_template_file (%s)\n", atlas_template_file);
				exit(EXIT_FAILURE);
			}
			sprintf(atlas_prior_list, "%s%s%s%satlas_priors11.lst", sPath, DIR_SEP, "atlas9", DIR_SEP);
			sprintf(atlas_mask_file, "%s%s%s%smask_256256128.nii.gz", sPath, DIR_SEP, "atlas9", DIR_SEP);
			sprintf(atlas_label_map_s, "%s%s%s%slabel_646464.img", sPath, DIR_SEP, "atlas9", DIR_SEP);
		}
		if (scan0_image_list[0] == 0 || scan2_image_list[0] == 0 || (scan0_init_mean[0] == 0 && b_scan0_point_info == false) || (scan2_init_mean[0] == 0 && b_scan2_point_info == false) || scan0_init_seed[0] == 0 ||
			atlas_template_file[0] == 0 || atlas_prior_list[0] == 0 || atlas_mask_file[0] == 0 || atlas_label_map_s[0] == 0 || out_folder[0] == 0)
		{
			printf("error: essential arguments are not specified\n");
			printf("use option -h or --help for help\n");
			exit(EXIT_FAILURE);
		}
		if (tmp_folder[0] == 0) {
			char tmpn[1024] = {0,};
#if defined(WIN32) || defined(WIN64)
			sprintf(tmpn, "tmp_XXXXXX");
			if (mktemp(tmpn) == NULL) {
				sprintf(tmpn, "tmp");
			}			
			sprintf(tmp_folder, "%s%s%s", out_folder, DIR_SEP, tmpn);
#else
			char *tmpn_res = NULL;
			if (getenv("SBIA_TMPDIR") != NULL) {
				sprintf(tmp_folder, "%s", getenv("SBIA_TMPDIR"));
				CreateDirectory(tmp_folder, NULL);
				//
				sprintf(tmpn, "%s%stmp_XXXXXX", getenv("SBIA_TMPDIR"), DIR_SEP);
				tmpn_res = mkdtemp(tmpn);
				if (tmpn_res != NULL) {
					sprintf(tmp_folder, "%s", tmpn_res);
				} else {
					// making temporary directory failed: let's use subdir of out_folder
					sprintf(tmp_folder, "%s%stmp", out_folder, DIR_SEP);
				}
			} else if (getenv("USER") != NULL) {
				sprintf(tmp_folder, "%stmp%s%s", DIR_SEP, DIR_SEP, getenv("USER"));
				CreateDirectory(tmp_folder, NULL);
				//
				sprintf(tmpn, "%stmp%s%s%stmp_XXXXXX", DIR_SEP, DIR_SEP, getenv("USER"), DIR_SEP);
				tmpn_res = mkdtemp(tmpn);
				if (tmpn_res != NULL) {
					sprintf(tmp_folder, "%s", tmpn_res);
				} else {
					// making temporary directory failed: let's use subdir of out_folder
					sprintf(tmp_folder, "%s%stmp", out_folder, DIR_SEP);
				}
			} else {
				// making temporary directory failed: let's use subdir of out_folder
				sprintf(tmp_folder, "%s%stmp", out_folder, DIR_SEP);
			}
#endif
		} else {
			delete_tmp_folder = false;
		}
		if (out2_folder[0] == 0) {
			sprintf(out2_folder, "%s", out_folder);
		}
		if (tmp2_folder[0] == 0) {
			sprintf(tmp2_folder, "%s", tmp_folder);
		}
		if (!b_scan0_point_info) {
			for (i = 0; i < NumberOfPriorChannels; i++) {
				b_scan0_valid_label[i] = 1;
			}
		} else {
			for (i = 0; i < NumberOfPriorChannels; i++) {
				b_scan0_valid_label[i] = 0;
			}
			b_scan0_valid_label[BG] = 1;
		}
		if (!b_scan2_point_info) {
			for (i = 0; i < NumberOfPriorChannels; i++) {
				b_scan2_valid_label[i] = 1;
			}
		} else {
			for (i = 0; i < NumberOfPriorChannels; i++) {
				b_scan2_valid_label[i] = 0;
			}
			b_scan2_valid_label[BG] = 1;
		}
	}

#if (!defined(WIN32) && !defined(WIN64)) || !defined(_DEBUG)
	{
		char MODULE_PATH[1024];
#if 0
		str_strip_file(argv[0], MODULE_PATH);
#else
		if (!GetModulePath(MODULE_PATH)) {
			printf("GetModulePath failed\n");
			exit(EXIT_FAILURE);
		}
#endif
		//
#if defined(WIN32) || defined(WIN64)
		sprintf(EVALUATE_Q_PATH, "%sEvaluateQ.exe", MODULE_PATH);
		//
		if (!b_run_test) {
			if (!FindExecutableInPath("HOPSPACK_main_threaded.exe", MODULE_PATH, HOPSPACK_PATH))  { printf("error: HOPSPACK_main_threaded.exe is not existing.\n"); exit(EXIT_FAILURE); }
			if (!FindExecutableInPath("flirt.exe", MODULE_PATH, FLIRT_PATH))					  { printf("error: flirt.exe is not existing.\n"); exit(EXIT_FAILURE); }
			if (!FindExecutableInPath("convert_xfm.exe", MODULE_PATH, CONVERT_XFM_PATH))		  { printf("error: convert_xfm.exe is not existing.\n"); exit(EXIT_FAILURE); }
			if (!FindExecutableInPath("ForwardSolverDiffusion.exe", MODULE_PATH, SIMULATOR_PATH)) { printf("error: ForwardSolverDiffusion.exe is not existing.\n"); exit(EXIT_FAILURE); }
		} else {
			sprintf(HOPSPACK_PATH, "%sHOPSPACK_main_threaded.exe", MODULE_PATH);
			sprintf(FLIRT_PATH, "%sflirt.exe", MODULE_PATH);
			sprintf(CONVERT_XFM_PATH, "%sconvert_xfm.exe", MODULE_PATH);
			sprintf(SIMULATOR_PATH, "%sForwardSolverDiffusion.exe", MODULE_PATH);
		}
		//
		sprintf(RESAMPLE_IMAGE_PATH, "%sResampleImage.exe", MODULE_PATH);
		sprintf(RESAMPLE_DEFORMATION_FIELD_PATH, "%sResampleDeformationField.exe", MODULE_PATH);
		sprintf(REVERSE_DEFORMATION_FIELD_PATH, "%sReverseDeformationField.exe", MODULE_PATH);
		sprintf(WARP_IMAGE_PATH, "%sWarpImage.exe", MODULE_PATH);
		sprintf(CONCATENATE_FIELD_PATH, "%sConcatenateFields.exe", MODULE_PATH);
#else
		sprintf(EVALUATE_Q_PATH, "%sEvaluateQ", MODULE_PATH);
		//
		if (!b_run_test) {
			if (!FindExecutableInPath("HOPSPACK_main_threaded", MODULE_PATH, HOPSPACK_PATH))  { printf("error: HOPSPACK_main_threaded is not existing.\n"); exit(EXIT_FAILURE); }
			if (!FindExecutableInPath("flirt", MODULE_PATH, FLIRT_PATH))					  { printf("error: flirt is not existing.\n"); exit(EXIT_FAILURE); }
			if (!FindExecutableInPath("convert_xfm", MODULE_PATH, CONVERT_XFM_PATH))		  { printf("error: convert_xfm is not existing.\n"); exit(EXIT_FAILURE); }
			if (!FindExecutableInPath("ForwardSolverDiffusion", MODULE_PATH, SIMULATOR_PATH)) { printf("error: ForwardSolverDiffusion is not existing.\n"); exit(EXIT_FAILURE); }
		} else {
			sprintf(HOPSPACK_PATH, "%sHOPSPACK_main_threaded", MODULE_PATH);
			sprintf(FLIRT_PATH, "%sflirt", MODULE_PATH);
			sprintf(CONVERT_XFM_PATH, "%sconvert_xfm", MODULE_PATH);
			sprintf(SIMULATOR_PATH, "%sForwardSolverDiffusion", MODULE_PATH);
		}
		//
		sprintf(RESAMPLE_IMAGE_PATH, "%sResampleImage", MODULE_PATH);
		sprintf(RESAMPLE_DEFORMATION_FIELD_PATH, "%sResampleDeformationField", MODULE_PATH);
		sprintf(REVERSE_DEFORMATION_FIELD_PATH, "%sReverseDeformationField", MODULE_PATH);
		sprintf(WARP_IMAGE_PATH, "%sWarpImage", MODULE_PATH);
		sprintf(CONCATENATE_FIELD_PATH, "%sConcatenateFields", MODULE_PATH);
#endif
	}
#endif

#ifdef USE_ALIGN_SCAN
	char scan2_align_image_files[NumberOfImageChannels][1024];
	char scan2_align_mat_file[1024];
	char scan2_align_hdr_file[1024];
	char scan2_align_inv_mat_file[1024];
	char scan2_align_inv_hdr_file[1024];
	char scan0_to_scan2_align_hdr_file[1024];
	char scan2_align_to_scan0_hdr_file[1024];
#endif
	//
	char scan0_image_files[NumberOfImageChannels][1024];
	char scan2_image_files[NumberOfImageChannels][1024];
	char scan0_image_masked_files[NumberOfImageChannels][1024];
	char scan2_image_masked_files[NumberOfImageChannels][1024];
	//
	char atlas_prior_files[NumberOfPriorChannels][1024];
	char scan2_to_atlas_mat_file[1024];
	char scan2_to_atlas_hdr_file[1024];
	char atlas_to_scan2_mat_file[1024];
	char atlas_to_scan2_hdr_file[1024];
	char scan0_to_scan2_hdr_file[1024];
	char scan2_to_scan0_hdr_file[1024];
	//
	char scan0_atlas_reg_image_list[1024];
	char scan0_atlas_reg_image_files[NumberOfImageChannels][1024];
	char scan0_atlas_reg_image_masked_list[1024];
	char scan0_atlas_reg_image_masked_files[NumberOfImageChannels][1024];
	char scan2_atlas_reg_image_list[1024];
	char scan2_atlas_reg_image_files[NumberOfImageChannels][1024];
	char scan2_atlas_reg_image_masked_list[1024];
	char scan2_atlas_reg_image_masked_files[NumberOfImageChannels][1024];
	//
	char scan0_atlas_reg_label_map_file[1024];
	char scan2_atlas_reg_label_map_file[1024];
	char scan2_atlas_reg_label_map_s_file[1024];
	char scan2_atlas_reg_label_map_s_hdr_file[1024];
	char scan2_atlas_reg_label_map_s_img_file[1024];
	char scan0_atlas_reg_prior_map_file[1024];
	char scan0_atlas_reg_prior_map_s_file[1024];
	char scan0_atlas_reg_prior_map_s_hdr_file[1024];
	char scan0_atlas_reg_prior_map_s_img_file[1024];
	//
	char scan0_atlas_reg_tumor_mask_file[1024];
	char scan2_atlas_reg_tumor_mask_file[1024];
	char scan0_atlas_reg_tumor_prob_file[1024];
	char scan2_atlas_reg_tumor_prob_file[1024];
	char scan0_atlas_reg_abnor_mask_file[1024];
	char scan2_atlas_reg_abnor_mask_file[1024];
	char scan0_atlas_reg_abnor_prob_file[1024];
	char scan2_atlas_reg_abnor_prob_file[1024];
	//
	char scan0_init_means_file[1024];
	char scan0_init_variances_file[1024];
	char scan0_atlas_reg_init_prior_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_init_posterior_files[NumberOfPriorChannels][1024];
	char scan0_h_init_hdr_file[1024];
	//
	char scan0_means_file[1024];
	char scan0_variances_file[1024];
	char scan0_atlas_reg_prior_list[1024];
	char scan0_atlas_reg_prior_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_prior_s_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_mass_prior_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_warp_prior_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_posterior_list[1024];
	char scan0_atlas_reg_posterior_files[NumberOfPriorChannels][1024];
	//
	char scan2_init_means_file[1024];
	char scan2_init_variances_file[1024];
	char scan2_means_file[1024];
	char scan2_variances_file[1024];
	char scan2_atlas_reg_prior_list[1024];
	char scan2_atlas_reg_prior_files[NumberOfPriorChannels][1024];
	char scan2_atlas_reg_warp_prior_files[NumberOfPriorChannels][1024];
	char scan2_atlas_reg_posterior_list[1024];
	char scan2_atlas_reg_posterior_files[NumberOfPriorChannels][1024];
	char scan2_atlas_reg_posterior_s_files[NumberOfPriorChannels][1024];
	//
	char scan0_u_hdr_file[1024];
	char scan0_u_inv_hdr_file[1024];
	char scan0_h_hdr_file[1024];
	char scan0_h2_hdr_file[1024];
	//
	char scan2_u_hdr_file[1024];
	char scan2_h_hdr_file[1024];
	//
	char scales_file[1024];
	char simulator_input_file[1024];
	//
	double scan0_seeds_info[MaxNumberOFTumorSeeds][4];
	double scan2_seeds_info[MaxNumberOFTumorSeeds][4];
	double scan0_atlas_reg_seeds_info[MaxNumberOFTumorSeeds][4];
	double scan2_atlas_reg_seeds_info[MaxNumberOFTumorSeeds][4];
	int scan0_seeds_num, scan2_seeds_num;
	//
	double scan0_points_info[MaxNumberOFPoints][4];
	double scan2_points_info[MaxNumberOFPoints][4];
	int scan0_points_num = 0;
	int scan2_points_num = 0;
	//
	char scan0_prior_list[1024];
	char scan0_prior_files[NumberOfPriorChannels][1024];
	char scan0_posterior_list[1024];
	char scan0_posterior_files[NumberOfPriorChannels][1024];
	char scan0_label_map_file[1024];
#ifdef USE_ALIGN_SCAN
	char scan2_align_prior_files[NumberOfPriorChannels][1024];
	char scan2_align_posterior_files[NumberOfPriorChannels][1024];
#endif
	char scan2_prior_list[1024];
	char scan2_prior_files[NumberOfPriorChannels][1024];
	char scan2_posterior_list[1024];
	char scan2_posterior_files[NumberOfPriorChannels][1024];
	char scan2_label_map_file[1024];
	//
	BVolume atlas_mask;
	//FVolume atlas_priors[NumberOfPriorChannels];
	//
	FVolume scan0_atlas_reg_images[NumberOfImageChannels];
	FVolume scan0_atlas_reg_images_masked[NumberOfImageChannels];
	FVolume scan2_atlas_reg_images[NumberOfImageChannels];
	FVolume scan2_atlas_reg_images_masked[NumberOfImageChannels];
	//
	BVolume scan0_atlas_reg_label_map;
	BVolume scan2_atlas_reg_label_map;
	BVolume scan2_atlas_reg_label_map_s;
	BVolume scan0_atlas_reg_prior_map;
	BVolume scan0_atlas_reg_prior_map_s;
	//
	BVolume scan0_atlas_reg_tumor_mask;
	BVolume scan2_atlas_reg_tumor_mask;
	BVolume scan0_atlas_reg_abnor_mask;
	BVolume scan2_atlas_reg_abnor_mask;
	FVolume scan0_atlas_reg_tumor_prob;
	FVolume scan2_atlas_reg_tumor_prob;
	FVolume scan0_atlas_reg_abnor_prob;
	FVolume scan2_atlas_reg_abnor_prob;
	//
	FVolume scan0_atlas_reg_priors[NumberOfPriorChannels];
	FVolume scan0_atlas_reg_priors_s[NumberOfPriorChannels];
	FVolume scan0_atlas_reg_mass_priors[NumberOfPriorChannels];
	FVolume scan0_atlas_reg_warp_priors[NumberOfPriorChannels];
	FVolume scan0_atlas_reg_posteriors[NumberOfPriorChannels];
	//
	FVolume scan2_atlas_reg_priors[NumberOfPriorChannels];
	FVolume scan2_atlas_reg_warp_priors[NumberOfPriorChannels];
	FVolume scan2_atlas_reg_posteriors[NumberOfPriorChannels];
	FVolume scan2_atlas_reg_posteriors_s[NumberOfPriorChannels];
	//
	MeanVectorType scan0_MeanVector, scan2_MeanVector;
	VarianceVectorType scan0_VarianceVector, scan2_VarianceVector;
	//
	RVolume scan0_u_x, scan0_u_y, scan0_u_z;
	RVolume scan0_u_inv_x, scan0_u_inv_y, scan0_u_inv_z;
	RVolume scan0_h_x, scan0_h_y, scan0_h_z;
	RVolume scan2_u_x, scan2_u_y, scan2_u_z;
	RVolume scan2_h_x, scan2_h_y, scan2_h_z;
	//
	char szCmdLine[2048];
	int i, j, k, l, m, n, c;
	//
	int scan0_jsr_max_iter = 3;

#ifdef _DEBUG
	/*
	{
		int tmp;
		// Get the current bits
		tmp = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
		//
		_CrtSetDbgFlag(_CRTDBG_CHECK_DEFAULT_DF);
	}
	//*/
#endif

	{
		char str_tmp[1024];
		FILE* fp;

		sprintf(str_tmp, "%s%sscan0_to_scan2.mhd", out2_folder, DIR_SEP);
		fp = fopen(str_tmp, "rb");
		if (fp != NULL) {
			fclose(fp);
			exit(EXIT_FAILURE);
		}

		sprintf(str_tmp, "del %s%ss0_h2u*.*", tmp2_folder, DIR_SEP);
		ExecuteProcess(str_tmp);
	}
		
	// set intermediate folders & files
	CreateDirectory((char*)tmp_folder, NULL);
	CreateDirectory((char*)out_folder, NULL);
	CreateDirectory((char*)tmp2_folder, NULL);
	CreateDirectory((char*)out2_folder, NULL);
	//
	SetCurrentDirectory((char*)tmp_folder);

	/////////////////////////////////////////////////////////////////////////////
	sprintf(g_trace_file, "%s%slog.txt", out2_folder, DIR_SEP);
	if (g_fp_trace != NULL) {
		fclose(g_fp_trace);
		g_fp_trace = NULL;
	}
	g_bTrace = FALSE;
	g_bTraceStdOut = (verbose>=1)?TRUE:FALSE;
	g_verbose = verbose;
	/////////////////////////////////////////////////////////////////////////////

	/////////////////////////////////////////////////////////////////////////////
	TRACE("==========================================================================\n");
	TRACE("PORTR: Pre-Operative and post-Recurrence brain Tumor Registration\n");
#ifdef SW_VER
	TRACE("  Version %s\n", SW_VER);
#endif
#ifdef SW_REV
	TRACE("  Revision %s\n", SW_REV);
#endif
	TRACE("Copyright (c) 2014 University of Pennsylvania. All rights reserved.\n");
	TRACE("See http://www.cbica.upenn.edu/sbia/software/license.html or COPYING file.\n");
	TRACE("==========================================================================\n");
	/////////////////////////////////////////////////////////////////////////////
	TRACE("\n");
	/////////////////////////////////////////////////////////////////////////////
	if (!b_run_test) {
		TRACE("EVALUATE_Q_PATH = %s\n", EVALUATE_Q_PATH);
		//
		TRACE("HOPSPACK_PATH = %s\n", HOPSPACK_PATH);
		TRACE("FLIRT_PATH = %s\n", FLIRT_PATH);
		TRACE("CONVERT_XFM_PATH = %s\n", CONVERT_XFM_PATH);
		TRACE("SIMULATOR_PATH = %s\n", SIMULATOR_PATH);
		//
		TRACE("RESAMPLE_IMAGE_PATH = %s\n", RESAMPLE_IMAGE_PATH);
		TRACE("RESAMPLE_DEFORMATION_FIELD_PATH = %s\n", RESAMPLE_DEFORMATION_FIELD_PATH);
		TRACE("REVERSE_DEFORMATION_FIELD_PATH = %s\n", REVERSE_DEFORMATION_FIELD_PATH);
		TRACE("WARP_IMAGE_PATH = %s\n", WARP_IMAGE_PATH);
		TRACE("CONCATENATE_FIELD_PATH = %s\n", CONCATENATE_FIELD_PATH);
		//
		if (!IsFileExist(EVALUATE_Q_PATH))					{ TRACE("error: %s is not existing.\n", EVALUATE_Q_PATH); exit(EXIT_FAILURE); }
		//
		if (!IsFileExist(HOPSPACK_PATH))					{ TRACE("error: %s is not existing.\n", HOPSPACK_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(FLIRT_PATH))						{ TRACE("error: %s is not existing.\n", FLIRT_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(CONVERT_XFM_PATH))					{ TRACE("error: %s is not existing.\n", CONVERT_XFM_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(SIMULATOR_PATH))					{ TRACE("error: %s is not existing.\n", SIMULATOR_PATH); exit(EXIT_FAILURE); }
		//
		if (!IsFileExist(RESAMPLE_IMAGE_PATH))				{ TRACE("error: %s is not existing.\n", RESAMPLE_IMAGE_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(RESAMPLE_DEFORMATION_FIELD_PATH))	{ TRACE("error: %s is not existing.\n", RESAMPLE_DEFORMATION_FIELD_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(REVERSE_DEFORMATION_FIELD_PATH))	{ TRACE("error: %s is not existing.\n", REVERSE_DEFORMATION_FIELD_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(WARP_IMAGE_PATH))					{ TRACE("error: %s is not existing.\n", WARP_IMAGE_PATH); exit(EXIT_FAILURE); }
		if (!IsFileExist(CONCATENATE_FIELD_PATH))			{ TRACE("error: %s is not existing.\n", CONCATENATE_FIELD_PATH); exit(EXIT_FAILURE); }
	}
	/////////////////////////////////////////////////////////////////////////////
	TRACE("\n");
	/////////////////////////////////////////////////////////////////////////////
	TRACE("out_folder = %s\n", out_folder);
	TRACE("tmp_folder = %s\n", tmp_folder);
	TRACE2("delete_tmp_folder = %d\n", (delete_tmp_folder)?1:0);
	/////////////////////////////////////////////////////////////////////////////
	TRACE("\n");
	/////////////////////////////////////////////////////////////////////////////
#ifdef _OPENMP
	if (num_omp_threads > 0) {
		omp_set_num_threads(num_omp_threads);
	}
	TRACE("num_omp_threads = %d\n", num_omp_threads);
#else
	TRACE("OpenMP is not supported.\n");
#endif
	TRACE("num_itk_threads = %d\n", num_itk_threads);
	TRACE("num_hop_threads = %d\n", num_hop_threads);
	/////////////////////////////////////////////////////////////////////////////
	TRACE("\n");
	/////////////////////////////////////////////////////////////////////////////
	TRACE("atlas_template_file = %s\n", atlas_template_file);
	TRACE("atlas_prior_list = %s\n", atlas_prior_list);
	TRACE("atlas_mask_file = %s\n", atlas_mask_file);
	TRACE("atlas_label_map_s = %s\n", atlas_label_map_s);
	//
	TRACE2("lambda_D = %f\n", lambda_D);
	TRACE2("lambda_P = %f\n", lambda_P);
	//
	TRACE2("scan0_seed_of_scan0 = %d\n", (scan0_seed_of_scan0)?1:0);
	TRACE2("scan0_reinit_means = %d\n", (scan0_reinit_means)?1:0);
#ifdef USE_ALIGN_SCAN
	TRACE2("align_scan = %d\n", (align_scan)?1:0);
#endif
	/////////////////////////////////////////////////////////////////////////////
	TRACE("\n");
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// load files
	/////////////////////////////////////////////////////////////////////////////
	{
		char str_path[1024];
		char str_tmp[1024];
		FILE* fp;
		//
		str_strip_file((char*)scan0_image_list, str_path);
		fp = fopen(scan0_image_list, "r");
		if (fp == NULL) {
			TRACE("Cannot open %s\n", scan0_image_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfImageChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(scan0_image_files[i], "%s%s", str_path, str_tmp);
		}
		fclose(fp);
		str_strip_file((char*)scan2_image_list, str_path);
		fp = fopen(scan2_image_list, "r");
		if (fp == NULL) {
			TRACE("Cannot open %s\n", scan2_image_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfImageChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(scan2_image_files[i], "%s%s", str_path, str_tmp);
		}
		fclose(fp);
		str_strip_file((char*)atlas_prior_list, str_path);
		fp = fopen(atlas_prior_list, "r");
		if (fp == NULL) {
			TRACE("Cannot open %s\n", atlas_prior_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(atlas_prior_files[i], "%s%s", str_path, str_tmp);
		}
		fclose(fp);
		//
		{
			fp = fopen(scan0_init_seed, "r");
			if (fp == NULL) {
				TRACE("Cannot open %s\n", scan0_init_seed);
				exit(EXIT_FAILURE);
			}
			fscanf(fp, "%s", str_tmp);
			scan0_seeds_num = atoi(str_tmp);
			TRACE("scan0_seeds_num = %d\n", scan0_seeds_num);
			for (i = 0; i < scan0_seeds_num; i++) {
				fscanf(fp, "%lf %lf %lf %lf", &scan0_seeds_info[i][0], &scan0_seeds_info[i][1], &scan0_seeds_info[i][2], &scan0_seeds_info[i][3]);
				TRACE("seed %d: %f %f %f %f\n", i, scan0_seeds_info[i][0], scan0_seeds_info[i][1], scan0_seeds_info[i][2], scan0_seeds_info[i][3]);
			}
			fclose(fp);
			fp = fopen(scan2_init_seed, "r");
			if (fp == NULL) {
				TRACE("Cannot open %s\n", scan2_init_seed);
				exit(EXIT_FAILURE);
			}
			fscanf(fp, "%s", str_tmp);
			scan2_seeds_num = atoi(str_tmp);
			TRACE("scan2_seeds_num = %d\n", scan2_seeds_num);
			for (i = 0; i < scan2_seeds_num; i++) {
				fscanf(fp, "%lf %lf %lf %lf", &scan2_seeds_info[i][0], &scan2_seeds_info[i][1], &scan2_seeds_info[i][2], &scan2_seeds_info[i][3]);
				TRACE("seed %d: %f %f %f %f\n", i, scan2_seeds_info[i][0], scan2_seeds_info[i][1], scan2_seeds_info[i][2], scan2_seeds_info[i][3]);
			}
			fclose(fp);
		}
		//
		if (b_scan0_point_info) {
			fp = fopen(scan0_init_point, "r");
			if (fp == NULL) {
				TRACE("Cannot open %s\n", scan0_init_point);
				exit(EXIT_FAILURE);
			}
			scan0_points_num = 0;
			while (TRUE) {
				int res;
				char class_id[1024];
				double fx, fy, fz;
				//
				res = fscanf(fp, "%s %lf %lf %lf", class_id, &fx, &fy, &fz);
				if (res == 0 || res == EOF) { break; }
				//
				for (j = 0; j < NumberOfPriorChannels; j++) {
#if defined(USE_11A_PRIORS)
					if ((strcmp(class_id, label[j]) == 0) ||
						(strcmp(class_id, label2[j]) == 0) ||
						(strcmp(class_id, label3[j]) == 0) ||
						(strcmp(class_id, label4[j]) == 0)) {
#else
					if (strcmp(class_id, label[j]) == 0) {
#endif
						scan0_points_info[scan0_points_num][0] = fx;
						scan0_points_info[scan0_points_num][1] = fy;
						scan0_points_info[scan0_points_num][2] = fz;
						scan0_points_info[scan0_points_num][3] = j;
						scan0_points_num++;
						if (scan0_points_num > MaxNumberOFPoints) {
							TRACE("scan0_points_num = %d > %d\n", scan0_points_num, MaxNumberOFPoints);
							exit(EXIT_FAILURE);
						}
						//
						b_scan0_valid_label[j] = 1;
						//
						break;
					}
				}
				if (j == NumberOfPriorChannels) {
					TRACE("class id is wrong");
					exit(EXIT_FAILURE);
				}
			}
			for (i = 0; i < scan0_points_num; i++) {
				TRACE("point %d: %s %f %f %f\n", i, label[(int)(scan0_points_info[i][3] + 0.1)], scan0_points_info[i][0], scan0_points_info[i][1], scan0_points_info[i][2]);
			}
			fclose(fp);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			TRACE2("b_scan0_valid_label[%s] = %d\n", label[i], b_scan0_valid_label[i]);
		}
		if (b_scan2_point_info) {
			fp = fopen(scan2_init_point, "r");
			if (fp == NULL) {
				TRACE("Cannot open %s\n", scan2_init_point);
				exit(EXIT_FAILURE);
			}
			scan2_points_num = 0;
			while (TRUE) {
				int res;
				char class_id[1024];
				double fx, fy, fz;
				//
				res = fscanf(fp, "%s %lf %lf %lf", class_id, &fx, &fy, &fz);
				if (res == 0 || res == EOF) { break; }
				//
				for (j = 0; j < NumberOfPriorChannels; j++) {
#if defined(USE_11A_PRIORS)
					if ((strcmp(class_id, label[j]) == 0) ||
						(strcmp(class_id, label2[j]) == 0) ||
						(strcmp(class_id, label3[j]) == 0) ||
						(strcmp(class_id, label4[j]) == 0)) {
#else
					if (strcmp(class_id, label[j]) == 0) {
#endif
						scan2_points_info[scan2_points_num][0] = fx;
						scan2_points_info[scan2_points_num][1] = fy;
						scan2_points_info[scan2_points_num][2] = fz;
						scan2_points_info[scan2_points_num][3] = j;
						scan2_points_num++;
						if (scan2_points_num > MaxNumberOFPoints) {
							TRACE("scan2_points_num = %d > %d\n", scan2_points_num, MaxNumberOFPoints);
							exit(EXIT_FAILURE);
						}
						//
						b_scan2_valid_label[j] = 1;
						//
						break;
					}
				}
				if (j == NumberOfPriorChannels) {
					TRACE("class id is wrong");
					exit(EXIT_FAILURE);
				}
			}
			for (i = 0; i < scan2_points_num; i++) {
				TRACE("point %d: %s %f %f %f\n", i, label[(int)(scan2_points_info[i][3] + 0.1)], scan2_points_info[i][0], scan2_points_info[i][1], scan2_points_info[i][2]);
			}
			fclose(fp);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			TRACE2("b_scan2_valid_label[%s] = %d\n", label[i], b_scan2_valid_label[i]);
		}
		//
#ifdef USE_ALIGN_SCAN
		sprintf(scan2_align_mat_file, "%s%sscan2_align.mat", tmp_folder, DIR_SEP);
		sprintf(scan2_align_hdr_file, "%s%sscan2_align.mhd", tmp_folder, DIR_SEP);
		sprintf(scan2_align_inv_mat_file, "%s%sscan2_align_inv.mat", tmp_folder, DIR_SEP);
		sprintf(scan2_align_inv_hdr_file, "%s%sscan2_align_inv.mhd", tmp_folder, DIR_SEP);
		sprintf(scan0_to_scan2_align_hdr_file, "%s%sscan0_to_scan2_align.mhd", tmp_folder, DIR_SEP);
		sprintf(scan2_align_to_scan0_hdr_file, "%s%sscan2_align_to_scan0.mhd", tmp_folder, DIR_SEP);
#endif
		sprintf(scan2_to_atlas_mat_file, "%s%sscan2_to_atlas.mat", tmp_folder, DIR_SEP);
		sprintf(scan2_to_atlas_hdr_file, "%s%sscan2_to_atlas.mhd", tmp_folder, DIR_SEP);
		sprintf(atlas_to_scan2_mat_file, "%s%satlas_to_scan2.mat", tmp_folder, DIR_SEP);
		sprintf(atlas_to_scan2_hdr_file, "%s%satlas_to_scan2.mhd", tmp_folder, DIR_SEP);
		sprintf(scan0_to_scan2_hdr_file, "%s%sscan0_to_scan2.mhd", out2_folder, DIR_SEP);
		sprintf(scan2_to_scan0_hdr_file, "%s%sscan2_to_scan0.mhd", out2_folder, DIR_SEP);
		//
		for (i = 0; i < NumberOfImageChannels; i++) {
			sprintf(scan0_image_masked_files[i]          , "%s%sscan0_image_masked_%d.nii.gz"          , tmp_folder, DIR_SEP, i);
			sprintf(scan2_image_masked_files[i]          , "%s%sscan2_image_masked_%d.nii.gz"          , tmp_folder, DIR_SEP, i);
			//
#ifdef USE_ALIGN_SCAN
			sprintf(scan2_align_image_files[i]           , "%s%sscan2_align_image_%d.nii.gz"           , tmp_folder, DIR_SEP, i);
#endif
			sprintf(scan0_atlas_reg_image_files[i]       , "%s%sscan0_atlas_reg_image_%d.nii.gz"       , tmp_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_image_masked_files[i], "%s%sscan0_atlas_reg_image_masked_%d.nii.gz", tmp_folder, DIR_SEP, i);
			sprintf(scan2_atlas_reg_image_files[i]       , "%s%sscan2_atlas_reg_image_%d.nii.gz"       , tmp_folder, DIR_SEP, i);
			sprintf(scan2_atlas_reg_image_masked_files[i], "%s%sscan2_atlas_reg_image_masked_%d.nii.gz", tmp_folder, DIR_SEP, i);
		}
		sprintf(scan0_atlas_reg_image_list, "%s%sscan0_atlas_reg_image.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan0_atlas_reg_image_list, "r");
		if (fp == NULL) {
			fp = fopen(scan0_atlas_reg_image_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan0_atlas_reg_image_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfImageChannels; i++) {
				fprintf(fp, "scan0_atlas_reg_image_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan0_atlas_reg_image_masked_list, "%s%sscan0_atlas_reg_image_masked.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan0_atlas_reg_image_masked_list, "r");
		if (fp == NULL) {
			fp = fopen(scan0_atlas_reg_image_masked_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan0_atlas_reg_image_masked_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfImageChannels; i++) {
				fprintf(fp, "scan0_atlas_reg_image_masked_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan2_atlas_reg_image_list, "%s%sscan2_atlas_reg_image.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan2_atlas_reg_image_list, "r");
		if (fp == NULL) {
			fp = fopen(scan2_atlas_reg_image_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan2_atlas_reg_image_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfImageChannels; i++) {
				fprintf(fp, "scan2_atlas_reg_image_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan2_atlas_reg_image_masked_list, "%s%sscan2_atlas_reg_image_masked.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan2_atlas_reg_image_masked_list, "r");
		if (fp == NULL) {
			fp = fopen(scan2_atlas_reg_image_masked_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan2_atlas_reg_image_masked_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfImageChannels; i++) {
				fprintf(fp, "scan2_atlas_reg_image_masked_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		//
		sprintf(scan0_atlas_reg_label_map_file      , "%s%sscan0_atlas_reg_label_map.nii.gz"  , tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_label_map_file      , "%s%sscan2_atlas_reg_label_map.nii.gz"  , tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_label_map_s_file    , "%s%sscan2_atlas_reg_label_map_s.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_label_map_s_hdr_file, "%s%sscan2_atlas_reg_label_map_s.hdr"   , tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_label_map_s_img_file, "%s%sscan2_atlas_reg_label_map_s.img"   , tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_prior_map_file      , "%s%sscan0_atlas_reg_prior_map.nii.gz"  , tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_prior_map_s_file    , "%s%sscan0_atlas_reg_prior_map_s.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_prior_map_s_hdr_file, "%s%sscan0_atlas_reg_prior_map_s.hdr"   , tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_prior_map_s_img_file, "%s%sscan0_atlas_reg_prior_map_s.img"   , tmp_folder, DIR_SEP);
		//
		sprintf(scan0_atlas_reg_tumor_mask_file, "%s%sscan0_atlas_reg_tumor_mask.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_tumor_mask_file, "%s%sscan2_atlas_reg_tumor_mask.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_tumor_prob_file, "%s%sscan0_atlas_reg_tumor_prob.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_tumor_prob_file, "%s%sscan2_atlas_reg_tumor_prob.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_abnor_mask_file, "%s%sscan0_atlas_reg_abnor_mask.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_abnor_mask_file, "%s%sscan2_atlas_reg_abnor_mask.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan0_atlas_reg_abnor_prob_file, "%s%sscan0_atlas_reg_abnor_prob.nii.gz", tmp_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_abnor_prob_file, "%s%sscan2_atlas_reg_abnor_prob.nii.gz", tmp_folder, DIR_SEP);
		//
		sprintf(scan0_init_means_file    , "%s%sscan0_init_means.txt"    , tmp_folder, DIR_SEP);
		sprintf(scan0_init_variances_file, "%s%sscan0_init_variances.txt", tmp_folder, DIR_SEP);
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(scan0_atlas_reg_init_prior_files[i]    , "%s%sscan0_atlas_reg_init_prior_%d.nii.gz"    , tmp_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_init_posterior_files[i], "%s%sscan0_atlas_reg_init_posterior_%d.nii.gz", tmp_folder, DIR_SEP, i);
		}
		sprintf(scan0_h_init_hdr_file, "%s%sscan0_h_init.mhd", tmp_folder, DIR_SEP);
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(scan0_atlas_reg_prior_files[i]     , "%s%sscan0_atlas_reg_prior_%d.nii.gz"     , tmp_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_prior_s_files[i]   , "%s%sscan0_atlas_reg_prior_s_%d.nii.gz"   , tmp_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_mass_prior_files[i], "%s%sscan0_atlas_reg_mass_prior_%d.nii.gz", tmp_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_warp_prior_files[i], "%s%sscan0_atlas_reg_warp_prior_%d.nii.gz", tmp_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_posterior_files[i] , "%s%sscan0_atlas_reg_posterior_%d.nii.gz" , tmp_folder, DIR_SEP, i);
		}
		sprintf(scan0_atlas_reg_prior_list, "%s%sscan0_atlas_reg_prior.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan0_atlas_reg_prior_list, "r");
		if (fp == NULL) {
			fp = fopen(scan0_atlas_reg_prior_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan0_atlas_reg_prior_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfPriorChannels; i++) {
				fprintf(fp, "scan0_atlas_reg_prior_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan0_atlas_reg_posterior_list, "%s%sscan0_atlas_reg_posterior.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan0_atlas_reg_posterior_list, "r");
		if (fp == NULL) {
			fp = fopen(scan0_atlas_reg_posterior_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan0_atlas_reg_posterior_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfPriorChannels; i++) {
				fprintf(fp, "scan0_atlas_reg_posterior_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan0_means_file    , "%s%sscan0_means.txt"    , tmp_folder, DIR_SEP);
		sprintf(scan0_variances_file, "%s%sscan0_variances.txt", tmp_folder, DIR_SEP);
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(scan2_atlas_reg_prior_files[i]      , "%s%sscan2_atlas_reg_prior_%d.nii.gz"      , tmp_folder, DIR_SEP, i);
			sprintf(scan2_atlas_reg_warp_prior_files[i] , "%s%sscan2_atlas_reg_warp_prior_%d.nii.gz" , tmp_folder, DIR_SEP, i);
			sprintf(scan2_atlas_reg_posterior_files[i]  , "%s%sscan2_atlas_reg_posterior_%d.nii.gz"  , tmp_folder, DIR_SEP, i);
			sprintf(scan2_atlas_reg_posterior_s_files[i], "%s%sscan2_atlas_reg_posterior_s_%d.nii.gz", tmp_folder, DIR_SEP, i);
		}
		sprintf(scan2_atlas_reg_prior_list, "%s%sscan2_atlas_reg_prior.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan2_atlas_reg_prior_list, "r");
		if (fp == NULL) {
			fp = fopen(scan2_atlas_reg_prior_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan2_atlas_reg_prior_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfPriorChannels; i++) {
				fprintf(fp, "scan2_atlas_reg_prior_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan2_atlas_reg_posterior_list, "%s%sscan2_atlas_reg_posterior.lst", tmp_folder, DIR_SEP);
		fp = fopen(scan2_atlas_reg_posterior_list, "r");
		if (fp == NULL) {
			fp = fopen(scan2_atlas_reg_posterior_list, "w");
			if (fp == NULL) {
				TRACE("Cannot make %s\n", scan2_atlas_reg_posterior_list);
				exit(EXIT_FAILURE);
			}
			for (i = 0; i < NumberOfPriorChannels; i++) {
				fprintf(fp, "scan2_atlas_reg_posterior_%d.nii.gz\n", i);
			}
			fclose(fp);
		} else {
			fclose(fp);
		}
		sprintf(scan2_init_means_file    , "%s%sscan2_init_means.txt"    , tmp_folder, DIR_SEP);
		sprintf(scan2_init_variances_file, "%s%sscan2_init_variances.txt", tmp_folder, DIR_SEP);
		sprintf(scan2_means_file         , "%s%sscan2_means.txt"         , tmp_folder, DIR_SEP);
		sprintf(scan2_variances_file     , "%s%sscan2_variances.txt"     , tmp_folder, DIR_SEP);
		//
		sprintf(scan0_u_hdr_file    , "%s%sscan0_u.mhd"    , tmp_folder, DIR_SEP);
		sprintf(scan0_u_inv_hdr_file, "%s%sscan0_u_inv.mhd", tmp_folder, DIR_SEP);
		sprintf(scan0_h_hdr_file    , "%s%sscan0_h.mhd"    , tmp_folder, DIR_SEP);
		sprintf(scan0_h2_hdr_file   , "%s%sscan0_h2.mhd"   , tmp_folder, DIR_SEP);
		//
		sprintf(scan2_u_hdr_file, "%s%sscan2_u.mhd", tmp_folder, DIR_SEP);
		sprintf(scan2_h_hdr_file, "%s%sscan2_h.mhd", tmp_folder, DIR_SEP);
		//
		sprintf(scales_file, "%s%sscales.txt", tmp_folder, DIR_SEP);
		sprintf(simulator_input_file, "%s%sForwardSolver.in", tmp_folder, DIR_SEP);
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(scan0_prior_files[i], "%s%sscan0_prior_%d.nii.gz", out_folder, DIR_SEP, i);
			sprintf(scan0_posterior_files[i], "%s%sscan0_posterior_%d.nii.gz", out_folder, DIR_SEP, i);
		}
		sprintf(scan0_prior_list, "%s%sscan0_prior.lst", out_folder, DIR_SEP);
		fp = fopen(scan0_prior_list, "w");
		if (fp == NULL) {
			TRACE("Cannot make %s\n", scan0_prior_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fprintf(fp, "scan0_prior_%d.nii.gz\n", i);
		}
		fclose(fp);
		sprintf(scan0_posterior_list, "%s%sscan0_posterior.lst", out_folder, DIR_SEP);
		fp = fopen(scan0_posterior_list, "w");
		if (fp == NULL) {
			TRACE("Cannot make %s\n", scan0_posterior_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fprintf(fp, "scan0_posterior_%d.nii.gz\n", i);
		}
		fclose(fp);
		//
		sprintf(scan0_label_map_file, "%s%sscan0_label_map.nii.gz", out_folder, DIR_SEP);
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
#ifdef USE_ALIGN_SCAN
			sprintf(scan2_align_prior_files[i], "%s%sscan2_align_prior_%d.nii.gz", tmp_folder, DIR_SEP, i);
			sprintf(scan2_align_posterior_files[i], "%s%sscan2_align_posterior_%d.nii.gz", tmp_folder, DIR_SEP, i);
#endif
			sprintf(scan2_prior_files[i], "%s%sscan2_prior_%d.nii.gz", out_folder, DIR_SEP, i);
			sprintf(scan2_posterior_files[i], "%s%sscan2_posterior_%d.nii.gz", out_folder, DIR_SEP, i);
		}
		sprintf(scan2_prior_list, "%s%sscan2_prior.lst", out_folder, DIR_SEP);
		fp = fopen(scan2_prior_list, "w");
		if (fp == NULL) {
			TRACE("Cannot make %s\n", scan2_prior_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fprintf(fp, "scan2_prior_%d.nii.gz\n", i);
		}
		fclose(fp);
		sprintf(scan2_posterior_list, "%s%sscan2_posterior.lst", out_folder, DIR_SEP);
		fp = fopen(scan2_posterior_list, "w");
		if (fp == NULL) {
			TRACE("Cannot make %s\n", scan2_posterior_list);
			exit(EXIT_FAILURE);
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fprintf(fp, "scan2_posterior_%d.nii.gz\n", i);
		}
		fclose(fp);
		//
		sprintf(scan2_label_map_file, "%s%sscan2_label_map.nii.gz", out_folder, DIR_SEP);
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	

	if (b_run_test) {
		/////////////////////////////////////////////////////////////////////////////
#ifdef USE_TRACE
		if (g_fp_trace != NULL) {
			fclose(g_fp_trace);
			g_fp_trace = NULL;
		}
		g_bTrace = FALSE;
		g_bTraceStdOut = FALSE;
		g_verbose = 0;
#endif
		/////////////////////////////////////////////////////////////////////////////

		/////////////////////////////////////////////////////////////////////////////
		//DeleteAll(tmp_folder, TRUE);
		//DeleteAll(out_folder, TRUE);
		/////////////////////////////////////////////////////////////////////////////

		exit(EXIT_SUCCESS);
	}


	/////////////////////////////////////////////////////////////////////////////
	// adjust the maximum intensity value to 255
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan0_image_masked_files[0])) {
		FVolume scan0_images[NumberOfImageChannels];
		FVolume scan2_images[NumberOfImageChannels];

		for (i = 0; i < NumberOfImageChannels; i++) {
			scan0_images[i].load(scan0_image_files[i], 1);
			scan2_images[i].load(scan2_image_files[i], 1);
		}

		// adjust the maximum value of images to 255
		{
			int vd_x, vd_y, vd_z;
			
			vd_x = scan0_images[0].m_vd_x;
			vd_y = scan0_images[0].m_vd_y;
			vd_z = scan0_images[0].m_vd_z;
		
			for (l = 0; l < NumberOfImageChannels; l++) {
				float scale_min = 0;
				float scale_max = 0;
				float scale, val;

				for (k = 0; k < vd_z; k++) {
					for (j = 0; j < vd_y; j++) {
						for (i = 0; i < vd_x; i++) {
							val = scan0_images[l].m_pData[k][j][i][0];
							if (val > scale_max) {
								scale_max = val;
							}
							if (val < scale_min) {
								scale_min = val;
							}
						}
					}
				}

				TRACE2("%s: scale_min = %f, scale_max = %f\n", scan0_image_files[l], scale_min, scale_max);
				if (scale_min < 0) {
					TRACE("%s has minus values: check value ranges\n", scan0_image_files[l]);
					exit(EXIT_FAILURE);
				}
				if (scale_max > 255.0) {
					TRACE("%s: adjust scale_max to 255\n", scan0_image_files[l]);

					scale = 255.0 / scale_max;

					for (k = 0; k < vd_z; k++) {
						for (j = 0; j < vd_y; j++) {
							for (i = 0; i < vd_x; i++) {
								val = scan0_images[l].m_pData[k][j][i][0] * scale;
								scan0_images[l].m_pData[k][j][i][0] = val;
							}
						}
					}
				}
			}
		}
		{
			int vd_x, vd_y, vd_z;
			
			vd_x = scan2_images[0].m_vd_x;
			vd_y = scan2_images[0].m_vd_y;
			vd_z = scan2_images[0].m_vd_z;
		
			for (l = 0; l < NumberOfImageChannels; l++) {
				float scale_min = 0;
				float scale_max = 0;
				float scale, val;

				for (k = 0; k < vd_z; k++) {
					for (j = 0; j < vd_y; j++) {
						for (i = 0; i < vd_x; i++) {
							val = scan2_images[l].m_pData[k][j][i][0];
							if (val > scale_max) {
								scale_max = val;
							}
							if (val < scale_min) {
								scale_min = val;
							}
						}
					}
				}

				TRACE2("%s: scale_min = %f, scale_max = %f\n", scan2_image_files[l], scale_min, scale_max);
				if (scale_min < 0) {
					TRACE("%s has minus values: check value ranges\n", scan2_image_files[l]);
					exit(EXIT_FAILURE);
				}
				if (scale_max > 255.0) {
					TRACE("%s: adjust scale_max to 255\n", scan2_image_files[l]);

					scale = 255.0 / scale_max;

					for (k = 0; k < vd_z; k++) {
						for (j = 0; j < vd_y; j++) {
							for (i = 0; i < vd_x; i++) {
								val = scan2_images[l].m_pData[k][j][i][0] * scale;
								scan2_images[l].m_pData[k][j][i][0] = val;
							}
						}
					}
				}
			}
		}

		for (i = 0; i < NumberOfImageChannels; i++) {
			scan0_images[i].save(scan0_image_masked_files[i], 1);
			ChangeNIIHeader(scan0_image_masked_files[i], scan0_image_files[0]);
			//
			scan0_images[i].clear();

			scan2_images[i].save(scan2_image_masked_files[i], 1);
			ChangeNIIHeader(scan2_image_masked_files[i], scan2_image_files[0]);
			//
			scan2_images[i].clear();
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


#ifdef USE_ALIGN_SCAN
	/////////////////////////////////////////////////////////////////////////////
	// align Scan 2 to Scan 0
	/////////////////////////////////////////////////////////////////////////////
	if (align_scan) {
		if (!IsFileExist(scan2_align_mat_file)) {
			putenv((char*)"FSLOUTPUTTYPE=NIFTI_GZ");
			//
			// align scan 2 to scan 0
			{
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -omat %s -cost mutualinfo -searchcost mutualinfo -searchrx -180 180 -searchry -180 180 -searchrz -180 180 -datatype float",
					FLIRT_PATH, scan2_image_masked_files[0], scan0_image_masked_files[0], scan2_align_image_files[0], scan2_align_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			//
			// generate align images for scan 2
			for (i = 1; i < NumberOfImageChannels; i++) {
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_image_masked_files[i], scan0_image_masked_files[0], scan2_align_image_files[i], scan2_align_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
		}
		if (!IsFileExist(scan2_align_inv_mat_file)) {
			putenv((char*)"FSLOUTPUTTYPE=NIFTI_GZ");
			//
			// get reverse mapping
			{
				sprintf(szCmdLine, "%s -omat %s -inverse %s",
					CONVERT_XFM_PATH, scan2_align_inv_mat_file, scan2_align_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			//
			GetFieldFromXFM(scan2_align_mat_file, scan2_image_masked_files[0], scan0_image_masked_files[0], scan2_align_hdr_file);
			GetFieldFromXFM(scan2_align_inv_mat_file, scan0_image_masked_files[0], scan2_image_masked_files[0], scan2_align_inv_hdr_file);
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
#endif


	/////////////////////////////////////////////////////////////////////////////
	// affine registration to atlas
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan2_to_atlas_mat_file)) {
		putenv((char*)"FSLOUTPUTTYPE=NIFTI_GZ");
		//
		// align scan 2 to atlas template
		{
			//sprintf(szCmdLine, "%s -in %s -ref %s -out %s -omat %s -searchrx -180 180 -searchry -180 180 -searchrz -180 180 -datatype float",
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -omat %s -cost mutualinfo -searchcost mutualinfo -searchrx -180 180 -searchry -180 180 -searchrz -180 180 -datatype float",
				FLIRT_PATH, 
#ifdef USE_ALIGN_SCAN
				(align_scan)?scan2_align_image_files[0]:scan2_image_masked_files[0],
#else
				scan2_image_masked_files[0], 
#endif
				atlas_template_file, scan2_atlas_reg_image_files[0], scan2_to_atlas_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
		}
		//
		// generate atlas reg images for scan 2
		for (i = 1; i < NumberOfImageChannels; i++) {
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
				FLIRT_PATH, 
#ifdef USE_ALIGN_SCAN
				(align_scan)?scan2_align_image_files[i]:scan2_image_masked_files[i],
#else
				scan2_image_masked_files[i], 
#endif
				atlas_template_file, scan2_atlas_reg_image_files[i], scan2_to_atlas_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
		}
		//
		// generate atlas reg images for scan 0
		for (i = 0; i < NumberOfImageChannels; i++) {
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
				FLIRT_PATH, scan0_image_masked_files[i], atlas_template_file, scan0_atlas_reg_image_files[i], scan2_to_atlas_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
		}
	}
	if (!IsFileExist(atlas_to_scan2_mat_file)) {
		putenv((char*)"FSLOUTPUTTYPE=NIFTI_GZ");
		//
		// get reverse mapping
		{
			sprintf(szCmdLine, "%s -omat %s -inverse %s",
				CONVERT_XFM_PATH, atlas_to_scan2_mat_file, scan2_to_atlas_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
		}
		//
		GetFieldFromXFM(scan2_to_atlas_mat_file, 
#ifdef USE_ALIGN_SCAN
			(align_scan)?scan2_align_image_files[0]:scan2_image_masked_files[0],
#else
			scan2_image_masked_files[0], 
#endif
			(char*)atlas_template_file, scan2_to_atlas_hdr_file);
		GetFieldFromXFM(atlas_to_scan2_mat_file, (char*)atlas_template_file, 
#ifdef USE_ALIGN_SCAN
			(align_scan)?scan2_align_image_files[0]:scan2_image_masked_files[0],
#else
			scan2_image_masked_files[0], 
#endif
			atlas_to_scan2_hdr_file);
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// apply brain mask
	/////////////////////////////////////////////////////////////////////////////
	{
		atlas_mask.load((char*)atlas_mask_file, 1);
		//
		if (!IsFileExist(scan2_atlas_reg_image_masked_files[0])) {
#ifdef USE_APPLY_BRAIN_MASK
			// apply brain mask
			for (i = 0; i < NumberOfImageChannels; i++) {
				scan2_atlas_reg_images[i].load(scan2_atlas_reg_image_files[i], 1);
				scan2_atlas_reg_images_masked[i].allocate(scan2_atlas_reg_images[i].m_vd_x, scan2_atlas_reg_images[i].m_vd_y, scan2_atlas_reg_images[i].m_vd_z);
				//
				for (n = 0; n < scan2_atlas_reg_images[i].m_vd_z; n++) {
					for (m = 0; m < scan2_atlas_reg_images[i].m_vd_y; m++) {
						for (l = 0; l < scan2_atlas_reg_images[i].m_vd_x; l++) {
							if (atlas_mask.m_pData[n][m][l][0] == 1) {
								scan2_atlas_reg_images_masked[i].m_pData[n][m][l][0] = scan2_atlas_reg_images[i].m_pData[n][m][l][0];
							} else {
								scan2_atlas_reg_images_masked[i].m_pData[n][m][l][0] = 0;
							}
						}
					}
				}
				//
				scan2_atlas_reg_images_masked[i].save(scan2_atlas_reg_image_masked_files[i], 1);
				ChangeNIIHeader(scan2_atlas_reg_image_masked_files[i], scan2_atlas_reg_image_files[i]);
				//
				scan2_atlas_reg_images[i].clear();
				scan2_atlas_reg_images_masked[i].clear();
			}
#else
			for (i = 0; i < NumberOfImageChannels; i++) {
				CopyFile(scan2_atlas_reg_image_files[i], scan2_atlas_reg_image_masked_files[i], FALSE);
			}
#endif
		}
		//
		if (!IsFileExist(scan0_atlas_reg_image_masked_files[0])) {
#ifdef USE_APPLY_BRAIN_MASK
			// apply brain mask
			for (i = 0; i < NumberOfImageChannels; i++) {
				scan0_atlas_reg_images[i].load(scan0_atlas_reg_image_files[i], 1);
				scan0_atlas_reg_images_masked[i].allocate(scan0_atlas_reg_images[i].m_vd_x, scan0_atlas_reg_images[i].m_vd_y, scan0_atlas_reg_images[i].m_vd_z);
				//
				for (n = 0; n < scan0_atlas_reg_images[i].m_vd_z; n++) {
					for (m = 0; m < scan0_atlas_reg_images[i].m_vd_y; m++) {
						for (l = 0; l < scan0_atlas_reg_images[i].m_vd_x; l++) {
							if (atlas_mask.m_pData[n][m][l][0] == 1) {
								scan0_atlas_reg_images_masked[i].m_pData[n][m][l][0] = scan0_atlas_reg_images[i].m_pData[n][m][l][0];
							} else {
								scan0_atlas_reg_images_masked[i].m_pData[n][m][l][0] = 0;
							}
						}
					}
				}
				//
				scan0_atlas_reg_images_masked[i].save(scan0_atlas_reg_image_masked_files[i], 1);
				ChangeNIIHeader(scan0_atlas_reg_image_masked_files[i], scan0_atlas_reg_image_files[i]);
				//
				scan0_atlas_reg_images[i].clear();
				scan0_atlas_reg_images_masked[i].clear();
			}
#else
			for (i = 0; i < NumberOfImageChannels; i++) {
				CopyFile(scan0_atlas_reg_image_files[i], scan0_atlas_reg_image_masked_files[i], FALSE);
			}
#endif
		}
		//
		atlas_mask.clear();
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	

	/////////////////////////////////////////////////////////////////////////////
	// transform seeds info
	/////////////////////////////////////////////////////////////////////////////
	{
#ifdef USE_ALIGN_SCAN
		if (align_scan) {
			if (!TransformPoints2(scan2_image_masked_files[0], scan2_align_image_files[0], scan2_atlas_reg_image_files[0], scan2_align_mat_file, scan2_to_atlas_mat_file, 
				scan2_seeds_info, scan2_atlas_reg_seeds_info, scan2_seeds_num, check_orientation)) {
				exit(EXIT_FAILURE);
			}
		} else {
			if (!TransformPoints(scan2_image_masked_files[0], scan2_atlas_reg_image_files[0], scan2_to_atlas_mat_file, 
				scan2_seeds_info, scan2_atlas_reg_seeds_info, scan2_seeds_num, check_orientation)) {
				exit(EXIT_FAILURE);
			}
		}
#else
		if (!TransformPoints(scan2_image_masked_files[0], scan2_atlas_reg_image_files[0], scan2_to_atlas_mat_file, 
			scan2_seeds_info, scan2_atlas_reg_seeds_info, scan2_seeds_num, check_orientation)) {
			exit(EXIT_FAILURE);
		}
#endif
		if (!TransformPoints(scan0_image_masked_files[0], scan0_atlas_reg_image_files[0], scan2_to_atlas_mat_file, 
			scan0_seeds_info, scan0_atlas_reg_seeds_info, scan0_seeds_num, check_orientation)) {
			exit(EXIT_FAILURE);
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	

	/////////////////////////////////////////////////////////////////////////////
	// make priors for scan 2
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan2_atlas_reg_prior_files[0])) {
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan2_atlas_reg_priors[i].load(atlas_prior_files[i], 1);
		}
		//
		// modify priors regaring tumor and edema
		if (!ModifyPriors(scan2_atlas_reg_priors, scan2_atlas_reg_seeds_info, scan2_seeds_num)) {
			exit(EXIT_FAILURE);
		} else {
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan2_atlas_reg_priors[i].save(scan2_atlas_reg_prior_files[i], 1);
				ChangeNIIHeader(scan2_atlas_reg_prior_files[i], scan2_atlas_reg_image_files[0]);
			}
		}
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan2_atlas_reg_priors[i].clear();
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// make posteriors for scan 2
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan2_h_hdr_file)) {
		{
			// init means and variances
			if (b_scan2_point_info) {
				if (!InitializeMeansAndVariancesFromPoints(scan2_points_info, scan2_points_num, scan2_image_masked_files, &scan2_MeanVector, &scan2_VarianceVector, check_orientation, b_scan2_valid_label)) {
					TRACE("InitializeMeansAndVariancesFromPoints - failed\n");
					exit(EXIT_FAILURE);
				}
				SaveMeansAndVariances(scan2_init_means_file, scan2_init_variances_file, &scan2_MeanVector, &scan2_VarianceVector);
			} else {
				if (!InitializeMeansAndVariances(scan2_init_mean, &scan2_MeanVector, &scan2_VarianceVector)) {
					TRACE("InitializeMeansAndVariances - failed\n");
					exit(EXIT_FAILURE);
				}
			}
			SaveMeansAndVariances(scan2_means_file, scan2_variances_file, &scan2_MeanVector, &scan2_VarianceVector);

			TRACE("Running JSR...\n");
			//
			/////////////////////////////////////////////////////////////////////////////
			char trace_file_out[1024];
			fflush(stdout);
			sprintf(trace_file_out, "%s%ss2_log_jsr_0.txt", tmp_folder, DIR_SEP);
#if defined(WIN32) || defined(WIN64)
			FILE *fp_trace_out;
			fp_trace_out = freopen(trace_file_out, "w", stdout);
#else
			int fd_trace_out;
			fpos_t pos_out; 
			fgetpos(stdout, &pos_out);
			fd_trace_out = _dup(fileno(stdout));
			freopen(trace_file_out, "w", stdout);
#endif
			/////////////////////////////////////////////////////////////////////////////
			//
			if (!EstimateDeformationFieldJSR(scan2_atlas_reg_image_masked_files, scan2_atlas_reg_prior_files, 
				&scan2_MeanVector, scan2_h_hdr_file, scan2_means_file, scan2_variances_file, (char*)"s2_cost_jsr_0.txt", 0, 1, num_itk_threads, false)) {
				exit(EXIT_FAILURE);
			}
			//
			/////////////////////////////////////////////////////////////////////////////
			fflush(stdout);
#if defined(WIN32) || defined(WIN64)
			fp_trace_out = freopen("CON", "w", stdout);
#else
			_dup2(fd_trace_out, fileno(stdout));
			close(fd_trace_out);
			clearerr(stdout);
			fsetpos(stdout, &pos_out);
#endif
			/////////////////////////////////////////////////////////////////////////////
			//
			TRACE("Running JSR...done\n");

#if 0
			for (j = 0; j < num_of_iter[2]; j++) {
				char file1[1024], file2[1024];
				//
				sprintf(file1, "jsr_means_%d.txt", j);
				sprintf(file2, "s2_means_jsr_0_%d.txt", j);
				DeleteFile(file2);
				MoveFile(file1, file2);
				//
				sprintf(file1, "jsr_variances_%d.txt", j);
				sprintf(file2, "s2_variances_jsr_0_%d.txt", j);
				DeleteFile(file2);
				MoveFile(file1, file2);
				//
				for (i = 0; i < NumberOfPriorChannels; i++) {
					sprintf(file1, "jsr_weight_image_%d_%d.nii.gz", j, i);
					sprintf(file2, "s2_weight_image_jsr_0_%d_%d.nii.gz", j, i);
					DeleteFile(file2);
					MoveFile(file1, file2);
					//
					sprintf(file1, "jsr_warped_image_%d_%d.nii.gz", j, i);
					sprintf(file2, "s2_warped_image_jsr_0_%d_%d.nii.gz", j, i);
					DeleteFile(file2);
					MoveFile(file1, file2);
				}
			}
#endif

			// refresh means and variances
			LoadMeansAndVariances(scan2_means_file, scan2_variances_file, &scan2_MeanVector, &scan2_VarianceVector);
		}
		{
			float vd_ox, vd_oy, vd_oz;
			FVolume v3;

			for (i = 0; i < NumberOfImageChannels; i++) {
				scan2_atlas_reg_images[i].load(scan2_atlas_reg_image_files[i], 1);
				scan2_atlas_reg_images_masked[i].load(scan2_atlas_reg_image_masked_files[i], 1);
			}
			//
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan2_atlas_reg_priors[i].load(scan2_atlas_reg_prior_files[i], 1);
				scan2_atlas_reg_warp_priors[i].load(scan2_atlas_reg_prior_files[i], 1);
				scan2_atlas_reg_posteriors[i].load(scan2_atlas_reg_prior_files[i], 1);
			}

			if (!LoadMHDData(NULL, (char*)scan2_h_hdr_file, &v3.m_pData, v3.m_vd_x, v3.m_vd_y, v3.m_vd_z, v3.m_vd_s, v3.m_vd_dx, v3.m_vd_dy, v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
				TRACE("Loading %s failed..\n", scan2_h_hdr_file);
			} else  {
				v3.computeDimension();
			}

			// apply warp
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[CSF], scan2_atlas_reg_priors[CSF], v3, 0.0f, false);
#ifdef USE_11A_PRIORS
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[VT ], scan2_atlas_reg_priors[VT ], v3, 0.0f, false);
#endif
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[GM ], scan2_atlas_reg_priors[GM ], v3, 0.0f, false);
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[WM ], scan2_atlas_reg_priors[WM ], v3, 0.0f, false);
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[TU ], scan2_atlas_reg_priors[TU ], v3, 0.0f, false);
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[ED ], scan2_atlas_reg_priors[ED ], v3, 0.0f, false);
#ifdef USE_11A_PRIORS
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[RTE], scan2_atlas_reg_priors[RTE], v3, 0.0f, false);
#endif
			//
#ifdef USE_WARPED_BG
			GenerateBackwardWarpVolume(scan2_atlas_reg_warp_priors[BG ], scan2_atlas_reg_priors[BG ], v3, 1.0f, false);
#else
			scan2_atlas_reg_warp_priors[BG ].load(atlas_prior_files[BG], 1);
#endif
			//
			scan2_atlas_reg_warp_priors[VS ].copy(scan2_atlas_reg_warp_priors[CSF]);
			scan2_atlas_reg_warp_priors[NCR].copy(scan2_atlas_reg_warp_priors[TU ]);
#ifdef USE_11A_PRIORS
			scan2_atlas_reg_warp_priors[RTN].copy(scan2_atlas_reg_warp_priors[RTE]);
#endif

			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan2_atlas_reg_warp_priors[i].save(scan2_atlas_reg_warp_prior_files[i], 1);
				ChangeNIIHeader(scan2_atlas_reg_warp_prior_files[i], scan2_atlas_reg_image_files[0]);
			}

			v3.clear();

			// store output posteriors for scan 2
			if (!ComputePosteriors(scan2_atlas_reg_images_masked, scan2_atlas_reg_warp_priors, &scan2_MeanVector, &scan2_VarianceVector, scan2_atlas_reg_posteriors)) {
				TRACE("ComputePosteriors failed..\n");
				exit(EXIT_FAILURE);
			} else {
				for (i = 0; i < NumberOfPriorChannels; i++) {
#ifdef USE_SMOOTH_POSTERIOR
					FVolume post_g;
					scan2_atlas_reg_posteriors[i].GaussianSmoothing(post_g, 0.67, 5);
					post_g.save(scan2_atlas_reg_posterior_files[i], 1);
					post_g.clear();
#else
					scan2_atlas_reg_posteriors[i].save(scan2_atlas_reg_posterior_files[i], 1);
#endif
					ChangeNIIHeader(scan2_atlas_reg_posterior_files[i], scan2_atlas_reg_image_files[0]);
				}
			}

			for (i = 0; i < NumberOfImageChannels; i++) {
				scan2_atlas_reg_images[i].clear();
				scan2_atlas_reg_images_masked[i].clear();
			}
			//
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan2_atlas_reg_priors[i].clear();
				scan2_atlas_reg_warp_priors[i].clear();
				scan2_atlas_reg_posteriors[i].clear();
			}
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// generate atlas reg label map for scan 2 using posteriors
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan2_atlas_reg_label_map_s_hdr_file)) {
		double ratio_x = 1.0 / 4.0;
		double ratio_y = 1.0 / 4.0;
		double ratio_z = 1.0 / 2.0;
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan2_atlas_reg_posteriors[i].load(scan2_atlas_reg_posterior_files[i], 1);
			//
			// resize posteriors
			{
				sprintf(szCmdLine, "%s -i %s -o %s -r %s -x %.15f -y %.15f -z %.15f", RESAMPLE_IMAGE_PATH, scan2_atlas_reg_posterior_files[i], scan2_atlas_reg_posterior_s_files[i], 
					scan2_atlas_reg_posterior_files[i], ratio_x, ratio_y, ratio_z);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			//
			scan2_atlas_reg_posteriors_s[i].load(scan2_atlas_reg_posterior_s_files[i], 1);
		}
		//
		scan2_atlas_reg_label_map.allocate(scan2_atlas_reg_posteriors[0].m_vd_x, scan2_atlas_reg_posteriors[0].m_vd_y, scan2_atlas_reg_posteriors[0].m_vd_z);
		scan2_atlas_reg_label_map_s.allocate(scan2_atlas_reg_posteriors_s[0].m_vd_x, scan2_atlas_reg_posteriors_s[0].m_vd_y, scan2_atlas_reg_posteriors_s[0].m_vd_z);
		//
		for (k = 0; k < scan2_atlas_reg_label_map.m_vd_z; k++) {
			for (j = 0; j < scan2_atlas_reg_label_map.m_vd_y; j++) {
				for (i = 0; i < scan2_atlas_reg_label_map.m_vd_x; i++) {
					float max_l_val = scan2_atlas_reg_posteriors[0].m_pData[k][j][i][0];
					int max_l_idx = 0;
					for (l = 1; l < NumberOfPriorChannels; l++) {
						// prefer higher labels
						if (max_l_val <= scan2_atlas_reg_posteriors[l].m_pData[k][j][i][0]) {
							max_l_val = scan2_atlas_reg_posteriors[l].m_pData[k][j][i][0];
							max_l_idx = l;
						}
					}
					//*
					scan2_atlas_reg_label_map.m_pData[k][j][i][0] = label_idx[max_l_idx];
					/*/
					if (max_l_idx == 0) {
						if (scan2_atlas_reg_posteriors[1].m_pData[k][j][i][0] > 0) {
							scan2_atlas_reg_label_map.m_pData[k][j][i][0] = label_idx[1];
						} else {
							scan2_atlas_reg_label_map.m_pData[k][j][i][0] = label_idx[0];
						}
					} else {
						scan2_atlas_reg_label_map.m_pData[k][j][i][0] = label_idx[max_l_idx];
					}
					//*/
				}
			}
		}
		for (k = 0; k < scan2_atlas_reg_label_map_s.m_vd_z; k++) {
			for (j = 0; j < scan2_atlas_reg_label_map_s.m_vd_y; j++) {
				for (i = 0; i < scan2_atlas_reg_label_map_s.m_vd_x; i++) {
					float max_l_val = scan2_atlas_reg_posteriors_s[0].m_pData[k][j][i][0];
					int max_l_idx = 0;
					for (l = 1; l < NumberOfPriorChannels; l++) {
						// prefer higher labels
						if (max_l_val <= scan2_atlas_reg_posteriors_s[l].m_pData[k][j][i][0]) {
							max_l_val = scan2_atlas_reg_posteriors_s[l].m_pData[k][j][i][0];
							max_l_idx = l;
						}
					}
					//*
					scan2_atlas_reg_label_map_s.m_pData[k][j][i][0] = label_s_idx[max_l_idx];
					/*/
					if (max_l_idx == 0) {
						if (scan2_atlas_reg_posteriors_s[1].m_pData[k][j][i][0] > 0) {
							scan2_atlas_reg_label_map_s.m_pData[k][j][i][0] = label_s_idx[1];
						} else {
							scan2_atlas_reg_label_map_s.m_pData[k][j][i][0] = label_s_idx[0];
						}
					} else {
						scan2_atlas_reg_label_map_s.m_pData[k][j][i][0] = label_s_idx[max_l_idx];
					}
					//*/
				}
			}
		}
		//
		scan2_atlas_reg_label_map.save(scan2_atlas_reg_label_map_file, 1);
		ChangeNIIHeader(scan2_atlas_reg_label_map_file, scan2_atlas_reg_posterior_files[0]);
		//
		scan2_atlas_reg_label_map_s.save(scan2_atlas_reg_label_map_s_file, 1);
		ChangeNIIHeader(scan2_atlas_reg_label_map_s_file, scan2_atlas_reg_posterior_s_files[0]);
		//
		{
			nifti_image* pNII;
			pNII = nifti_image_read(scan2_atlas_reg_label_map_s_file, 1);
			pNII->nifti_type = NIFTI_FTYPE_ANALYZE;
			strcpy(pNII->fname, scan2_atlas_reg_label_map_s_hdr_file);
			strcpy(pNII->iname, scan2_atlas_reg_label_map_s_img_file);
			nifti_image_write(pNII);
			nifti_image_free(pNII);
		}
		
		// free data
		{
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan2_atlas_reg_posteriors[i].clear();
				scan2_atlas_reg_posteriors_s[i].clear();
			}
			//
			scan2_atlas_reg_label_map.clear();
			scan2_atlas_reg_label_map_s.clear();
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// generate atlas reg prior map for scan 0
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan0_atlas_reg_prior_map_s_hdr_file)) {
		double ratio_x = 1.0 / 4.0;
		double ratio_y = 1.0 / 4.0;
		double ratio_z = 1.0 / 2.0;
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			{
				sprintf(szCmdLine, "%s -i %s -r %s -o %s -d %s", WARP_IMAGE_PATH, atlas_prior_files[i], scan0_atlas_reg_image_files[0], scan0_atlas_reg_prior_files[i], scan2_h_hdr_file);
				//
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			//
			scan0_atlas_reg_priors[i].load(scan0_atlas_reg_prior_files[i], 1);
			//
			// resize priors
			{
				sprintf(szCmdLine, "%s -i %s -o %s -r %s -x %.15f -y %.15f -z %.15f", RESAMPLE_IMAGE_PATH, scan0_atlas_reg_prior_files[i], scan0_atlas_reg_prior_s_files[i], 
					scan0_atlas_reg_prior_files[i], ratio_x, ratio_y, ratio_z);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			//
			scan0_atlas_reg_priors_s[i].load(scan0_atlas_reg_prior_s_files[i], 1);
		}
		//
		scan0_atlas_reg_prior_map.allocate(scan0_atlas_reg_priors[0].m_vd_x, scan0_atlas_reg_priors[0].m_vd_y, scan0_atlas_reg_priors[0].m_vd_z);
		scan0_atlas_reg_prior_map_s.allocate(scan0_atlas_reg_priors_s[0].m_vd_x, scan0_atlas_reg_priors_s[0].m_vd_y, scan0_atlas_reg_priors_s[0].m_vd_z);
		//
		for (k = 0; k < scan0_atlas_reg_prior_map.m_vd_z; k++) {
			for (j = 0; j < scan0_atlas_reg_prior_map.m_vd_y; j++) {
				for (i = 0; i < scan0_atlas_reg_prior_map.m_vd_x; i++) {
					float max_l_val = scan0_atlas_reg_priors[0].m_pData[k][j][i][0];
					int max_l_idx = 0;
					for (l = 1; l < 5; l++) {
						// prefer higher labels
						if (max_l_val <= scan0_atlas_reg_priors[l].m_pData[k][j][i][0]) {
							max_l_val = scan0_atlas_reg_priors[l].m_pData[k][j][i][0];
							max_l_idx = l;
						}
					}
					scan0_atlas_reg_prior_map.m_pData[k][j][i][0] = label_idx[max_l_idx];
				}
			}
		}
		for (k = 0; k < scan0_atlas_reg_prior_map_s.m_vd_z; k++) {
			for (j = 0; j < scan0_atlas_reg_prior_map_s.m_vd_y; j++) {
				for (i = 0; i < scan0_atlas_reg_prior_map_s.m_vd_x; i++) {
					float max_l_val = scan0_atlas_reg_priors_s[0].m_pData[k][j][i][0];
					int max_l_idx = 0;
					for (l = 1; l < 5; l++) {
						// prefer higher labels
						if (max_l_val <= scan0_atlas_reg_priors_s[l].m_pData[k][j][i][0]) {
							max_l_val = scan0_atlas_reg_priors_s[l].m_pData[k][j][i][0];
							max_l_idx = l;
						}
					}
					scan0_atlas_reg_prior_map_s.m_pData[k][j][i][0] = label_idx[max_l_idx];
				}
			}
		}
		//
		scan0_atlas_reg_prior_map.save(scan0_atlas_reg_prior_map_file, 1);
		ChangeNIIHeader(scan0_atlas_reg_prior_map_file, scan0_atlas_reg_prior_files[0]);
		//
		scan0_atlas_reg_prior_map_s.save(scan0_atlas_reg_prior_map_s_file, 1);
		ChangeNIIHeader(scan0_atlas_reg_prior_map_s_file, scan0_atlas_reg_prior_s_files[0]);
		//
		{
			nifti_image* pNII;
			pNII = nifti_image_read(scan0_atlas_reg_prior_map_s_file, 1);
			pNII->nifti_type = NIFTI_FTYPE_ANALYZE;
			strcpy(pNII->fname, scan0_atlas_reg_prior_map_s_hdr_file);
			strcpy(pNII->iname, scan0_atlas_reg_prior_map_s_img_file);
			nifti_image_write(pNII);
			nifti_image_free(pNII);
		}
		
		// free data
		{
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_atlas_reg_priors[i].clear();
				scan0_atlas_reg_priors_s[i].clear();
			}
			//
			scan0_atlas_reg_prior_map.clear();
			scan0_atlas_reg_prior_map_s.clear();
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// make initial priors for scan 0
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan2_atlas_reg_tumor_mask_file)) {
		int vd_x, vd_y, vd_z;

		// make abnormal regions mask
		scan2_atlas_reg_label_map.load(scan2_atlas_reg_label_map_file, 1);
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan2_atlas_reg_posteriors[i].load(scan2_atlas_reg_posterior_files[i], 1);
			scan2_atlas_reg_warp_priors[i].load(scan2_atlas_reg_warp_prior_files[i], 1);
		}

		vd_x = scan2_atlas_reg_label_map.m_vd_x;
		vd_y = scan2_atlas_reg_label_map.m_vd_y;
		vd_z = scan2_atlas_reg_label_map.m_vd_z;

		scan2_atlas_reg_tumor_mask.allocate(vd_x, vd_y, vd_z);
		scan2_atlas_reg_tumor_prob.allocate(vd_x, vd_y, vd_z);
		scan2_atlas_reg_abnor_mask.allocate(vd_x, vd_y, vd_z);
		scan2_atlas_reg_abnor_prob.allocate(vd_x, vd_y, vd_z);

		for (k = 0; k < vd_z; k++) {
			for (j = 0; j < vd_y; j++) {
				for (i = 0; i < vd_x; i++) {
					int label = scan2_atlas_reg_label_map.m_pData[k][j][i][0];
					if ((label == label_idx[TU]) || (label == label_idx[NCR])) {
						scan2_atlas_reg_tumor_mask.m_pData[k][j][i][0] = 1;
					} else {
						scan2_atlas_reg_tumor_mask.m_pData[k][j][i][0] = 0;
					}   
#ifndef USE_11A_PRIORS
					if ((label == label_idx[TU]) || (label == label_idx[NCR]) || (label == label_idx[ED])) {
#else
					if ((label == label_idx[TU]) || (label == label_idx[NCR]) || (label == label_idx[ED]) || (label == label_idx[RTE]) || (label == label_idx[RTN])) {
#endif
						scan2_atlas_reg_abnor_mask.m_pData[k][j][i][0] = 1;
					} else {
						scan2_atlas_reg_abnor_mask.m_pData[k][j][i][0] = 0;
					}
					//
					scan2_atlas_reg_tumor_prob.m_pData[k][j][i][0] = 0;
					scan2_atlas_reg_tumor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[TU ].m_pData[k][j][i][0];
					scan2_atlas_reg_tumor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[NCR].m_pData[k][j][i][0];
					//
					//scan2_atlas_reg_tumor_prob.m_pData[k][j][i][0] *= scan2_atlas_reg_warp_priors[TU].m_pData[k][j][i][0] + scan2_atlas_reg_warp_priors[NCR].m_pData[k][j][i][0];
					//
					if (scan2_atlas_reg_tumor_prob.m_pData[k][j][i][0] > 1.0f) {
						scan2_atlas_reg_tumor_prob.m_pData[k][j][i][0] = 1.0f;
					}
					scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] = 0;
					scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[TU ].m_pData[k][j][i][0];
					scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[NCR].m_pData[k][j][i][0];
					scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[ED ].m_pData[k][j][i][0];
#ifdef USE_11A_PRIORS
					scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[RTE].m_pData[k][j][i][0];
					scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] += scan2_atlas_reg_posteriors[RTN].m_pData[k][j][i][0];
#endif
					//
					//scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] *= scan2_atlas_reg_warp_priors[TU].m_pData[k][j][i][0] + scan2_atlas_reg_warp_priors[NCR].m_pData[k][j][i][0] + scan2_atlas_reg_warp_priors[ED].m_pData[k][j][i][0];
					//
					if (scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] > 1.0f) {
						scan2_atlas_reg_abnor_prob.m_pData[k][j][i][0] = 1.0f;
					}
				}
			}
		}

		scan2_atlas_reg_tumor_mask.save(scan2_atlas_reg_tumor_mask_file, 1);
		ChangeNIIHeader(scan2_atlas_reg_tumor_mask_file, scan2_atlas_reg_image_files[0]);
		scan2_atlas_reg_tumor_prob.save(scan2_atlas_reg_tumor_prob_file, 1);
		ChangeNIIHeader(scan2_atlas_reg_tumor_prob_file, scan2_atlas_reg_image_files[0]);
		scan2_atlas_reg_abnor_mask.save(scan2_atlas_reg_abnor_mask_file, 1);
		ChangeNIIHeader(scan2_atlas_reg_abnor_mask_file, scan2_atlas_reg_image_files[0]);
		scan2_atlas_reg_abnor_prob.save(scan2_atlas_reg_abnor_prob_file, 1);
		ChangeNIIHeader(scan2_atlas_reg_abnor_prob_file, scan2_atlas_reg_image_files[0]);
		
		if (!IsFileExist(scan0_atlas_reg_prior_files[0])) {
			//*
			// making priors for scan 0, apply h of scan 2 on atlas priors
			for (i = 0; i < NumberOfPriorChannels; i++) {
				sprintf(szCmdLine, "%s -i %s -r %s -o %s -d %s", WARP_IMAGE_PATH, atlas_prior_files[i], scan0_atlas_reg_image_files[0], scan0_atlas_reg_prior_files[i], scan2_h_hdr_file);
				//
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			/*/
			for (i = 0; i < NumberOfPriorChannels; i++) {
				CopyFile(atlas_prior_files[i], scan0_atlas_reg_prior_files[i], FALSE);
			}
			//*/
		}

		scan2_atlas_reg_label_map.clear();
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan2_atlas_reg_posteriors[i].clear();
			scan2_atlas_reg_warp_priors[i].clear();
		}
		scan2_atlas_reg_tumor_mask.clear();
		scan2_atlas_reg_tumor_prob.clear();
		scan2_atlas_reg_abnor_mask.clear();
		scan2_atlas_reg_abnor_prob.clear();
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


#if 1
	/////////////////////////////////////////////////////////////////////////////
	// make initial priors and posteriors for scan 0
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan0_h_init_hdr_file)) {
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan0_atlas_reg_priors[i].load(atlas_prior_files[i], 1);
		}
		//
		// modify priors regaring tumor and edema
		//if (!ModifyPriors(scan0_atlas_reg_priors, scan0_atlas_reg_seeds_info, scan0_seeds_num, a_TU, a_ED, b_ED)) {
		if (!ModifyPriors(scan0_atlas_reg_priors, scan0_atlas_reg_seeds_info, scan0_seeds_num)) {
			exit(EXIT_FAILURE);
		} else {
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_atlas_reg_priors[i].save(scan0_atlas_reg_init_prior_files[i], 1);
				ChangeNIIHeader(scan0_atlas_reg_init_prior_files[i], scan0_atlas_reg_image_files[0]);
			}
		}
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan0_atlas_reg_priors[i].clear();
		}
	}
	if (!IsFileExist(scan0_atlas_reg_init_posterior_files[0])) {
		{
			char cost_file[1024];

			sprintf(cost_file, "%s%ss0_cost_jsr_init.txt", tmp_folder, DIR_SEP);

			if (b_scan0_point_info) {
				if (!InitializeMeansAndVariancesFromPoints(scan0_points_info, scan0_points_num, scan0_image_masked_files, &scan0_MeanVector, &scan0_VarianceVector, check_orientation, b_scan0_valid_label)) {
					TRACE("InitializeMeansAndVariancesFromPoints - failed\n");
					exit(EXIT_FAILURE);
				}
				SaveMeansAndVariances(scan0_init_means_file, scan0_init_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
			} else {
				if (!InitializeMeansAndVariances(scan0_init_mean, &scan0_MeanVector, &scan0_VarianceVector)) {
					TRACE("InitializeMeansAndVariances - failed\n");
					exit(EXIT_FAILURE);
				}
			}

			TRACE("Running JSR...\n");
			//
			/////////////////////////////////////////////////////////////////////////////
			char trace_file_out[1024];
			fflush(stdout);
			sprintf(trace_file_out, "%s%ss0_log_jsr_init.txt", tmp_folder, DIR_SEP);
#if defined(WIN32) || defined(WIN64)
			FILE *fp_trace_out;
			fp_trace_out = freopen(trace_file_out, "w", stdout);
#else
			int fd_trace_out;
			fpos_t pos_out; 
			fgetpos(stdout, &pos_out);
			fd_trace_out = _dup(fileno(stdout));
			freopen(trace_file_out, "w", stdout);
#endif
			/////////////////////////////////////////////////////////////////////////////
			//
			if (!EstimateDeformationFieldJSR(scan0_atlas_reg_image_masked_files, scan0_atlas_reg_init_prior_files, 
				&scan0_MeanVector, scan0_h_init_hdr_file, scan0_init_means_file, scan0_init_variances_file, cost_file, 0, 1, num_itk_threads, false)) {
				exit(EXIT_FAILURE);
			}
			//
			/////////////////////////////////////////////////////////////////////////////
			fflush(stdout);
#if defined(WIN32) || defined(WIN64)
			fp_trace_out = freopen("CON", "w", stdout);
#else
			_dup2(fd_trace_out, fileno(stdout));
			close(fd_trace_out);
			clearerr(stdout);
			fsetpos(stdout, &pos_out);
#endif
			/////////////////////////////////////////////////////////////////////////////
			//
			TRACE("Running JSR...done\n");

#if 0
			for (j = 0; j < num_of_iter[2]; j++) {
				char file1[1024], file2[1024];
				//
				sprintf(file1, "jsr_means_%d.txt", j);
				sprintf(file2, "s2_means_jsr_0_%d.txt", j);
				DeleteFile(file2);
				MoveFile(file1, file2);
				//
				sprintf(file1, "jsr_variances_%d.txt", j);
				sprintf(file2, "s2_variances_jsr_0_%d.txt", j);
				DeleteFile(file2);
				MoveFile(file1, file2);
				//
				for (i = 0; i < NumberOfPriorChannels; i++) {
					sprintf(file1, "jsr_weight_image_%d_%d.nii.gz", j, i);
					sprintf(file2, "s2_weight_image_jsr_0_%d_%d.nii.gz", j, i);
					DeleteFile(file2);
					MoveFile(file1, file2);
					//
					sprintf(file1, "jsr_warped_image_%d_%d.nii.gz", j, i);
					sprintf(file2, "s2_warped_image_jsr_0_%d_%d.nii.gz", j, i);
					DeleteFile(file2);
					MoveFile(file1, file2);
				}
			}
#endif

			// refresh means and variances
			LoadMeansAndVariances(scan0_init_means_file, scan0_init_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
		}
		{
			float vd_ox, vd_oy, vd_oz;
			FVolume v3;

			for (i = 0; i < NumberOfImageChannels; i++) {
				scan0_atlas_reg_images[i].load(scan0_atlas_reg_image_files[i], 1);
				scan0_atlas_reg_images_masked[i].load(scan0_atlas_reg_image_masked_files[i], 1);
			}
			//
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_atlas_reg_priors[i].load(scan0_atlas_reg_init_prior_files[i], 1);
				scan0_atlas_reg_warp_priors[i].load(scan0_atlas_reg_init_prior_files[i], 1);
				scan0_atlas_reg_posteriors[i].load(scan0_atlas_reg_init_prior_files[i], 1);
			}

			if (!LoadMHDData(NULL, (char*)scan0_h_init_hdr_file, &v3.m_pData, v3.m_vd_x, v3.m_vd_y, v3.m_vd_z, v3.m_vd_s, v3.m_vd_dx, v3.m_vd_dy, v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
				TRACE("Loading %s failed..\n", scan0_h_init_hdr_file);
			} else  {
				v3.computeDimension();
			}

			// apply warp
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[CSF], scan0_atlas_reg_priors[CSF], v3, 0.0f, false);
#ifdef USE_11A_PRIORS
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[VT ], scan0_atlas_reg_priors[VT ], v3, 0.0f, false);
#endif
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[GM ], scan0_atlas_reg_priors[GM ], v3, 0.0f, false);
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[WM ], scan0_atlas_reg_priors[WM ], v3, 0.0f, false);
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[TU ], scan0_atlas_reg_priors[TU ], v3, 0.0f, false);
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[ED ], scan0_atlas_reg_priors[ED ], v3, 0.0f, false);
#ifdef USE_11A_PRIORS
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[RTE], scan0_atlas_reg_priors[RTE], v3, 0.0f, false);
#endif
			//
#ifdef USE_WARPED_BG
			GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[BG ], scan0_atlas_reg_priors[BG ], v3, 1.0f, false);
#else
			scan0_atlas_reg_warp_priors[BG ].load(atlas_prior_files[BG], 1);
#endif
			//
			scan0_atlas_reg_warp_priors[VS ].copy(scan0_atlas_reg_warp_priors[CSF]);
			scan0_atlas_reg_warp_priors[NCR].copy(scan0_atlas_reg_warp_priors[TU ]);
#ifdef USE_11A_PRIORS
			scan0_atlas_reg_warp_priors[RTN].copy(scan0_atlas_reg_warp_priors[RTE]);
#endif

			/*
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_atlas_reg_warp_priors[i].save(scan0_atlas_reg_warp_prior_files[i], 1);
				ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[i], scan0_atlas_reg_image_files[0]);
			}
			//*/

			v3.clear();

			// store output posteriors for scan 0
			if (!ComputePosteriors(scan0_atlas_reg_images_masked, scan0_atlas_reg_warp_priors, &scan0_MeanVector, &scan0_VarianceVector, scan0_atlas_reg_posteriors)) {
				TRACE("ComputePosteriors failed..\n");
				exit(EXIT_FAILURE);
			} else {
				for (i = 0; i < NumberOfPriorChannels; i++) {
#ifdef USE_SMOOTH_POSTERIOR
					FVolume post_g;
					scan0_atlas_reg_posteriors[i].GaussianSmoothing(post_g, 0.67, 5);
					post_g.save(scan0_atlas_reg_init_posterior_files[i], 1);
					post_g.clear();
#else
					scan0_atlas_reg_posteriors[i].save(scan0_atlas_reg_init_posterior_files[i], 1);
#endif
					ChangeNIIHeader(scan0_atlas_reg_init_posterior_files[i], scan0_atlas_reg_image_files[0]);
				}
			}

			for (i = 0; i < NumberOfImageChannels; i++) {
				scan0_atlas_reg_images[i].clear();
				scan0_atlas_reg_images_masked[i].clear();
			}
			//
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_atlas_reg_priors[i].clear();
				scan0_atlas_reg_warp_priors[i].clear();
				scan0_atlas_reg_posteriors[i].clear();
			}
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
#endif


	/////////////////////////////////////////////////////////////////////////////
	// joint segmentation & registration for scan 0
	/////////////////////////////////////////////////////////////////////////////
#if 1
	if (!IsFileExist(scan0_atlas_reg_posterior_files[0])) {
		int vd_x, vd_y, vd_z;
		float vd_dx, vd_dy, vd_dz;
		float _vd_dx, _vd_dy, _vd_dz;
		float vd_ox, vd_oy, vd_oz;

		{
			scan0_atlas_reg_images[0].load(scan0_atlas_reg_image_files[0], 1);
			//
			vd_x = scan0_atlas_reg_images[0].m_vd_x;
			vd_y = scan0_atlas_reg_images[0].m_vd_y;
			vd_z = scan0_atlas_reg_images[0].m_vd_z;
			vd_dx = scan0_atlas_reg_images[0].m_vd_dx;
			vd_dy = scan0_atlas_reg_images[0].m_vd_dy;
			vd_dz = scan0_atlas_reg_images[0].m_vd_dz;
			_vd_dx = 1.0 / vd_dx;
			_vd_dy = 1.0 / vd_dy;
			_vd_dz = 1.0 / vd_dz;
			//
			scan0_atlas_reg_images[0].clear();
		}

		for (k = 0; k < scan0_jsr_max_iter; k++) {
			char hops_file[1024];
			char solution_file[1024];
			//
			char cost_file[1024];
			char u_hdr_file[1024];
			char u_inv_hdr_file[1024];
			char h_hdr_file[1024];
			char means_file[1024];
			char variances_file[1024];
			char atlas_reg_mass_prior_files[NumberOfPriorChannels][1024];
			char atlas_reg_warp_prior_files[NumberOfPriorChannels][1024];
			char atlas_reg_posterior_files[NumberOfPriorChannels][1024];
			char label_map_s_img_file[1024];

			sprintf(hops_file    , "%s%ss0_Optim_%d.txt"   , tmp_folder, DIR_SEP, k);
			sprintf(solution_file, "%s%ss0_solution_%d.txt", tmp_folder, DIR_SEP, k);
			//
			sprintf(cost_file     , "%s%ss0_cost_jsr_%d.txt"     , tmp_folder, DIR_SEP, k);
			sprintf(u_hdr_file    , "%s%ss0_u_jsr_%d.mhd"        , tmp_folder, DIR_SEP, k);
			sprintf(u_inv_hdr_file, "%s%ss0_u_inv_jsr_%d.mhd"    , tmp_folder, DIR_SEP, k);
			sprintf(h_hdr_file    , "%s%ss0_h_jsr_%d.mhd"        , tmp_folder, DIR_SEP, k);
			sprintf(means_file    , "%s%ss0_means_jsr_%d.txt"    , tmp_folder, DIR_SEP, k);
			sprintf(variances_file, "%s%ss0_variances_jsr_%d.txt", tmp_folder, DIR_SEP, k);
			for (i = 0; i < NumberOfPriorChannels; i++) {
				sprintf(atlas_reg_mass_prior_files[i], "%s%ss0_atlas_reg_mass_prior_jsr_%d_%d.nii.gz", tmp_folder, DIR_SEP, k, i);
				sprintf(atlas_reg_warp_prior_files[i], "%s%ss0_atlas_reg_warp_prior_jsr_%d_%d.nii.gz", tmp_folder, DIR_SEP, k, i);
				sprintf(atlas_reg_posterior_files[i] , "%s%ss0_atlas_reg_posterior_jsr_%d_%d.nii.gz" , tmp_folder, DIR_SEP, k, i);
			}
			//
			//strcpy(label_map_s_img_file, scan2_atlas_reg_label_map_s_img_file);
			strcpy(label_map_s_img_file, scan0_atlas_reg_prior_map_s_img_file);

			if (k == 0) {
				// init means and variances
				if (scan0_reinit_means) {
					if (b_scan0_point_info) {
						if (!InitializeMeansAndVariancesFromPoints(scan0_points_info, scan0_points_num, scan0_image_masked_files, &scan0_MeanVector, &scan0_VarianceVector, check_orientation, b_scan0_valid_label)) {
							TRACE("InitializeMeansAndVariancesFromPoints - failed\n");
							exit(EXIT_FAILURE);
						}
						SaveMeansAndVariances(scan0_init_means_file, scan0_init_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
					} else {
						if (!InitializeMeansAndVariances(scan0_init_mean, &scan0_MeanVector, &scan0_VarianceVector)) {
							TRACE("InitializeMeansAndVariances - failed\n");
							exit(EXIT_FAILURE);
						}
					}
				} else {
					LoadMeansAndVariances(scan0_init_means_file, scan0_init_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
				}
				SaveMeansAndVariances(scan0_means_file, scan0_variances_file, &scan0_MeanVector, &scan0_VarianceVector);

				// make zero init deformation field
				{
					FVolume v0;
					v0.allocate(vd_x, vd_y, vd_z, 3);
					for (n = 0; n < vd_z; n++) {
						for (m = 0; m < vd_y; m++) {
							for (l = 0; l < vd_x; l++) {
								v0.m_pData[n][m][l][0] = 0;
								v0.m_pData[n][m][l][1] = 0;
								v0.m_pData[n][m][l][2] = 0;
							}
						}
					}
					SaveMHDData(NULL, scan0_h_hdr_file, v0.m_pData, vd_x, vd_y, vd_z, 3, vd_dx, vd_dy, vd_dz, 0, 0, 0);
					//SaveMHDData(NULL, scan0_h2_hdr_file, v0.m_pData, vd_x, vd_y, vd_z, 3, vd_dx, vd_dy, vd_dz, 0, 0, 0);
				}
			}
			
			// get optimal tumor parameters
			if (!IsFileExist(atlas_reg_mass_prior_files[0])) {
				double gxc, gyc, gzc, T;

				if (scan0_seed_of_scan0) {
					// for AAAB
					gxc = scan0_atlas_reg_seeds_info[0][0];
					gyc = scan0_atlas_reg_seeds_info[0][1];
					gzc = scan0_atlas_reg_seeds_info[0][2];
				} else {
					// for others
					gxc = scan2_atlas_reg_seeds_info[0][0];
					gyc = scan2_atlas_reg_seeds_info[0][1];
					gzc = scan2_atlas_reg_seeds_info[0][2];
				}

				// tumor days
				T = 0.004 * pow(scan0_atlas_reg_seeds_info[0][3] / 2, 3) * 4;

				TRACE2("tp: xc = %f, yc = %f, zc = %f, T = %f, r = %f\n", gxc, gyc, gzc, T, scan0_atlas_reg_seeds_info[0][3]);

				MakeScalesFile(scales_file);
				//
				{
					BVolume label_map_s;
					label_map_s.load(label_map_s_img_file, 1);
					//
					MakeSimulatorInputFile(simulator_input_file, 
						vd_dx, vd_dy, vd_dz, 
						label_map_s.m_vd_x, label_map_s.m_vd_y, label_map_s.m_vd_z, 
						label_map_s.m_vd_dx, label_map_s.m_vd_dy, label_map_s.m_vd_dz);
					//
					label_map_s.clear();
				}
				//
				{
					int max_eval;
					BOOL optim_xyz;
					//*
					if (k == 0) {
						optim_xyz = FALSE;
						max_eval = 40;
					} else if (k == 1) {
						optim_xyz = FALSE;
						max_eval = 70;
					} else {
						//optim_xyz = TRUE;
						optim_xyz = FALSE;
						max_eval = 100;
					}
					/*/
					//optim_xyz = TRUE;
					optim_xyz = FALSE;
					max_eval = 100;
					//*/
					//
					MakeHOPSFile(hops_file, gxc, gyc, gzc, T, atlas_mask_file, scan2_atlas_reg_abnor_mask_file,
						scan0_atlas_reg_image_masked_list, scan0_atlas_reg_prior_list, scan0_means_file, scan0_variances_file, scan0_h_hdr_file, scan0_h_hdr_file, //scan0_h2_hdr_file,
						scan2_atlas_reg_image_masked_list, scan2_atlas_reg_posterior_list, label_map_s_img_file, 
						simulator_input_file, scales_file, out_folder, tmp_folder, solution_file, max_eval, optim_xyz, num_hop_threads);
				}

				if (!IsFileExist(solution_file)) {
					sprintf(szCmdLine, "%s %s", HOPSPACK_PATH, hops_file);
					//
					TRACE("%s\n", szCmdLine);
					//
					/////////////////////////////////////////////////////////////////////////////
					char trace_file_out[1024];
					fflush(stdout);
					sprintf(trace_file_out, "%s%sOptim_%d_log.txt", tmp_folder, DIR_SEP, k);
#if defined(WIN32) || defined(WIN64)
					FILE *fp_trace_out;
					fp_trace_out = freopen(trace_file_out, "w", stdout);
#else
					int fd_trace_out;
					fpos_t pos_out; 
					fgetpos(stdout, &pos_out);
					fd_trace_out = _dup(fileno(stdout));
					freopen(trace_file_out, "w", stdout);
#endif
					/////////////////////////////////////////////////////////////////////////////
					//
					if (!ExecuteProcess(szCmdLine)) {
						TRACE("failed\n");
						exit(EXIT_FAILURE);
					}
					//
					/////////////////////////////////////////////////////////////////////////////
					fflush(stdout);
#if defined(WIN32) || defined(WIN64)
					fp_trace_out = freopen("CON", "w", stdout);
#else
					_dup2(fd_trace_out, fileno(stdout));
					close(fd_trace_out);
					clearerr(stdout);
					fsetpos(stdout, &pos_out);
#endif
					/////////////////////////////////////////////////////////////////////////////
				}

#if 1
				// get optimal params from solution file and run simulator
				if (!IsFileExist(atlas_reg_mass_prior_files[0])) {
					char input_file[1024];
					char output_file[1024] = "SOL";
					int tag = k;
					char tag_F[1024] = "SOL";
					char tmp[9+5][1024];
					double val[9];
					FILE* fp;

					sprintf(input_file, "%s%sinput.%d_%s.txt", tmp_folder, DIR_SEP, tag, tag_F);
#ifdef USE_OPTIM_DG
					fp = fopen(solution_file, "r");
					for (i = 0; i < 9+5; i++) {
						fscanf(fp, "%s", (char*)&tmp[i]);
						if (i >= 4 && i < 9+5) {
							val[i-4] = atof(tmp[i]);
						}
					}
					fclose(fp);
					fp = fopen(input_file, "w");
					fprintf(fp, "F\n9\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g", val[0], val[1], val[2], val[3], val[4], val[5], val[6], val[7], val[8]);
					fclose(fp);
#else
					fp = fopen(solution_file, "r");
					for (i = 0; i < 8+5; i++) {
						fscanf(fp, "%s", &tmp[i]);
						if (i >= 4 && i < 8+5) {
							val[i-4] = atof(tmp[i]);
						}
					}
					fclose(fp);
					fp = fopen(input_file, "w");
					fprintf(fp, "F\n8\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g\n%.15g", val[0], val[1], val[2], val[3], val[4], val[5], val[6], val[7]);
					fclose(fp);
#endif
					sprintf(szCmdLine, "%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %d %s",
						EVALUATE_Q_PATH, atlas_mask_file, scan2_atlas_reg_abnor_mask_file,
						scan0_atlas_reg_image_masked_list, scan0_atlas_reg_prior_list, scan0_means_file, scan0_variances_file, scan0_h_hdr_file, scan0_h_hdr_file, //scan0_h2_hdr_file,
						scan2_atlas_reg_image_masked_list, scan2_atlas_reg_posterior_list, label_map_s_img_file,
						simulator_input_file, scales_file, out_folder, tmp_folder,
						input_file, output_file, tag, tag_F);
					//
					TRACE("%s\n", szCmdLine);
					//
					if (!ExecuteProcess(szCmdLine)) {
						exit(EXIT_FAILURE);
					}

					// refresh tumor deformation field
					CopyMHDData(NULL, u_hdr_file, NULL, scan0_u_hdr_file, true);
					CopyMHDData(NULL, u_inv_hdr_file, NULL, scan0_u_inv_hdr_file, true);
				}
#endif
			}

			// update deformation, means, variances
			if (!IsFileExist(h_hdr_file)) {
				TRACE("Running JSR...\n");
				//
				/////////////////////////////////////////////////////////////////////////////
				char trace_file_out[1024];
				fflush(stdout);
				sprintf(trace_file_out, "%s%ss0_log_jsr_%d.txt", tmp_folder, DIR_SEP, k);
#if defined(WIN32) || defined(WIN64)
				FILE *fp_trace_out;
				fp_trace_out = freopen(trace_file_out, "w", stdout);
#else
				int fd_trace_out;
				fpos_t pos_out; 
				fgetpos(stdout, &pos_out);
				fd_trace_out = _dup(fileno(stdout));
				freopen(trace_file_out, "w", stdout);
#endif
				/////////////////////////////////////////////////////////////////////////////
				//
				if (!EstimateDeformationFieldJSR(scan0_atlas_reg_image_masked_files, atlas_reg_mass_prior_files, 
					&scan0_MeanVector, h_hdr_file, means_file, variances_file, cost_file, k, scan0_jsr_max_iter, num_itk_threads, false)) {
					exit(EXIT_FAILURE);
				}
				//
				/////////////////////////////////////////////////////////////////////////////
				fflush(stdout);
#if defined(WIN32) || defined(WIN64)
				fp_trace_out = freopen("CON", "w", stdout);
#else
				_dup2(fd_trace_out, fileno(stdout));
				close(fd_trace_out);
				clearerr(stdout);
				fsetpos(stdout, &pos_out);
#endif
				/////////////////////////////////////////////////////////////////////////////
				//
				TRACE("Running JSR...done\n");

#if 0
				for (j = 0; j < num_of_iter[2]; j++) {
					char file1[1024], file2[1024];
					//
					sprintf(file1, "jsr_means_%d.txt", j);
					sprintf(file2, "s0_means_jsr_%d_%d.txt", k, j);
					DeleteFile(file2);
					MoveFile(file1, file2);
					//
					sprintf(file1, "jsr_variances_%d.txt", j);
					sprintf(file2, "s0_variances_jsr_%d_%d.txt", k, j);
					DeleteFile(file2);
					MoveFile(file1, file2);
					//
					for (i = 0; i < NumberOfPriorChannels; i++) {
						sprintf(file1, "jsr_weight_image_%d_%d.nii.gz", j, i);
						sprintf(file2, "s0_weight_image_jsr_%d_%d_%d.nii.gz", k, j, i);
						DeleteFile(file2);
						MoveFile(file1, file2);
						//
						sprintf(file1, "jsr_warped_image_%d_%d.nii.gz", j, i);
						sprintf(file2, "s0_warped_image_jsr_%d_%d_%d.nii.gz", k, j, i);
						DeleteFile(file2);
						MoveFile(file1, file2);
					}
				}
#endif

				// refresh deformation field
				CopyMHDData(NULL, h_hdr_file, NULL, scan0_h_hdr_file, true);
				// refresh means and variances
				LoadMeansAndVariances(means_file, variances_file, &scan0_MeanVector, &scan0_VarianceVector);
				SaveMeansAndVariances(scan0_means_file, scan0_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
			} else {
				CopyMHDData(NULL, h_hdr_file, NULL, scan0_h_hdr_file, true);
				//
				LoadMeansAndVariances(means_file, variances_file, &scan0_MeanVector, &scan0_VarianceVector);
				SaveMeansAndVariances(scan0_means_file, scan0_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
			}

			// get posteriors
			if (!IsFileExist(atlas_reg_posterior_files[0])) {
				FVolume v3;

				// load data
				scan0_atlas_reg_mass_priors[CSF].load(atlas_reg_mass_prior_files[CSF], 1);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_mass_priors[VT ].load(atlas_reg_mass_prior_files[VT ], 1);
#endif
				scan0_atlas_reg_mass_priors[GM ].load(atlas_reg_mass_prior_files[GM ], 1);
				scan0_atlas_reg_mass_priors[WM ].load(atlas_reg_mass_prior_files[WM ], 1);
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_mass_priors[VS ].load(atlas_reg_mass_prior_files[VS ], 1);
#endif
				scan0_atlas_reg_mass_priors[TU ].load(atlas_reg_mass_prior_files[TU ], 1);
				scan0_atlas_reg_mass_priors[ED ].load(atlas_reg_mass_prior_files[ED ], 1);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_mass_priors[RTE].load(atlas_reg_mass_prior_files[RTE], 1);
#endif
				//
				if (!LoadMHDData(NULL, h_hdr_file, &v3.m_pData, v3.m_vd_x, v3.m_vd_y, v3.m_vd_z, v3.m_vd_s, v3.m_vd_dx, v3.m_vd_dy, v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
					TRACE("Loading %s failed..\n", h_hdr_file);
				} else  {
					v3.computeDimension();
				}

				scan0_atlas_reg_warp_priors[CSF].allocate(vd_x, vd_y, vd_z);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_warp_priors[VT ].allocate(vd_x, vd_y, vd_z);
#endif
				scan0_atlas_reg_warp_priors[GM ].allocate(vd_x, vd_y, vd_z);
				scan0_atlas_reg_warp_priors[WM ].allocate(vd_x, vd_y, vd_z);
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_warp_priors[VS ].allocate(vd_x, vd_y, vd_z);
#endif
#ifdef USE_WARPED_BG
				scan0_atlas_reg_warp_priors[BG ].allocate(vd_x, vd_y, vd_z);
#endif
				scan0_atlas_reg_warp_priors[TU ].allocate(vd_x, vd_y, vd_z);
				scan0_atlas_reg_warp_priors[ED ].allocate(vd_x, vd_y, vd_z);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_warp_priors[RTE].allocate(vd_x, vd_y, vd_z);
#endif

				// apply warp
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[CSF], scan0_atlas_reg_mass_priors[CSF], v3, 0.0f, false);
#ifdef USE_11A_PRIORS
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[VT ], scan0_atlas_reg_mass_priors[VT ], v3, 0.0f, false);
#endif
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[GM ], scan0_atlas_reg_mass_priors[GM ], v3, 0.0f, false);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[WM ], scan0_atlas_reg_mass_priors[WM ], v3, 0.0f, false);
#ifdef USE_VS_PRIOR
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[VS ], scan0_atlas_reg_mass_priors[VS ], v3, 0.0f, false);
#endif
#ifdef USE_WARPED_BG
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[BG ], scan0_atlas_reg_mass_priors[BG ], v3, 1.0f, false);
#endif
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[TU ], scan0_atlas_reg_mass_priors[TU ], v3, 0.0f, false);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[ED ], scan0_atlas_reg_mass_priors[ED ], v3, 0.0f, false);
#ifdef USE_11A_PRIORS
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[RTE], scan0_atlas_reg_mass_priors[RTE], v3, 0.0f, false);
#endif

				v3.clear();
				scan0_atlas_reg_mass_priors[CSF].clear();
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_mass_priors[VT ].clear();
#endif
				scan0_atlas_reg_mass_priors[GM ].clear();
				scan0_atlas_reg_mass_priors[WM ].clear();
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_mass_priors[VS ].clear();
#endif
#ifdef USE_WARPED_BG
				scan0_atlas_reg_mass_priors[BG ].clear();
#endif
				scan0_atlas_reg_mass_priors[TU ].clear();
				scan0_atlas_reg_mass_priors[ED ].clear();
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_mass_priors[RTE].clear();
#endif

#ifndef USE_WARPED_BG
				scan0_atlas_reg_warp_priors[BG ].load(scan0_atlas_reg_prior_files[BG ], 1);
#endif
#ifndef USE_VS_PRIOR
				scan0_atlas_reg_warp_priors[VS ].copy(scan0_atlas_reg_warp_priors[CSF]);
#endif
				scan0_atlas_reg_warp_priors[NCR].copy(scan0_atlas_reg_warp_priors[TU ]);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_warp_priors[RTN].copy(scan0_atlas_reg_warp_priors[RTE]);
#endif

				for (i = 0; i < NumberOfImageChannels; i++) {
					//scan0_atlas_reg_images[i].load(scan0_atlas_reg_image_files[i], 1);
					scan0_atlas_reg_images[i].load(scan0_atlas_reg_image_masked_files[i], 1);
				}
				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan0_atlas_reg_posteriors[i].allocate(vd_x, vd_y, vd_z);
				}

				// store output posteriors for scan 0
				if (!ComputePosteriors(scan0_atlas_reg_images, scan0_atlas_reg_warp_priors, &scan0_MeanVector, &scan0_VarianceVector, scan0_atlas_reg_posteriors)) {
					TRACE("ComputePosteriors failed..\n");
					exit(EXIT_FAILURE);
				} else {
					for (i = 0; i < NumberOfPriorChannels; i++) {
						scan0_atlas_reg_warp_priors[i].save(atlas_reg_warp_prior_files[i], 1);
						ChangeNIIHeader(atlas_reg_warp_prior_files[i], scan0_atlas_reg_image_files[0]);
						//
#ifdef USE_SMOOTH_POSTERIOR
						FVolume post_g;
						scan0_atlas_reg_posteriors[i].GaussianSmoothing(post_g, 0.67, 5);
						post_g.save(atlas_reg_posterior_files[i], 1);
						post_g.clear();
#else
						scan0_atlas_reg_posteriors[i].save(atlas_reg_posterior_files[i], 1);
#endif
						ChangeNIIHeader(atlas_reg_posterior_files[i], scan0_atlas_reg_image_files[0]);
					}
				}

				if (k == scan0_jsr_max_iter-1) {
					for (i = 0; i < NumberOfPriorChannels; i++) {
						CopyFile(atlas_reg_mass_prior_files[i], scan0_atlas_reg_mass_prior_files[i], FALSE);
						//
						scan0_atlas_reg_warp_priors[i].save(scan0_atlas_reg_warp_prior_files[i], 1);
						ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[i], scan0_atlas_reg_image_files[0]);
						//
						scan0_atlas_reg_posteriors[i].save(scan0_atlas_reg_posterior_files[i], 1);
						ChangeNIIHeader(scan0_atlas_reg_posterior_files[i], scan0_atlas_reg_image_files[0]);
					}
				}

				// free data
				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan0_atlas_reg_warp_priors[i].clear();
					scan0_atlas_reg_posteriors[i].clear();
				}
				for (i = 0; i < NumberOfImageChannels; i++) {
					scan0_atlas_reg_images[i].clear();
				}
			}
		} // k
	}
#endif
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// generate atlas reg label map for scan 0
	/////////////////////////////////////////////////////////////////////////////
	if (!IsFileExist(scan0_atlas_reg_label_map_file)) {
		for (i = 0; i < NumberOfPriorChannels; i++) {
			scan0_atlas_reg_posteriors[i].load(scan0_atlas_reg_posterior_files[i], 1);
		}
		//
		scan0_atlas_reg_label_map.allocate(scan0_atlas_reg_posteriors[0].m_vd_x, scan0_atlas_reg_posteriors[0].m_vd_y, scan0_atlas_reg_posteriors[0].m_vd_z);
		//
		for (k = 0; k < scan0_atlas_reg_label_map.m_vd_z; k++) {
			for (j = 0; j < scan0_atlas_reg_label_map.m_vd_y; j++) {
				for (i = 0; i < scan0_atlas_reg_label_map.m_vd_x; i++) {
					float max_l_val = scan0_atlas_reg_posteriors[0].m_pData[k][j][i][0];
					int max_l_idx = 0;
					for (l = 1; l < NumberOfPriorChannels; l++) {
						// prefer higher labels
						if (max_l_val <= scan0_atlas_reg_posteriors[l].m_pData[k][j][i][0]) {
							max_l_val = scan0_atlas_reg_posteriors[l].m_pData[k][j][i][0];
							max_l_idx = l;
						}
					}
					scan0_atlas_reg_label_map.m_pData[k][j][i][0] = label_idx[max_l_idx];
				}
			}
		}
		//
		scan0_atlas_reg_label_map.save(scan0_atlas_reg_label_map_file, 1);
		ChangeNIIHeader(scan0_atlas_reg_label_map_file, scan0_atlas_reg_posterior_files[0]);

		// free data
		{
			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_atlas_reg_posteriors[i].clear();
			}
			//
			scan0_atlas_reg_label_map.clear();
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// Update deformation field
	/////////////////////////////////////////////////////////////////////////////
	{
		int vd_x, vd_y, vd_z;
		float vd_dx, vd_dy, vd_dz;
		float _vd_dx, _vd_dy, _vd_dz;
		float vd_ox, vd_oy, vd_oz;
		char h1_hdr_file[1024];
		char h1u_hdr_file[1024];
		char h1u_r_hdr_file[1024];
		char h2u_hdr_file[1024];
		char h2u_r_hdr_file[1024];
		char h2u_log_name[1024];
		//
		FVolume scan0_sc;
		FVolume scan2_sc;
		FVolume scan0_hu_x, scan0_hu_y, scan0_hu_z;
		FVolume scan0_hu_r_x, scan0_hu_r_y, scan0_hu_r_z;

		{
			scan0_atlas_reg_images[0].load(scan0_atlas_reg_image_files[0], 1);
			//
			vd_x = scan0_atlas_reg_images[0].m_vd_x;
			vd_y = scan0_atlas_reg_images[0].m_vd_y;
			vd_z = scan0_atlas_reg_images[0].m_vd_z;
			vd_dx = scan0_atlas_reg_images[0].m_vd_dx;
			vd_dy = scan0_atlas_reg_images[0].m_vd_dy;
			vd_dz = scan0_atlas_reg_images[0].m_vd_dz;
			_vd_dx = 1.0 / vd_dx;
			_vd_dy = 1.0 / vd_dy;
			_vd_dz = 1.0 / vd_dz;
			//
			scan0_atlas_reg_images[0].clear();
		}

		sprintf(h1_hdr_file   , "%s%ss0_h1.mhd"   , tmp2_folder, DIR_SEP);
		sprintf(h1u_hdr_file  , "%s%ss0_h1u.mhd"  , tmp2_folder, DIR_SEP);
		sprintf(h1u_r_hdr_file, "%s%ss0_h1u_r.mhd", tmp2_folder, DIR_SEP);
		sprintf(h2u_hdr_file  , "%s%ss0_h2u.mhd"  , tmp2_folder, DIR_SEP);
		sprintf(h2u_r_hdr_file, "%s%ss0_h2u_r.mhd", tmp2_folder, DIR_SEP);
		sprintf(h2u_log_name  , "%s%ss0_h2u"      , tmp2_folder, DIR_SEP);

		for (i = 0; i < NumberOfImageChannels; i++) {
			scan0_atlas_reg_images[i].load(scan0_atlas_reg_image_masked_files[i], 1);
			scan2_atlas_reg_images[i].load(scan2_atlas_reg_image_masked_files[i], 1);
		}
		//
		scan2_atlas_reg_tumor_mask.load(scan2_atlas_reg_tumor_mask_file, 1);
		scan2_atlas_reg_tumor_prob.load(scan2_atlas_reg_tumor_prob_file, 1);
		scan0_atlas_reg_tumor_mask.allocate(vd_x, vd_y, vd_z);
		scan0_atlas_reg_tumor_prob.allocate(vd_x, vd_y, vd_z);
		//
		scan2_atlas_reg_abnor_mask.load(scan2_atlas_reg_abnor_mask_file, 1);
		scan2_atlas_reg_abnor_prob.load(scan2_atlas_reg_abnor_prob_file, 1);
		scan0_atlas_reg_abnor_mask.allocate(vd_x, vd_y, vd_z);
		scan0_atlas_reg_abnor_prob.allocate(vd_x, vd_y, vd_z);
		scan0_sc.allocate(vd_x, vd_y, vd_z);
		scan2_sc.allocate(vd_x, vd_y, vd_z);

		{
			// get tumor mask for scan 0
			{
				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan0_atlas_reg_posteriors[i].load(scan0_atlas_reg_posterior_files[i], 1);
				}

				for (n = 0; n < vd_z; n++) {
					for (m = 0; m < vd_y; m++) {
						for (l = 0; l < vd_x; l++) {
							float max_l_val = scan0_atlas_reg_posteriors[0].m_pData[n][m][l][0];
							int max_l_idx = 0;
							for (c = 1; c < NumberOfPriorChannels; c++) {
								// prefer higher labels
								if (max_l_val <= scan0_atlas_reg_posteriors[c].m_pData[n][m][l][0]) {
									max_l_val = scan0_atlas_reg_posteriors[c].m_pData[n][m][l][0];
									max_l_idx = c;
								}
							}
							if (max_l_idx == TU || max_l_idx == NCR) {
								scan0_atlas_reg_tumor_mask.m_pData[n][m][l][0] = 1;
							} else {
								scan0_atlas_reg_tumor_mask.m_pData[n][m][l][0] = 0;
							}
							if (max_l_idx == TU || max_l_idx == NCR || max_l_idx == ED) {
								scan0_atlas_reg_abnor_mask.m_pData[n][m][l][0] = 1;
							} else {
								scan0_atlas_reg_abnor_mask.m_pData[n][m][l][0] = 0;
							}
							//
							scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] = 0;
							scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] += scan0_atlas_reg_posteriors[TU ].m_pData[n][m][l][0];
							scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] += scan0_atlas_reg_posteriors[NCR].m_pData[n][m][l][0];
							//
							//scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] *= scan0_atlas_reg_warp_priors[TU].m_pData[n][m][l][0] + scan0_atlas_reg_warp_priors[NCR].m_pData[n][m][l][0];
							//
							if (scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] > 1.0f) {
								scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] = 1.0f;
							}
							scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0] = 0;
							scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0] += scan0_atlas_reg_posteriors[TU ].m_pData[n][m][l][0];
							scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0] += scan0_atlas_reg_posteriors[NCR].m_pData[n][m][l][0];
							scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0] += scan0_atlas_reg_posteriors[ED ].m_pData[n][m][l][0];
							//
							//scan0_atlas_reg_tumor_prob.m_pData[n][m][l][0] *= scan0_atlas_reg_warp_priors[TU].m_pData[n][m][l][0] + scan0_atlas_reg_warp_priors[NCR].m_pData[n][m][l][0] + scan0_atlas_reg_warp_priors[ED].m_pData[n][m][l][0];
							//
							if (scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0] > 1.0f) {
								scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0] = 1.0f;
							}
						}
					}
				}

				scan0_atlas_reg_tumor_mask.save(scan0_atlas_reg_tumor_mask_file, 1);
				ChangeNIIHeader(scan0_atlas_reg_tumor_mask_file, scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_tumor_prob.save(scan0_atlas_reg_tumor_prob_file, 1);
				ChangeNIIHeader(scan0_atlas_reg_tumor_prob_file, scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_abnor_mask.save(scan0_atlas_reg_abnor_mask_file, 1);
				ChangeNIIHeader(scan0_atlas_reg_abnor_mask_file, scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_abnor_prob.save(scan0_atlas_reg_abnor_prob_file, 1);
				ChangeNIIHeader(scan0_atlas_reg_abnor_prob_file, scan0_atlas_reg_image_files[0]);

				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan0_atlas_reg_posteriors[i].clear();
				}
			}

			// smoothness cost scale
			{
				//scan0_atlas_reg_posteriors[VT].load(scan0_atlas_reg_posterior_files[VT], 1);
				//scan2_atlas_reg_posteriors[VT].load(scan2_atlas_reg_posterior_files[VT], 1);

				for (n = 0; n < vd_z; n++) {
					for (m = 0; m < vd_y; m++) {
						for (l = 0; l < vd_x; l++) {
#ifdef USE_11A_PRIORS
							//scan0_sc.m_pData[n][m][l][0] = 1.0f - 0.2f * scan0_atlas_reg_posteriors[VT].m_pData[n][m][l][0];
							scan0_sc.m_pData[n][m][l][0] = 1;
							//
							//scan2_sc.m_pData[n][m][l][0] = 1.0f - 0.2f * scan2_atlas_reg_posteriors[VT].m_pData[n][m][l][0];
							scan2_sc.m_pData[n][m][l][0] = 1;
#else
							scan0_sc.m_pData[n][m][l][0] = 1;
							scan2_sc.m_pData[n][m][l][0] = 1;
#endif
						}
					}
				}

				//scan0_atlas_reg_posteriors[VT].clear();
				//scan2_atlas_reg_posteriors[VT].clear();
			}

			// fill abnormal region with average wm value
			{
				float scan0_mean_wm[NumberOfImageChannels], scan2_mean_wm[NumberOfImageChannels];

				LoadMeansAndVariances(scan0_means_file, scan0_variances_file, &scan0_MeanVector, &scan0_VarianceVector);
				LoadMeansAndVariances(scan2_means_file, scan2_variances_file, &scan2_MeanVector, &scan2_VarianceVector);

				for (i = 0; i < NumberOfImageChannels; i++) {
					scan0_mean_wm[i] = scan0_MeanVector.at(WM)[i];
					scan2_mean_wm[i] = scan2_MeanVector.at(WM)[i];
				}

				for (n = 0; n < vd_z; n++) {
					for (m = 0; m < vd_y; m++) {
						for (l = 0; l < vd_x; l++) {
							double alpha, v0;

							alpha = scan0_atlas_reg_abnor_prob.m_pData[n][m][l][0];
							if (alpha > 0) {
								for (i = 0; i < NumberOfImageChannels; i++) {
									v0 = scan0_atlas_reg_images[i].m_pData[n][m][l][0];
									scan0_atlas_reg_images[i].m_pData[n][m][l][0] = alpha * scan0_mean_wm[i] + (1.0-alpha) * v0;
								}
							}

							alpha = scan2_atlas_reg_abnor_prob.m_pData[n][m][l][0];
							if (alpha > 0) {
								for (i = 0; i < NumberOfImageChannels; i++) {
									v0 = scan2_atlas_reg_images[i].m_pData[n][m][l][0];
									scan2_atlas_reg_images[i].m_pData[n][m][l][0] = alpha * scan2_mean_wm[i] + (1.0-alpha) * v0;
								}
							}
						}
					}
				}
			}

			{
				FVolume scan0_u_v3;
				FVolume scan0_h_v3;

				scan0_hu_x.allocate(vd_x, vd_y, vd_z); scan0_hu_x.m_vd_dx = vd_dx; scan0_hu_x.m_vd_dy = vd_dy; scan0_hu_x.m_vd_dz = vd_dz;
				scan0_hu_y.allocate(vd_x, vd_y, vd_z); scan0_hu_y.m_vd_dx = vd_dx; scan0_hu_y.m_vd_dy = vd_dy; scan0_hu_y.m_vd_dz = vd_dz;
				scan0_hu_z.allocate(vd_x, vd_y, vd_z); scan0_hu_z.m_vd_dx = vd_dx; scan0_hu_z.m_vd_dy = vd_dy; scan0_hu_z.m_vd_dz = vd_dz;
				scan0_hu_r_x.allocate(vd_x, vd_y, vd_z); scan0_hu_r_x.m_vd_dx = vd_dx; scan0_hu_r_x.m_vd_dy = vd_dy; scan0_hu_r_x.m_vd_dz = vd_dz;
				scan0_hu_r_y.allocate(vd_x, vd_y, vd_z); scan0_hu_r_y.m_vd_dx = vd_dx; scan0_hu_r_y.m_vd_dy = vd_dy; scan0_hu_r_y.m_vd_dz = vd_dz;
				scan0_hu_r_z.allocate(vd_x, vd_y, vd_z); scan0_hu_r_z.m_vd_dx = vd_dx; scan0_hu_r_z.m_vd_dy = vd_dy; scan0_hu_r_z.m_vd_dz = vd_dz;

				if (!LoadMHDData(NULL, scan0_u_hdr_file, 
					&scan0_u_v3.m_pData, scan0_u_v3.m_vd_x, scan0_u_v3.m_vd_y, scan0_u_v3.m_vd_z, scan0_u_v3.m_vd_s, scan0_u_v3.m_vd_dx, scan0_u_v3.m_vd_dy, scan0_u_v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
					TRACE("Loading %s failed..\n", scan0_u_hdr_file);
				} else  {
					scan0_u_v3.computeDimension();
					//
					{
						scan0_u_x.allocate(vd_x, vd_y, vd_z); scan0_u_x.m_vd_dx = vd_dx; scan0_u_x.m_vd_dy = vd_dy; scan0_u_x.m_vd_dz = vd_dz;
						scan0_u_y.allocate(vd_x, vd_y, vd_z); scan0_u_y.m_vd_dx = vd_dx; scan0_u_y.m_vd_dy = vd_dy; scan0_u_y.m_vd_dz = vd_dz;
						scan0_u_z.allocate(vd_x, vd_y, vd_z); scan0_u_z.m_vd_dx = vd_dx; scan0_u_z.m_vd_dy = vd_dy; scan0_u_z.m_vd_dz = vd_dz;
						for (n = 0; n < vd_z; n++) {
							for (m = 0; m < vd_y; m++) {
								for (l = 0; l < vd_x; l++) {
									// mm -> voxel
									scan0_u_x.m_pData[n][m][l][0] = scan0_u_v3.m_pData[n][m][l][0] * _vd_dx;
									scan0_u_y.m_pData[n][m][l][0] = scan0_u_v3.m_pData[n][m][l][1] * _vd_dy;
									scan0_u_z.m_pData[n][m][l][0] = scan0_u_v3.m_pData[n][m][l][2] * _vd_dz;
								}
							}
						}
					}
					//
					scan0_u_v3.clear();
				}
				//if (!LoadMHDData(NULL, h1_hdr_file, 
				if (!LoadMHDData(NULL, scan0_h_hdr_file, 
					&scan0_h_v3.m_pData, scan0_h_v3.m_vd_x, scan0_h_v3.m_vd_y, scan0_h_v3.m_vd_z, scan0_h_v3.m_vd_s, scan0_h_v3.m_vd_dx, scan0_h_v3.m_vd_dy, scan0_h_v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
					TRACE("Loading %s failed..\n", scan0_h_hdr_file);
				} else  {
					scan0_h_v3.computeDimension();
					//
					{
						scan0_h_x.allocate(vd_x, vd_y, vd_z); scan0_h_x.m_vd_dx = vd_dx; scan0_h_x.m_vd_dy = vd_dy; scan0_h_x.m_vd_dz = vd_dz;
						scan0_h_y.allocate(vd_x, vd_y, vd_z); scan0_h_y.m_vd_dx = vd_dx; scan0_h_y.m_vd_dy = vd_dy; scan0_h_y.m_vd_dz = vd_dz;
						scan0_h_z.allocate(vd_x, vd_y, vd_z); scan0_h_z.m_vd_dx = vd_dx; scan0_h_z.m_vd_dy = vd_dy; scan0_h_z.m_vd_dz = vd_dz;
						for (n = 0; n < vd_z; n++) {
							for (m = 0; m < vd_y; m++) {
								for (l = 0; l < vd_x; l++) {
									// mm -> voxel
									scan0_h_x.m_pData[n][m][l][0] = scan0_h_v3.m_pData[n][m][l][0] * _vd_dx;
									scan0_h_y.m_pData[n][m][l][0] = scan0_h_v3.m_pData[n][m][l][1] * _vd_dy;
									scan0_h_z.m_pData[n][m][l][0] = scan0_h_v3.m_pData[n][m][l][2] * _vd_dz;
								}
							}
						}
					}
					//
					scan0_h_v3.clear();
				}

				ConcatenateFields(scan0_h_x, scan0_h_y, scan0_h_z, scan0_u_x, scan0_u_y, scan0_u_z, scan0_hu_x, scan0_hu_y, scan0_hu_z);
				//ComposeDiffs(scan0_h_x, scan0_h_y, scan0_h_z, scan0_u_x, scan0_u_y, scan0_u_z, scan0_hu_x, scan0_hu_y, scan0_hu_z, 1.0);

#ifdef USE_INIT_FIELD
				//*
				// weight abnormal region
				{
					FVolume prob_g;
					prob_g.allocate(vd_x, vd_y, vd_z);

					//scan0_atlas_reg_abnor_prob.GaussianSmoothing(prob_g, 3.0, 9);
					scan0_atlas_reg_tumor_prob.GaussianSmoothing(prob_g, 3.0, 9);

					for (n = 0; n < vd_z; n++) {
						for (m = 0; m < vd_y; m++) {
							for (l = 0; l < vd_x; l++) {
								double abnor_prob;

								abnor_prob  = prob_g.m_pData[n][m][l][0];

								scan0_hu_x.m_pData[n][m][l][0] = scan0_hu_x.m_pData[n][m][l][0] * abnor_prob;
								scan0_hu_y.m_pData[n][m][l][0] = scan0_hu_y.m_pData[n][m][l][0] * abnor_prob;
								scan0_hu_z.m_pData[n][m][l][0] = scan0_hu_z.m_pData[n][m][l][0] * abnor_prob;
							}
						}
					}

					prob_g.clear();

					/*
					FVolume scan0_hu_x_t = scan0_hu_x;
					FVolume scan0_hu_y_t = scan0_hu_y;
					FVolume scan0_hu_z_t = scan0_hu_z;
					scan0_hu_x_t.GaussianSmoothing(scan0_hu_x, 3.0, 5);
					scan0_hu_y_t.GaussianSmoothing(scan0_hu_y, 3.0, 5);
					scan0_hu_z_t.GaussianSmoothing(scan0_hu_z, 3.0, 5);
					/*/
					SmoothField(scan0_hu_x, scan0_hu_y, scan0_hu_z, scan0_hu_x, scan0_hu_y, scan0_hu_z, 3.0);
					//*/

					/*
					// caution: this function performs bad
					ReverseDeformationField(scan0_hu_x, scan0_hu_y, scan0_hu_z, scan0_hu_r_x, scan0_hu_r_y, scan0_hu_r_z);
					ReverseDeformationField(scan0_hu_r_x, scan0_hu_r_y, scan0_hu_r_z, scan0_hu_x, scan0_hu_y, scan0_hu_z);
					/*/
					ReverseField(scan0_hu_x, scan0_hu_y, scan0_hu_z, scan0_hu_r_x, scan0_hu_r_y, scan0_hu_r_z);
					ReverseField(scan0_hu_r_x, scan0_hu_r_y, scan0_hu_r_z, scan0_hu_x, scan0_hu_y, scan0_hu_z);
					//*/
					//
					SaveMHDData(NULL, h1u_r_hdr_file, scan0_hu_r_x.m_pData, scan0_hu_r_y.m_pData, scan0_hu_r_z.m_pData, scan0_hu_r_x.m_vd_x, scan0_hu_r_x.m_vd_y, scan0_hu_r_x.m_vd_z, scan0_hu_r_x.m_vd_dx, scan0_hu_r_x.m_vd_dy, scan0_hu_r_x.m_vd_dz, 0, 0, 0);
					SaveMHDData(NULL, h1u_hdr_file, scan0_hu_x.m_pData, scan0_hu_y.m_pData, scan0_hu_z.m_pData, scan0_hu_x.m_vd_x, scan0_hu_x.m_vd_y, scan0_hu_x.m_vd_z, scan0_hu_x.m_vd_dx, scan0_hu_x.m_vd_dy, scan0_hu_x.m_vd_dz, 0, 0, 0);
				}
				//*/
#else
				{
					for (n = 0; n < vd_z; n++) {
						for (m = 0; m < vd_y; m++) {
							for (l = 0; l < vd_x; l++) {
								scan0_hu_x.m_pData[n][m][l][0] = 0;
								scan0_hu_y.m_pData[n][m][l][0] = 0;
								scan0_hu_z.m_pData[n][m][l][0] = 0;
							}
						}
					}
				}
#endif

				scan0_h_x.clear(); scan0_h_y.clear(); scan0_h_z.clear();
				scan0_u_x.clear(); scan0_u_y.clear(); scan0_u_z.clear();
			}

			{
				UpdateDeformationFieldSyM(scan0_atlas_reg_images, scan2_atlas_reg_images, scan0_sc, scan2_sc,
					scan0_atlas_reg_tumor_prob, scan2_atlas_reg_tumor_prob, scan0_atlas_reg_abnor_prob, scan2_atlas_reg_abnor_prob, 
					scan0_hu_x, scan0_hu_y, scan0_hu_z, scan0_hu_r_x, scan0_hu_r_y, scan0_hu_r_z,
					h2u_log_name, 
					//20, 11, 3, 5, 1, false);
					//20, 0, 3, 5, 1, false);
					20, 11, 3, 5, 1, false, save_int, save_pyrd, lambda_D, lambda_P);
			}

			/*
			{
				char filename[1024];
				sprintf(filename, "s0_h2u_[l0_m1_s49]_[d20_m11]_h_r.vx3d");
				scan0_hu_r_x.load(filename, 1);
				sprintf(filename, "s0_h2u_[l0_m1_s49]_[d20_m11]_h_r.vy3d");
				scan0_hu_r_y.load(filename, 1);
				sprintf(filename, "s0_h2u_[l0_m1_s49]_[d20_m11]_h_r.vz3d");
				scan0_hu_r_z.load(filename, 1);
			}
			//*/

			SaveMHDData(NULL, h2u_hdr_file, scan0_hu_x.m_pData, scan0_hu_y.m_pData, scan0_hu_z.m_pData, scan0_hu_x.m_vd_x, scan0_hu_x.m_vd_y, scan0_hu_x.m_vd_z, scan0_hu_x.m_vd_dx, scan0_hu_x.m_vd_dy, scan0_hu_x.m_vd_dz, 0, 0, 0);
			SaveMHDData(NULL, h2u_r_hdr_file, scan0_hu_r_x.m_pData, scan0_hu_r_y.m_pData, scan0_hu_r_z.m_pData, scan0_hu_r_x.m_vd_x, scan0_hu_r_x.m_vd_y, scan0_hu_r_x.m_vd_z, scan0_hu_r_x.m_vd_dx, scan0_hu_r_x.m_vd_dy, scan0_hu_r_x.m_vd_dz, 0, 0, 0);

#ifdef USE_ALIGN_SCAN
			if (align_scan) {
				ConcatenateAllFields(scan0_image_masked_files[0], scan2_align_image_files[0], scan2_to_atlas_hdr_file, h2u_hdr_file, atlas_to_scan2_hdr_file, scan0_to_scan2_align_hdr_file);
				ConcatenateAllFields(scan2_align_image_files[0], scan0_image_masked_files[0], scan2_to_atlas_hdr_file, h2u_r_hdr_file, atlas_to_scan2_hdr_file, scan2_align_to_scan0_hdr_file);
				//
				{
					sprintf(szCmdLine, "%s -fi %s -im %s -fm %s", CONCATENATE_FIELD_PATH, scan0_to_scan2_align_hdr_file, scan2_align_inv_hdr_file, scan0_to_scan2_hdr_file);
					TRACE("%s\n", szCmdLine);
					//
					if (!ExecuteProcess(szCmdLine)) {
						exit(EXIT_FAILURE);
					}
					//
					sprintf(szCmdLine, "%s -fi %s -im %s -fm %s", CONCATENATE_FIELD_PATH, scan2_align_hdr_file, scan2_align_to_scan0_hdr_file, scan2_to_scan0_hdr_file);
					TRACE("%s\n", szCmdLine);
					//
					if (!ExecuteProcess(szCmdLine)) {
						exit(EXIT_FAILURE);
					}
				}
			} else {
				ConcatenateAllFields(scan0_image_masked_files[0], scan2_image_masked_files[0], scan2_to_atlas_hdr_file, h2u_hdr_file, atlas_to_scan2_hdr_file, scan0_to_scan2_hdr_file);
				ConcatenateAllFields(scan2_image_masked_files[0], scan0_image_masked_files[0], scan2_to_atlas_hdr_file, h2u_r_hdr_file, atlas_to_scan2_hdr_file, scan2_to_scan0_hdr_file);
			}
#else
			ConcatenateAllFields(scan0_image_masked_files[0], scan2_image_masked_files[0], scan2_to_atlas_hdr_file, h2u_hdr_file, atlas_to_scan2_hdr_file, scan0_to_scan2_hdr_file);
			ConcatenateAllFields(scan2_image_masked_files[0], scan0_image_masked_files[0], scan2_to_atlas_hdr_file, h2u_r_hdr_file, atlas_to_scan2_hdr_file, scan2_to_scan0_hdr_file);
#endif

			for (i = 0; i < NumberOfImageChannels; i++) {
				scan0_atlas_reg_images[i].clear();
				scan2_atlas_reg_images[i].clear();
			}
			scan2_atlas_reg_tumor_mask.clear();
			scan2_atlas_reg_tumor_prob.clear();
			scan0_atlas_reg_tumor_mask.clear();
			scan0_atlas_reg_tumor_prob.clear();
			//
			scan0_hu_x.clear(); scan0_hu_y.clear(); scan0_hu_z.clear();
			scan0_hu_r_x.clear(); scan0_hu_r_y.clear(); scan0_hu_r_z.clear();
			//
			scan0_sc.clear();
			scan2_sc.clear();
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	// transform results
	/////////////////////////////////////////////////////////////////////////////
	{
		putenv((char*)"FSLOUTPUTTYPE=NIFTI_GZ");
		//
		// transform prior and posterior
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
				FLIRT_PATH, scan0_atlas_reg_warp_prior_files[i], scan0_image_masked_files[0], scan0_prior_files[i], atlas_to_scan2_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
			//
#ifdef USE_ALIGN_SCAN
			if (align_scan) {
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_atlas_reg_warp_prior_files[i], scan2_align_image_files[0], scan2_align_prior_files[i], atlas_to_scan2_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
				//
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_align_prior_files[i], scan2_image_masked_files[0], scan2_prior_files[i], scan2_align_inv_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			} else {
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_atlas_reg_warp_prior_files[i], scan2_image_masked_files[0], scan2_prior_files[i], atlas_to_scan2_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
#else
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
				FLIRT_PATH, scan2_atlas_reg_warp_prior_files[i], scan2_image_masked_files[0], scan2_prior_files[i], atlas_to_scan2_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
#endif
		}
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
				FLIRT_PATH, scan0_atlas_reg_posterior_files[i], scan0_image_masked_files[0], scan0_posterior_files[i], atlas_to_scan2_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
			//
#ifdef USE_ALIGN_SCAN
			if (align_scan) {
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_atlas_reg_posterior_files[i], scan2_align_image_files[0], scan2_align_posterior_files[i], atlas_to_scan2_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
				//
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_align_posterior_files[i], scan2_image_masked_files[0], scan2_posterior_files[i], scan2_align_inv_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			} else {
				sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
					FLIRT_PATH, scan2_atlas_reg_posterior_files[i], scan2_image_masked_files[0], scan2_posterior_files[i], atlas_to_scan2_mat_file);
				TRACE("%s\n", szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
#else
			sprintf(szCmdLine, "%s -in %s -ref %s -out %s -init %s -datatype float -applyxfm", 
				FLIRT_PATH, scan2_atlas_reg_posterior_files[i], scan2_image_masked_files[0], scan2_posterior_files[i], atlas_to_scan2_mat_file);
			TRACE("%s\n", szCmdLine);
			//
			if (!ExecuteProcess(szCmdLine)) {
				exit(EXIT_FAILURE);
			}
#endif
		}
		//
		// make label map
		{
			FVolume scan0_posteriors[NumberOfPriorChannels];
			BVolume scan0_label_map;

			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan0_posteriors[i].load(scan0_posterior_files[i], 1);
			}
			//
			scan0_label_map.allocate(scan0_posteriors[0].m_vd_x, scan0_posteriors[0].m_vd_y, scan0_posteriors[0].m_vd_z);
			//
			for (k = 0; k < scan0_label_map.m_vd_z; k++) {
				for (j = 0; j < scan0_label_map.m_vd_y; j++) {
					for (i = 0; i < scan0_label_map.m_vd_x; i++) {
						float max_l_val = scan0_posteriors[0].m_pData[k][j][i][0];
						int max_l_idx = 0;
						for (l = 1; l < NumberOfPriorChannels; l++) {
							// prefer higher labels
							if (max_l_val <= scan0_posteriors[l].m_pData[k][j][i][0]) {
								max_l_val = scan0_posteriors[l].m_pData[k][j][i][0];
								max_l_idx = l;
							}
						}
						scan0_label_map.m_pData[k][j][i][0] = label_idx[max_l_idx];
					}
				}
			}
			//
			scan0_label_map.save(scan0_label_map_file, 1);
			// To do: this works only for the LPS header
			ChangeNIIHeader(scan0_label_map_file, scan0_posterior_files[0]);

			// free data
			{
				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan0_posteriors[i].clear();
				}
				//
				scan0_label_map.clear();
			}
		}
		{
			FVolume scan2_posteriors[NumberOfPriorChannels];
			BVolume scan2_label_map;

			for (i = 0; i < NumberOfPriorChannels; i++) {
				scan2_posteriors[i].load(scan2_posterior_files[i], 1);
			}
			//
			scan2_label_map.allocate(scan2_posteriors[0].m_vd_x, scan2_posteriors[0].m_vd_y, scan2_posteriors[0].m_vd_z);
			//
			for (k = 0; k < scan2_label_map.m_vd_z; k++) {
				for (j = 0; j < scan2_label_map.m_vd_y; j++) {
					for (i = 0; i < scan2_label_map.m_vd_x; i++) {
						float max_l_val = scan2_posteriors[0].m_pData[k][j][i][0];
						int max_l_idx = 0;
						for (l = 1; l < NumberOfPriorChannels; l++) {
							// prefer higher labels
							if (max_l_val <= scan2_posteriors[l].m_pData[k][j][i][0]) {
								max_l_val = scan2_posteriors[l].m_pData[k][j][i][0];
								max_l_idx = l;
							}
						}
						scan2_label_map.m_pData[k][j][i][0] = label_idx[max_l_idx];
					}
				}
			}
			//
			scan2_label_map.save(scan2_label_map_file, 1);
			// To do: this works only for the LPS header
			ChangeNIIHeader(scan2_label_map_file, scan2_posterior_files[0]);

			// free data
			{
				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan2_posteriors[i].clear();
				}
				//
				scan2_label_map.clear();
			}
		}
	}
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	/////////////////////////////////////////////////////////////////////////////
	SetCurrentDirectory((char*)out_folder);
	//
	if (delete_tmp_folder) {
		DeleteAll(tmp_folder, TRUE);
	}
	/////////////////////////////////////////////////////////////////////////////

	TRACE("finished\n");

	/////////////////////////////////////////////////////////////////////////////
#ifdef USE_TRACE
	if (g_fp_trace != NULL) {
		fclose(g_fp_trace);
		g_fp_trace = NULL;
	}
	g_bTrace = FALSE;
	g_bTraceStdOut = FALSE;
	g_verbose = 0;
#endif
	/////////////////////////////////////////////////////////////////////////////

	exit(EXIT_SUCCESS);
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL TransformPoints(char* scan_image_file, char* scan_atlas_reg_image_file, char* scan_to_atlas_mat_file, double (*scan_seeds_info)[4], double (*scan_atlas_reg_seeds_info)[4], int scan_seeds_num, int check_orientation)
{
	nifti_image* pNII = NULL;
	double cx, cy, cz;
	double gxc, gyc, gzc;
	double wx, wy, wz;
	int l;
	//
	pNII = nifti_image_read(scan_image_file, 1);
	if (pNII == NULL) {
		return FALSE;
	}
	wx = pNII->nx-1;
	wy = pNII->ny-1;
	wz = pNII->nz-1;
	//
	for (l = 0; l < scan_seeds_num; l++) {
		cx = scan_seeds_info[l][0];
		cy = scan_seeds_info[l][1];
		cz = scan_seeds_info[l][2];
		//
#if 1
		// points = physical coordinates of scan 2
		// 1. convert spatial input coordinates to voxel indices
		if (check_orientation == 0) {
			// We assume RAS header
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
		} else if (check_orientation == 1) {
			// For general header
			int icod, jcod, kcod;
			nifti_mat44_to_orientation(pNII->qto_ijk, &icod, &jcod, &kcod);
			if (icod != NIFTI_L2R) { cx = -cx; }
			if (jcod != NIFTI_P2A) { cy = -cy; }
			if (kcod != NIFTI_S2I) { cz = -cz; }
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
			if (icod != NIFTI_L2R) { gxc = -gxc; }
			if (jcod != NIFTI_P2A) { gyc = -gyc; }
			if (kcod != NIFTI_S2I) { gzc = -gzc; }
		} else {
			if (check_orientation == 211) {
				cx = -cx;
			}
			if (check_orientation == 121) {
				cy = -cy;
			}
			if (check_orientation == 112) {
				cz = -cz;
			}
			if (check_orientation == 221) {
				cx = -cx;
				cy = -cy;
			}
			if (check_orientation == 212) {
				cx = -cx;
				cz = -cz;
			}
			if (check_orientation == 122) {
				cy = -cy;
				cz = -cz;
			}
			if (check_orientation == 222) {
				cx = -cx;
				cy = -cy;
				cz = -cz;
			}
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
			if (check_orientation == 211) {
				gxc = -gxc;
			}
			if (check_orientation == 121) {
				gyc = -gyc;
			}
			if (check_orientation == 112) {
				gzc = -gzc;
			}
			if (check_orientation == 221) {
				gxc = -gxc;
				gyc = -gyc;
			}
			if (check_orientation == 212) {
				gxc = -gxc;
				gzc = -gzc;
			}
			if (check_orientation == 122) {
				gyc = -gyc;
				gzc = -gzc;
			}
			if (check_orientation == 222) {
				gxc = -gxc;
				gyc = -gyc;
				gzc = -gzc;
			}
		}
		//*/
#else
		// points = voxel indices of scan 0 or scan 2
		gxc = cx;
		gyc = cy;
		gzc = cz;
#endif
		// 2. multiply by voxel size and change units to meters
		gxc = gxc * pNII->dx;
		gyc = gyc * pNII->dy;
		gzc = gzc * pNII->dz;
		//
		// 3. transform coordinates to atlas space
		{
			nifti_image* pNIIa;
			float scan_to_atlas_mat[4][4];
			double xc, yc, zc;
			//
			pNIIa = nifti_image_read(scan_atlas_reg_image_file, 1);
			if (pNIIa == NULL) {
				return FALSE;
			}
			ReadXFMData(scan_to_atlas_mat_file, scan_to_atlas_mat);
			//
			gxc = (pNII->nx-1)*pNII->dx - gxc;
			// apply transform
			xc = scan_to_atlas_mat[0][0] * gxc + scan_to_atlas_mat[0][1] * gyc + scan_to_atlas_mat[0][2] * gzc + scan_to_atlas_mat[0][3];
			yc = scan_to_atlas_mat[1][0] * gxc + scan_to_atlas_mat[1][1] * gyc + scan_to_atlas_mat[1][2] * gzc + scan_to_atlas_mat[1][3];
			zc = scan_to_atlas_mat[2][0] * gxc + scan_to_atlas_mat[2][1] * gyc + scan_to_atlas_mat[2][2] * gzc + scan_to_atlas_mat[2][3];
			//
			// voxel indices on atlas reg scan 2
			gxc = (pNIIa->nx-1) - xc / pNIIa->dx;
			gyc = yc / pNIIa->dy;
			gzc = zc / pNIIa->dz;
			//
			// multiply by voxel size and change units to meters
			gxc = gxc * pNIIa->dx;
			gyc = gyc * pNIIa->dy;
			gzc = gzc * pNIIa->dz;
			//
			nifti_image_free(pNIIa);
		}
		//
		scan_atlas_reg_seeds_info[l][0] = gxc;
		scan_atlas_reg_seeds_info[l][1] = gyc;
		scan_atlas_reg_seeds_info[l][2] = gzc;
		//
		scan_atlas_reg_seeds_info[l][3] = scan_seeds_info[l][3];
	}
	//
	nifti_image_free(pNII);

	return TRUE;
}
BOOL TransformPoints2(char* scan_image_file, char* scan_align_image_file, char* scan_atlas_reg_image_file, char* scan_align_mat_file, char* scan_to_atlas_mat_file, double (*scan_seeds_info)[4], double (*scan_atlas_reg_seeds_info)[4], int scan_seeds_num, int check_orientation)
{
	nifti_image* pNII = NULL;
	double cx, cy, cz;
	double gxc, gyc, gzc;
	double wx, wy, wz;
	int l;
	//
	pNII = nifti_image_read(scan_image_file, 1);
	if (pNII == NULL) {
		return FALSE;
	}
	wx = pNII->nx-1;
	wy = pNII->ny-1;
	wz = pNII->nz-1;
	//
	for (l = 0; l < scan_seeds_num; l++) {
		cx = scan_seeds_info[l][0];
		cy = scan_seeds_info[l][1];
		cz = scan_seeds_info[l][2];
		//
#if 1
		// points = physical coordinates of scan 2
		// 1. convert spatial input coordinates to voxel indices
		if (check_orientation == 0) {
			// We assume RAS header
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
		} else if (check_orientation == 1) {
			// For general header
			int icod, jcod, kcod;
			nifti_mat44_to_orientation(pNII->qto_ijk, &icod, &jcod, &kcod);
			if (icod != NIFTI_L2R) { cx = -cx; }
			if (jcod != NIFTI_P2A) { cy = -cy; }
			if (kcod != NIFTI_S2I) { cz = -cz; }
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
			if (icod != NIFTI_L2R) { gxc = -gxc; }
			if (jcod != NIFTI_P2A) { gyc = -gyc; }
			if (kcod != NIFTI_S2I) { gzc = -gzc; }
		} else {
			if (check_orientation == 211) {
				cx = -cx;
			}
			if (check_orientation == 121) {
				cy = -cy;
			}
			if (check_orientation == 112) {
				cz = -cz;
			}
			if (check_orientation == 221) {
				cx = -cx;
				cy = -cy;
			}
			if (check_orientation == 212) {
				cx = -cx;
				cz = -cz;
			}
			if (check_orientation == 122) {
				cy = -cy;
				cz = -cz;
			}
			if (check_orientation == 222) {
				cx = -cx;
				cy = -cy;
				cz = -cz;
			}
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
			if (check_orientation == 211) {
				gxc = -gxc;
			}
			if (check_orientation == 121) {
				gyc = -gyc;
			}
			if (check_orientation == 112) {
				gzc = -gzc;
			}
			if (check_orientation == 221) {
				gxc = -gxc;
				gyc = -gyc;
			}
			if (check_orientation == 212) {
				gxc = -gxc;
				gzc = -gzc;
			}
			if (check_orientation == 122) {
				gyc = -gyc;
				gzc = -gzc;
			}
			if (check_orientation == 222) {
				gxc = -gxc;
				gyc = -gyc;
				gzc = -gzc;
			}
		}
		//*/
#else
		// points = voxel indices of scan 0 or scan 2
		gxc = cx;
		gyc = cy;
		gzc = cz;
#endif
		// 2. multiply by voxel size and change units to meters
		gxc = gxc * pNII->dx;
		gyc = gyc * pNII->dy;
		gzc = gzc * pNII->dz;
		//
		// 3-1. transform coordinates to scan align space
		{
			nifti_image* pNIIa;
			float scan_align_mat[4][4];
			double xc, yc, zc;
			//
			pNIIa = nifti_image_read(scan_align_image_file, 1);
			if (pNIIa == NULL) {
				return FALSE;
			}
			ReadXFMData(scan_align_mat_file, scan_align_mat);
			//
			gxc = (pNII->nx-1)*pNII->dx - gxc;
			// apply transform
			xc = scan_align_mat[0][0] * gxc + scan_align_mat[0][1] * gyc + scan_align_mat[0][2] * gzc + scan_align_mat[0][3];
			yc = scan_align_mat[1][0] * gxc + scan_align_mat[1][1] * gyc + scan_align_mat[1][2] * gzc + scan_align_mat[1][3];
			zc = scan_align_mat[2][0] * gxc + scan_align_mat[2][1] * gyc + scan_align_mat[2][2] * gzc + scan_align_mat[2][3];
			//
			// voxel indices on atlas reg scan 2
			gxc = (pNIIa->nx-1) - xc / pNIIa->dx;
			gyc = yc / pNIIa->dy;
			gzc = zc / pNIIa->dz;
			//
			// multiply by voxel size and change units to meters
			gxc = gxc * pNIIa->dx;
			gyc = gyc * pNIIa->dy;
			gzc = gzc * pNIIa->dz;
			//
			nifti_image_free(pNIIa);
		}
		//
		// 3-2. transform coordinates to atlas space
		{
			nifti_image* pNIIa;
			float scan_to_atlas_mat[4][4];
			double xc, yc, zc;
			//
			pNIIa = nifti_image_read(scan_atlas_reg_image_file, 1);
			if (pNIIa == NULL) {
				return FALSE;
			}
			ReadXFMData(scan_to_atlas_mat_file, scan_to_atlas_mat);
			//
			gxc = (pNII->nx-1)*pNII->dx - gxc;
			// apply transform
			xc = scan_to_atlas_mat[0][0] * gxc + scan_to_atlas_mat[0][1] * gyc + scan_to_atlas_mat[0][2] * gzc + scan_to_atlas_mat[0][3];
			yc = scan_to_atlas_mat[1][0] * gxc + scan_to_atlas_mat[1][1] * gyc + scan_to_atlas_mat[1][2] * gzc + scan_to_atlas_mat[1][3];
			zc = scan_to_atlas_mat[2][0] * gxc + scan_to_atlas_mat[2][1] * gyc + scan_to_atlas_mat[2][2] * gzc + scan_to_atlas_mat[2][3];
			//
			// voxel indices on atlas reg scan 2
			gxc = (pNIIa->nx-1) - xc / pNIIa->dx;
			gyc = yc / pNIIa->dy;
			gzc = zc / pNIIa->dz;
			//
			// multiply by voxel size and change units to meters
			gxc = gxc * pNIIa->dx;
			gyc = gyc * pNIIa->dy;
			gzc = gzc * pNIIa->dz;
			//
			nifti_image_free(pNIIa);
		}
		//
		scan_atlas_reg_seeds_info[l][0] = gxc;
		scan_atlas_reg_seeds_info[l][1] = gyc;
		scan_atlas_reg_seeds_info[l][2] = gzc;
		//
		scan_atlas_reg_seeds_info[l][3] = scan_seeds_info[l][3];
	}
	//
	nifti_image_free(pNII);

	return TRUE;
}
BOOL GetPoints(char* scan_image_file, double (*scan_points_info)[4], double (*scan_points_info_out)[4], int scan_points_num, int check_orientation)
{
	nifti_image* pNII = NULL;
	double cx, cy, cz;
	double gxc, gyc, gzc;
	int l;
	//
	pNII = nifti_image_read(scan_image_file, 1);
	if (pNII == NULL) {
		return FALSE;
	}
	//
	for (l = 0; l < scan_points_num; l++) {
		cx = scan_points_info[l][0];
		cy = scan_points_info[l][1];
		cz = scan_points_info[l][2];
		//
		// convert spatial input coordinates to voxel indices
		if (check_orientation == 0) {
			// We assume RAS header
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
		} else if (check_orientation == 1) {
			// For general header
			int icod, jcod, kcod;
			nifti_mat44_to_orientation(pNII->qto_ijk, &icod, &jcod, &kcod);
			if (icod != NIFTI_L2R) { cx = -cx; }
			if (jcod != NIFTI_P2A) { cy = -cy; }
			if (kcod != NIFTI_S2I) { cz = -cz; }
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
			if (icod != NIFTI_L2R) { gxc = -gxc; }
			if (jcod != NIFTI_P2A) { gyc = -gyc; }
			if (kcod != NIFTI_S2I) { gzc = -gzc; }
		} else {
			if (check_orientation == 211) {
				cx = -cx;
			}
			if (check_orientation == 121) {
				cy = -cy;
			}
			if (check_orientation == 112) {
				cz = -cz;
			}
			if (check_orientation == 221) {
				cx = -cx;
				cy = -cy;
			}
			if (check_orientation == 212) {
				cx = -cx;
				cz = -cz;
			}
			if (check_orientation == 122) {
				cy = -cy;
				cz = -cz;
			}
			if (check_orientation == 222) {
				cx = -cx;
				cy = -cy;
				cz = -cz;
			}
			gxc = pNII->qto_ijk.m[0][0] * cx + pNII->qto_ijk.m[0][1] * cy + pNII->qto_ijk.m[0][2] * cz + pNII->qto_ijk.m[0][3];
			gyc = pNII->qto_ijk.m[1][0] * cx + pNII->qto_ijk.m[1][1] * cy + pNII->qto_ijk.m[1][2] * cz + pNII->qto_ijk.m[1][3];
			gzc = pNII->qto_ijk.m[2][0] * cx + pNII->qto_ijk.m[2][1] * cy + pNII->qto_ijk.m[2][2] * cz + pNII->qto_ijk.m[2][3];
			if (check_orientation == 211) {
				gxc = -gxc;
			}
			if (check_orientation == 121) {
				gyc = -gyc;
			}
			if (check_orientation == 112) {
				gzc = -gzc;
			}
			if (check_orientation == 221) {
				gxc = -gxc;
				gyc = -gyc;
			}
			if (check_orientation == 212) {
				gxc = -gxc;
				gzc = -gzc;
			}
			if (check_orientation == 122) {
				gyc = -gyc;
				gzc = -gzc;
			}
			if (check_orientation == 222) {
				gxc = -gxc;
				gyc = -gyc;
				gzc = -gzc;
			}
		}
		//*/
		//
		scan_points_info_out[l][0] = gxc;
		scan_points_info_out[l][1] = gyc;
		scan_points_info_out[l][2] = gzc;
		//
		scan_points_info_out[l][3] = scan_points_info[l][3];
	}
	//
	nifti_image_free(pNII);

	return TRUE;
}

// modify priors regaring tumor and edema
BOOL ModifyPriors(FVolume* priors, double seeds_info[MaxNumberOFTumorSeeds][4], int seeds_num)
{
	double dx, dy, dz;
	double x, y, z;
	double cx, cy, cz;
	double d2, d, p, p_1, p_max_TU, p_max_ED;
	double seeds_d_TU[MaxNumberOFTumorSeeds];
	double seeds_d_ED[MaxNumberOFTumorSeeds];
	// sigmoid function
	// assume y = 1 / (1 + exp(a * (x - b)))
	double seeds_a_TU[MaxNumberOFTumorSeeds];
	double seeds_a_ED[MaxNumberOFTumorSeeds];
	int i, j, k, l;
	int vd_x, vd_y, vd_z;

	vd_x = priors[0].m_vd_x;
	vd_y = priors[0].m_vd_y;
	vd_z = priors[0].m_vd_z;

	dx = priors[0].m_vd_dx;
	dy = priors[0].m_vd_dy;
	dz = priors[0].m_vd_dz;

	for (l = 0; l < seeds_num; l++) {
		seeds_d_TU[l] = seeds_info[l][3] / 2;
		seeds_d_ED[l] = seeds_info[l][3] * 2;
		//
		seeds_a_TU[l] = 0.8;
		seeds_a_ED[l] = 0.2;
	}

	z = 0;
	for (k = 0; k < vd_z; k++) {
		y = 0;
		for (j = 0; j < vd_y; j++) {
			x = 0;
			for (i = 0; i < vd_x; i++) {
				priors[ED].m_pData[k][j][i][0] = 0;
				priors[NCR].m_pData[k][j][i][0] = 0;
				priors[TU].m_pData[k][j][i][0] = 0;
				//
				priors[CSF].m_pData[k][j][i][0] = priors[CSF].m_pData[k][j][i][0] / 2;
				priors[VS].m_pData[k][j][i][0] = priors[CSF].m_pData[k][j][i][0];
#ifdef USE_11A_PRIORS
				priors[VT].m_pData[k][j][i][0] = priors[VT].m_pData[k][j][i][0];
#endif
#ifdef USE_11A_PRIORS
				priors[RTN].m_pData[k][j][i][0] = 0;
				priors[RTE].m_pData[k][j][i][0] = 0;
#endif
				//
				if (priors[BG].m_pData[k][j][i][0] >= 1.0) {
					x += dx;
					continue;
				}
				//
				p_max_TU = 0;
				p_max_ED = 0;
				for (l = 0; l < seeds_num; l++) {
					cx = seeds_info[l][0];
					cy = seeds_info[l][1];
					cz = seeds_info[l][2];
					d2 = (x-cx)*(x-cx) + (y-cy)*(y-cy) + (z-cz)*(z-cz);
					d =	sqrt(d2);
					//
					p = 1.0 / (1.0 + exp(seeds_a_TU[l] * (d - seeds_d_TU[l])));
					if (p > p_max_TU) {
						p_max_TU = p;
					}
					//
					p = 1.0 / (1.0 + exp(seeds_a_ED[l] * (d - seeds_d_ED[l])));
					if (p > p_max_ED) {
						p_max_ED = p;
					}
				}
				//
				if (p_max_TU > 1e-6 || p_max_ED > 1e-6) {
					p = p_max_TU;
					p_1 = 1.0 - p;
					//
					priors[TU].m_pData[k][j][i][0] = p / 2;
					priors[NCR].m_pData[k][j][i][0] = priors[TU].m_pData[k][j][i][0];
					//
#ifndef USE_ED_INC_GM
					{
						double p2, wm;
						//double w;
#ifdef USE_ED_0_5
						p2 = p_max_ED / 2;
#else
						p2 = p_max_ED / 4;
#endif
						if (p2 > p_1) {
							p2 = p_1;
						}
						/*
						w = 1.0 - p - p2;
						wm = priors[WM].m_pData[k][j][i][0];
						priors[WM].m_pData[k][j][i][0] = w  * wm;
						priors[ED].m_pData[k][j][i][0] = p2 * wm;
						/*/
						wm = priors[WM].m_pData[k][j][i][0];
						priors[WM].m_pData[k][j][i][0] = (1.0 - p) * (1.0 - p2) * wm;
						priors[ED].m_pData[k][j][i][0] = (1.0 - p) * (      p2) * wm;
						//*/
					}
					priors[GM].m_pData[k][j][i][0] = p_1 * priors[GM].m_pData[k][j][i][0];
#else
					{
						double p2, w, wm, gm;
#ifdef USE_ED_0_5
						p2 = p_max_ED / 2;
#else
						p2 = p_max_ED / 4;
#endif
						if (p2 > p_1) {
							p2 = p_1;
						}
						/*
						wm = priors[WM].m_pData[k][j][i][0];
						gm = priors[GM].m_pData[k][j][i][0];
						priors[WM].m_pData[k][j][i][0] = (1.0 - p) * ((1.0 - p2) * wm);
						priors[GM].m_pData[k][j][i][0] = (1.0 - p) * ((1.0 - p2) * gm);
						priors[ED].m_pData[k][j][i][0] = (1.0 - p) * (p2 * (wm + gm));
						/*/
						w = 1.0 - p - p2;
						wm = priors[WM].m_pData[k][j][i][0];
						gm = priors[GM].m_pData[k][j][i][0];
						priors[WM].m_pData[k][j][i][0] = w  * wm;
						priors[GM].m_pData[k][j][i][0] = w  * gm;
						priors[ED].m_pData[k][j][i][0] = p2 * (wm + gm);
						//*/
					}
#endif
					//
					priors[CSF].m_pData[k][j][i][0] = p_1 * priors[CSF].m_pData[k][j][i][0];
					priors[VS].m_pData[k][j][i][0] = priors[CSF].m_pData[k][j][i][0];
#ifdef USE_11A_PRIORS
					priors[VT].m_pData[k][j][i][0] = p_1 * priors[VT].m_pData[k][j][i][0];
#endif
				}
				//
#ifdef USE_11A_PRIORS
				double p_max_CV = 0;
				double p_max_RT = 0;
				{
					cx = seeds_info[0][0];
					cy = seeds_info[0][1];
					cz = seeds_info[0][2];
					d2 = (x-cx)*(x-cx) + (y-cy)*(y-cy) + (z-cz)*(z-cz);
					d =	sqrt(d2);
					//
					p = 1.0 / (1.0 + exp(seeds_a_TU[0] * (d - seeds_d_TU[0])));
					if (p > p_max_CV) {
						p_max_CV = p;
					}
				}
				for (l = 1; l < seeds_num; l++) {
					cx = seeds_info[l][0];
					cy = seeds_info[l][1];
					cz = seeds_info[l][2];
					d2 = (x-cx)*(x-cx) + (y-cy)*(y-cy) + (z-cz)*(z-cz);
					d =	sqrt(d2);
					//
					p = 1.0 / (1.0 + exp(seeds_a_TU[l] * (d - seeds_d_TU[l])));
					if (p > p_max_RT) {
						p_max_RT = p;
					}
				}
				//
				if (p_max_CV > 10e-6 || p_max_RT > 1e-6) {
					priors[TU ].m_pData[k][j][i][0] = p_max_CV / 2;
					priors[NCR].m_pData[k][j][i][0] = priors[TU].m_pData[k][j][i][0];
					//
					priors[RTE].m_pData[k][j][i][0] = max(p_max_RT - p_max_CV, 0.0) / 2;
					priors[RTN].m_pData[k][j][i][0] = priors[RTE].m_pData[k][j][i][0];
				}
#endif
				//
				x += dx;
			}
			y += dy;
		}
		z += dz;
	}

	return TRUE;
}

void GetMDist(float x, float y, float z, float cx, float cy, float cz, float R[9], float sx, float sy, float sz, float* dist) {
	float mx, my, mz;
	float rx, ry, rz;

	mx = x - cx;
	my = y - cy;
	mz = z - cz;
	rx = R[0] * mx + R[1] * my + R[2] * mz;
	ry = R[3] * mx + R[4] * my + R[5] * mz;
	rz = R[6] * mx + R[7] * my + R[8] * mz;

	*dist = (rx*rx) / (sx*sx) + (ry*ry) / (sy*sy) + (rz*rz) / (sz*sz);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL MakeScalesFile(const char* scales_file)
{
	FILE* fp;

	fp = fopen(scales_file, "w");
#ifdef USE_OPTIM_DG
	//           gcx  gcy  gcz  p1   p2   rho  dw    dg    T
	//fprintf(fp, "1e0\n1e0\n1e0\n1e2\n1e1\n1e0\n1e-6\n1e-7\n1e3\n");
	fprintf(fp, "1e0\n1e0\n1e0\n1e2\n1e-2\n1e0\n1e-6\n1e-7\n1e3\n");
#else
	fprintf(fp, "1e0\n1e0\n1e0\n1e2\n1e1\n1e0\n1e-6\n1e3\n");
#endif
	fclose(fp);

	return TRUE;
}
BOOL MakeSimulatorInputFile(const char* simulator_input_file, double o_dx, double o_dy, double o_dz, int x, int y, int z, double dx, double dy, double dz)
{
	FILE* fp;

	fp = fopen(simulator_input_file, "w");
	if (fp == NULL) {
		return FALSE;
	}

	fprintf(fp, "-T 100\n");
	fprintf(fp, "-ntimesteps 5\n");
	fprintf(fp, "-nstore 5\n");
	fprintf(fp, "-cfln 0.5\n");
	//
	fprintf(fp, "-gstiffwm 1.0\n");
	fprintf(fp, "-gstiffgm 1.0\n");
	fprintf(fp, "-gstiffvent 0.238\n");
#ifdef USE_11A_PRIORS
	fprintf(fp, "-gstiffcsf 1.0\n");
#else
	fprintf(fp, "-gstiffcsf 0.238\n");
#endif
	fprintf(fp, "-gdiffwm 0.65e-6\n");
	fprintf(fp, "-gdiffgm 0.13e-6\n");
	fprintf(fp, "-gdiffvent 0.0\n");
	fprintf(fp, "-gcompresbrain 0.45\n");
	//fprintf(fp, "-gcompresvent 0.1\n");
	fprintf(fp, "-gcompresvent 0.02\n");
	//
	fprintf(fp, "-gres_x %.15f\n", o_dx / 1000);
	fprintf(fp, "-gres_y %.15f\n", o_dy / 1000);
	fprintf(fp, "-gres_z %.15f\n", o_dz / 1000);
	//
	fprintf(fp, "-grho 0.01\n");
	fprintf(fp, "-gp1 2.5\n");
	fprintf(fp, "-gp2 0.0\n");
	fprintf(fp, "-gse 2.0\n");
	//
	fprintf(fp, "-gcinit 1\n");
	fprintf(fp, "-gsigsq 2.25e-5\n");
	//
	fprintf(fp, "-nsd 3\n");
	fprintf(fp, "-ndimx %d\n", (x / 8) + 1);
	fprintf(fp, "-ndimy %d\n", (y / 8) + 1);
	fprintf(fp, "-ndimz %d\n", (z / 8) + 1);
	fprintf(fp, "-imgx %d\n", x);
	fprintf(fp, "-imgy %d\n", y);
	fprintf(fp, "-imgz %d\n", z);
	fprintf(fp, "-imgdx %.15f\n", dx);
	fprintf(fp, "-imgdy %.15f\n", dy);
	fprintf(fp, "-imgdz %.15f\n", dz);
	fprintf(fp, "-Lx %.15f\n", x * dx / 1000);
	fprintf(fp, "-Ly %.15f\n", y * dy / 1000);
	fprintf(fp, "-Lz %.15f\n", z * dz / 1000);
	//
	fprintf(fp, "-material_projection_required 1\n");
	fprintf(fp, "-steady 1\n");
	fprintf(fp, "-ksp_type cg\n");
	fprintf(fp, "-ksp_rtol 1.0e-1\n");
	fprintf(fp, "-ksp_max_it 30\n");
	fprintf(fp, "-ksp_monitor\n");
	fprintf(fp, "-youngs_global 2\n");
	fprintf(fp, "-matrixfree 1\n");
	fprintf(fp, "-youngs_min 1\n");
	fprintf(fp, "-mgnlevels 4\n");
	fprintf(fp, "-restype 2\n");
	fprintf(fp, "-pcShell 2\n");
	fprintf(fp, "-mg_levels_pc_type shell\n");
	fprintf(fp, "-mg_levels_ksp_type cg\n");
	fprintf(fp, "-mg_coarse_pc_type lu\n");
	//fprintf(fp, "-mg_coarse_ksp_monitor\n");
	fprintf(fp, "-mg_coarse_ksp_max_it 10\n");
	fprintf(fp, "-mg_coarse_ksp_rtol 1e-12\n");
	fprintf(fp, "-mg_levels_ksp_max_it 5\n");
	fprintf(fp, "-mg_levels_ksp_rtol 1.0e-1\n");

	fclose(fp);

	return TRUE;
}
BOOL MakeHOPSFile(const char* hops_file, double gxc, double gyc, double gzc, double T, const char* mask_file, const char* scan2_atlas_reg_abnor_mask_file,
	const char* scan0_atlas_reg_image_list, const char* scan0_atlas_reg_init_prior_list, const char* scan0_means_file, const char* scan0_variances_file, const char* scan0_h_hdr_file, const char* scan0_h2_hdr_file,
	const char* scan2_atlas_reg_image_list, const char* scan2_atlas_reg_posterior_list, const char* label_map_s_img_file, 
	const char* simulator_input_file, const char* scales_file, const char* out_folder, const char* tmp_folder, char* solution_file, int max_eval, BOOL optim_xyz, int num_hop_threads)
{
	BOOL optim_xc = optim_xyz;
	BOOL optim_yc = optim_xyz;
	BOOL optim_zc = optim_xyz;
	BOOL optim_T = TRUE;
	BOOL optim_p1 = TRUE;
	//BOOL optim_p2 = FALSE;
	BOOL optim_p2 = TRUE;
	BOOL optim_rho = FALSE;
	BOOL optim_dw = TRUE;
#ifdef USE_OPTIM_DG
	//BOOL optim_dg = TRUE;
	BOOL optim_dg = FALSE;
#endif
	double minx, maxx, miny, maxy, minz, maxz;
	double min_growth_time, max_growth_time;
	double minp1, maxp1;
	double minp2, maxp2;
	double minrho, maxrho;
	double mindw, maxdw;
#ifdef USE_OPTIM_DG
	double mindg, maxdg;
#endif
	double rw;
	FILE* fp;

    // calcurate range weight for xc, yc, zc and growth_time
    if (max_eval < 1) {
		max_eval = 1;
	}
    rw = (double)max_eval / 100;

	// convert units
	gxc = gxc / 1000;
	gyc = gyc / 1000;
	gzc = gzc / 1000;
	T   = T   / 1000;

	// set valid ranges of parameters
	// assuming label image of atlas of size 64x64x64 and voxel size 3.5x3.5x3mm^3
	if (optim_xc) {					// scale: 1e0
		minx = gxc - (0.0105 * rw);	// -10.5 mm
		maxx = gxc + (0.0105 * rw);	// +10.5 mm
	} else {
		minx = gxc;
		maxx = gxc;
	}
	if (optim_yc) {					// scale: 1e0
		miny = gyc - (0.0105 * rw);	// -10.5 mm
		maxy = gyc + (0.0105 * rw);	// +10.5 mm
	} else {
		miny = gyc;
		maxy = gyc;
	}
	if (optim_zc) {					// scale: 1e0 
		minz = gzc - (0.0105 * rw);	// -10.5 mm
		maxz = gzc + (0.0105 * rw);	// +10.5 mm
	} else {
		minz = gzc;
		maxz = gzc;
	}
	if (optim_T) {				// scale: 1e3
		min_growth_time = (1.0 - (0.5 * rw)) * T;
		max_growth_time = (1.0 + (0.5 * rw)) * T;
	} else {
		min_growth_time = T;
		max_growth_time = T;
	}
	if (optim_p1) {				// scale: 1e2
		minp1 = 0.01;
		//maxp1 = 0.20;
		maxp1 = 0.08;
	} else {
		minp1 = 0.03;
		maxp1 = 0.03;
	}
	if (optim_p2) {				// scale: 1e1 -> 1e-2
		minp2 = 0.00;
		maxp2 = 0.20;
	} else {
		minp2 = 0.00;
		maxp2 = 0.00;
	}
	if (optim_rho) {			// scale: 1e0
		minrho = 0.01;
		maxrho = 0.05;
	} else {
		minrho = 0.03;
		maxrho = 0.03;
	}
#ifdef USE_OPTIM_DG
	/*
	if (optim_dw) {				// scale: 1e-6
		mindw = 0.0001;
		//mindw = 0.00;
		maxdw = 0.80;
		mindg = 0.02;
		maxdg = 0.02;
	} else {
		mindw = 0.10;
		maxdw = 0.10;
		mindg = 0.02;
		maxdg = 0.02;
	}
	if (optim_dg) {				// scale: 1e-7
		mindg = 0.0001;
		//mindg = 0.00;
		maxdg = 0.15;
	}
	/*/
	if (optim_dw) {				// scale: 1e-6
		mindw = 0.001;
		maxdw = 0.75;
		mindg = 0.001;
		maxdg = 0.001;
	} else {
		mindw = 0.15;
		maxdw = 0.15;
		mindg = 0.03;
		maxdg = 0.03;
	}
	if (optim_dg) {				// scale: 1e-7
		mindg = 0.001;
		maxdg = 0.03;
	}
	//*/
#else
	if (optim_dw) {				// scale: 1e-6
		//mindw = 0.075;
		mindw = 0.001;
		maxdw = 0.75;
	} else {
		// low diffusivity
		mindw = 0.075;
		maxdw = 0.075;
	}
#endif
	
	// compose input for HOPSPACK
	fp = fopen(hops_file, "w");
	if (fp == NULL) {
		return FALSE;
	}

	fprintf(fp, "@ \"Problem Definition\"\n");
	fprintf(fp, "  \"Objective Type\" string \"Minimize\"\n");
#ifdef USE_OPTIM_DG
	fprintf(fp, "  \"Number Unknowns\" int 9\n");
	fprintf(fp, "  \"Upper Bounds\" vector 9  %.15g %.15g %.15g %.15g %.15g %.15g %.15g %.15g %.15g\n", maxx, maxy, maxz, maxp1, maxp2, maxrho, maxdw, maxdg, max_growth_time);
	fprintf(fp, "  \"Lower Bounds\" vector 9  %.15g %.15g %.15g %.15g %.15g %.15g %.15g %.15g %.15g\n", minx, miny, minz, minp1, minp2, minrho, mindw, mindg, min_growth_time);
	fprintf(fp, "  \"Scaling\" vector 9  1 1 1 1 1 1 1 1 1\n");
#else
	fprintf(fp, "  \"Number Unknowns\" int 8\n");
	fprintf(fp, "  \"Upper Bounds\" vector 8  %.15g %.15g %.15g %.15g %.15g %.15g %.15g %.15g\n", maxx, maxy, maxz, maxp1, maxp2, maxrho, maxdw, max_growth_time);
	fprintf(fp, "  \"Lower Bounds\" vector 8  %.15g %.15g %.15g %.15g %.15g %.15g %.15g %.15g\n", minx, miny, minz, minp1, minp2, minrho, mindw, min_growth_time);
	fprintf(fp, "  \"Scaling\" vector 8  1 1 1 1 1 1 1 1\n");
#endif
	fprintf(fp, "@@\n");

	fprintf(fp, "@ \"Evaluator\"\n");
	fprintf(fp, "  \"Evaluator Type\" string \"System Call\"\n");
	fprintf(fp, "  \"Executable Name\" string \"%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s\"\n", 
		EVALUATE_Q_PATH, mask_file, scan2_atlas_reg_abnor_mask_file,
		scan0_atlas_reg_image_list, scan0_atlas_reg_init_prior_list, scan0_means_file, scan0_variances_file, scan0_h_hdr_file, scan0_h2_hdr_file,
		scan2_atlas_reg_image_list, scan2_atlas_reg_posterior_list, label_map_s_img_file, 
		simulator_input_file, scales_file, out_folder, tmp_folder);
	fprintf(fp, "  \"Save IO Files\" bool false\n");
	fprintf(fp, "@@\n");

	fprintf(fp, "@ \"Mediator\"\n");
	fprintf(fp, "  \"Citizen Count\" int 1\n");
	fprintf(fp, "  \"Number Processors\" int 1\n");
	fprintf(fp, "  \"Number Threads\" int %d\n", num_hop_threads+1);
	//fprintf(fp, "  \"Maximum Evaluations\" int 100\n");
	fprintf(fp, "  \"Maximum Evaluations\" int %d\n", max_eval);
	//fprintf(fp, "  \"Synchronous Evaluations\" bool true\n");
	fprintf(fp, "  \"Synchronous Evaluations\" bool false\n");
	fprintf(fp, "  \"Solution File\" string \"%s\"\n", solution_file);
	fprintf(fp, "  \"Display\" int 3\n");
	fprintf(fp, "@@\n");

	fprintf(fp, "@ \"Citizen 1\"\n");
	fprintf(fp, "  \"Type\" string \"GSS\"\n");
	fprintf(fp, "  \"Step Tolerance\" double 5.0e-3\n");
	fprintf(fp, "  \"Use Random Order\" bool true\n");
	//fprintf(fp, "  \"Use Random Order\" bool false\n");
	fprintf(fp, "  \"Display\" int 1\n");
	fprintf(fp, "@@\n");

	fclose(fp);

	return TRUE;
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL LoadMeansAndVariances(const char* means_file, const char* variances_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector)
{
	FILE* fp;
	int i, j, k;

	////////////////////////////////////////////////////////////////////////////////
	// initialize
	for (i = 0; i < NumberOfPriorChannels; i++) {
		MeanType v(0.0);
		pMeanVector->push_back(v); 
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		VarianceType s;
		s.set_identity();
		pVarianceVector->push_back(s);
	}
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// read means
	fp = fopen(means_file, "r");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", means_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		char class_id[1024];
		MeanType mean;
		float meanj;
		//
		fscanf(fp, "%s", class_id);
		for (j = 0; j < NumberOfImageChannels; j++) {
			fscanf(fp, "%f", &meanj);
			mean(j) = meanj;
		}
		for (j = 0; j < NumberOfPriorChannels; j++) {
#if defined(USE_11A_PRIORS)
			if ((strcmp(class_id, label[j]) == 0) ||
				(strcmp(class_id, label2[j]) == 0) ||
				(strcmp(class_id, label3[j]) == 0) ||
				(strcmp(class_id, label4[j]) == 0)) {
#else
			if (strcmp(class_id, label[j]) == 0) {
#endif
				pMeanVector->at(j) = mean;
				break;
			}
		}
		if (j == NumberOfPriorChannels) {
			TRACE("class id is wrong");
			fclose(fp);
			return FALSE;
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// read variances
	fp = fopen(variances_file, "r");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", variances_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		char class_id[1024];
		VarianceType var;
		float varjk;
		//
		fscanf(fp, "%s", class_id);
		for (j = 0; j < NumberOfImageChannels; j++) {
			for (k = 0; k < NumberOfImageChannels; k++) {
				fscanf(fp, "%f", &varjk);
				var(j, k) = varjk;
			}
		}
		for (j = 0; j < NumberOfPriorChannels; j++) {
#if defined(USE_11A_PRIORS)
			if ((strcmp(class_id, label[j]) == 0) ||
				(strcmp(class_id, label2[j]) == 0) ||
				(strcmp(class_id, label3[j]) == 0) ||
				(strcmp(class_id, label4[j]) == 0)) {
#else
			if (strcmp(class_id, label[j]) == 0) {
#endif
				pVarianceVector->at(j) = var;
				break;
			}
		}
		if (j == NumberOfPriorChannels) {
			TRACE("class id is wrong");
			fclose(fp);
			return FALSE;
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	return TRUE;
}

BOOL SaveMeansAndVariances(const char* means_file, const char* variances_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector)
{
	FILE* fp;
	int i, j, k;

	////////////////////////////////////////////////////////////////////////////////
	// save means
	fp = fopen(means_file, "w");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", means_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		MeanType mean;
		//
		mean = pMeanVector->at(i);
		//
		fprintf(fp, "%s\n", label[i]);
		//
		for (j = 0; j < NumberOfImageChannels; j++) {
			fprintf(fp, "%f", mean(j));
			if (j == NumberOfImageChannels-1) {
				fprintf(fp, "\n");
			} else {
				fprintf(fp, " ");
			}
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// save variances
	fp = fopen(variances_file, "w");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", variances_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		VarianceType var;
		//
		var = pVarianceVector->at(i);
		//
		fprintf(fp, "%s\n", label[i]);
		//
		for (j = 0; j < NumberOfImageChannels; j++) {
			for (k = 0; k < NumberOfImageChannels; k++) {
				fprintf(fp, "%f", var(j, k));
				if (k == NumberOfImageChannels-1) {
					fprintf(fp, "\n");
				} else {
					fprintf(fp, " ");
				}
			}
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	return TRUE;
}

BOOL InitializeMeansAndVariances(const char* means_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector)
{
	FILE* fp;
	int i, j;

	////////////////////////////////////////////////////////////////////////////////
	for(i = 0; i < NumberOfPriorChannels; i++) {
		MeanType v(0.0);
		pMeanVector->push_back(v); 
	}
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// read means
	fp = fopen(means_file, "r");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", means_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		char class_id[1024];
		MeanType mean;
		float meanj;
		//
		fscanf(fp, "%s", class_id);
		for (j = 0; j < NumberOfImageChannels; j++) {
			fscanf(fp, "%f", &meanj);
			mean(j) = meanj;
		}
		for (j = 0; j < NumberOfPriorChannels; j++) {
#if defined(USE_11A_PRIORS)
			if ((strcmp(class_id, label[j]) == 0) ||
				(strcmp(class_id, label2[j]) == 0) ||
				(strcmp(class_id, label3[j]) == 0) ||
				(strcmp(class_id, label4[j]) == 0)) {
#else
			if (strcmp(class_id, label[j]) == 0) {
#endif
				pMeanVector->at(j) = mean;
				break;
			}
		}
		if (j == NumberOfPriorChannels) {
			TRACE("class id is wrong");
			fclose(fp);
			return FALSE;
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	for(i = 0; i < NumberOfPriorChannels; i++) {
		VarianceType s;
		s.set_identity();
		s *= 10.0;
		pVarianceVector->push_back(s);
	}
	////////////////////////////////////////////////////////////////////////////////

	return TRUE;
}
BOOL InitializeMeansAndVariancesFromPoints(double (*scan_points_info)[4], int scan_points_num, char (*image_files)[1024], MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, int check_orientation, int* b_valid_label)
{
	FVolume images[NumberOfImageChannels];
	double scan_points_info_out[MaxNumberOFPoints][4];
	double color_mean[NumberOfPriorChannels][NumberOfImageChannels];
	int color_num[NumberOfPriorChannels];
	int i, j;

	for (i = 0; i < NumberOfImageChannels; i++) {
		images[i].load(image_files[i], 1);
	}

	////////////////////////////////////////////////////////////////////////////////
	pMeanVector->clear();
	for (i = 0; i < NumberOfPriorChannels; i++) {
		MeanType v(0.0);
		pMeanVector->push_back(v); 
	}
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	for (i = 0; i < NumberOfPriorChannels; i++) {
		color_num[i] = 0;
		for (j = 0; j < NumberOfImageChannels; j++) {
			color_mean[i][j] = 0;
		}
	}

	if (!GetPoints(image_files[0], scan_points_info, scan_points_info_out, scan_points_num, check_orientation)) {
		return FALSE;
	}

	for (i = 0; i < scan_points_num; i++) {
		int ix, iy, iz, type;
		ix = (int)(scan_points_info_out[i][0] + 0.1);
		iy = (int)(scan_points_info_out[i][1] + 0.1);
		iz = (int)(scan_points_info_out[i][2] + 0.1);
		type = (int)(scan_points_info_out[i][3] + 0.1);
		if (ix >= 0 && ix < images[0].m_vd_x && iy >= 0 && iy < images[0].m_vd_y && iz >= 0 && iz < images[0].m_vd_z) {
			TRACE2("point %d: (%d, %d, %d)\n", i, ix, iy, iz);
			//
			for (j = 0; j < NumberOfImageChannels; j++) {
				color_mean[type][j] += images[j].m_pData[iz][iy][ix][0];
			}
			color_num[type]++;
		} else {
			TRACE("wrong point: (%d, %d, %d), use check orientation\n", ix, iy, iz);
			return FALSE;
		}
	}

	for (i = 0; i < NumberOfPriorChannels; i++) {
		if (color_num[i] > 0) {
			for (j = 0; j < NumberOfImageChannels; j++) {
				color_mean[i][j] /= color_num[i];
			}
		}
	}
	if (color_num[BG] == 0) {
		for (j = 0; j < NumberOfImageChannels; j++) {
			color_mean[BG][j] = 0.05;
		}
		color_num[BG] = 1;
	}
	if (color_num[TU] == 0) {
		if (color_num[NCR] != 0) {
			for (j = 0; j < NumberOfImageChannels; j++) {
				color_mean[TU][j] = color_mean[NCR][j];
			}
			color_num[TU] = color_num[NCR];
		} else {
			TRACE("no point on tumor labels\n");
			return FALSE;
		}
	}
	if (color_num[NCR] == 0) {
		for (j = 0; j < NumberOfImageChannels; j++) {
			color_mean[NCR][j] = color_mean[TU][j];
		}
		color_num[NCR] = color_num[TU];
	}
#if defined(USE_11A_PRIORS)
	if (color_num[RTE] == 0) {
		if (color_num[RTN] != 0) {
			for (j = 0; j < NumberOfImageChannels; j++) {
				color_mean[RTE][j] = color_mean[RTN][j];
			}
			color_num[RTE] = color_num[RTN];
		} else {
			for (j = 0; j < NumberOfImageChannels; j++) {
				color_mean[RTE][j] = color_mean[TU][j];
				color_mean[RTN][j] = color_mean[NCR][j];
			}
			color_num[RTE] = color_num[TU];
			color_num[RTN] = color_num[NCR];
		}
	}
	if (color_num[RTN] == 0) {
		for (j = 0; j < NumberOfImageChannels; j++) {
			color_mean[RTN][j] = color_mean[RTE][j];
		}
		color_num[RTN] = color_num[RTE];
	}
#endif
	if (color_num[ED] == 0) {
		for (j = 0; j < NumberOfImageChannels; j++) {
			color_mean[ED][j] = color_mean[WM][j];
		}
		color_num[ED] = color_num[WM];
	}
	if (color_num[VT] == 0) {
		for (j = 0; j < NumberOfImageChannels; j++) {
			color_mean[VT][j] = color_mean[CSF][j];
		}
		color_num[VT] = color_num[CSF];
	}

	for (i = 0; i < NumberOfPriorChannels; i++) {
		if (color_num[i] == 0) {
			TRACE("no point on %s\n", label[i]);
			return FALSE;
		}
	}
	//
	for (i = 0; i < NumberOfPriorChannels; i++) {
		MeanType mean;
		for (j = 0; j < NumberOfImageChannels; j++) {
			mean(j) = color_mean[i][j];
		}
		pMeanVector->at(i) = mean;
	}
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	pVarianceVector->clear();
	for(i = 0; i < NumberOfPriorChannels; i++) {
		VarianceType s;
		s.set_identity();
		s *= 10.0;
		pVarianceVector->push_back(s);
	}
	////////////////////////////////////////////////////////////////////////////////

	for (i = 0; i < NumberOfImageChannels; i++) {
		images[i].clear();
	}

	return TRUE;
}

BOOL UpdateMeansAndVariances(FVolume* vd, FVolume* posteriors, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, BOOL bFirst)
{
	int i, j, k, l, m, n;
	//
	int vd_x = vd[0].m_vd_x;
	int vd_y = vd[0].m_vd_y;
	int vd_z = vd[0].m_vd_z;

	// define a vector object to keep sum of weighted vectors for each class
	double mv_sum[NumberOfPriorChannels][NumberOfImageChannels];
	// define a vector object to keep sum of weighted outer products for each class
	double vv_sum[NumberOfPriorChannels][NumberOfImageChannels][NumberOfImageChannels];
	// define a vector object to keep sum of the weights for each class
	double w_sum[NumberOfPriorChannels];
	double mv[NumberOfPriorChannels][NumberOfImageChannels];

	for (k = 0; k < NumberOfPriorChannels; k++) {
		w_sum[k] = eps;
		for (j = 0; j < NumberOfImageChannels; j++) {
			mv_sum[k][j] = eps;
			for (i = 0; i < NumberOfImageChannels; i++) {
				if (i == j) {
					vv_sum[k][j][i] = eps;
				} else {
					vv_sum[k][j][i] = 0;
				}
			}
		}
	}

	// sum loop for means
	for (n = 0; n < vd_z; n++) {
		for (m = 0; m < vd_y; m++) {
			for (l = 0; l < vd_x; l++) {
				double y[NumberOfImageChannels];
				double ys = 0;
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					y[i] = vd[i].m_pData[n][m][l][0];
					ys += y[i];
				}
				if (ys > 0) {
					// to the number of classes
					for (k = 0; k < NumberOfPriorChannels; k++) {
						double p = (double)posteriors[k].m_pData[n][m][l][0];
						for (i = 0; i < NumberOfImageChannels; i++) {
							mv_sum[k][i] += p * y[i];
						}
						w_sum[k] += p;
					}
				}
			}
		}
	}

	if (!bFirst) {
		for (k = 0; k < NumberOfPriorChannels; k++) {
			double w_1 = 1.0 / (w_sum[k] + eps);
			for (i = 0; i < NumberOfImageChannels; i++) {
				mv[k][i] = w_1 * mv_sum[k][i] + eps;
				pMeanVector->at(k)[i] = mv[k][i];
			}
		}
	} else {
		// in the first iteration we want to favor user supplied values
		for (k = 0; k < NumberOfPriorChannels; k++) {
			for (i = 0; i < NumberOfImageChannels; i++) {
				mv[k][i] = pMeanVector->at(k)[i];
			}
		}
	}

	// sum loop for variances
	for (n = 0; n < vd_z; n++) {
		for (m = 0; m < vd_y; m++) {
			for (l = 0; l < vd_x; l++) {
				double y[NumberOfImageChannels];
				double ym[NumberOfImageChannels];
				double ys = 0;
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					y[i] = vd[i].m_pData[n][m][l][0];
					ys += y[i];
				}
				if (ys > 0) {
					// to the number of classes
					for (k = 0; k < NumberOfPriorChannels; k++) {
						double p = (double)posteriors[k].m_pData[n][m][l][0];
						for (i = 0; i < NumberOfImageChannels; i++) {
							ym[i] = y[i] - mv[k][i];
						}
						for (j = 0; j < NumberOfImageChannels; j++) {
							for (i = 0; i < NumberOfImageChannels; i++) {
								vv_sum[k][j][i] += p * ym[j] * ym[i];
							}
						}
					}
				}
			}
		}
	}

	// computing updated variances to the number of classes
	for (k = 0; k < NumberOfPriorChannels; k++) {
		double w_1 = 1.0 / (w_sum[k] + eps);
		for (j = 0; j < NumberOfImageChannels; j++) {
			for (i = 0; i < NumberOfImageChannels; i++) {
				pVarianceVector->at(k)[j][i] = w_1 * vv_sum[k][j][i];
			}
		}
	}

	return TRUE;
}

static
inline void ComputeLikelihood4(double m[16], double ym[4], double* like) {
	double inv[16], det, det_1, ySIy;
	
	inv[ 0] =  m[ 5] * m[10] * m[15] - m[ 5] * m[11] * m[14] - m[ 9] * m[ 6] * m[15] + m[ 9] * m[ 7] * m[14] + m[13] * m[ 6] * m[11] - m[13] * m[ 7] * m[10];
	inv[ 4] = -m[ 4] * m[10] * m[15] + m[ 4] * m[11] * m[14] + m[ 8] * m[ 6] * m[15] - m[ 8] * m[ 7] * m[14] - m[12] * m[ 6] * m[11] + m[12] * m[ 7] * m[10];
	inv[ 8] =  m[ 4] * m[ 9] * m[15] - m[ 4] * m[11] * m[13] - m[ 8] * m[ 5] * m[15] + m[ 8] * m[ 7] * m[13] + m[12] * m[ 5] * m[11] - m[12] * m[ 7] * m[ 9];
	inv[12] = -m[ 4] * m[ 9] * m[14] + m[ 4] * m[10] * m[13] + m[ 8] * m[ 5] * m[14] - m[ 8] * m[ 6] * m[13] - m[12] * m[ 5] * m[10] + m[12] * m[ 6] * m[ 9];
	inv[ 1] = -m[ 1] * m[10] * m[15] + m[ 1] * m[11] * m[14] + m[ 9] * m[ 2] * m[15] - m[ 9] * m[ 3] * m[14] - m[13] * m[ 2] * m[11] + m[13] * m[ 3] * m[10];
	inv[ 5] =  m[ 0] * m[10] * m[15] - m[ 0] * m[11] * m[14] - m[ 8] * m[ 2] * m[15] + m[ 8] * m[ 3] * m[14] + m[12] * m[ 2] * m[11] - m[12] * m[ 3] * m[10];
	inv[ 9] = -m[ 0] * m[ 9] * m[15] + m[ 0] * m[11] * m[13] + m[ 8] * m[ 1] * m[15] - m[ 8] * m[ 3] * m[13] - m[12] * m[ 1] * m[11] + m[12] * m[ 3] * m[ 9];
	inv[13] =  m[ 0] * m[ 9] * m[14] - m[ 0] * m[10] * m[13] - m[ 8] * m[ 1] * m[14] + m[ 8] * m[ 2] * m[13] + m[12] * m[ 1] * m[10] - m[12] * m[ 2] * m[ 9];
	inv[ 2] =  m[ 1] * m[ 6] * m[15] - m[ 1] * m[ 7] * m[14] - m[ 5] * m[ 2] * m[15] + m[ 5] * m[ 3] * m[14] + m[13] * m[ 2] * m[ 7] - m[13] * m[ 3] * m[ 6];   
	inv[ 6] = -m[ 0] * m[ 6] * m[15] + m[ 0] * m[ 7] * m[14] + m[ 4] * m[ 2] * m[15] - m[ 4] * m[ 3] * m[14] - m[12] * m[ 2] * m[ 7] + m[12] * m[ 3] * m[ 6];  
	inv[10] =  m[ 0] * m[ 5] * m[15] - m[ 0] * m[ 7] * m[13] - m[ 4] * m[ 1] * m[15] + m[ 4] * m[ 3] * m[13] + m[12] * m[ 1] * m[ 7] - m[12] * m[ 3] * m[ 5];   
	inv[14] = -m[ 0] * m[ 5] * m[14] + m[ 0] * m[ 6] * m[13] + m[ 4] * m[ 1] * m[14] - m[ 4] * m[ 2] * m[13] - m[12] * m[ 1] * m[ 6] + m[12] * m[ 2] * m[ 5];   
	inv[ 3] = -m[ 1] * m[ 6] * m[11] + m[ 1] * m[ 7] * m[10] + m[ 5] * m[ 2] * m[11] - m[ 5] * m[ 3] * m[10] - m[ 9] * m[ 2] * m[ 7] + m[ 9] * m[ 3] * m[ 6];  
	inv[ 7] =  m[ 0] * m[ 6] * m[11] - m[ 0] * m[ 7] * m[10] - m[ 4] * m[ 2] * m[11] + m[ 4] * m[ 3] * m[10] + m[ 8] * m[ 2] * m[ 7] - m[ 8] * m[ 3] * m[ 6];  
	inv[11] = -m[ 0] * m[ 5] * m[11] + m[ 0] * m[ 7] * m[ 9] + m[ 4] * m[ 1] * m[11] - m[ 4] * m[ 3] * m[ 9] - m[ 8] * m[ 1] * m[ 7] + m[ 8] * m[ 3] * m[ 5];   
	inv[15] =  m[ 0] * m[ 5] * m[10] - m[ 0] * m[ 6] * m[ 9] - m[ 4] * m[ 1] * m[10] + m[ 4] * m[ 2] * m[ 9] + m[ 8] * m[ 1] * m[ 6] - m[ 8] * m[ 2] * m[ 5];

	det = m[0] * inv[0] + m[1] * inv[4] + m[2] * inv[8] + m[3] * inv[12] + epss;  
	
	det_1 = 1.0 / det;

	ySIy = ym[0] * (inv[ 0] * ym[0] + inv[ 1] * ym[1] + inv[ 2] * ym[2] + inv[ 3] * ym[3]) +
	       ym[1] * (inv[ 4] * ym[0] + inv[ 5] * ym[1] + inv[ 6] * ym[2] + inv[ 7] * ym[3]) +
	       ym[2] * (inv[ 8] * ym[0] + inv[ 9] * ym[1] + inv[10] * ym[2] + inv[11] * ym[3]) +
	       ym[3] * (inv[12] * ym[0] + inv[13] * ym[1] + inv[14] * ym[2] + inv[15] * ym[3]);
	ySIy *= det_1;

	*like = 1.0 / vcl_sqrt(2.0 * PI * det) * exp (-0.5 * ySIy);
}

// faster version
BOOL ComputePosteriors(FVolume* vd, FVolume* priors, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, FVolume* posteriors)
{
	double vv[NumberOfPriorChannels][16];
	double mv[NumberOfPriorChannels][4];
	BOOL res = TRUE;
	//
	int i, j, k, l, m, n;
	//
	int vd_x = vd[0].m_vd_x;
	int vd_y = vd[0].m_vd_y;
	int vd_z = vd[0].m_vd_z;
	//
	// assume 4 channels for image
	if (NumberOfImageChannels != 4) {
		return FALSE;
	}
	//
	for (i = 0; i < NumberOfPriorChannels; i++) {
		for (j = 0; j < 4; j++) {
			for (k = 0; k < 4; k++) {
				vv[i][j*4+k] = pVarianceVector->at(i)(j, k);
			}
			mv[i][j] = pMeanVector->at(i)(j);
		}
	}
	//
	for (n = 0; n < vd_z; n++) {
		for (m = 0; m < vd_y; m++) {
			for (l = 0; l < vd_x; l++) {
				float priors_val[NumberOfPriorChannels];
				double like[NumberOfPriorChannels];
				double prior_like[NumberOfPriorChannels];
				double denum, denum_1;
				double y[4], ym[4];
				double ys = 0;
				//
				for (i = 0; i < NumberOfPriorChannels; i++) {
					priors_val[i] = priors[i].m_pData[n][m][l][0];
				}
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					y[i] = vd[i].m_pData[n][m][l][0];
					ys += y[i];
				}
				//
				// computing likelihood and posterior values
				// if foreground...
				if (ys > 0) {
					// to the number of classes
					for (i = 0; i < NumberOfPriorChannels; i++) {
						ym[0] = y[0] - mv[i][0]; 
						ym[1] = y[1] - mv[i][1]; 
						ym[2] = y[2] - mv[i][2]; 
						ym[3] = y[3] - mv[i][3]; 
						ComputeLikelihood4(vv[i], ym, &like[i]);
					}
					// compute the denum
					denum = 0.0;
					for (i = 0; i < NumberOfPriorChannels; i++) {
						prior_like[i] = priors_val[i] * like[i];
						denum += prior_like[i];
					}
					// compute the posterior
					if (denum != 0) {
						denum_1 = 1.0 / denum;
						for (i = 0; i < NumberOfPriorChannels; i++) {
							posteriors[i].m_pData[n][m][l][0] = prior_like[i] * denum_1;
						}
					} else {
						for (i = 0; i < NumberOfPriorChannels; i++) {
							posteriors[i].m_pData[n][m][l][0] = 0.0;
						}
					}
				// if background...
				} else {
					// compute the posterior
					for (i = 0; i < NumberOfPriorChannels; i++) {
						posteriors[i].m_pData[n][m][l][0] = 0.0;
					}
					posteriors[BG].m_pData[n][m][l][0] = 1.0;
				}
			} // l
		} // m
	} // n

	return res;
}

// Make 4 class posteriors: BG, CSF (including VS), VT, GM, WM (including ED, NCR, TU)
BOOL ComputePosteriors5(FVolume* vd, FVolume* priors, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector, FVolume* posteriors)
{
	double vv[NumberOfPriorChannels][16];
	double mv[NumberOfPriorChannels][4];
	BOOL res = TRUE;
	//
	int i, j, k, l, m, n;
	//
	int vd_x = vd[0].m_vd_x;
	int vd_y = vd[0].m_vd_y;
	int vd_z = vd[0].m_vd_z;
	//
	// assume 4 channels for image
	if (NumberOfImageChannels != 4) {
		return FALSE;
	}
	//
	for (i = 0; i < NumberOfPriorChannels; i++) {
		for (j = 0; j < 4; j++) {
			for (k = 0; k < 4; k++) {
				vv[i][j*4+k] = pVarianceVector->at(i)(j, k);
			}
			mv[i][j] = pMeanVector->at(i)(j);
		}
	}
	//
	for (n = 0; n < vd_z; n++) {
		for (m = 0; m < vd_y; m++) {
			for (l = 0; l < vd_x; l++) {
				float priors_val[NumberOfPriorChannels];
				double like[NumberOfPriorChannels];
				double prior_like[NumberOfPriorChannels];
				double denum, denum_1;
				double y[4], ym[4];
				double ys = 0;
				//
				for (i = 0; i < NumberOfPriorChannels; i++) {
					priors_val[i] = priors[i].m_pData[n][m][l][0];
				}
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					y[i] = vd[i].m_pData[n][m][l][0];
					ys += y[i];
				}
				//
				// computing likelihood and posterior values
				// if foreground...
				if (ys > 0) {
					// to the number of classes
					for (i = 0; i < NumberOfPriorChannels; i++) {
						ym[0] = y[0] - mv[i][0]; 
						ym[1] = y[1] - mv[i][1]; 
						ym[2] = y[2] - mv[i][2]; 
						ym[3] = y[3] - mv[i][3]; 
						ComputeLikelihood4(vv[i], ym, &like[i]);
					}
					// compute the denum
					denum = 0.0;
					{
						prior_like[BG ] = priors_val[BG] * like[BG];
						denum += prior_like[BG];
						prior_like[CSF] = max(priors_val[CSF] * like[CSF], priors_val[VS] * like[VS]);
						denum += prior_like[CSF];
						prior_like[VT ] = priors_val[VT] * like[VT];
						denum += prior_like[VT];
						prior_like[GM ] = priors_val[GM] * like[GM];
						denum += prior_like[GM];
						prior_like[WM ] = max(priors_val[WM] * like[WM], max(priors_val[ED] * like[ED], max(priors_val[NCR] * like[NCR], priors_val[TU] * like[TU])));
						denum += prior_like[WM];
					}
					// compute the posterior
					if (denum != 0) {
						denum_1 = 1.0 / denum;
						posteriors[BG ].m_pData[n][m][l][0] = prior_like[BG ] * denum_1;
						posteriors[CSF].m_pData[n][m][l][0] = prior_like[CSF] * denum_1;
						posteriors[VT ].m_pData[n][m][l][0] = prior_like[VT ] * denum_1;
						posteriors[GM ].m_pData[n][m][l][0] = prior_like[GM ] * denum_1;
						posteriors[WM ].m_pData[n][m][l][0] = prior_like[WM ] * denum_1;
					} else {
						posteriors[BG ].m_pData[n][m][l][0] = 1.0;
						posteriors[CSF].m_pData[n][m][l][0] = 0.0;
						posteriors[VT ].m_pData[n][m][l][0] = 0.0;
						posteriors[GM ].m_pData[n][m][l][0] = 0.0;
						posteriors[WM ].m_pData[n][m][l][0] = 0.0;
					}
				// if background...
				} else {
					// compute the posterior
					posteriors[BG ].m_pData[n][m][l][0] = 1.0;
					posteriors[CSF].m_pData[n][m][l][0] = 0.0;
					posteriors[VT ].m_pData[n][m][l][0] = 0.0;
					posteriors[GM ].m_pData[n][m][l][0] = 0.0;
					posteriors[WM ].m_pData[n][m][l][0] = 0.0;
				}
			} // l
		} // m
	} // n

	return res;
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
void MultiplyXFM(float A[4][4], float B[4][4], float C[4][4])
{
	int i, j;
	float AB[4][4];
	for (i = 0; i < 4; i++) {
		for (j = 0; j < 4; j++) {
			AB[i][j] = A[i][0] * B[0][j] + A[i][1] * B[1][j] + A[i][2] * B[2][j] + A[i][3] * B[3][j];
		}
	}
	for (i = 0; i < 4; i++) {
		for (j = 0; j < 4; j++) {
			C[i][j] = AB[i][j];
		}
	}
}
void GetCoordXFM(float A[4][4], float b[4], float c[4])
{
	int i;
	float Ab[4];
	for (i = 0; i < 4; i++) {
		Ab[i] = A[i][0] * b[0] + A[i][1] * b[1] + A[i][2] * b[2] + A[i][3] * b[3];
	}
	for (i = 0; i < 4; i++) {
		c[i] = Ab[i];
	}
}
BOOL GetFieldFromXFM(char* m2f_xfm_file, char* m_file, char* f_file, char* m2f_v_hdr_file)
{
	float m2f_xfm[4][4];
	FVolume m, f;
	FVolume m2f_v;
	nifti_image *m_nii, *f_nii;
	float f_ijk[4][4];
	float m_xyz[4][4];
	float m2f[4][4];
	int i, j, k;

	m.loadNII(m_file);
	f.loadNII(f_file);
	m_nii = nifti_image_read(m_file, 1);
	f_nii = nifti_image_read(f_file, 1);

	ReadXFMData(m2f_xfm_file, m2f_xfm);

	m2f_v.allocate(m.m_vd_x, m.m_vd_y, m.m_vd_z, 3);

	for (j = 0; j < 4; j++) {
		for (i = 0; i < 4; i++) {
			f_ijk[j][i] = 0;
			m_xyz[j][i] = 0;
		}
	}
	f_ijk[0][0] = 1 / f_nii->dx;
	f_ijk[1][1] = 1 / f_nii->dy;
	f_ijk[2][2] = 1 / f_nii->dz;
	f_ijk[3][3] = 1;
	m_xyz[0][0] = m_nii->dx;
	m_xyz[1][1] = m_nii->dy;
	m_xyz[2][2] = m_nii->dz;
	m_xyz[3][3] = 1;
	
	MultiplyXFM(m2f_xfm, m_xyz, m2f);
	MultiplyXFM(f_ijk, m2f, m2f);

	for (k = 0; k < m.m_vd_z; k++) {
		for (j = 0; j < m.m_vd_y; j++) {
			for (i = 0; i < m.m_vd_x; i++) {
				float fv[4], mv[4];
				//
				mv[0] = m.m_vd_x-1 - i;
				mv[1] = j;
				mv[2] = k;
				mv[3] = 1;
				//
				GetCoordXFM(m2f, mv, fv);
				//
				fv[0] = f.m_vd_x-1 - fv[0];
				//
				m2f_v.m_pData[k][j][i][0] = (fv[0] - i) * m_nii->dx;
				m2f_v.m_pData[k][j][i][1] = (fv[1] - j) * m_nii->dy;
				m2f_v.m_pData[k][j][i][2] = (fv[2] - k) * m_nii->dz;
			}
		}
	}

	SaveMHDData(NULL, m2f_v_hdr_file, m2f_v.m_pData, m2f_v.m_vd_x, m2f_v.m_vd_y, m2f_v.m_vd_z, m2f_v.m_vd_s, m_nii->dx, m_nii->dy, m_nii->dz,
		m_nii->qoffset_x, m_nii->qoffset_y, m_nii->qoffset_z);
		
	nifti_image_free(m_nii);
	nifti_image_free(f_nii);

	return TRUE;
}

// concatenating whole field
// all unit is mm
BOOL ConcatenateAllFields(char* p0_file, char* p2_file, char* a0_hdr_file, char* h_hdr_file, char* u_hdr_file, char* a2_inv_hdr_file, char* f_hdr_file) {
	FVolume p0, p2;
	FVolume a0_v, a2_inv_v;
	FVolume h_v, u_v;
	FVolume f_v;
	//
	float a0_v_vd_ox, a0_v_vd_oy, a0_v_vd_oz;
	float a2_inv_v_vd_ox, a2_inv_v_vd_oy, a2_inv_v_vd_oz;
	float h_v_vd_ox, h_v_vd_oy, h_v_vd_oz;
	float u_v_vd_ox, u_v_vd_oy, u_v_vd_oz;
	//
	int i, j, k;

	p0.load(p0_file, 1);
	p2.load(p2_file, 1);

	LoadMHDData(NULL, a0_hdr_file, &a0_v.m_pData, a0_v.m_vd_x, a0_v.m_vd_y, a0_v.m_vd_z, a0_v.m_vd_s, a0_v.m_vd_dx, a0_v.m_vd_dy, a0_v.m_vd_dz, a0_v_vd_ox, a0_v_vd_oy, a0_v_vd_oz);
	a0_v.computeDimension();
	LoadMHDData(NULL, h_hdr_file, &h_v.m_pData, h_v.m_vd_x, h_v.m_vd_y, h_v.m_vd_z, h_v.m_vd_s, h_v.m_vd_dx, h_v.m_vd_dy, h_v.m_vd_dz, h_v_vd_ox, h_v_vd_oy, h_v_vd_oz);
	h_v.computeDimension();
	LoadMHDData(NULL, u_hdr_file, &u_v.m_pData, u_v.m_vd_x, u_v.m_vd_y, u_v.m_vd_z, u_v.m_vd_s, u_v.m_vd_dx, u_v.m_vd_dy, u_v.m_vd_dz, u_v_vd_ox, u_v_vd_oy, u_v_vd_oz);
	u_v.computeDimension();
	LoadMHDData(NULL, a2_inv_hdr_file, &a2_inv_v.m_pData, a2_inv_v.m_vd_x, a2_inv_v.m_vd_y, a2_inv_v.m_vd_z, a2_inv_v.m_vd_s, a2_inv_v.m_vd_dx, a2_inv_v.m_vd_dy, a2_inv_v.m_vd_dz, a2_inv_v_vd_ox, a2_inv_v_vd_oy, a2_inv_v_vd_oz);
	a2_inv_v.computeDimension();

	f_v.allocate(a0_v.m_vd_x, a0_v.m_vd_y, a0_v.m_vd_z, 3);

	for (k = 0; k < a0_v.m_vd_z; k++) {
		for (j = 0; j < a0_v.m_vd_y; j++) {
			for (i = 0; i < a0_v.m_vd_x; i++) {
				float v0[3], v1[3], v2[3], v3[3];
				float x0[3], x1[3], x2[3], x3[3];
				//
				v0[0] = a0_v.m_pData[k][j][i][0];
				v0[1] = a0_v.m_pData[k][j][i][1];
				v0[2] = a0_v.m_pData[k][j][i][2];
				//
				x0[0] = i + v0[0] / a0_v.m_vd_dx;
				x0[1] = j + v0[1] / a0_v.m_vd_dy;
				x0[2] = k + v0[2] / a0_v.m_vd_dz;
				//
				if (x0[0] < 0 || x0[0] >= h_v.m_vd_x || x0[1] < 0 || x0[1] >= h_v.m_vd_y || x0[2] < 0 || x0[2] >= h_v.m_vd_z) {
					f_v.m_pData[k][j][i][0] = 0;
					f_v.m_pData[k][j][i][1] = 0;
					f_v.m_pData[k][j][i][2] = 0;
					continue;
				}
				//
				h_v.GetAt(x0[0], x0[1], x0[2], v1);
				//
				x1[0] = x0[0] + v1[0] / h_v.m_vd_dx;
				x1[1] = x0[1] + v1[1] / h_v.m_vd_dy;
				x1[2] = x0[2] + v1[2] / h_v.m_vd_dz;
				//
				if (x1[0] < 0 || x1[0] >= u_v.m_vd_x || x1[1] < 0 || x1[1] >= u_v.m_vd_y || x1[2] < 0 || x1[2] >= u_v.m_vd_z) {
					f_v.m_pData[k][j][i][0] = 0;
					f_v.m_pData[k][j][i][1] = 0;
					f_v.m_pData[k][j][i][2] = 0;
					continue;
				}
				//
				u_v.GetAt(x1[0], x1[1], x1[2], v2);
				//
				x2[0] = x1[0] + v2[0] / u_v.m_vd_dx;
				x2[1] = x1[1] + v2[1] / u_v.m_vd_dy;
				x2[2] = x1[2] + v2[2] / u_v.m_vd_dz;
				//
				if (x2[0] < 0 || x2[0] >= a2_inv_v.m_vd_x || x2[1] < 0 || x2[1] >= a2_inv_v.m_vd_y || x2[2] < 0 || x2[2] >= a2_inv_v.m_vd_z) {
					f_v.m_pData[k][j][i][0] = 0;
					f_v.m_pData[k][j][i][1] = 0;
					f_v.m_pData[k][j][i][2] = 0;
					continue;
				}
				//
				a2_inv_v.GetAt(x2[0], x2[1], x2[2], v3);
				//
				x3[0] = x2[0] + v3[0] / a2_inv_v.m_vd_dx;
				x3[1] = x2[1] + v3[1] / a2_inv_v.m_vd_dy;
				x3[2] = x2[2] + v3[2] / a2_inv_v.m_vd_dz;
				//
				f_v.m_pData[k][j][i][0] = (x3[0] - i) * a0_v.m_vd_dx;
				f_v.m_pData[k][j][i][1] = (x3[1] - j) * a0_v.m_vd_dy;
				f_v.m_pData[k][j][i][2] = (x3[2] - k) * a0_v.m_vd_dz;
			}
		}
	}

	SaveMHDData(NULL, f_hdr_file, f_v.m_pData, a0_v.m_vd_x, a0_v.m_vd_y, a0_v.m_vd_z, a0_v.m_vd_s, a0_v.m_vd_dx, a0_v.m_vd_dy, a0_v.m_vd_dz, a0_v_vd_ox, a0_v_vd_oy, a0_v_vd_oz);

	return TRUE;
}
BOOL ConcatenateAllFields(char* p0_file, char* p2_file, char* a0_hdr_file, char* hu_hdr_file, char* a2_inv_hdr_file, char* f_hdr_file) {
	FVolume p0, p2;
	FVolume a0_v, a2_inv_v;
	FVolume hu_v;
	FVolume f_v;
	//
	float a0_v_vd_ox, a0_v_vd_oy, a0_v_vd_oz;
	float a2_inv_v_vd_ox, a2_inv_v_vd_oy, a2_inv_v_vd_oz;
	float hu_v_vd_ox, hu_v_vd_oy, hu_v_vd_oz;
	//
	int i, j, k;

	p0.load(p0_file, 1);
	p2.load(p2_file, 1);

	LoadMHDData(NULL, a0_hdr_file, &a0_v.m_pData, a0_v.m_vd_x, a0_v.m_vd_y, a0_v.m_vd_z, a0_v.m_vd_s, a0_v.m_vd_dx, a0_v.m_vd_dy, a0_v.m_vd_dz, a0_v_vd_ox, a0_v_vd_oy, a0_v_vd_oz);
	a0_v.computeDimension();
	LoadMHDData(NULL, hu_hdr_file, &hu_v.m_pData, hu_v.m_vd_x, hu_v.m_vd_y, hu_v.m_vd_z, hu_v.m_vd_s, hu_v.m_vd_dx, hu_v.m_vd_dy, hu_v.m_vd_dz, hu_v_vd_ox, hu_v_vd_oy, hu_v_vd_oz);
	hu_v.computeDimension();
	LoadMHDData(NULL, a2_inv_hdr_file, &a2_inv_v.m_pData, a2_inv_v.m_vd_x, a2_inv_v.m_vd_y, a2_inv_v.m_vd_z, a2_inv_v.m_vd_s, a2_inv_v.m_vd_dx, a2_inv_v.m_vd_dy, a2_inv_v.m_vd_dz, a2_inv_v_vd_ox, a2_inv_v_vd_oy, a2_inv_v_vd_oz);
	a2_inv_v.computeDimension();

	f_v.allocate(a0_v.m_vd_x, a0_v.m_vd_y, a0_v.m_vd_z, 3);

	for (k = 0; k < a0_v.m_vd_z; k++) {
		for (j = 0; j < a0_v.m_vd_y; j++) {
			for (i = 0; i < a0_v.m_vd_x; i++) {
				float v0[3], v1[3], v2[3];
				float x0[3], x1[3], x2[3];
				//
				v0[0] = a0_v.m_pData[k][j][i][0];
				v0[1] = a0_v.m_pData[k][j][i][1];
				v0[2] = a0_v.m_pData[k][j][i][2];
				//
				x0[0] = i + v0[0] / a0_v.m_vd_dx;
				x0[1] = j + v0[1] / a0_v.m_vd_dy;
				x0[2] = k + v0[2] / a0_v.m_vd_dz;
				//
				if (x0[0] < 0 || x0[0] >= hu_v.m_vd_x || x0[1] < 0 || x0[1] >= hu_v.m_vd_y || x0[2] < 0 || x0[2] >= hu_v.m_vd_z) {
					f_v.m_pData[k][j][i][0] = 0;
					f_v.m_pData[k][j][i][1] = 0;
					f_v.m_pData[k][j][i][2] = 0;
					continue;
				}
				//
				hu_v.GetAt(x0[0], x0[1], x0[2], v1);
				//
				x1[0] = x0[0] + v1[0] / hu_v.m_vd_dx;
				x1[1] = x0[1] + v1[1] / hu_v.m_vd_dy;
				x1[2] = x0[2] + v1[2] / hu_v.m_vd_dz;
				//
				if (x1[0] < 0 || x1[0] >= a2_inv_v.m_vd_x || x1[1] < 0 || x1[1] >= a2_inv_v.m_vd_y || x1[2] < 0 || x1[2] >= a2_inv_v.m_vd_z) {
					f_v.m_pData[k][j][i][0] = 0;
					f_v.m_pData[k][j][i][1] = 0;
					f_v.m_pData[k][j][i][2] = 0;
					continue;
				}
				//
				a2_inv_v.GetAt(x1[0], x1[1], x1[2], v2);
				//
				x2[0] = x1[0] + v2[0] / a2_inv_v.m_vd_dx;
				x2[1] = x1[1] + v2[1] / a2_inv_v.m_vd_dy;
				x2[2] = x1[2] + v2[2] / a2_inv_v.m_vd_dz;
				//
				f_v.m_pData[k][j][i][0] = (x2[0] - i) * a0_v.m_vd_dx;
				f_v.m_pData[k][j][i][1] = (x2[1] - j) * a0_v.m_vd_dy;
				f_v.m_pData[k][j][i][2] = (x2[2] - k) * a0_v.m_vd_dz;
			}
		}
	}

	SaveMHDData(NULL, f_hdr_file, f_v.m_pData, a0_v.m_vd_x, a0_v.m_vd_y, a0_v.m_vd_z, a0_v.m_vd_s, a0_v.m_vd_dx, a0_v.m_vd_dy, a0_v.m_vd_dz, a0_v_vd_ox, a0_v_vd_oy, a0_v_vd_oz);

	return TRUE;
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
BOOL EstimateDeformationFieldJSR(char atlas_reg_image_files[NumberOfImageChannels][1024], char atlas_reg_prior_files[NumberOfPriorChannels][1024], 
	MeanVectorType* pMeanVector, char* h_hdr_file, char* means_file, char* variances_file, char* cost_file, int iter, int max_iter, int num_itk_threads, bool bEstimateTumorRegionOnly, float fTumorRegionThreshold, bool bElastic)
{
	int i, j;
	//
	if (num_itk_threads > 0) {	
		char str_num_itk_threads[1024];
		sprintf(str_num_itk_threads, "ITK_GLOBAL_DEFAULT_NUMBER_OF_THREADS=%d", num_itk_threads);
		TRACE2("%s\n", str_num_itk_threads);
		putenv(str_num_itk_threads);
	}
	//
	// read fixed images
	FixedImageReaderVectorType FixedReaderVector(NumberOfImageChannels);
	for (i = 0; i < NumberOfImageChannels; i++) {
		FixedReaderVector.at(i) = FixedImageReaderType::New();
		FixedReaderVector.at(i)->SetFileName(atlas_reg_image_files[i]);
		try {
			FixedReaderVector.at(i)->UpdateLargestPossibleRegion();
		} catch (itk::ExceptionObject& e) {
			std::cerr << e << std::endl;
			return FALSE;
		}
	}

	// read moving images (atlas priors)
	MovingImageReaderVectorType MovingReaderVector(NumberOfPriorChannels);
	for (i = 0; i < NumberOfPriorChannels; i++) {
		MovingReaderVector.at(i) = MovingImageReaderType::New();
		MovingReaderVector.at(i)->SetFileName(atlas_reg_prior_files[i]);
		try {
			MovingReaderVector.at(i)->UpdateLargestPossibleRegion();
		} catch (itk::ExceptionObject& e) {
			std::cerr << e << std::endl;
			return FALSE;
		}
	}

	// perform joint segmentation & registration
	TRACE2("creating j_seg_reg_filter..\n");
	JointSegmentationRegistrationType::Pointer j_seg_reg_filter = JointSegmentationRegistrationType::New();
	JointSegmentationRegistrationFunctionType* drfp = dynamic_cast<JointSegmentationRegistrationFunctionType*>(j_seg_reg_filter->GetSegmentationRegistrationFilter()->GetDifferenceFunction().GetPointer());
	TRACE2("creating j_seg_reg_filter.. done\n");
	
	// set input images
	TRACE2("setting input images..\n");
	for (i = 1; i < NumberOfImageChannels + 1; i++) {
		j_seg_reg_filter->SetNthFixedImage(FixedReaderVector.at(i-1)->GetOutput(), i);
	}
	for (i = 1; i < NumberOfPriorChannels + 1; i++) {
		j_seg_reg_filter->SetNthMovingImage(MovingReaderVector.at(i-1)->GetOutput(), i);
	}
	TRACE2("setting input images.. done\n");
					
	// set parameters
	TRACE2("setting parameters..\n");
	#define NumberOfLevels 3
	double sigma2 = 0.2;
	unsigned int num_of_iter[NumberOfLevels];
	if (iter == 0) {
		num_of_iter[0] = 0;
		num_of_iter[1] = 0;
		num_of_iter[2] = 20;
	} else if (iter == 1) {
		num_of_iter[0] = 0;
		num_of_iter[1] = 0;
		num_of_iter[2] = 50;
	} else if (iter == -1) {
		num_of_iter[0] = 50;
		num_of_iter[1] = 50;
		num_of_iter[2] = 50;
	} else {
		num_of_iter[0] = 0;
		num_of_iter[1] = 0;
		num_of_iter[2] = 50;
	}
	//
	j_seg_reg_filter->SetNumberOfLevels(NumberOfLevels);
	j_seg_reg_filter->SetNumberOfIterations(num_of_iter);
	drfp->SetSigma2(sigma2);
	//
	drfp->m_bEstimateTumorRegionOnly = bEstimateTumorRegionOnly;
	drfp->m_fTumorRegionThreshold = fTumorRegionThreshold;
	TRACE2("setting parameters.. done\n");

	TRACE2("setting registration..\n");
	if (bElastic) {
		// setting for a elastic registration performance
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SmoothUpdateFieldOff();
#if ITK_VERSION_MAJOR >= 4
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SmoothDisplacementFieldOn();
#else
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SmoothDeformationFieldOn();
#endif
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SetStandardDeviations(2.0);
	} else {
		// setting for a viscos registration performance
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SmoothUpdateFieldOn();
#if ITK_VERSION_MAJOR >= 4
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SmoothDisplacementFieldOff();
#else
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SmoothDeformationFieldOff();
#endif
		j_seg_reg_filter->GetSegmentationRegistrationFilter()->SetStandardDeviations(2.0);
	}
	TRACE2("setting registration.. done\n");

	// set initial mean values
	for (j = 0; j < NumberOfPriorChannels; j++) {
		drfp->GetMeanVector()->at(j) = pMeanVector->at(j);
	}
	
	//j_seg_reg_filter->Print(std::cout); 

	// update filter
	TRACE2("updating filter..\n");
	try {
		j_seg_reg_filter->Update();
	} catch (itk::ExceptionObject& e) {
		std::cerr << e << std::endl;
		return FALSE;
	}
	TRACE2("updating filter.. done\n");

	// write output deformation field
	TRACE2("writing output deformation field..\n");
	DeformationFieldWriterType::Pointer field_writer = DeformationFieldWriterType::New();
	field_writer->SetInput(j_seg_reg_filter->GetOutput());
	field_writer->SetFileName(h_hdr_file);
	field_writer->SetUseCompression(true);
	try {
		field_writer->Update();
	} catch (itk::ExceptionObject& e) {
		std::cerr << e << std::endl;
		return FALSE;
	}
	TRACE2("writing output deformation field.. done\n");

	if (means_file != NULL && variances_file != NULL) {
		// write means and variances
		if (!SaveMeansAndVariances(means_file, variances_file, drfp->GetMeanVector(), drfp->GetVarianceVector())) {
			TRACE("SaveMeansAndVariances failed\n");
			return FALSE;
		}
	}
	if (cost_file != NULL) {
		// write cost to file
		std::ofstream fcost(cost_file);
		if (!fcost) {
			TRACE("Failed to open file %s for writing\n", cost_file);
			return FALSE;
		} else {
			fcost << j_seg_reg_filter->GetMetric();
			fcost.close();
		}
	}

	return TRUE;
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
#define edema_eps 1e-8

BOOL MakeProbForTumorAndEdema2(FVolume& priors_TU, FVolume& priors_ED, FVolume& priors_CSF, 
#ifdef USE_11A_PRIORS
	FVolume& priors_VT, 
#endif
	FVolume& priors_GM, FVolume& priors_WM, 
#ifdef USE_WARPED_BG
	FVolume& priors_BG, 
#endif
	FVolume& tumor_density)
{
	int vd_x, vd_y, vd_z;
	int i, j, k;

	vd_x = tumor_density.m_vd_x;
	vd_y = tumor_density.m_vd_y;
	vd_z = tumor_density.m_vd_z;

	for (k = 0; k < vd_z; k++) {
		for (j = 0; j < vd_y; j++) {
			for (i = 0; i < vd_x; i++) {
				float c, c_1, wm, csf, gm;
#ifdef USE_11A_PRIORS
				float vt;
#endif
				//
				c = tumor_density.m_pData[k][j][i][0];
				c_1 = 1.0 - c;
				//
				priors_TU.m_pData[k][j][i][0] = 0.5 * c;
				//
				csf = priors_CSF.m_pData[k][j][i][0] * c_1;
				priors_CSF.m_pData[k][j][i][0] = 0.5 * csf;
				//
#ifdef USE_11A_PRIORS
				vt = priors_VT.m_pData[k][j][i][0] * c_1;
				priors_VT.m_pData[k][j][i][0] = vt;
#endif
				//
#ifndef USE_ED_INC_GM
				gm = priors_GM.m_pData[k][j][i][0] * c_1;
				priors_GM.m_pData[k][j][i][0] = gm;
				//
				wm = priors_WM.m_pData[k][j][i][0] * c_1;
				if (c > edema_eps && c < 1.0) {
					priors_WM.m_pData[k][j][i][0] = 0.5 * wm;
					priors_ED.m_pData[k][j][i][0] = 0.5 * wm;
				} else {
					priors_WM.m_pData[k][j][i][0] = wm;
					priors_ED.m_pData[k][j][i][0] = 0;
				}
#else
				gm = priors_GM.m_pData[k][j][i][0] * c_1;
				wm = priors_WM.m_pData[k][j][i][0] * c_1;
				if (c > edema_eps && c < 1.0) {
					priors_GM.m_pData[k][j][i][0] = 0,5 * gm;
					priors_WM.m_pData[k][j][i][0] = 0.5 * wm;
					priors_ED.m_pData[k][j][i][0] = 0.5 * (gm + wm);
				} else {
					priors_GM.m_pData[k][j][i][0] = gm;
					priors_WM.m_pData[k][j][i][0] = wm;
					priors_ED.m_pData[k][j][i][0] = 0;
				}
#endif
			}
		}
	}

	return TRUE;
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
