///////////////////////////////////////////////////////////////////////////////////////
// EvaluateQ.cpp for BrainTumorRegistration
// Developed by Dongjin Kwon
///////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2014 University of Pennsylvania. All rights reserved.
// See http://www.cbica.upenn.edu/sbia/software/license.html or COYPING file.
//
// Contact: SBIA Group <sbia-software at uphs.upenn.edu>
///////////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
//
#undef TRACE
#define TRACE printf
//
#include "MyUtils.h"
#include "Volume.h"
#include "ENT_table.h"
#include "VolRegOpt.h"
//
#include "itkImageFunction.h"
#include "vnl/algo/vnl_qr.h"
#include "vnl/vnl_math.h"


#if (!defined(WIN32) && !defined(WIN64)) || !defined(_DEBUG)
char SIMULATOR_PATH[1024];
char RESAMPLE_IMAGE_PATH[1024];
char RESAMPLE_DEFORMATION_FIELD_PATH[1024];
char REVERSE_DEFORMATION_FIELD_PATH[1024];
#endif


#define PI						3.1415926535897932384626433832795
#define eps						1e-8
#define epss					1e-32


typedef vnl_matrix_fixed<double, NumberOfImageChannels, NumberOfImageChannels> VarianceType;
typedef vnl_vector_fixed<double, NumberOfImageChannels> MeanType;
typedef std::vector<VarianceType> VarianceVectorType;
typedef std::vector<MeanType> MeanVectorType;


BOOL MakeProbForTumorAndEdema(FVolume* priors, FVolume& tumor_density);
BOOL MakeProbForTumorAndEdema2(FVolume& priors_TU, FVolume& priors_ED, FVolume& priors_CSF, 
#ifdef USE_11A_PRIORS
	FVolume& priors_VT,
#endif
	FVolume& priors_GM, FVolume& priors_WM, 
#ifdef USE_VS_PRIOR	
	FVolume& priors_VS,
#endif
#ifdef USE_WARPED_BG
	FVolume& priors_BG, 
#endif
	FVolume& tumor_density);
BOOL MakeProbForTumorAndEdema3(FVolume& priors_TU, FVolume& priors_ED, FVolume& priors_CSF, 
#ifdef USE_11A_PRIORS
	FVolume& priors_VT,
#endif
	FVolume& priors_GM, FVolume& priors_WM, 
#ifdef USE_VS_PRIOR	
	FVolume& priors_VS, 
#endif
#ifdef USE_WARPED_BG
	FVolume& priors_BG, 
#endif
	FVolume& tumor_density, float gxc, float gyc, float gzc, float T);

BOOL ComputeQ(FVolume* scan0_atlas_reg_images, FVolume* scan0_atlas_reg_warp_prior, const char* scan0_means_file, const char* scan0_variances_file, double* pScore, int tag, char* tag_F);
#ifdef USE_COMPUTEQ2
BOOL ComputeQ2(FVolume* scan0_atlas_reg_images, FVolume* scan0_atlas_reg_warp_prior, FVolume* scan0_atlas_reg_posterior, const char* scan0_means_file, const char* scan0_variances_file, double* pScore, int tag, char* tag_F);
#endif


void version()
{
	printf("==========================================================================\n");
	printf("EvaluateQ (PORTR): Internal Procedure\n");
#ifdef SW_VER
	printf("  Version %s\n", SW_VER);
#endif
#ifdef SW_REV
	printf("  Revision %s\n", SW_REV);
#endif
	printf("Copyright (c) 2014 University of Pennsylvania. All rights reserved.\n");
	printf("See http://www.cbica.upenn.edu/sbia/software/license.html or COPYING file.\n");
	printf("==========================================================================\n");
}
void usage()
{
}

int main(int argc, char* argv[])
{
	// intput: atlas reg scan 0, atlas reg scan 2, simulator input, scales
	// output: score
	const char* atlas_mask_file = NULL;
	const char* scan2_atlas_reg_abnor_mask_file = NULL;
    const char* scan0_atlas_reg_image_list = NULL;
    const char* scan0_atlas_reg_init_prior_list = NULL;
    const char* scan0_means_file = NULL;
    const char* scan0_variances_file = NULL;
    const char* scan0_h_hdr_file = NULL;
    const char* scan0_h2_hdr_file = NULL;
    const char* scan2_atlas_reg_image_list = NULL;
    const char* scan2_atlas_reg_posterior_list = NULL;
    const char* label_map_s_img_file = NULL;
	const char* simulator_input_file = NULL;
	const char* scales_file = NULL;
	const char* out_folder = NULL;
	const char* tmp_folder = NULL;
	const char* input_file = NULL;
	const char* output_file = NULL;
	int tag = 0;
	const char* tag_F = NULL;
	BOOL sol = FALSE;
	int i;
#ifdef USE_COMPUTEQ2
	int j, k, l, m, n;
#endif

	if (argc != 20) {
		version();
        usage();
        exit(EXIT_SUCCESS);
	}

	printf("argc = %d\n", argc);
	for (i = 0; i < argc; i++) {
		printf("argv[%d] = %s\n", i, argv[i]);
	}

#if (!defined(WIN32) && !defined(WIN64)) || !defined(_DEBUG)
	{
		char MODULE_PATH[1024];
		str_strip_file(argv[0], MODULE_PATH);
		//
#if defined(WIN32) || defined(WIN64)
		if (!FindExecutableInPath("ForwardSolverDiffusion.exe", MODULE_PATH, SIMULATOR_PATH)) { printf("error: ForwardSolverDiffusion.exe is not existing.\n"); exit(EXIT_FAILURE); }
		//
		sprintf(RESAMPLE_IMAGE_PATH, "%sResampleImage.exe", MODULE_PATH);
		if (!IsFileExist(RESAMPLE_IMAGE_PATH)) { TRACE("error: %s is not existing.\n", RESAMPLE_IMAGE_PATH); exit(EXIT_FAILURE); }
		sprintf(RESAMPLE_DEFORMATION_FIELD_PATH, "%sResampleDeformationField.exe", MODULE_PATH);
		if (!IsFileExist(RESAMPLE_DEFORMATION_FIELD_PATH)) { TRACE("error: %s is not existing.\n", RESAMPLE_DEFORMATION_FIELD_PATH); exit(EXIT_FAILURE); }
		sprintf(REVERSE_DEFORMATION_FIELD_PATH, "%sReverseDeformationField.exe", MODULE_PATH);
		if (!IsFileExist(REVERSE_DEFORMATION_FIELD_PATH)) { TRACE("error: %s is not existing.\n", REVERSE_DEFORMATION_FIELD_PATH); exit(EXIT_FAILURE); }
#else
		if (!FindExecutableInPath("ForwardSolverDiffusion", MODULE_PATH, SIMULATOR_PATH)) { printf("error: ForwardSolverDiffusion.exe is not existing.\n"); exit(EXIT_FAILURE); }
		//
		sprintf(RESAMPLE_IMAGE_PATH, "%sResampleImage", MODULE_PATH);
		if (!IsFileExist(RESAMPLE_IMAGE_PATH)) { TRACE("error: %s is not existing.\n", RESAMPLE_IMAGE_PATH); exit(EXIT_FAILURE); }
		sprintf(RESAMPLE_DEFORMATION_FIELD_PATH, "%sResampleDeformationField", MODULE_PATH);
		if (!IsFileExist(RESAMPLE_DEFORMATION_FIELD_PATH)) { TRACE("error: %s is not existing.\n", RESAMPLE_DEFORMATION_FIELD_PATH); exit(EXIT_FAILURE); }
		sprintf(REVERSE_DEFORMATION_FIELD_PATH, "%sReverseDeformationField", MODULE_PATH);
		if (!IsFileExist(REVERSE_DEFORMATION_FIELD_PATH)) { TRACE("error: %s is not existing.\n", REVERSE_DEFORMATION_FIELD_PATH); exit(EXIT_FAILURE); }
#endif
	}
#endif

	/////////////////////////////////////////////////////////////////////////////
	printf("SIMULATOR_PATH = %s\n", SIMULATOR_PATH);
	printf("RESAMPLE_IMAGE_PATH = %s\n", RESAMPLE_IMAGE_PATH);
	printf("RESAMPLE_DEFORMATION_FIELD_PATH = %s\n", RESAMPLE_DEFORMATION_FIELD_PATH);
	printf("REVERSE_DEFORMATION_FIELD_PATH = %s\n", REVERSE_DEFORMATION_FIELD_PATH);
	/////////////////////////////////////////////////////////////////////////////

	atlas_mask_file = argv[1];
	scan2_atlas_reg_abnor_mask_file = argv[2];
	scan0_atlas_reg_image_list = argv[3];
	scan0_atlas_reg_init_prior_list = argv[4];
	scan0_means_file = argv[5];
	scan0_variances_file = argv[6];
	scan0_h_hdr_file = argv[7];
	scan0_h2_hdr_file = argv[8];
	scan2_atlas_reg_image_list = argv[9];
	scan2_atlas_reg_posterior_list = argv[10];
	label_map_s_img_file = argv[11];
	simulator_input_file = argv[12];
	scales_file = argv[13];
	out_folder = argv[14];
	tmp_folder = argv[15];
	input_file = argv[16];
	output_file = argv[17];
	tag = atoi(argv[18]);
	tag_F = argv[19];

	if (strcmp(tag_F, "SOL") == 0) {
		sol = TRUE;
	}

	char simulator_folder[1024];
	//
	char scan0_atlas_reg_abnor_mask_file[1024];
	char scan2_atlas_reg_mass_abnor_mask_file[1024];
	char scan2_atlas_reg_warp_abnor_mask_file[1024];
	//
	char scan0_atlas_reg_image_files[NumberOfImageChannels][1024];
	char scan0_atlas_reg_init_prior_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_mass_prior_files[NumberOfPriorChannels][1024];
	char scan0_atlas_reg_warp_prior_files[NumberOfPriorChannels][1024];
	//char scan0_atlas_reg_posterior_files[NumberOfPriorChannels][1024];
	//
	char scan2_atlas_reg_image_files[NumberOfImageChannels][1024];
	char scan2_atlas_reg_mass_image_files[NumberOfImageChannels][1024];
	char scan2_atlas_reg_warp_image_files[NumberOfImageChannels][1024];
	char scan2_atlas_reg_posterior_files[NumberOfPriorChannels][1024];
	char tumor_density_s_file[1024];
	char tumor_deformation_field_s_file[1024];
	char tumor_deformation_field_s_r_file[1024];
	char tumor_density_file[1024];
	char tumor_deformation_field_r_file[1024];
	char tumor_deformation_field_file[1024];
	char simulator_input_copy_file[1024];
	char parameter_cost_file[1024];
	//
	BVolume atlas_mask;
#ifdef USE_COMPUTEQ2
	BVolume scan0_atlas_reg_abnor_mask;
	BVolume scan2_atlas_reg_abnor_mask;
	BVolume scan2_atlas_reg_mass_abnor_mask;
	BVolume scan2_atlas_reg_warp_abnor_mask;
#endif
	//
	FVolume scan0_atlas_reg_images[NumberOfImageChannels];
	FVolume scan0_atlas_reg_priors[NumberOfPriorChannels];
	FVolume scan0_atlas_reg_mass_priors[NumberOfPriorChannels];
	FVolume scan0_atlas_reg_warp_priors[NumberOfPriorChannels];
#ifdef USE_COMPUTEQ2
	FVolume scan0_atlas_reg_posteriors[NumberOfPriorChannels];
	//
	FVolume scan2_atlas_reg_images[NumberOfImageChannels];
	FVolume scan2_atlas_reg_mass_images[NumberOfImageChannels];
	FVolume scan2_atlas_reg_warp_images[NumberOfImageChannels];
#endif
	//
	FVolume tumor_density;
	//
	FVolume scan0_u_v3;
	FVolume scan0_h_v3;
	FVolume scan0_h2_v3;
	//
	//int vd_x, vd_y, vd_z;
	char szCmdLine[1024];
	//
	double scales[9];
	double gxc, gyc, gzc, gp1, gp2, grho, gdiffwm, gdiffgm, T;
	//
	int vd_x, vd_y, vd_z;
	float vd_dx, vd_dy, vd_dz;
	float vd_ox, vd_oy, vd_oz;
	
	TRACE("%d_%s: EvaluateQ...\n", tag, tag_F);

	// load files
	{
		char str_path[1024];
		char str_tmp[1024];
		FILE* fp;
		//
		sprintf(simulator_folder, "%s%s%d_%s", tmp_folder, DIR_SEP, tag, tag_F);
		//
		sprintf(scan0_atlas_reg_abnor_mask_file     , "%s%sscan0_atlas_reg_abnor_mask.nii.gz"     , simulator_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_mass_abnor_mask_file, "%s%sscan2_atlas_reg_mass_abnor_mask.nii.gz", simulator_folder, DIR_SEP);
		sprintf(scan2_atlas_reg_warp_abnor_mask_file, "%s%sscan2_atlas_reg_warp_abnor_mask.nii.gz", simulator_folder, DIR_SEP);
		//
		str_strip_file((char*)scan0_atlas_reg_image_list, str_path);
		fp = fopen(scan0_atlas_reg_image_list, "r");
		for (i = 0; i < NumberOfImageChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(scan0_atlas_reg_image_files[i], "%s%s", str_path, str_tmp);
		}
		fclose(fp);
		str_strip_file((char*)scan2_atlas_reg_image_list, str_path);
		fp = fopen(scan2_atlas_reg_image_list, "r");
		for (i = 0; i < NumberOfImageChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(scan2_atlas_reg_image_files[i], "%s%s", str_path, str_tmp);
		}
		fclose(fp);
		//
		for (i = 0; i < NumberOfImageChannels; i++) {
			sprintf(scan2_atlas_reg_mass_image_files[i], "%s%sscan2_atlas_reg_mass_image_%d.nii.gz", simulator_folder, DIR_SEP, i);
			sprintf(scan2_atlas_reg_warp_image_files[i], "%s%sscan2_atlas_reg_warp_image_%d.nii.gz", simulator_folder, DIR_SEP, i);
		}
		//
		str_strip_file((char*)scan0_atlas_reg_init_prior_list, str_path);
		fp = fopen(scan0_atlas_reg_init_prior_list, "r");
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(scan0_atlas_reg_init_prior_files[i], "%s%s", str_path, str_tmp);
		}
		//
		for (i = 0; i < NumberOfPriorChannels; i++) {
			sprintf(scan0_atlas_reg_mass_prior_files[i], "%s%sscan0_atlas_reg_mass_prior_%d.nii.gz", simulator_folder, DIR_SEP, i);
			sprintf(scan0_atlas_reg_warp_prior_files[i], "%s%sscan0_atlas_reg_warp_prior_%d.nii.gz", simulator_folder, DIR_SEP, i);
		}
		//
		str_strip_file((char*)scan2_atlas_reg_posterior_list, str_path);
		fp = fopen(scan2_atlas_reg_posterior_list, "r");
		for (i = 0; i < NumberOfPriorChannels; i++) {
			fscanf(fp, "%s", str_tmp);
			sprintf(scan2_atlas_reg_posterior_files[i], "%s%s", str_path, str_tmp);
		}
		fclose(fp);
		//
		sprintf(tumor_density_s_file            , "%s%sTumorDensity.001.mhd"        , simulator_folder, DIR_SEP);
		sprintf(tumor_deformation_field_s_file  , "%s%sDeformationField.001.mhd"    , simulator_folder, DIR_SEP);
		sprintf(tumor_deformation_field_s_r_file, "%s%sDeformationField_inv.001.mhd", simulator_folder, DIR_SEP);
		sprintf(tumor_density_file            , "%s%stumor_density.nii.gz"           , simulator_folder, DIR_SEP);
		sprintf(tumor_deformation_field_r_file, "%s%stumor_deformation_field.mhd"    , simulator_folder, DIR_SEP);
		sprintf(tumor_deformation_field_file  , "%s%stumor_deformation_field_inv.mhd", simulator_folder, DIR_SEP);
		//
		str_strip_path((char*)simulator_input_file, str_tmp);
		sprintf(simulator_input_copy_file, "%s%s%s", simulator_folder, DIR_SEP, str_tmp);
		//
		sprintf(parameter_cost_file, "%s%sparameter_cost.txt", tmp_folder, DIR_SEP);
	}

	// read input file and get params
	{
		FILE* fp;
		char tmp[1024];
		fp = fopen(input_file, "r");
		if (fp == NULL) {
			exit(EXIT_FAILURE);
		}
		fscanf(fp, "%s", tmp);
		//if (strcmp(tmp, "F") != 0) {
		//	exit(EXIT_FAILURE);
		//}
		fscanf(fp, "%s", tmp);
#ifdef USE_OPTIM_DG
		if (strcmp(tmp, "9") != 0) {
#else
		if (strcmp(tmp, "8") != 0) {
#endif
			fclose(fp);
			exit(EXIT_FAILURE);
		}
		fscanf(fp, "%s", tmp); gxc = atof(tmp);
		fscanf(fp, "%s", tmp); gyc = atof(tmp);
		fscanf(fp, "%s", tmp); gzc = atof(tmp);
		fscanf(fp, "%s", tmp); gp1 = atof(tmp);
		fscanf(fp, "%s", tmp); gp2 = atof(tmp);
		fscanf(fp, "%s", tmp); grho = atof(tmp);
		fscanf(fp, "%s", tmp); gdiffwm = atof(tmp);
#ifdef USE_OPTIM_DG
		fscanf(fp, "%s", tmp); gdiffgm = atof(tmp);
#endif
		fscanf(fp, "%s", tmp); T = atof(tmp);
		fclose(fp);
	}

	// read and apply scales
	{
		FILE* fp;
		char tmp[1024];
		fp = fopen(scales_file, "r");
		if (fp == NULL) {
			exit(EXIT_FAILURE);
		}
#ifdef USE_OPTIM_DG
		for (i = 0; i < 9; i++) {
#else
		for (i = 0; i < 8; i++) {
#endif
			fscanf(fp, "%s", tmp); scales[i] = atof(tmp);
		}
		fclose(fp);
		//
		gxc *= scales[0];
		gyc *= scales[1];
		gzc *= scales[2];
		gp1 *= scales[3];
		gp2 *= scales[4];
		grho *= scales[5];
		gdiffwm *= scales[6];
#ifdef USE_OPTIM_DG
		gdiffgm *= scales[7];
		T *= scales[8];
#else
		// let's set dg = dw / 5
		gdiffgm = gdiffwm / 5;
		T *= scales[7];
#endif
	}
	
	// run simulator with params and get density and deform field
	{
		char simlulator_path[1024];
#ifdef COPY_SIMULATOR
#if !defined(WIN32) && !defined(WIN64)
		char simulator_name[1024];
		char simulator_copy_file[1024];

		str_strip_path(SIMULATOR_PATH, simulator_name);
		sprintf(simulator_copy_file, "%s%s%s", simulator_folder, DIR_SEP, simulator_name);

		//TRACE("SIMULATOR_PATH = %s\n", SIMULATOR_PATH);
		//TRACE("simulator_name = %s\n", simulator_name);
		//TRACE("simulator_copy_file = %s\n", simulator_copy_file);
#endif
#endif

		CreateDirectory(simulator_folder, NULL);
		//
		SetCurrentDirectory(simulator_folder);

		CopyFile(simulator_input_file, simulator_input_copy_file, FALSE);
#ifdef COPY_SIMULATOR
#if !defined(WIN32) && !defined(WIN64)
		CopyFile(SIMULATOR_PATH, simulator_copy_file, FALSE);
#endif
#endif

#if defined(WIN32) || defined(WIN64)
		sprintf(simlulator_path, "%s", SIMULATOR_PATH);
#else
#ifdef COPY_SIMULATOR
		sprintf(simlulator_path, "%s%s%s", simulator_folder, DIR_SEP, simulator_name);
#else
		sprintf(simlulator_path, "%s", SIMULATOR_PATH);
#endif
#endif
		//TRACE("simlulator_path = %s\n", simlulator_path);

		sprintf(szCmdLine, "%s -gfileInput %s -T %.15g -gdiffgm %.15g -gdiffwm %.15g -grho %.15g -gp2 %.15g -gp1 %.15g -gzc %.15g -gyc %.15g -gxc %.15g",
			simlulator_path, label_map_s_img_file, T, gdiffgm, gdiffwm, grho, gp2, gp1, gzc, gyc, gxc);
		//
		TRACE("%d_%s: %s\n", tag, tag_F, szCmdLine);
		//
		/////////////////////////////////////////////////////////////////////////////
		//freopen(g_trace_file, "a", stdout);
		/////////////////////////////////////////////////////////////////////////////
		//
		if (!ExecuteProcess(szCmdLine)) {
			exit(EXIT_FAILURE);
		}

#ifdef COPY_SIMULATOR
#if !defined(WIN32) && !defined(WIN64)
		DeleteFile(simulator_copy_file);
#endif
#endif

		SetCurrentDirectory(tmp_folder);
	}

	// resample density and deform field
	{
		sprintf(szCmdLine, "%s -i %s -o %s -r %s -x %.15f -y %.15f -z %.15f", RESAMPLE_IMAGE_PATH, tumor_density_s_file, tumor_density_file, scan2_atlas_reg_image_files[0], 1.0, 1.0, 1.0);
		//
		TRACE("%d_%s: %s\n", tag, tag_F, szCmdLine);
		//
		if (!ExecuteProcess(szCmdLine)) {
			exit(EXIT_FAILURE);
		}

		if (!sol) {
			sprintf(szCmdLine, "%s -i %s -o %s -n %d -s %e", REVERSE_DEFORMATION_FIELD_PATH, tumor_deformation_field_s_file, tumor_deformation_field_s_r_file, 15, 1e-6);
		} else {
			sprintf(szCmdLine, "%s -i %s -o %s -n %d -s %e", REVERSE_DEFORMATION_FIELD_PATH, tumor_deformation_field_s_file, tumor_deformation_field_s_r_file, 30, 1e-6);
		}
		//
		TRACE("%d_%s: %s\n", tag, tag_F, szCmdLine);
		//
		if (!ExecuteProcess(szCmdLine)) {
			exit(EXIT_FAILURE);
		}

		sprintf(szCmdLine, "%s -i %s -o %s -r %s -x %.15f -y %.15f -z %.15f", RESAMPLE_DEFORMATION_FIELD_PATH, tumor_deformation_field_s_r_file, tumor_deformation_field_r_file, scan2_atlas_reg_image_files[0], 1.0, 1.0, 1.0);
		//
		TRACE("%d_%s: %s\n", tag, tag_F, szCmdLine);
		//
		if (!ExecuteProcess(szCmdLine)) {
			exit(EXIT_FAILURE);
		}
	}

	// deform scan 2
	{
		{
			if (!LoadMHDData(NULL, tumor_deformation_field_r_file, 
				&scan0_u_v3.m_pData, scan0_u_v3.m_vd_x, scan0_u_v3.m_vd_y, scan0_u_v3.m_vd_z, scan0_u_v3.m_vd_s, scan0_u_v3.m_vd_dx, scan0_u_v3.m_vd_dy, scan0_u_v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
				TRACE("Loading %s failed..\n", tumor_deformation_field_r_file);
			} else  {
				scan0_u_v3.computeDimension();
			}

			// apply mass
			scan0_atlas_reg_priors[CSF].load(scan0_atlas_reg_init_prior_files[CSF], 1);
			//
			vd_x = scan0_atlas_reg_priors[CSF].m_vd_x;
			vd_y = scan0_atlas_reg_priors[CSF].m_vd_y;
			vd_z = scan0_atlas_reg_priors[CSF].m_vd_z;
			vd_dx = scan0_atlas_reg_priors[CSF].m_vd_dx;
			vd_dy = scan0_atlas_reg_priors[CSF].m_vd_dy;
			vd_dz = scan0_atlas_reg_priors[CSF].m_vd_dz;
			//
			scan0_atlas_reg_mass_priors[CSF].allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan0_atlas_reg_mass_priors[CSF], scan0_atlas_reg_priors[CSF], scan0_u_v3, 0.0f, false);
			scan0_atlas_reg_priors[CSF].clear();
			//
#ifdef USE_11A_PRIORS
			scan0_atlas_reg_priors[VT ].load(scan0_atlas_reg_init_prior_files[VT ], 1);
			scan0_atlas_reg_mass_priors[VT ].allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan0_atlas_reg_mass_priors[VT ], scan0_atlas_reg_priors[VT ], scan0_u_v3, 0.0f, false);
			scan0_atlas_reg_priors[VT ].clear();
#endif
			//
			scan0_atlas_reg_priors[GM ].load(scan0_atlas_reg_init_prior_files[GM ], 1);
			scan0_atlas_reg_mass_priors[GM ].allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan0_atlas_reg_mass_priors[GM ], scan0_atlas_reg_priors[GM ], scan0_u_v3, 0.0f, false);
			scan0_atlas_reg_priors[GM ].clear();
			//
			scan0_atlas_reg_priors[WM ].load(scan0_atlas_reg_init_prior_files[WM ], 1);
			scan0_atlas_reg_mass_priors[WM ].allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan0_atlas_reg_mass_priors[WM ], scan0_atlas_reg_priors[WM ], scan0_u_v3, 0.0f, false);
			scan0_atlas_reg_priors[WM ].clear();
			//
#ifdef USE_VS_PRIOR
			scan0_atlas_reg_priors[VS ].load(scan0_atlas_reg_init_prior_files[VS ], 1);
			scan0_atlas_reg_mass_priors[VS ].allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan0_atlas_reg_mass_priors[VS ], scan0_atlas_reg_priors[VS ], scan0_u_v3, 0.0f, false);
			scan0_atlas_reg_priors[VS ].clear();
#endif
			//
#ifdef USE_WARPED_BG
			scan0_atlas_reg_priors[BG ].load(scan0_atlas_reg_init_prior_files[BG ], 1);
			scan0_atlas_reg_mass_priors[BG ].allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan0_atlas_reg_mass_priors[BG ], scan0_atlas_reg_priors[BG ], scan0_u_v3, 1.0f, false);
			scan0_atlas_reg_priors[BG ].clear();
#endif
			//
#ifdef USE_COMPUTEQ2
			scan2_atlas_reg_abnor_mask.load((char*)scan2_atlas_reg_abnor_mask_file, 1);
			scan2_atlas_reg_mass_abnor_mask.allocate(vd_x, vd_y, vd_z);
			GenerateBackwardWarpVolume(scan2_atlas_reg_mass_abnor_mask, scan2_atlas_reg_abnor_mask, scan0_u_v3, (unsigned char)0, true);
			scan2_atlas_reg_abnor_mask.clear();
			//
			for (i = 0; i < NumberOfImageChannels; i++) {
				scan2_atlas_reg_images[i].load(scan2_atlas_reg_image_files[i], 1);
				scan2_atlas_reg_mass_images[i].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan2_atlas_reg_mass_images[i], scan2_atlas_reg_images[i], scan0_u_v3, 0.0f, false);
				scan2_atlas_reg_images[i].clear();
			}
#endif

			scan0_u_v3.clear();

			scan0_atlas_reg_mass_priors[TU].allocate(vd_x, vd_y, vd_z);
			scan0_atlas_reg_mass_priors[ED].allocate(vd_x, vd_y, vd_z);
#ifdef USE_11A_PRIORS
			scan0_atlas_reg_mass_priors[RTE].allocate(vd_x, vd_y, vd_z);
#endif

			// combine tumor and edema prior
			tumor_density.load(tumor_density_file, 1);
			//
			//MakeProbForTumorAndEdema2(scan0_atlas_reg_mass_priors[TU], scan0_atlas_reg_mass_priors[ED], scan0_atlas_reg_mass_priors[CSF],
			MakeProbForTumorAndEdema3(scan0_atlas_reg_mass_priors[TU], scan0_atlas_reg_mass_priors[ED], scan0_atlas_reg_mass_priors[CSF],
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_mass_priors[VT],
#endif
				scan0_atlas_reg_mass_priors[GM], scan0_atlas_reg_mass_priors[WM], 
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_mass_priors[VS],
#endif
#ifdef USE_WARPED_BG
				scan0_atlas_reg_mass_priors[BG],
#endif
				tumor_density, gxc, gyc, gzc, T);

			//
			tumor_density.clear();
		}
	}

	// compute matching cost regarding density information
	{
		if (!sol) {
			double score_sum = 0.0;
			//
#ifdef USE_COMPUTEQ2
			double score1_sum = 0;
			double score2_sum = 0;
			double score2[NumberOfImageChannels];
#endif

#if 1
			// calcluate posterior * log (prior * image)
			{
				if (!LoadMHDData(NULL, (char*)scan0_h2_hdr_file, 
					&scan0_h2_v3.m_pData, scan0_h2_v3.m_vd_x, scan0_h2_v3.m_vd_y, scan0_h2_v3.m_vd_z, scan0_h2_v3.m_vd_s, scan0_h2_v3.m_vd_dx, scan0_h2_v3.m_vd_dy, scan0_h2_v3.m_vd_dz, vd_ox, vd_oy, vd_oz)) {
					TRACE("Loading %s failed..\n", scan0_h_hdr_file);
				} else  {
					scan0_h2_v3.computeDimension();
				}

				// apply warp
				scan0_atlas_reg_warp_priors[CSF].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[CSF], scan0_atlas_reg_mass_priors[CSF], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[CSF].clear();
				//
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_warp_priors[VT ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[VT ], scan0_atlas_reg_mass_priors[VT ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[VT ].clear();
#endif
				//
				scan0_atlas_reg_warp_priors[GM ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[GM ], scan0_atlas_reg_mass_priors[GM ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[GM ].clear();
				//
				scan0_atlas_reg_warp_priors[WM ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[WM ], scan0_atlas_reg_mass_priors[WM ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[WM ].clear();
				//
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_warp_priors[VS ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[VS ], scan0_atlas_reg_mass_priors[VS ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[VS ].clear();
#else
				scan0_atlas_reg_warp_priors[VS ].copy(scan0_atlas_reg_warp_priors[CSF]);
#endif
				//
#ifdef USE_WARPED_BG
				scan0_atlas_reg_warp_priors[BG ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[BG ], scan0_atlas_reg_mass_priors[BG ], scan0_h2_v3, 1.0f, false);
				scan0_atlas_reg_mass_priors[BG ].clear();
#else
				scan0_atlas_reg_warp_priors[BG ].load(scan0_atlas_reg_init_prior_files[BG], 1);
#endif
				//
				scan0_atlas_reg_warp_priors[TU ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[TU ], scan0_atlas_reg_mass_priors[TU ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[TU ].clear();
				scan0_atlas_reg_warp_priors[NCR].copy(scan0_atlas_reg_warp_priors[TU ]);
				//
				scan0_atlas_reg_warp_priors[ED ].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[ED ], scan0_atlas_reg_mass_priors[ED ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[ED ].clear();
				//
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_warp_priors[RTE].allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan0_atlas_reg_warp_priors[RTE], scan0_atlas_reg_mass_priors[TU ], scan0_h2_v3, 0.0f, false);
				scan0_atlas_reg_mass_priors[RTE].clear();
				scan0_atlas_reg_warp_priors[RTN].copy(scan0_atlas_reg_warp_priors[RTE]);
#endif
				//
#ifdef USE_COMPUTEQ2
				scan2_atlas_reg_warp_abnor_mask.allocate(vd_x, vd_y, vd_z);
				GenerateBackwardWarpVolume(scan2_atlas_reg_warp_abnor_mask, scan2_atlas_reg_mass_abnor_mask, scan0_h2_v3, (unsigned char)0, true);
				scan2_atlas_reg_mass_abnor_mask.clear();
				// add grown tumor region
				{
					for (n = 0; n < vd_z; n++) {
						for (m = 0; m < vd_y; m++) {
							for (l = 0; l < vd_x; l++) {
								if (scan0_atlas_reg_warp_priors[TU].m_pData[n][m][l][0] > 0.1) {
									scan2_atlas_reg_warp_abnor_mask.m_pData[n][m][l][0] = 1;
								}
							}
						}
					}
				}
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					scan2_atlas_reg_warp_images[i].allocate(vd_x, vd_y, vd_z);
					GenerateBackwardWarpVolume(scan2_atlas_reg_warp_images[i], scan2_atlas_reg_mass_images[i], scan0_h2_v3, 0.0f, false);
					scan2_atlas_reg_mass_images[i].clear();
				}
#endif

				/*
				scan0_atlas_reg_mass_priors[CSF].save(scan0_atlas_reg_mass_prior_files[CSF], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[CSF], scan0_atlas_reg_image_files[0]);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_mass_priors[VT ].save(scan0_atlas_reg_mass_prior_files[VT ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[VT ], scan0_atlas_reg_image_files[0]);
#endif
				scan0_atlas_reg_mass_priors[GM ].save(scan0_atlas_reg_mass_prior_files[GM ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[GM ], scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_mass_priors[WM ].save(scan0_atlas_reg_mass_prior_files[WM ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[WM ], scan0_atlas_reg_image_files[0]);
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_mass_priors[VS ].save(scan0_atlas_reg_mass_prior_files[VS ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[VS ], scan0_atlas_reg_image_files[0]);
#endif
#ifdef USE_WARPED_BG
				scan0_atlas_reg_mass_priors[BG ].save(scan0_atlas_reg_mass_prior_files[BG ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[BG ], scan0_atlas_reg_image_files[0]);
#endif
				scan0_atlas_reg_mass_priors[TU ].save(scan0_atlas_reg_mass_prior_files[TU ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[TU ], scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_mass_priors[ED ].save(scan0_atlas_reg_mass_prior_files[ED ], 1); ChangeNIIHeader(scan0_atlas_reg_mass_prior_files[ED ], scan0_atlas_reg_image_files[0]);
				//
				scan2_atlas_reg_mass_abnor_mask.save(scan2_atlas_reg_mass_abnor_mask_file, 1); ChangeNIIHeader(scan2_atlas_reg_mass_abnor_mask_file, scan0_atlas_reg_image_files[0]);
				//
				scan0_atlas_reg_warp_priors[CSF].save(scan0_atlas_reg_warp_prior_files[CSF], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[CSF], scan0_atlas_reg_image_files[0]);
#ifdef USE_11A_PRIORS
				scan0_atlas_reg_warp_priors[VT ].save(scan0_atlas_reg_warp_prior_files[VT ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[VT ], scan0_atlas_reg_image_files[0]);
#endif
				scan0_atlas_reg_warp_priors[GM ].save(scan0_atlas_reg_warp_prior_files[GM ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[GM ], scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_warp_priors[WM ].save(scan0_atlas_reg_warp_prior_files[WM ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[WM ], scan0_atlas_reg_image_files[0]);
#ifdef USE_VS_PRIOR
				scan0_atlas_reg_warp_priors[VS ].save(scan0_atlas_reg_warp_prior_files[VS ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[VS ], scan0_atlas_reg_image_files[0]);
#endif
#ifdef USE_WARPED_BG
				scan0_atlas_reg_warp_priors[BG ].save(scan0_atlas_reg_warp_prior_files[BG ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[BG ], scan0_atlas_reg_image_files[0]);
#endif
				scan0_atlas_reg_warp_priors[TU ].save(scan0_atlas_reg_warp_prior_files[TU ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[TU ], scan0_atlas_reg_image_files[0]);
				scan0_atlas_reg_warp_priors[ED ].save(scan0_atlas_reg_warp_prior_files[ED ], 1); ChangeNIIHeader(scan0_atlas_reg_warp_prior_files[ED ], scan0_atlas_reg_image_files[0]);
				//
#ifdef USE_COMPUTEQ2
				scan2_atlas_reg_warp_abnor_mask.save(scan2_atlas_reg_warp_abnor_mask_file, 1); ChangeNIIHeader(scan2_atlas_reg_warp_abnor_mask_file, scan0_atlas_reg_image_files[0]);
#endif
				//*/

				scan0_h2_v3.clear();

				for (i = 0; i < NumberOfImageChannels; i++) {
					scan0_atlas_reg_images[i].load(scan0_atlas_reg_image_files[i], 1);
				}

#ifndef USE_COMPUTEQ2
				ComputeQ(scan0_atlas_reg_images, scan0_atlas_reg_warp_priors, scan0_means_file, scan0_variances_file, &score_sum, tag, (char*)tag_F);
#else
				{
					for (i = 0; i < NumberOfPriorChannels; i++) {
						scan0_atlas_reg_posteriors[i].allocate(vd_x, vd_y, vd_z);
					}
					scan0_atlas_reg_abnor_mask.allocate(vd_x, vd_y, vd_z);

					ComputeQ2(scan0_atlas_reg_images, scan0_atlas_reg_warp_priors, scan0_atlas_reg_posteriors, scan0_means_file, scan0_variances_file, &score1_sum, tag, (char*)tag_F);
					//
					for (k = 0; k < scan0_atlas_reg_abnor_mask.m_vd_z; k++) {
						for (j = 0; j < scan0_atlas_reg_abnor_mask.m_vd_y; j++) {
							for (i = 0; i < scan0_atlas_reg_abnor_mask.m_vd_x; i++) {
								float max_l_val = scan0_atlas_reg_posteriors[0].m_pData[k][j][i][0];
								int max_l_idx = 0;
								for (l = 1; l < NumberOfPriorChannels; l++) {
									// prefer higher labels
									if (max_l_val <= scan0_atlas_reg_posteriors[l].m_pData[k][j][i][0]) {
										max_l_val = scan0_atlas_reg_posteriors[l].m_pData[k][j][i][0];
										max_l_idx = l;
									}
								}
#ifdef USE_11A_PRIORS
								if (max_l_idx == TU || max_l_idx == NCR || max_l_idx == ED || max_l_idx == RTN || max_l_idx == RTE ) {
#else
								if (max_l_idx == TU || max_l_idx == NCR || max_l_idx == ED) {
#endif
									scan0_atlas_reg_abnor_mask.m_pData[k][j][i][0] = 1;
								} else {
									scan0_atlas_reg_abnor_mask.m_pData[k][j][i][0] = 0;
								}
							}
						}
					}
					//
					//scan0_atlas_reg_abnor_mask.save(scan0_atlas_reg_abnor_mask_file, 1); ChangeNIIHeader(scan0_atlas_reg_abnor_mask_file, scan0_atlas_reg_image_files[0]);
					//
					//*
					for (i = 0; i < NumberOfImageChannels; i++) {
						ComputeQ_NMI(scan0_atlas_reg_images[i].m_pData, scan0_atlas_reg_abnor_mask.m_pData, scan2_atlas_reg_warp_images[i].m_pData, scan2_atlas_reg_warp_abnor_mask.m_pData, vd_x, vd_y, vd_z, &score2[i], tag, (char*)tag_F);
						score2_sum += score2[i];
					}
					score2_sum /= NumberOfImageChannels;
					/*/
					ComputeQ_NMI(scan0_atlas_reg_images[0].m_pData, scan0_atlas_reg_abnor_mask.m_pData, scan2_atlas_reg_warp_images[0].m_pData, scan2_atlas_reg_warp_abnor_mask.m_pData, vd_x, vd_y, vd_z, &score2[0], tag, (char*)tag_F);
					score2_sum += score2[0];
					//*/

					for (i = 0; i < NumberOfPriorChannels; i++) {
						scan0_atlas_reg_posteriors[i].clear();
					}
					scan0_atlas_reg_abnor_mask.clear();

					//score_sum = score1_sum + score2_sum;
					//score_sum = 0.5 * score1_sum + score2_sum;
					score_sum = score1_sum;
				}
#endif

				for (i = 0; i < NumberOfPriorChannels; i++) {
					scan0_atlas_reg_warp_priors[i].clear();
				}
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					scan0_atlas_reg_images[i].clear();
				}
				//
#ifdef USE_COMPUTEQ2
				scan2_atlas_reg_warp_abnor_mask.clear();
				//
				for (i = 0; i < NumberOfImageChannels; i++) {
					scan2_atlas_reg_warp_images[i].clear();
				}
#endif

				{
					while (TRUE) {
#if defined(WIN32) || defined(WIN64)
						FILE* fp;
						fp = _fsopen(parameter_cost_file, "a", _SH_DENYWR);
						if (fp == NULL) {
							Sleep(100);
							continue;
						}
#else
						FILE* fp;
						fp = fopen(parameter_cost_file, "a");
						if (fp == NULL) {
							continue;
						}
#endif
						//
#ifdef USE_COMPUTEQ2
						fprintf(fp, "no:%04d gxc:%10.8f (%4.0f) gyc:%10.8f (%4.0f) gzc:%10.8f (%4.0f) p1:%12g p2:%12g rho:%7g dw:%12g dg:%12g T:%8.3f  cost:%10.8f [%10.8f, %10.8f]\n",
							tag, gxc, 1000 * gxc / vd_dx, gyc, 1000 * gyc / vd_dy, gzc, 1000 * gzc / vd_dz, gp1, gp2, grho, gdiffwm, gdiffgm, T, score_sum, score1_sum, score2_sum);
#else
						fprintf(fp, "no:%04d gxc:%10.8f (%4.0f) gyc:%10.8f (%4.0f) gzc:%10.8f (%4.0f) p1:%12g p2:%12g rho:%7g dw:%12g dg:%12g T:%8.3f  cost:%10.8f\n",
							tag, gxc, 1000 * gxc / vd_dx, gyc, 1000 * gyc / vd_dy, gzc, 1000 * gzc / vd_dz, gp1, gp2, grho, gdiffwm, gdiffgm, T, score_sum);
#endif
						//
						fclose(fp);
						break;
					}
				}
			}
#endif
		
			// store output file
			{
				FILE* fp;
				fp = fopen(output_file, "w");
				fprintf (fp, "1\n");
				fprintf(fp, "%23.15e", score_sum);
				fclose(fp);
			}

			SetCurrentDirectory(tmp_folder);

			// remove working directory
			DeleteAll(simulator_folder, TRUE);
		} else {
			// copy mass prior and deformation field to the tmp_folder
			// tag is the iteration number
			char u_file[1024];
			char u_inv_file[1024];
			char atlas_reg_mass_prior_files[NumberOfPriorChannels][1024];
			
			sprintf(u_file, "%s%ss0_u_jsr_%d.mhd", tmp_folder, DIR_SEP, tag);
			sprintf(u_inv_file, "%s%ss0_u_inv_jsr_%d.mhd", tmp_folder, DIR_SEP, tag);
			for (i = 0; i < NumberOfPriorChannels; i++) {
				sprintf(atlas_reg_mass_prior_files[i], "%s%ss0_atlas_reg_mass_prior_jsr_%d_%d.nii.gz", tmp_folder, DIR_SEP, tag, i);
			}
			//
			CopyMHDData(NULL, tumor_deformation_field_r_file, NULL, u_file);
			//
			{
				sprintf(szCmdLine, "%s -i %s -o %s -r %s -x %.15f -y %.15f -z %.15f", RESAMPLE_DEFORMATION_FIELD_PATH, tumor_deformation_field_s_file, tumor_deformation_field_file, scan2_atlas_reg_image_files[0], 1.0, 1.0, 1.0);
				//
				TRACE("%d_%s: %s\n", tag, tag_F, szCmdLine);
				//
				if (!ExecuteProcess(szCmdLine)) {
					exit(EXIT_FAILURE);
				}
			}
			CopyMHDData(NULL, tumor_deformation_field_file, NULL, u_inv_file);
			//
#ifdef USE_VS_PRIOR
			scan0_atlas_reg_mass_priors[VS ].save(atlas_reg_mass_prior_files[VS ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[VS ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[VS ].clear();
#else
			scan0_atlas_reg_mass_priors[CSF].save(atlas_reg_mass_prior_files[VS ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[VS ], scan0_atlas_reg_image_files[0]);
#endif
			//
			scan0_atlas_reg_mass_priors[CSF].save(atlas_reg_mass_prior_files[CSF], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[CSF], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[CSF].clear();
			//
#ifdef USE_11A_PRIORS
			scan0_atlas_reg_mass_priors[VT ].save(atlas_reg_mass_prior_files[VT ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[VT ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[VT ].clear();
#endif
			scan0_atlas_reg_mass_priors[GM ].save(atlas_reg_mass_prior_files[GM ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[GM ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[GM ].clear();
			//
			scan0_atlas_reg_mass_priors[WM ].save(atlas_reg_mass_prior_files[WM ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[WM ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[WM ].clear();
			//
			scan0_atlas_reg_mass_priors[TU ].save(atlas_reg_mass_prior_files[TU ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[TU ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[TU ].save(atlas_reg_mass_prior_files[NCR], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[NCR], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[TU ].clear();
			//
			scan0_atlas_reg_mass_priors[ED ].save(atlas_reg_mass_prior_files[ED ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[ED ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[ED ].clear();
#ifdef USE_11A_PRIORS
			scan0_atlas_reg_mass_priors[RTE].save(atlas_reg_mass_prior_files[RTE], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[RTE], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[RTE].save(atlas_reg_mass_prior_files[RTN], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[RTN], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[RTE].clear();
#endif
			//
#ifndef USE_WARPED_BG
			scan0_atlas_reg_mass_priors[BG ].load(scan0_atlas_reg_init_prior_files[BG], 1);
#endif
			scan0_atlas_reg_mass_priors[BG ].save(atlas_reg_mass_prior_files[BG ], 1); ChangeNIIHeader(atlas_reg_mass_prior_files[BG ], scan0_atlas_reg_image_files[0]);
			scan0_atlas_reg_mass_priors[BG ].clear();

			// remove input file
			DeleteFile(input_file);

			// remove all remaining files
			DeleteFiles((char*)tmp_folder, "input.*_F", FALSE);
			DeleteFiles((char*)tmp_folder, "output.*_F", FALSE);
			DeleteSubDirs((char*)tmp_folder, "*_F");

			SetCurrentDirectory(tmp_folder);
		}
	}

	TRACE("%d_%s: EvaluateQ... - done\n", tag, tag_F);

	exit(EXIT_SUCCESS);
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//#define USE_ED_INC_GM

// modify priors regaring tumor and edema
#define edema_eps 1e-8

BOOL MakeProbForTumorAndEdema(FVolume* priors, FVolume& tumor_density)
{
	int vd_x, vd_y, vd_z;
	int i, j, k;

	vd_x = priors[0].m_vd_x;
	vd_y = priors[0].m_vd_y;
	vd_z = priors[0].m_vd_z;

	for (k = 0; k < vd_z; k++) {
		for (j = 0; j < vd_y; j++) {
			for (i = 0; i < vd_x; i++) {
				float c, c_1, wm, csf, gm, bg;
#ifdef USE_11A_PRIORS
				float vt;
#endif
				//
				c = tumor_density.m_pData[k][j][i][0];
				c_1 = 1.0 - c;
				//
				priors[TU].m_pData[k][j][i][0] = 0.5 * c;
				priors[NCR].m_pData[k][j][i][0] = 0.5 * c;
				//
				csf = priors[CSF].m_pData[k][j][i][0] * c_1;
				priors[CSF].m_pData[k][j][i][0] = 0.5 * csf;
				priors[VS ].m_pData[k][j][i][0] = 0.5 * csf;
				//
#ifdef USE_11A_PRIORS
				vt = priors[VT].m_pData[k][j][i][0] * c_1;
				priors[VT].m_pData[k][j][i][0] = vt;
#endif
				//
				bg = priors[BG].m_pData[k][j][i][0] * c_1;
				priors[BG].m_pData[k][j][i][0] = bg;
				//
#ifndef USE_ED_INC_GM
				gm = priors[GM].m_pData[k][j][i][0] * c_1;
				priors[GM].m_pData[k][j][i][0] = gm;
				//
				wm = priors[WM].m_pData[k][j][i][0] * c_1;
				if (c > edema_eps && c < 1.0) {
					priors[WM].m_pData[k][j][i][0] = 0.5 * wm;
					priors[ED].m_pData[k][j][i][0] = 0.5 * wm;
				} else {
					priors[WM].m_pData[k][j][i][0] = wm;
					priors[ED].m_pData[k][j][i][0] = 0;
				}
#else
				gm = priors[GM].m_pData[k][j][i][0] * c_1;
				wm = priors[WM].m_pData[k][j][i][0] * c_1;
				if (c > edema_eps && c < 1.0) {
					priors[GM].m_pData[k][j][i][0] = 0.5 * gm;
					priors[WM].m_pData[k][j][i][0] = 0.5 * wm;
					priors[ED].m_pData[k][j][i][0] = 0.5 * (gm + wm);
				} else {
					priors[GM].m_pData[k][j][i][0] = gm;
					priors[WM].m_pData[k][j][i][0] = wm;
					priors[ED].m_pData[k][j][i][0] = 0;
				}
#endif
			}
		}
	}

	return TRUE;
}
BOOL MakeProbForTumorAndEdema2(FVolume& priors_TU, FVolume& priors_ED, FVolume& priors_CSF, 
#ifdef USE_11A_PRIORS
	FVolume& priors_VT, 
#endif
	FVolume& priors_GM, FVolume& priors_WM, 
#ifdef USE_VS_PRIOR	
	FVolume& priors_VS, 
#endif
#ifdef USE_WARPED_BG
	FVolume& priors_BG, 
#endif
	FVolume& tumor_density)
{
	int vd_x, vd_y, vd_z;
	int i, j, k;

	vd_x = tumor_density.m_vd_x;
	vd_y = tumor_density.m_vd_y;
	vd_z = tumor_density.m_vd_z;

	for (k = 0; k < vd_z; k++) {
		for (j = 0; j < vd_y; j++) {
			for (i = 0; i < vd_x; i++) {
				float c, c_1, wm, csf, gm;
#ifdef USE_11A_PRIORS
				float vt;
#endif
#ifdef USE_VS_PRIOR
				float vs;
#endif
#ifdef USE_WARPED_BG
				float bg;
#endif
				//
				c = tumor_density.m_pData[k][j][i][0];
				c_1 = 1.0 - c;
				//
				priors_TU.m_pData[k][j][i][0] = 0.5 * c;
				//
				csf = priors_CSF.m_pData[k][j][i][0] * c_1;
				priors_CSF.m_pData[k][j][i][0] = csf;
				//
#ifdef USE_11A_PRIORS
				vt = priors_VT.m_pData[k][j][i][0] * c_1;
				priors_VT.m_pData[k][j][i][0] = vt;
#endif
				//
#ifdef USE_VS_PRIOR
				vs = priors_VS.m_pData[k][j][i][0] * c_1;
				priors_VS.m_pData[k][j][i][0] = vs;
#endif
				//
#ifdef USE_WARPED_BG
				bg = priors_BG.m_pData[k][j][i][0] * c_1;
				priors_BG.m_pData[k][j][i][0] = bg;
#endif
				//
#ifndef USE_ED_INC_GM
				gm = priors_GM.m_pData[k][j][i][0] * c_1;
				priors_GM.m_pData[k][j][i][0] = gm;
				//
				wm = priors_WM.m_pData[k][j][i][0] * c_1;
				if (c > edema_eps && c < 1.0) {
					priors_WM.m_pData[k][j][i][0] = 0.5 * wm;
					priors_ED.m_pData[k][j][i][0] = 0.5 * wm;
				} else {
					priors_WM.m_pData[k][j][i][0] = wm;
					priors_ED.m_pData[k][j][i][0] = 0;
				}
#else
				gm = priors_GM.m_pData[k][j][i][0] * c_1;
				wm = priors_WM.m_pData[k][j][i][0] * c_1;
				if (c > edema_eps && c < 1.0) {
					priors_GM.m_pData[k][j][i][0] = 0.5 * gm;
					priors_WM.m_pData[k][j][i][0] = 0.5 * wm;
					priors_ED.m_pData[k][j][i][0] = 0.5 * (gm + wm);
				} else {
					priors_GM.m_pData[k][j][i][0] = gm;
					priors_WM.m_pData[k][j][i][0] = wm;
					priors_ED.m_pData[k][j][i][0] = 0;
				}
#endif
			}
		}
	}

	return TRUE;
}
BOOL MakeProbForTumorAndEdema3(FVolume& priors_TU, FVolume& priors_ED, FVolume& priors_CSF, 
#ifdef USE_11A_PRIORS
	FVolume& priors_VT, 
#endif
	FVolume& priors_GM, FVolume& priors_WM, 
#ifdef USE_VS_PRIOR	
	FVolume& priors_VS, 
#endif
#ifdef USE_WARPED_BG
	FVolume& priors_BG, 
#endif
	FVolume& tumor_density, float gxc, float gyc, float gzc, float T)
{
	double dx, dy, dz;
	double x, y, z;
	double cx, cy, cz;
	//double d2, d,p_max_ED;
	double seeds_r;
	double seeds_d_TU;
	double seeds_d_ED;
	// sigmoid function
	// assume y = 1 / (1 + exp(a * (x - b)))
	double seeds_a_TU;
	double seeds_a_ED;
	int i, j, k;
	int vd_x, vd_y, vd_z;

	vd_x = tumor_density.m_vd_x;
	vd_y = tumor_density.m_vd_y;
	vd_z = tumor_density.m_vd_z;

	dx = tumor_density.m_vd_dx;
	dy = tumor_density.m_vd_dy;
	dz = tumor_density.m_vd_dz;

	cx = gxc * 1000;
	cy = gyc * 1000;
	cz = gzc * 1000;
	// tumor days
	seeds_r = pow(T / (0.004 * 4.0), 1.0/3.0) * 2;

	seeds_d_TU = seeds_r / 2;
	seeds_d_ED = seeds_r * 2;
	seeds_a_TU = 0.8;
	seeds_a_ED = 0.2;

	z = 0;
	for (k = 0; k < vd_z; k++) {
		y = 0;
		for (j = 0; j < vd_y; j++) {
			x = 0;
			for (i = 0; i < vd_x; i++) {
				float c, c_1, csf, gm;
#ifdef USE_11A_PRIORS
				float vt;
#endif
#ifdef USE_VS_PRIOR
				float vs;
#endif
#ifdef USE_WARPED_BG
				float bg;
#endif
				//
				c = tumor_density.m_pData[k][j][i][0];
				c_1 = 1.0 - c;
				//
#if 0
				{
					double p2, w, wm;
					//*
					d2 = (x-cx)*(x-cx) + (y-cy)*(y-cy) + (z-cz)*(z-cz);
					d =	sqrt(d2);
					/*/
					d = log(1.0 / c - 1.0) / seeds_a_TU + seeds_d_TU;
					//*/
					p_max_ED = 1.0 / (1.0 + exp(seeds_a_ED * (d - seeds_d_ED)));
					if (p_max_ED > 1e-6) {
						p2 = p_max_ED / 4;
						if (p2 > 0.5) {
							p2 = 0.5;
						}
					} else {
						p2 = 0;
					}
#ifndef USE_ED_INC_GM
					wm = priors_WM.m_pData[k][j][i][0] * c_1;
					priors_WM.m_pData[k][j][i][0] = (1.0 - p2)  * wm;
					priors_ED.m_pData[k][j][i][0] = p2 * wm;
#else
					gm = priors_GM.m_pData[k][j][i][0] * c_1;
					wm = priors_WM.m_pData[k][j][i][0] * c_1;
					priors_GM.m_pData[k][j][i][0] = (1.0 - p2) * gm;
					priors_WM.m_pData[k][j][i][0] = (1.0 - p2) * wm;
					priors_ED.m_pData[k][j][i][0] = p2 * (gm + wm);
#endif
				}
#else
				{
					double x1, x2;
					double p2, a, wm;
					x1 = 1e-6;
					x2 = 1e-10;
					if (c > x1) {
						p2 = 0.5;
					} else if (c > x2) {
						a = 0.5 / (log10(x1) - log10(x2));
						p2 = a * (log10(c) - log10(x2));
					} else {
						p2 = 0;
					}
#ifndef USE_ED_INC_GM
					wm = priors_WM.m_pData[k][j][i][0] * c_1;
					priors_WM.m_pData[k][j][i][0] = (1.0 - p2)  * wm;
					priors_ED.m_pData[k][j][i][0] = p2 * wm;
#else
					gm = priors_GM.m_pData[k][j][i][0] * c_1;
					wm = priors_WM.m_pData[k][j][i][0] * c_1;
					priors_GM.m_pData[k][j][i][0] = (1.0 - p2) * gm;
					priors_WM.m_pData[k][j][i][0] = (1.0 - p2) * wm;
					priors_ED.m_pData[k][j][i][0] = p2 * (gm + wm);
#endif
				}
#endif
				//
				if (c > 0) {
					priors_TU.m_pData[k][j][i][0] = 0.5 * c;
					//
					csf = priors_CSF.m_pData[k][j][i][0] * c_1;
					priors_CSF.m_pData[k][j][i][0] = csf;
					//
#ifdef USE_11A_PRIORS
					vt = priors_VT.m_pData[k][j][i][0] * c_1;
					priors_VT.m_pData[k][j][i][0] = vt;
#endif
					//
#ifdef USE_VS_PRIOR
					vs = priors_VS.m_pData[k][j][i][0] * c_1;
					priors_VS.m_pData[k][j][i][0] = vs;
#endif
					//
#ifdef USE_WARPED_BG
					bg = priors_BG.m_pData[k][j][i][0] * c_1;
					priors_BG.m_pData[k][j][i][0] = bg;
#endif
					//
#ifndef USE_ED_INC_GM
					gm = priors_GM.m_pData[k][j][i][0] * c_1;
					priors_GM.m_pData[k][j][i][0] = gm;
#endif
				}
				//
				x += dx;
			}
			y += dy;
		}
		z += dz;
	}

	return TRUE;
}

BOOL LoadMeansAndVariances(const char* means_file, const char* variances_file, MeanVectorType* pMeanVector, VarianceVectorType* pVarianceVector)
{
	FILE* fp;
	int i, j, k;

	////////////////////////////////////////////////////////////////////////////////
	// initialize
	for (i = 0; i < NumberOfPriorChannels; i++) {
		MeanType v(0.0);
		pMeanVector->push_back(v); 
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		VarianceType s;
		s.set_identity();
		pVarianceVector->push_back(s);
	}
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// read means
	fp = fopen(means_file, "r");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", means_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		char class_id[1024];
		MeanType mean;
		float meanj;
		//
		fscanf(fp, "%s", class_id);
		for (j = 0; j < NumberOfImageChannels; j++) {
			fscanf(fp, "%f", &meanj);
			mean(j) = meanj;
		}
		for (j = 0; j < NumberOfPriorChannels; j++) {
			if (strcmp(class_id, label[j]) == 0) {
				pMeanVector->at(j) = mean;
				break;
			}
		}
		if (j == NumberOfPriorChannels) {
			TRACE("class id is wrong");
			fclose(fp);
			return FALSE;
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	////////////////////////////////////////////////////////////////////////////////
	// read variances
	fp = fopen(variances_file, "r");
	if (fp == NULL) {
		TRACE("Failed to open file: '%s'\n", variances_file);
		return FALSE;
	}
	for(i = 0; i < NumberOfPriorChannels; i++) {
		char class_id[1024];
		VarianceType var;
		float varjk;
		//
		fscanf(fp, "%s", class_id);
		for (j = 0; j < NumberOfImageChannels; j++) {
			for (k = 0; k < NumberOfImageChannels; k++) {
				fscanf(fp, "%f", &varjk);
				var(j, k) = varjk;
			}
		}
		for (j = 0; j < NumberOfPriorChannels; j++) {
			if (strcmp(class_id, label[j]) == 0) {
				pVarianceVector->at(j) = var;
				break;
			}
		}
		if (j == NumberOfPriorChannels) {
			TRACE("class id is wrong");
			fclose(fp);
			return FALSE;
		}
	}
	fclose(fp);
	////////////////////////////////////////////////////////////////////////////////

	return TRUE;
}

static
inline void ComputeLikelihood4(double m[16], double ym[4], double* like) {
	double inv[16], det, det_1, ySIy;
	
	inv[ 0] =  m[ 5] * m[10] * m[15] - m[ 5] * m[11] * m[14] - m[ 9] * m[ 6] * m[15] + m[ 9] * m[ 7] * m[14] + m[13] * m[ 6] * m[11] - m[13] * m[ 7] * m[10];
	inv[ 4] = -m[ 4] * m[10] * m[15] + m[ 4] * m[11] * m[14] + m[ 8] * m[ 6] * m[15] - m[ 8] * m[ 7] * m[14] - m[12] * m[ 6] * m[11] + m[12] * m[ 7] * m[10];
	inv[ 8] =  m[ 4] * m[ 9] * m[15] - m[ 4] * m[11] * m[13] - m[ 8] * m[ 5] * m[15] + m[ 8] * m[ 7] * m[13] + m[12] * m[ 5] * m[11] - m[12] * m[ 7] * m[ 9];
	inv[12] = -m[ 4] * m[ 9] * m[14] + m[ 4] * m[10] * m[13] + m[ 8] * m[ 5] * m[14] - m[ 8] * m[ 6] * m[13] - m[12] * m[ 5] * m[10] + m[12] * m[ 6] * m[ 9];
	inv[ 1] = -m[ 1] * m[10] * m[15] + m[ 1] * m[11] * m[14] + m[ 9] * m[ 2] * m[15] - m[ 9] * m[ 3] * m[14] - m[13] * m[ 2] * m[11] + m[13] * m[ 3] * m[10];
	inv[ 5] =  m[ 0] * m[10] * m[15] - m[ 0] * m[11] * m[14] - m[ 8] * m[ 2] * m[15] + m[ 8] * m[ 3] * m[14] + m[12] * m[ 2] * m[11] - m[12] * m[ 3] * m[10];
	inv[ 9] = -m[ 0] * m[ 9] * m[15] + m[ 0] * m[11] * m[13] + m[ 8] * m[ 1] * m[15] - m[ 8] * m[ 3] * m[13] - m[12] * m[ 1] * m[11] + m[12] * m[ 3] * m[ 9];
	inv[13] =  m[ 0] * m[ 9] * m[14] - m[ 0] * m[10] * m[13] - m[ 8] * m[ 1] * m[14] + m[ 8] * m[ 2] * m[13] + m[12] * m[ 1] * m[10] - m[12] * m[ 2] * m[ 9];
	inv[ 2] =  m[ 1] * m[ 6] * m[15] - m[ 1] * m[ 7] * m[14] - m[ 5] * m[ 2] * m[15] + m[ 5] * m[ 3] * m[14] + m[13] * m[ 2] * m[ 7] - m[13] * m[ 3] * m[ 6];   
	inv[ 6] = -m[ 0] * m[ 6] * m[15] + m[ 0] * m[ 7] * m[14] + m[ 4] * m[ 2] * m[15] - m[ 4] * m[ 3] * m[14] - m[12] * m[ 2] * m[ 7] + m[12] * m[ 3] * m[ 6];  
	inv[10] =  m[ 0] * m[ 5] * m[15] - m[ 0] * m[ 7] * m[13] - m[ 4] * m[ 1] * m[15] + m[ 4] * m[ 3] * m[13] + m[12] * m[ 1] * m[ 7] - m[12] * m[ 3] * m[ 5];   
	inv[14] = -m[ 0] * m[ 5] * m[14] + m[ 0] * m[ 6] * m[13] + m[ 4] * m[ 1] * m[14] - m[ 4] * m[ 2] * m[13] - m[12] * m[ 1] * m[ 6] + m[12] * m[ 2] * m[ 5];   
	inv[ 3] = -m[ 1] * m[ 6] * m[11] + m[ 1] * m[ 7] * m[10] + m[ 5] * m[ 2] * m[11] - m[ 5] * m[ 3] * m[10] - m[ 9] * m[ 2] * m[ 7] + m[ 9] * m[ 3] * m[ 6];  
	inv[ 7] =  m[ 0] * m[ 6] * m[11] - m[ 0] * m[ 7] * m[10] - m[ 4] * m[ 2] * m[11] + m[ 4] * m[ 3] * m[10] + m[ 8] * m[ 2] * m[ 7] - m[ 8] * m[ 3] * m[ 6];  
	inv[11] = -m[ 0] * m[ 5] * m[11] + m[ 0] * m[ 7] * m[ 9] + m[ 4] * m[ 1] * m[11] - m[ 4] * m[ 3] * m[ 9] - m[ 8] * m[ 1] * m[ 7] + m[ 8] * m[ 3] * m[ 5];   
	inv[15] =  m[ 0] * m[ 5] * m[10] - m[ 0] * m[ 6] * m[ 9] - m[ 4] * m[ 1] * m[10] + m[ 4] * m[ 2] * m[ 9] + m[ 8] * m[ 1] * m[ 6] - m[ 8] * m[ 2] * m[ 5];

	det = m[0] * inv[0] + m[1] * inv[4] + m[2] * inv[8] + m[3] * inv[12] + epss;  
	
	det_1 = 1.0 / det;

	ySIy = ym[0] * (inv[ 0] * ym[0] + inv[ 1] * ym[1] + inv[ 2] * ym[2] + inv[ 3] * ym[3]) +
	       ym[1] * (inv[ 4] * ym[0] + inv[ 5] * ym[1] + inv[ 6] * ym[2] + inv[ 7] * ym[3]) +
	       ym[2] * (inv[ 8] * ym[0] + inv[ 9] * ym[1] + inv[10] * ym[2] + inv[11] * ym[3]) +
	       ym[3] * (inv[12] * ym[0] + inv[13] * ym[1] + inv[14] * ym[2] + inv[15] * ym[3]);
	ySIy *= det_1;

	*like = 1.0 / vcl_sqrt(2.0 * PI * det) * exp (-0.5 * ySIy);
} 

#define USE_FAST_LIKELIHOOD
BOOL ComputeQ(FVolume* scan0_atlas_reg_images, FVolume* scan0_atlas_reg_warp_prior, const char* scan0_means_file, const char* scan0_variances_file, double* pScore, int tag, char* tag_F)
{
	MeanVectorType scan0_MeanVector;
	VarianceVectorType scan0_VarianceVector;
	int vd_x, vd_y, vd_z;
	int i, j, k, l, m, n;
    double cost = 0.0;
    double number_of_pixels = 0.0;
#ifdef USE_FAST_LIKELIHOOD
	double vv[NumberOfPriorChannels][16];
	double mv[NumberOfPriorChannels][4];

	// assume 4 channels for image
	if (NumberOfImageChannels != 4) {
		return FALSE;
	}
#endif

	TRACE("%d_%s: ComputeQ...\n", tag, tag_F);

	vd_x = scan0_atlas_reg_images[0].m_vd_x;
	vd_y = scan0_atlas_reg_images[0].m_vd_y;
	vd_z = scan0_atlas_reg_images[0].m_vd_z;

	if (!LoadMeansAndVariances(scan0_means_file, scan0_variances_file, &scan0_MeanVector, &scan0_VarianceVector)) {
		return FALSE;
	}

#ifdef USE_FAST_LIKELIHOOD
	// assume 4 channels for image
	if (NumberOfImageChannels != 4) {
		return FALSE;
	}
	//
	for (i = 0; i < NumberOfPriorChannels; i++) {
		for (j = 0; j < 4; j++) {
			for (k = 0; k < 4; k++) {
				vv[i][j*4+k] = scan0_VarianceVector.at(i)(j, k);
			}
			mv[i][j] = scan0_MeanVector.at(i)(j);
		}
	}
#endif

	for (n = 0; n < vd_z; n++) {
		for (m = 0; m < vd_y; m++) {
			for (l = 0; l < vd_x; l++) {
				double prior[NumberOfPriorChannels];
				double post[NumberOfPriorChannels];
				double like[NumberOfPriorChannels];
#ifdef USE_FAST_LIKELIHOOD
				double y[4], ym[4];
				double ys = 0;
#else
				MeanType y;
#endif

				number_of_pixels++;

				for (i = 0; i < NumberOfPriorChannels; i++) {
					//scan0_atlas_reg_warp_prior[i].GetAt(l, m, n, &prior[i]);
					prior[i] = scan0_atlas_reg_warp_prior[i].m_pData[n][m][l][0];
				}
				//
				// making a vnl vector from fixed images
				for (i = 0; i < NumberOfImageChannels; i++) {
#ifdef USE_FAST_LIKELIHOOD
					y[i] = scan0_atlas_reg_images[i].m_pData[n][m][l][0];
					ys += y[i];
#else
					//float dv;
					//scan0_atlas_reg_images[i].GetAt(l, m, n, &dv);
					//y(i) = dv;
					y(i) = scan0_atlas_reg_images[i].m_pData[n][m][l][0];
#endif
				}

				// if foreground...
				// consider also if tumor and ncr priors have resonable value
#ifdef USE_FAST_LIKELIHOOD
				if (ys > 0 || (prior[TU]+prior[NCR]) > 0.01) {
#else
				if (!y.is_zero() || (prior[TU]+prior[NCR]) > 0.01) {
#endif
					// to the number of classes
					for (i = 0; i < NumberOfPriorChannels; i++) {
#ifdef USE_FAST_LIKELIHOOD
						ym[0] = y[0] - mv[i][0]; 
						ym[1] = y[1] - mv[i][1]; 
						ym[2] = y[2] - mv[i][2]; 
						ym[3] = y[3] - mv[i][3]; 
						ComputeLikelihood4(vv[i], ym, &like[i]);
#else
						MeanType SIy = vnl_qr<double>(scan0_VarianceVector.at(i)).solve(y - scan0_MeanVector.at(i));
						double ySIy = dot_product(y - scan0_MeanVector.at(i), SIy);
						double detS = vnl_determinant(scan0_VarianceVector.at(i)) + epss;
						like[i] = 1.0 / vcl_sqrt(2.0 * PI * detS) * exp(-0.5 * ySIy);
#endif
					}
					// compute the denum
					double denum = epss;
					for (i = 0; i < NumberOfPriorChannels; i++) {
						denum +=  prior[i] * like[i];
					}
					// compute the posterior
					for (i = 0; i < NumberOfPriorChannels; i++) {
						post[i] = prior[i] * like[i] / denum;
					}
				// if background...
				} else {
					// compute the posterior
					for (i = 0; i < NumberOfPriorChannels; i++) {
						post[i] = 0.0;
						like[i] = 1.0;
					}
				}

				// to the number of classes
				for (i = 0; i < NumberOfPriorChannels; i++) {
					// we have to take care of negative values
					double val = like[i] * prior[i];
					if (val <= 0.0) val = epss;
					cost -= post[i] * log(val);
				}
			} // l
		} // m
	} // n

	cost /= number_of_pixels;

	*pScore = cost;
	TRACE("%d_%s: cost = %f, number_of_pixels = %f\n", tag, tag_F, cost, number_of_pixels);

	TRACE("%d_%s: ComputeQ... - done\n", tag, tag_F);
	
	return TRUE;
}

#ifdef USE_COMPUTEQ2
// compute Q and store posteriors
BOOL ComputeQ2(FVolume* scan0_atlas_reg_images, FVolume* scan0_atlas_reg_warp_prior, FVolume* scan0_atlas_reg_posterior, const char* scan0_means_file, const char* scan0_variances_file, double* pScore, int tag, char* tag_F)
{
	MeanVectorType scan0_MeanVector;
	VarianceVectorType scan0_VarianceVector;
	int vd_x, vd_y, vd_z;
	int i, j, k, l, m, n;
    double cost = 0.0;
    double number_of_pixels = 0.0;
	double vv[NumberOfPriorChannels][16];
	double mv[NumberOfPriorChannels][4];

	// assume 4 channels for image
	if (NumberOfImageChannels != 4) {
		return FALSE;
	}

	TRACE("%d_%s: ComputeQ...\n", tag, tag_F);

	vd_x = scan0_atlas_reg_images[0].m_vd_x;
	vd_y = scan0_atlas_reg_images[0].m_vd_y;
	vd_z = scan0_atlas_reg_images[0].m_vd_z;

	if (!LoadMeansAndVariances(scan0_means_file, scan0_variances_file, &scan0_MeanVector, &scan0_VarianceVector)) {
		return FALSE;
	}

	// assume 4 channels for image
	if (NumberOfImageChannels != 4) {
		return FALSE;
	}
	//
	for (i = 0; i < NumberOfPriorChannels; i++) {
		for (j = 0; j < 4; j++) {
			for (k = 0; k < 4; k++) {
				vv[i][j*4+k] = scan0_VarianceVector.at(i)(j, k);
			}
			mv[i][j] = scan0_MeanVector.at(i)(j);
		}
	}

	for (n = 0; n < vd_z; n++) {
		for (m = 0; m < vd_y; m++) {
			for (l = 0; l < vd_x; l++) {
				double prior[NumberOfPriorChannels];
				double post[NumberOfPriorChannels];
				double like[NumberOfPriorChannels];
				double y[4], ym[4];
				double ys = 0;

				number_of_pixels++;

				for (i = 0; i < NumberOfPriorChannels; i++) {
					//scan0_atlas_reg_warp_prior[i].GetAt(l, m, n, &prior[i]);
					prior[i] = scan0_atlas_reg_warp_prior[i].m_pData[n][m][l][0];
				}
				//
				// making a vnl vector from fixed images
				for (i = 0; i < NumberOfImageChannels; i++) {
					y[i] = scan0_atlas_reg_images[i].m_pData[n][m][l][0];
					ys += y[i];
				}

				// if foreground...
				// consider also if tumor and ncr priors have resonable value
				if (ys > 0 || (prior[TU]+prior[NCR]) > 0.01) {
					// to the number of classes
					for (i = 0; i < NumberOfPriorChannels; i++) {
						ym[0] = y[0] - mv[i][0]; 
						ym[1] = y[1] - mv[i][1]; 
						ym[2] = y[2] - mv[i][2]; 
						ym[3] = y[3] - mv[i][3]; 
						ComputeLikelihood4(vv[i], ym, &like[i]);
					}
					// compute the denum
					double denum = epss;
					for (i = 0; i < NumberOfPriorChannels; i++) {
						denum +=  prior[i] * like[i];
					}
					// compute the posterior
					for (i = 0; i < NumberOfPriorChannels; i++) {
						post[i] = prior[i] * like[i] / denum;
						//
						scan0_atlas_reg_posterior[i].m_pData[n][m][l][0] = post[i];
					}
				// if background...
				} else {
					// compute the posterior
					for (i = 0; i < NumberOfPriorChannels; i++) {
						post[i] = 0.0;
						like[i] = 1.0;
						//
						scan0_atlas_reg_posterior[i].m_pData[n][m][l][0] = 0.0;
					}
					scan0_atlas_reg_posterior[BG].m_pData[n][m][l][0] = 1.0;
				}

				// to the number of classes
				for (i = 0; i < NumberOfPriorChannels; i++) {
					// we have to take care of negative values
					double val = like[i] * prior[i];
					if (val <= 0.0) val = epss;
					cost -= post[i] * log(val);
				}
			} // l
		} // m
	} // n

	cost /= number_of_pixels;

	*pScore = cost;
	TRACE("%d_%s: cost = %f, number_of_pixels = %f\n", tag, tag_F, cost, number_of_pixels);

	TRACE("%d_%s: ComputeQ... - done\n", tag, tag_F);
	
	return TRUE;
}
#endif
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
