%
%       Written by  Kayhan  Batmanghelich
%                   March 2012
%                   Section of Biomedical Image Analysis (SBIA)
%                   University of Pennsylvania
% This function performs all experiment steps.
%     gondola    [option Name]   [option Value]
%
%     [option Name]           [option Value]                                      obligatory ?
%  ------------------------------------------------------------------------------------------------
%     what                    Learn, Show, FeatureExt                             YES
%     expnum                  Experiment number                                   YES
%     ResFilename             full path to a .mat file                            YES
%     continueFlag            true OR false                                       if what=learn
%     ImageList               full path to the input image files                  if what=learn, FeatureExt
%     IdListFile              full path to the corresponding ID list              if what=FeatureExt
%     ArffFileName            full path to the .arff features file                if what=FeatureExt
%     saveBasisImgPath        full path to save basis image                       if what=show
%     randSeed                number for random seed for initialization           NO
%
%
%     Description
%  ---------------------------------------------------------------------
%     what            The experiment steps to perform
%						learn : learn basis vectors
%						FeatureExt : extract features
%						show : read the .mat file and save basis vectors
%
%     expnum          A name for the experiment. There must be a .config file corresponding to the specified experiment number (e.g. exp1001.config). This file specifies all experiment parameters, and should be placed in the same folder as the .mat file containing the results
%
%     ResFilename     A .mat file containing basis vectors and the rest of optimization problem variables
%
%     continueFlag    If set to "true", continue process from previous point. This can be used if the optimization process previously stopped before completion.
%
%     ImageList       A text file containing list of subjects to be processed. This list should also contain class labels. Notice that the class labels must start from 1 (i.e. 1,2,3,...), and 0 assigned to any unlabeled subjects (i.e. semi-supervised case). This file must conform to a certain format; please refer to the documentation for full details. You do not need to provide this file if you just want to save output basis images.
%                     
%
%     IdListFile      A text file containing corresponding ID's for each line in the ImageList file. Notice that the first three lines of the ImageList file contain formatting information, and hence do not correspond to any line in the IdListFile. An ID can be a number, string, or a combination of both. This file is only necessary for feature extraction (what=FeatureExt).
%
%     ArffFileName    Output file in .ARFF format (Weka format) containing extracted features. Number of features is equal to the number of basis vectors plus 2, where the last two spots are reserved for class labels and IDs (provided in IdListFile).
%
%
%
%   
%    Template for the experiment .config file:
%  ----------------------------------------------------------
%    algo:                         [algo]
%    expnum:                       [expnum]
%    DownSampleRatio:              [DownSampleRatio]
%    nrm_mode:                     [nrm_mode]
%    nrm_scale:                    [nrm_scale]
%    MAXITR:                       [MAXITR]
%    tol:                          [tol]
%    numBasisVectors:              [numBasisVectors]
%    lambda_gen:                   [lambda_gen]
%    lambda_disc:                  [lambda_disc]
%    lambda_const:                 [lambda_const]
%    lambda_stab:                  [lambda_stab]
%    numBatchBasisVector:          [numBatchBasisVector]
%    BBMethod:                     [BBMethod]
%    Monitor_Bsol:                 [Monitor_Bsol]
%    Monitor_Csol:                 [Monitor_Csol]
%    saveAfterEachIteration:       [saveAfterEachIteration]
%
%
%   **Description of the fields**
%   ****
%    algo: Specifies the algorithm. This can be one of the following:
%          MultiChannel_MultiView_BoxedSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; boxed-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, one B is shared between modalities. For more detail, see [2],[1].
%          MultiChannel_SingleView_BoxedSparsity_spg_Mosek : Can be used for uni- or multi-channel case; boxed-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, there is one B per modality. For more detail, see [2],[1].
%          MultiChannel_MultiView_GroupSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; group-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, one B is shared between modalities. Note that this algorithm only supports non-overlapping groups. For more detail, see [2],[1].
%          MultiChannel_SingleView_GroupSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; group-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, there is one B per modality. Note that this algorithm only supports non-overlapping groups. For more detail, see [2],[1].
%  MultiChannel_MultiView_freeC_BoxedSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; boxed-sparsity is used as a feasible set, but C is free to be any value (not necessarily positive), SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, there is one B per modality. For more detail, see [2],[1].
%
%    expnum: Specifies the experiment number. This should match the same number used for the file name; i.e. if the name of the file is exp1500.config, [expnum] should be replaced with 1500.
%
%    DownSampleRatio: Specify the down-sampling ratio: use 1 or 2 . When using group-sparsity, 1 is recommended. Down sampling speeds up execution by reducing the data dimension.
%
%    nrm_mode : Specify normalization method.
%         0 :  do nothing
%         1 :  for each modality, z-score normalization; i.e. subtract the mean and divide result by the standard deviation
%         2 :  for each modality, normalization between [0,1]
%         3 :  for each modality, multiplication by a constant defined by "nrm_scale". This is the recommended normalization method.
%
%    nrm_scale : If nrm_mode=3, this value is used for multiplication in normalization.
%
%    MAXITR :  maximum number of iterations per block; i.e. if MAXITR=30, the combined number of iterations for w,B, and C is set to 30 - i.e. 10 iterations per round.
%
%    tols : tolerance value for optimization of B. It is recommended to leave this as the default value (0.5).
%
%    numBasisVectors : number of basis vectors (columns of B).
%
%    lambda_gen : lambda coefficient to be multiplied with the generative term (lambda_1 in [1]).
%
%    lambda_disc : lambda coefficient to be multiplied with the discriminative term (lambda_2 in [1]).
%
%    lambda_const : lambda constraint to be used to specify sparsity (Boxed or Group) in the feasible set (lambda_3  in [1])
%
%    lambda_stab :  lambda of the stability. It is *NOT* recommended to change this. The default value is 0.01 .
%
%    numBatchBasisVector :  deprecated.
%
%    BBMethod : Specifies how to approximate the hessian using Barzilai-Borwein method. Can take values 1,2, or 3, but it is *NOT* recommended to change it from the default value 3.
%
%    Monitor_Bsol:  ADVANCED OPTION. It is recommended to leave this option as FALSE unless you are fully aware of the consequences. If the value is TRUE, the algorithm checks descend of the algorithm after optimizing B. If the cost function is not descending, B is not updated to the new value.
%
%    Monitor_Csol:  ADVANCED OPTION. It is recommended to leave this option as FALSE unless you are fully aware of the consequences. If the value is TRUE, the algorithm checks descend of the algorithm after optimizing C. If the cost function is not descending; C is not updated to the new value.
%
%    saveAfterEachIteration:  If TRUE (the recommended setting), the program saves intermediate results every 3 hours.
%
%   Here is an example configuration file, exp1200.config file:
%   ~ > cat   exp1200.config
%
%  # THIS IS A COMMENTED LINE THAT WON'T BE PARSED
%  algo:                           MultiChannel_MultiView_BoxedSparsity_spg_Mosek
%  expnum:                         1200
%  DownSampleRatio:                2
%  nrm_mode:                       3
%  nrm_scale:                       5
%  MAXITR:                         300
%  tol:                            0.5
%  numBasisVectors:                30
%  lambda_gen:                     10
%  lambda_disc:                    0.1
%  lambda_const:                   0.2
%  lambda_stab:                    0.01
%  numBatchBasisVector:            2
%  BBMethod:                       3
%  Monitor_Bsol:                   FALSE
%  Monitor_Csol:                   FALSE
%  saveAfterEachIteration:          TRUE
%
%   ~ >
%   Notice that the number value of expnum (1200) matches the .config filename.
%
%
%   Refrences:
%  =====
%    [1] K. N. Batmanghelich, B. Taskar, C. Davatzikos; Generative-Discriminative
%    Basis Learning for Medical Imagin; IEEE Trans Med Imaging. 2012 Jan;31(1):51-69. Epub 2011 Jul 25
%
%    [2] K. N. Batmanghelich, B. Taskar, D. Ye, C. Davatzikos; Regularized Tensor
%    factorization for multi-modality medical image classification, MICCAI 2011,
%    LNCS 6893, p17
%
%
%    Examples
%  -----------------------------------------------------------------------------
%    Ex.1 : Learning basis vectors
%        gondola   what learn  expnum 1001  continueFlag false  ImageList /home/batmangn/exp1001/imgFnList.lst   IdListFile  /home/batmangn/exp1001/imgIdList.lst     ResFilename /home/batmangn/exp1001/exp1001.mat
%
%     Description:  Learn the basis vectors for image filenames in imgFnList.lst and corresponding IDs in imgIdList.lst, and save both intermediate and final results in exp1001.mat.  Since continueFlag=false, the process begins from scratch. Providing IdList is not necessary for basis learning. <-- so why is it in the command line?
%
%
%    Ex.2 : Extracting basis vectors
%         gondola   what FeatureExt  expnum 1001   ImageList /home/batmangn/exp1001/imgFnList.lst   IdListFile  /home/batmangn/exp1001/imgIdList.lst     ResFilename /home/batmangn/exp1001/exp1001.mat  ArffFileName  /home/batmangn/exp1001/exp1001.arff
%
%     Description: Read the results saved in exp1001.mat, and save the extracted features in exp1001.arff.
%
%    Ex. 3: Saving basis vectors
%        gondola   what show  expnum 1001   ResFilename /home/batmangn/exp1001/exp1001.mat  saveBasisImgPath  /home/batmangn/exp1001/basisImg/
%
%   Description: Read results from the exp1001.mat file and save the basis vectors as images in the specified path
 

function  gondola(varargin)
      %[what, expnum, continiue, BasisVectorFilename, ImageList, ArffFileName, IdListFile] 
      Args = parseArguments(varargin) ;
     
      % this line initializes variables
      [Data,Options] = Initialization(Args)  ;  


      % Do one of the following: 1-Learn basis (learn),  2-Extract Features (featureextr),  3-Save basis vectors (show)
      switch(lower(Args.what))
          case 'learn',
            % learn basis vectors based on the algorithm
            switch strtrim(Options.algo),
                case 'MultiChannel_MultiView_BoxedSparsity_spg_Mosek',
                      %  Multi-Channel input:  V is tensot 
                      %  Multi-View basis : B is matrix; solver for B is Spectral Projected Gradient
                      %  C is also tensor and MOSEK is used for optimization.
                      %  Regularization is Boxed-Sparsity
                      MultiChannel_MultiView_BoxedSparsity_spg_Mosek(Data, Options) ;
                case 'MultiChannel_MultiView_GroupSparsity_spg_Mosek',
                      %  Multi-Channel input:  V is tensot 
                      %  Multi-View basis : B is matrix; solver for B is Spectral Projected Gradient 
                      %  C is also tensor and MOSEK is used for optimization.
                      %  Regularization is Group-Sparsity
                      MultiChannel_MultiView_GroupSparsity_spg_Mosek(Data, Options) ;
                case 'MultiChannel_SingleView_BoxedSparsity_spg_Mosek',
                      %  Multi-Channel input:  V is tensot 
                      %  Single-View basis : B is tensor; solver for B is Spectral Projected Gradient
                      %  C is also tensor and MOSEK is used for optimization.
                      %  Regularization is Boxed-Sparsity
                      MultiChannel_SingleView_BoxedSparsity_spg_Mosek(Data, Options) ;
                case 'MultiChannel_SingleView_GroupSparsity_spg_Mosek',
                      %  Multi-Channel input:  V is tensot 
                      %  Single-View basis : B is tensor; solver for B is Spectral Projected Gradient 
                      %  C is also tensor and MOSEK is used for optimization.
                      %  Regularization is Group-Sparsity
                      MultiChannel_SingleView_GroupSparsity_spg_Mosek(Data, Options) ;
                case 'MultiChannel_MultiView_freeC_BoxedSparsity_spg_Mosek',
                      %  Multi-Channel input:  V is tensot 
                      %  Multi-View basis : B is matrix; solver for B is Spectral Projected Gradient
                      %  C is also tensor and MOSEK is used for optimization. In this case C is free which means that it can also take nagative values.
                      %  Regularization is Boxed-Sparsity
                      MultiChannel_MultiView_freeC_BoxedSparsity_spg_Mosek(Data, Options) ;

                otherwise,
                    error('no such algorithm implemented !');
            end
          case 'featureextr',
            % extract features based on the setting of the experiment
            load(Args.ResFilename,'B','ConstVars') ;
            FtrExtroptions.ArffFileName  = Args.ArffFileName ;
            FtrExtroptions.ProjectionMode = Options.ProjectionMode ;
            FtrExtroptions.IsID_provided = true  ;    % I kept this only for historical reasons
            IDs = textread(Args.IdListFile,'%s','delimiter', '\n') ;
            FtrExtroptions.ClassName = {} ;
            for  classCnt=1:ConstVars.numClasses
                 FtrExtroptions.ClassName{classCnt} = num2str(classCnt) ;
            end
    
            FeatureExtr(IDs,Data.V,Data.y,B,FtrExtroptions) ;

          case 'show',
            % save basis vectors
            saveBasis_asImages(Options) ;

      end
               
end




% this function parses input arguments 
function Options = parseArguments(argOpt)
        numArgs = size(argOpt) ;
        % check for the required arguments
        Options = []  ;
        cnt = 1 ;
        % collect all options into a data structure
        while  (true)
            if (cnt > numArgs)
              break ;
            end
            optName = argOpt{cnt} ;
            if ~(isstr(optName))
                usage() ;
                error('option name must be string !! ') ;
            end
            if ((cnt+1) > numArgs)
                usage() ;
                error(['You probably forgot value for this option : ' optName ' ']) ;
            end
            optVal = argOpt{cnt+1} ;
            Options.(optName) = optVal ;
            cnt = cnt + 2 ;
        end
        %  Now we check 
        try 
            what = Options.what ;        
        catch exception
            disp(exception.message);
            usage() ;
            error('you may forgot to provide "what" option! Notice that it case sensitive ! ') ;
        end
        if ~ismember(lower(Options.what),{'show','learn','featureextr'})
            usage() ;
            error('what option can be either "learn", show or FeatureExt and nothing else !!!')  ;
        end

        % checking expnum option
        if ~isfield(Options,'expnum')
            usage() ;
            error('you may forgot to provide "expnum" option ! It is a mandatory option !!! ') ;
        else
            if isdeployed()
                try 
                    Options.expnum = num2str(Options.expnum) ;     
                catch exception
                    disp(exception.message);
                    usage() ;
                    error('experiment number must be a number (e.g. 1001) !!! ')  ;
                end
            else
                if ~isnumeric(Options.expnum)
                    usage() ;
                    error('experiment number must be a number (e.g. 1001) !!!  ')  ;
                end
            end 
        end
                    
        % checking ResFilename option
        if ~isfield(Options,'ResFilename') 
            usage() ;
            error(' you may forgot to provide "ResFilename" ! It is mandatory option  !!!  ')  ;
        end
        try
            [ResultFile_path,name,fileExt,~] = fileparts(Options.ResFilename) ;   % check that the folder exist and extension of the file is correct
        catch exception
            disp(exception.message);
            usage() ;
            error('make sure that the option value for the ResFilename is correct !!! \n') ;
        end
        if ~(exist(ResultFile_path,'dir') && isequal(fileExt,'.mat') )
            usage() ;
            error('make sure that ResFilename is a .mat file and the directory it is going to also exists !! ') ;
        end

        switch lower(what)
            case 'learn',
                % checking continueFlag options
                if ~isfield(Options,'continueFlag')
                    usage() ;
                    error(' you may forgot to provide "continueFlag" ! It is necessary for what=learn !!!   ')  ;
                end
                try
                    if isdeployed()
                            Options.continueFlag = str2num(lower(Options.continueFlag)) ; 
                    elseif ~islogical(Options.continueFlag)
                            error(' "continueFlag" must be either true or false  !!! ') ;    
                    end
                catch exception
                   disp(exception.message);
                   usage() ;
                   error(' "continueFlag" must be either true or false  !!! ') ;
                end
                if Options.continueFlag   
                      try 
                          load(Options.ResFilename,'iter') ;   % the result file should exist and number of iteration should be more than 1
                      catch exception
                          disp(exception.message);
                          usage() ;
                          error('if you want to continue from a previously saved results, make sure that ResFilename exists !!! ');   
                      end    
                      if exist('iter','var')
                           if (iter < 2)
                                 usage() ;
                                 error('if you want to continue from the previously save results, you must make sure that "ResFilename" exist and at least a few iteration has been passed !! ') ;
                           end
                      else
                           usage() ;
                           error('if you want to continue from the previously save results, you must make sure that "ResFilename" exist and at least a few iteration has been passed !! ') ;
                      end
                      
                end

                % checking ImageList
                if ~isfield(Options,'ImageList')
                    usage() ;
                    error(' you may forgot to provide "ImageList" ! It is necessary for what=learn !!!  ')  ;
                end
                if ~exist(Options.ImageList,'file') 
                    usage() ;
                    error('make sure that "ImageList" exists !!! ') ;
                end
 
            case 'featureext',
                % checking ImageList
                if ~isfield(Options,'ImageList')
                    usage() ;
                    error(' you may forgot to provide "ImageList" ! It is necessary for what=FeatureExt !!! !!!  ')  ;
                end
                if ~exist(Options.ImageList,'file') 
                    usage() ;
                    error('make sure that ImageList exists !!! \n') ;
                end
  
                % checking IdListFile
                if ~isfield(Options,'IdListFile')
                    usage() ;
                    error(' you may forgot to provide "IdListFile" ! It is necessary for what=FeatureExtr !!!  ')  ;
                end
                if ~exist(Options.IdListFile,'file') 
                    usage() ;
                    error('make sure that IdListFile exists !!! \n') ;
                end
  
                % checking the arff file
                if ~isfield(Options,'ArffFileName')
                    usage() ;
                    error(' you may forgot to provide "ArffFileName" ! It is necessary for what=FeatureExtr !!!  ')  ;
                end
                try
                    [pathstr,name,fileExt,~] = fileparts(Options.ArffFileName) ;   % check that the folder exist and extension of the file is correct
                catch exception
                    disp(exception.message);
                    usage() ;
                    error('make sure that the option value for the ArffFileName is correct !!! \n') ;
                end
                if (exist(pathstr,'dir') && isequal(fileExt,'.arff') )
                    usage() ;
                    error('make sure that "ArffFileName" is a .arff file and the directory it is going to also exists !! ') ;
                end

            case 'show',
                % checking the saveBasisImgPath file
                if ~isfield(Options,'saveBasisImgPath')
                    usage() ;
                    error(' you may forgot to provide "saveBasisImgPath" ! It is necessary for what=show !!! ')  ;
                end
                if ~exist(Options.saveBasisImgPath,'dir')
                      usage() ;
                      error(' make sure that the folder to save basis images exists !!!') ;
                end
        end
        % make sure that config also exist in the same folder as .mat file
        Options.configFile = [ResultFile_path '/exp' num2str(Options.expnum) '.config'] ;
        if ~exist(Options.configFile,'file')
            usage() ;
            error(['exp' num2str(Options.expnum) '.config must exist in the same folder as the result file !!!' ]) ;
        end
        %  print Option on stdout just for fun !
        Options 
end



% print the usage
function usage()
fprintf('  This function performs all experiment steps. \n');
fprintf('      gondola    [option Name]   [option Value] \n');
fprintf('  \n');
fprintf('      [option Name]           [option Value]                                      obligatory ? \n');
fprintf('   ------------------------------------------------------------------------------------------------ \n');
fprintf('      what                    Learn, Show, FeatureExt                             YES \n');
fprintf('      expnum                  Experiment number                                   YES \n');
fprintf('      ResFilename             full path to a .mat file                            YES \n');
fprintf('      continueFlag            true OR false                                       if what=learn \n');
fprintf('      ImageList               full path to the input image files                  if what=learn, FeatureExt \n');
fprintf('      IdListFile              full path to the corresponding ID list              if what=FeatureExt \n');
fprintf('      ArffFileName            full path to the .arff features file                if what=FeatureExt \n');
fprintf('      saveBasisImgPath        full path to save basis image                       if what=show \n');
fprintf('      randSeed                number for random seed for initialization           NO \n');
fprintf('  \n');
fprintf('  \n');
fprintf('      Description \n');
fprintf('   --------------------------------------------------------------------- \n');
fprintf('      what            The experiment steps to perform \n');
fprintf(' 						learn : learn basis vectors \n');
fprintf(' 						FeatureExt : extract features \n');
fprintf(' 						show : read the .mat file and save basis vectors \n');
fprintf('  \n');
fprintf('      expnum          A name for the experiment. There must be a .config file corresponding to the specified experiment number (e.g. exp1001.config). This file specifies all experiment parameters, and should be placed in the same folder as the .mat file containing the results \n');
fprintf('  \n');
fprintf('      ResFilename     A .mat file containing basis vectors and the rest of optimization problem variables \n');
fprintf('  \n');
fprintf('      continueFlag    If set to "true", continue process from previous point. This can be used if the optimization process previously stopped before completion. \n');
fprintf('  \n');
fprintf('      ImageList       A text file containing list of subjects to be processed. This list should also contain class labels. Notice that the class labels must start from 1 (i.e. 1,2,3,...), and 0 assigned to any unlabeled subjects (i.e. semi-supervised case). This file must conform to a certain format; please refer to the documentation for full details. You do not need to provide this file if you just want to save output basis images. \n');
fprintf('                       \n');
fprintf('  \n');
fprintf('      IdListFile      A text file containing corresponding ID''s for each line in the ImageList file. Notice that the first three lines of the ImageList file contain formatting information, and hence do not correspond to any line in the IdListFile. An ID can be a number, string, or a combination of both. This file is only necessary for feature extraction (what=FeatureExt). \n');
fprintf('  \n');
fprintf('      ArffFileName    Output file in .ARFF format (Weka format) containing extracted features. Number of features is equal to the number of basis vectors plus 2, where the last two spots are reserved for class labels and IDs (provided in IdListFile). \n');
fprintf('  \n');
fprintf('  \n');
fprintf('  \n');
fprintf('     \n');
fprintf('     Template for the experiment .config file: \n');
fprintf('   ---------------------------------------------------------- \n');
fprintf('     algo:                         [algo] \n');
fprintf('     expnum:                       [expnum] \n');
fprintf('     DownSampleRatio:              [DownSampleRatio] \n');
fprintf('     nrm_mode:                     [nrm_mode] \n');
fprintf('     nrm_scale:                    [nrm_scale] \n');
fprintf('     MAXITR:                       [MAXITR] \n');
fprintf('     tol:                          [tol] \n');
fprintf('     numBasisVectors:              [numBasisVectors] \n');
fprintf('     lambda_gen:                   [lambda_gen] \n');
fprintf('     lambda_disc:                  [lambda_disc] \n');
fprintf('     lambda_const:                 [lambda_const] \n');
fprintf('     lambda_stab:                  [lambda_stab] \n');
fprintf('     numBatchBasisVector:          [numBatchBasisVector] \n');
fprintf('     BBMethod:                     [BBMethod] \n');
fprintf('     Monitor_Bsol:                 [Monitor_Bsol] \n');
fprintf('     Monitor_Csol:                 [Monitor_Csol] \n');
fprintf('     saveAfterEachIteration:       [saveAfterEachIteration] \n');
fprintf('  \n');
fprintf('  \n');
fprintf('    **Description of the fields** \n');
fprintf('    **** \n');
fprintf('     algo: Specifies the algorithm. This can be one of the following: \n');
fprintf('           MultiChannel_MultiView_BoxedSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; boxed-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, one B is shared between modalities. For more detail, see [2],[1]. \n');
fprintf('           MultiChannel_SingleView_BoxedSparsity_spg_Mosek : Can be used for uni- or multi-channel case; boxed-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, there is one B per modality. For more detail, see [2],[1]. \n');
fprintf('           MultiChannel_MultiView_GroupSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; group-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, one B is shared between modalities. Note that this algorithm only supports non-overlapping groups. For more detail, see [2],[1]. \n');
fprintf('           MultiChannel_SingleView_GroupSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; group-sparsity is used as a feasible set. SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, there is one B per modality. Note that this algorithm only supports non-overlapping groups. For more detail, see [2],[1]. \n');
fprintf('   MultiChannel_MultiView_freeC_BoxedSparsity_spg_Mosek  : Can be used for uni- or multi-channel case; boxed-sparsity is used as a feasible set, but C is free to be any value (not necessarily positive), SPG is used to optimize B (basis vectors) and MOSEK is used to optimize C (coefficients). In the multi-channel case, there is one B per modality. For more detail, see [2],[1]. \n');
fprintf('  \n');
fprintf('     expnum: Specifies the experiment number. This should match the same number used for the file name; i.e. if the name of the file is exp1500.config, [expnum] should be replaced with 1500. \n');
fprintf('  \n');
fprintf('     DownSampleRatio: Specify the down-sampling ratio: use 1 or 2 . When using group-sparsity, 1 is recommended. Down sampling speeds up execution by reducing the data dimension. \n');
fprintf('  \n');
fprintf('     nrm_mode : Specify normalization method. \n');
fprintf('          0 :  do nothing \n');
fprintf('          1 :  for each modality, z-score normalization; i.e. subtract the mean and divide result by the standard deviation \n');
fprintf('          2 :  for each modality, normalization between [0,1] \n');
fprintf('          3 :  for each modality, multiplication by a constant defined by "nrm_scale". This is the recommended normalization method. \n');
fprintf('  \n');
fprintf('     nrm_scale : If nrm_mode=3, this value is used for multiplication in normalization. \n');
fprintf('  \n');
fprintf('     MAXITR :  maximum number of iterations per block; i.e. if MAXITR=30, the combined number of iterations for w,B, and C is set to 30 - i.e. 10 iterations per round. \n');
fprintf('  \n');
fprintf('     tols : tolerance value for optimization of B. It is recommended to leave this as the default value (0.5). \n');
fprintf('  \n');
fprintf('     numBasisVectors : number of basis vectors (columns of B). \n');
fprintf('  \n');
fprintf('     lambda_gen : lambda coefficient to be multiplied with the generative term (lambda_1 in [1]). \n');
fprintf('  \n');
fprintf('     lambda_disc : lambda coefficient to be multiplied with the discriminative term (lambda_2 in [1]). \n');
fprintf('  \n');
fprintf('     lambda_const : lambda constraint to be used to specify sparsity (Boxed or Group) in the feasible set (lambda_3  in [1]) \n');
fprintf('  \n');
fprintf('     lambda_stab :  lambda of the stability. It is *NOT* recommended to change this. The default value is 0.01 . \n');
fprintf('  \n');
fprintf('     numBatchBasisVector :  deprecated. \n');
fprintf('  \n');
fprintf('     BBMethod : Specifies how to approximate the hessian using Barzilai-Borwein method. Can take values 1,2, or 3, but it is *NOT* recommended to change it from the default value 3. \n');
fprintf('  \n');
fprintf('     Monitor_Bsol:  ADVANCED OPTION. It is recommended to leave this option as FALSE unless you are fully aware of the consequences. If the value is TRUE, the algorithm checks descend of the algorithm after optimizing B. If the cost function is not descending, B is not updated to the new value. \n');
fprintf('  \n');
fprintf('     Monitor_Csol:  ADVANCED OPTION. It is recommended to leave this option as FALSE unless you are fully aware of the consequences. If the value is TRUE, the algorithm checks descend of the algorithm after optimizing C. If the cost function is not descending; C is not updated to the new value. \n');
fprintf('  \n');
fprintf('     saveAfterEachIteration:  If TRUE (the recommended setting), the program saves intermediate results every 3 hours. \n');
fprintf('  \n');
fprintf('    Here is an example configuration file, exp1200.config file: \n');
fprintf('    ~ > cat   exp1200.config \n');
fprintf('  \n');
fprintf('   # THIS IS A COMMENTED LINE THAT WON''T BE PARSED \n');
fprintf('   algo:                           MultiChannel_MultiView_BoxedSparsity_spg_Mosek \n');
fprintf('   expnum:                         1200 \n');
fprintf('   DownSampleRatio:                2 \n');
fprintf('   nrm_mode:                       3 \n');
fprintf('   nrm_scale:                       5 \n');
fprintf('   MAXITR:                         300 \n');
fprintf('   tol:                            0.5 \n');
fprintf('   numBasisVectors:                30 \n');
fprintf('   lambda_gen:                     10 \n');
fprintf('   lambda_disc:                    0.1 \n');
fprintf('   lambda_const:                   0.2 \n');
fprintf('   lambda_stab:                    0.01 \n');
fprintf('   numBatchBasisVector:            2 \n');
fprintf('   BBMethod:                       3 \n');
fprintf('   Monitor_Bsol:                   FALSE \n');
fprintf('   Monitor_Csol:                   FALSE \n');
fprintf('   saveAfterEachIteration:          TRUE \n');
fprintf('  \n');
fprintf('    ~ > \n');
fprintf('    Notice that the number value of expnum (1200) matches the .config filename. \n');
fprintf('  \n');
fprintf('  \n');
fprintf('    Refrences: \n');
fprintf('   ===== \n');
fprintf('     [1] K. N. Batmanghelich, B. Taskar, C. Davatzikos; Generative-Discriminative \n');
fprintf('     Basis Learning for Medical Imagin; IEEE Trans Med Imaging. 2012 Jan;31(1):51-69. Epub 2011 Jul 25 \n');
fprintf('  \n');
fprintf('     [2] K. N. Batmanghelich, B. Taskar, D. Ye, C. Davatzikos; Regularized Tensor \n');
fprintf('     factorization for multi-modality medical image classification, MICCAI 2011, \n');
fprintf('     LNCS 6893, p17 \n');
fprintf('  \n');
fprintf('  \n');
fprintf('     Examples \n');
fprintf('   ----------------------------------------------------------------------------- \n');
fprintf('     Ex.1 : Learning basis vectors \n');
fprintf('         gondola   what learn  expnum 1001  continueFlag false  ImageList /home/batmangn/exp1001/imgFnList.lst   IdListFile  /home/batmangn/exp1001/imgIdList.lst     ResFilename /home/batmangn/exp1001/exp1001.mat \n');
fprintf('  \n');
fprintf('      Description:  Learn the basis vectors for image filenames in imgFnList.lst and corresponding IDs in imgIdList.lst, and save both intermediate and final results in exp1001.mat.  Since continueFlag=false, the process begins from scratch. Providing IdList is not necessary for basis learning. <-- so why is it in the command line? \n');
fprintf('  \n');
fprintf('  \n');
fprintf('     Ex.2 : Extracting basis vectors \n');
fprintf('          gondola   what FeatureExt  expnum 1001   ImageList /home/batmangn/exp1001/imgFnList.lst   IdListFile  /home/batmangn/exp1001/imgIdList.lst     ResFilename /home/batmangn/exp1001/exp1001.mat  ArffFileName  /home/batmangn/exp1001/exp1001.arff \n');
fprintf('  \n');
fprintf('      Description: Read the results saved in exp1001.mat, and save the extracted features in exp1001.arff. \n');
fprintf('  \n');
fprintf('     Ex. 3: Saving basis vectors \n');
fprintf('         gondola   what show  expnum 1001   ResFilename /home/batmangn/exp1001/exp1001.mat  saveBasisImgPath  /home/batmangn/exp1001/basisImg/ \n');
fprintf('  \n');
fprintf('    Description: Read results from the exp1001.mat file and save the basis vectors as images in the specified path \n');
 
end



