/* Copyright (c) 2001-2005, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dicom;

import java.util.Iterator;

/**
 * <p>A class with methods for constructing a {@link com.pixelmed.dicom.ContentItem ContentItem} of the appropriate class from a list of attributes.</p>
 *
 * <p>The sub-classes of {@link com.pixelmed.dicom.ContentItem ContentItem} are public internal classes of this class,
 * but specialize the methods, specifically the extractors and the string representation methods.</p>
 *
 * <p>This is not an abstract class, an the content item factory method is not static; an instance of
 * the factory needs to be created.</p>
 *
 * @see com.pixelmed.dicom.ContentItem
 * @see com.pixelmed.dicom.StructuredReport
 * @see com.pixelmed.dicom.StructuredReportBrowser
 *
 * @author	dclunie
 */
public class ContentItemFactory {

	/***/
	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/dicom/ContentItemFactory.java,v 1.9 2008/02/18 18:14:40 dclunie Exp $";

	/***/
	public class UnrecognizedContentItem extends ContentItem {

		/**
		 * @param	parent
		 */
		public UnrecognizedContentItem(ContentItem parent) {
			super(parent,null);
		}

		/**
		 * @param	parent
		 * @param	list
		 */
		public UnrecognizedContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
		}

		/**
		 * @param	parent
		 * @param	list
		 * @param	name
		 */
		public UnrecognizedContentItem(ContentItem parent,AttributeList list,String name) {
			super(parent,list);
		}

		/***/
		public String getConceptValue()      { return ""; }
	}

	/***/
	public class ContainerContentItem extends ContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public ContainerContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
		}

		/***/
		public String getConceptValue()      { return ""; }
	}

	/***/
	public class CompositeContentItem extends ContentItem {

		/***/
		protected AttributeList referencedSOPSequenceItemAttributeList;		// subclasses will use this to extract macro-specific attributes
		/***/
		protected String referencedSOPClassUID;
		/***/
		protected String referencedSOPInstanceUID;

		/**
		 * @param	parent
		 * @param	list
		 */
		public CompositeContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			referencedSOPSequenceItemAttributeList = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(list,TagFromName.ReferencedSOPSequence);
			if (referencedSOPSequenceItemAttributeList != null) {
				referencedSOPClassUID=Attribute.getSingleStringValueOrEmptyString(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedSOPClassUID);
				referencedSOPInstanceUID=Attribute.getSingleStringValueOrEmptyString(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedSOPInstanceUID);
			}
			if (referencedSOPClassUID == null) referencedSOPClassUID="";		// just for consistency with other string content items
			if (referencedSOPInstanceUID == null) referencedSOPInstanceUID="";
		}

		/***/
		public String getConceptValue()      { return ""; }

		/***/
		public String toString() {
			return super.toString()+" = "+referencedSOPClassUID+" : "+referencedSOPInstanceUID;
		}

		/***/
		public String getReferencedSOPClassUID()    { return referencedSOPClassUID; }
		/***/
		public String getReferencedSOPInstanceUID() { return referencedSOPInstanceUID; }
	}

	/***/
	public class ImageContentItem extends CompositeContentItem {

		/***/
		protected int referencedFrameNumber;
		/***/
		protected int referencedSegmentNumber;
		/***/
		protected String presentationStateSOPClassUID;
		/***/
		protected String presentationStateSOPInstanceUID;
		/***/
		protected String realWorldValueMappingSOPClassUID;
		/***/
		protected String realWorldValueMappingSOPInstanceUID;

		/**
		 * @param	parent
		 * @param	list
		 */
		public ImageContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			if (referencedSOPSequenceItemAttributeList != null) {
				referencedFrameNumber=Attribute.getSingleIntegerValueOrDefault(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedFrameNumber,0);
				referencedSegmentNumber=Attribute.getSingleIntegerValueOrDefault(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedSegmentNumber,0);

				{
					AttributeList psl = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedSOPSequence);
					if (psl != null) {
						presentationStateSOPClassUID=Attribute.getSingleStringValueOrEmptyString(psl,TagFromName.ReferencedSOPClassUID);
						presentationStateSOPInstanceUID=Attribute.getSingleStringValueOrEmptyString(psl,TagFromName.ReferencedSOPInstanceUID);
					}
					if (presentationStateSOPClassUID == null) presentationStateSOPClassUID="";
					if (presentationStateSOPInstanceUID == null) presentationStateSOPInstanceUID="";
				}
				{
					AttributeList rwvl = SequenceAttribute.getAttributeListFromWithinSequenceWithSingleItem(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedRealWorldValueMappingInstanceSequence);
					if (rwvl != null) {
						realWorldValueMappingSOPClassUID=Attribute.getSingleStringValueOrEmptyString(rwvl,TagFromName.ReferencedSOPClassUID);
						realWorldValueMappingSOPInstanceUID=Attribute.getSingleStringValueOrEmptyString(rwvl,TagFromName.ReferencedSOPInstanceUID);
					}
					if (realWorldValueMappingSOPClassUID == null) realWorldValueMappingSOPClassUID="";
					if (realWorldValueMappingSOPInstanceUID == null) realWorldValueMappingSOPInstanceUID="";
				}
				// forget about Icon Image Sequence for now :(
			}
		}

		/***/
		public String toString() {
			return super.toString()
				+ (referencedFrameNumber == 0 ? "" : ("[Frame "+Integer.toString(referencedFrameNumber)+"]"))
				+ (referencedSegmentNumber == 0 ? "" : ("[Segment "+Integer.toString(referencedSegmentNumber)+"]"))
				+ (presentationStateSOPInstanceUID == null || presentationStateSOPInstanceUID.length() == 0 ? "" : (" (PS "+presentationStateSOPClassUID+" : "+presentationStateSOPInstanceUID+")"))
				+ (realWorldValueMappingSOPInstanceUID == null || realWorldValueMappingSOPInstanceUID.length() == 0 ? "" : (" (RWV "+realWorldValueMappingSOPClassUID+" : "+realWorldValueMappingSOPInstanceUID+")"))
				;
		}

		/***/
		public int getReferencedFrameNumber()    { return referencedFrameNumber; }
		/***/
		public int getReferencedSegmentNumber()    { return referencedSegmentNumber; }
		/***/
		public String getPresentationStateSOPClassUID()    { return presentationStateSOPClassUID; }
		/***/
		public String getPresentationStateSOPInstanceUID() { return presentationStateSOPInstanceUID; }
		/***/
		public String getRealWorldValueMappingSOPClassUID()    { return realWorldValueMappingSOPClassUID; }
		/***/
		public String getRealWorldValueMappingSOPInstanceUID() { return realWorldValueMappingSOPInstanceUID; }
	}

	/***/
	public class WaveformContentItem extends CompositeContentItem {

		/***/
		protected int[] referencedWaveformChannels;

		/**
		 * @param	parent
		 * @param	list
		 */
		public WaveformContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			if (referencedSOPSequenceItemAttributeList != null) {
				referencedWaveformChannels=Attribute.getIntegerValues(referencedSOPSequenceItemAttributeList,TagFromName.ReferencedWaveformChannels);
			}
		}

		/***/
		public String toString() {
			StringBuffer str = new StringBuffer();
			str.append(super.toString());
			str.append(" = [");
			if (referencedWaveformChannels != null) {
				for (int j=0; j<referencedWaveformChannels.length; ++j) {
					if (j > 0) str.append(",");
					str.append(referencedWaveformChannels[j]);
				}
			}
			str.append("]");
			return str.toString();
		}

		/***/
		public int[] getReferencedWaveformChannels()    { return referencedWaveformChannels; }
	}

	/***/
	public class SpatialCoordinatesContentItem extends ContentItem {

		/***/
		protected String graphicType;
		/***/
		protected float[] graphicData;

		/**
		 * @param	parent
		 * @param	list
		 */
		public SpatialCoordinatesContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			graphicType=Attribute.getSingleStringValueOrDefault(list,TagFromName.GraphicType,"");
			try {
				Attribute a = list.get(TagFromName.GraphicData);
				if (a != null) {
					graphicData = a.getFloatValues();
				}
			}
			catch (DicomException e) {
			}
		}

		/***/
		public String getConceptValue()      { return ""; }

		/***/
		public String toString() {
			StringBuffer str = new StringBuffer();
			str.append(super.toString());
			str.append(" = ");
			str.append(graphicType);
			str.append(" (");
			if (graphicData != null) {
				for (int j=0; j<graphicData.length; ++j) {
					if (j > 0) str.append(",");
					str.append(graphicData[j]);
				}
			}
			str.append(")");
			return str.toString();
		}

		/***/
		public String getGraphicType()              { return graphicType; }
		/***/
		public float[] getGraphicData()             { return graphicData; }
	}


	/***/
	public class TemporalCoordinatesContentItem extends ContentItem {

		/***/
		protected String temporalRangeType;
		/***/
		protected int[] referencedSamplePositions;
		/***/
		protected float[] referencedTimeOffsets;
		/***/
		protected String[] referencedDateTimes;

		/**
		 * @param	parent
		 * @param	list
		 */
		public TemporalCoordinatesContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			temporalRangeType=Attribute.getSingleStringValueOrDefault(list,TagFromName.TemporalRangeType,"");
			try {
				{
					Attribute a = list.get(TagFromName.ReferencedSamplePositions);
					if (a != null) {
						referencedSamplePositions = a.getIntegerValues();
					}
				}
				{
					Attribute a = list.get(TagFromName.ReferencedTimeOffsets);
					if (a != null) {
						referencedTimeOffsets = a.getFloatValues();
					}
				}
				{
					Attribute a = list.get(TagFromName.ReferencedDateTime);
					if (a != null) {
						referencedDateTimes = a.getStringValues();
					}
				}
			}
			catch (DicomException e) {
			}
		}

		/***/
		public String getConceptValue()      { return ""; }

		/***/
		public String toString() {
			StringBuffer str = new StringBuffer();
			str.append(super.toString());
			str.append(" = ");
			str.append(temporalRangeType);
			if (referencedSamplePositions != null) {
				str.append(" Sample Positions (");
				for (int j=0; j<referencedSamplePositions.length; ++j) {
					if (j > 0) str.append(",");
					str.append(referencedSamplePositions[j]);
				}
				str.append(")");
			}
			if (referencedTimeOffsets != null) {
				str.append(" Time Offsets (");
				for (int j=0; j<referencedTimeOffsets.length; ++j) {
					if (j > 0) str.append(",");
					str.append(referencedTimeOffsets[j]);
				}
				str.append(")");
			}
			if (referencedDateTimes != null) {
				str.append(" DateTimes (");
				for (int j=0; j<referencedDateTimes.length; ++j) {
					if (j > 0) str.append(",");
					str.append(referencedDateTimes[j]);
				}
				str.append(")");
			}
			return str.toString();
		}

		/***/
		public String getTemporalRangeType()		{ return temporalRangeType; }
		/***/
		public int[] getReferencedSamplePositions()	{ return referencedSamplePositions; }
		/***/
		public float[] getReferencedTimeOffsets()	{ return referencedTimeOffsets; }
		/***/
		public String[] getReferencedDateTimes()	{ return referencedDateTimes; }
	}

	/***/
	public class NumericContentItem extends ContentItem {

		/***/
		protected String numericValue;
		/***/
		protected CodedSequenceItem units;

		/**
		 * @param	parent
		 * @param	list
		 */
		public NumericContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			SequenceAttribute a=(SequenceAttribute)(list.get(TagFromName.MeasuredValueSequence));
			if (a != null) {
//System.err.println("NumericContentItem: MeasuredValueSequence="+a);
				Iterator i = a.iterator();
				if (i.hasNext()) {
					SequenceItem item = ((SequenceItem)i.next());
					if (item != null) {
//System.err.println("NumericContentItem: item="+item);
						AttributeList l = item.getAttributeList();
						numericValue=Attribute.getSingleStringValueOrEmptyString(l,TagFromName.NumericValue);
						units=CodedSequenceItem.getSingleCodedSequenceItemOrNull(l,TagFromName.MeasurementUnitsCodeSequence);
					}
				}
			}
			if (numericValue == null) numericValue="";	// just for consistency with other string content items
		}

		
		/***/
		public CodedSequenceItem getUnits()		{ return units; }

		/***/
		public String getNumericValue()			{ return numericValue; }

		/***/
		public String getConceptValue() {
			return numericValue+" "+(units == null ? "" : units.getCodeMeaning());
		}

		/***/
		public String getConceptNameAndValue() {
			return getConceptNameCodeMeaning()+" = "+numericValue+" "+(units == null ? "" : units.getCodeMeaning());
		}

		/***/
		public String toString() {
			return super.toString()+" = "+numericValue+" "+(units == null ? "" : units.getCodeMeaning());
		}
	}

	/***/
	public class CodeContentItem extends ContentItem {

		/***/
		protected CodedSequenceItem conceptCode;

		/**
		 * @param	parent
		 * @param	list
		 */
		public CodeContentItem(ContentItem parent,AttributeList list) {
			super(parent,list);
			conceptCode=CodedSequenceItem.getSingleCodedSequenceItemOrNull(list,TagFromName.ConceptCodeSequence);
		}

		/***/
		public String getConceptValue() {
			return (conceptCode == null ? "" : conceptCode.getCodeMeaning());
		}

		/***/
		public String toString() {
			return super.toString()+" = "+(conceptCode == null ? "" : conceptCode.getCodeMeaning());
		}
		
		/***/
		public CodedSequenceItem getConceptCode()    { return conceptCode; }
	}

	/***/
	abstract protected class StringContentItem extends ContentItem {

		/***/
		protected String stringValue;

		/**
		 * @param	parent
		 * @param	list
		 * @param	tag
		 */
		public StringContentItem(ContentItem parent,AttributeList list,AttributeTag tag) {
			super(parent,list);
			stringValue=Attribute.getSingleStringValueOrDefault(list,tag,"");
		}


		/***/
		public String getConceptValue() {
			return stringValue;
		}

		/***/
		public String toString() {
			return super.toString()+" = "+stringValue;
		}
	}

	/***/
	public class DateTimeContentItem extends StringContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public DateTimeContentItem(ContentItem parent,AttributeList list) {
			super(parent,list,TagFromName.DateTime);
		}
	}

	/***/
	public class DateContentItem extends StringContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public DateContentItem(ContentItem parent,AttributeList list) {
			super(parent,list,TagFromName.Date);
		}
	}

	/***/
	public class TimeContentItem extends StringContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public TimeContentItem(ContentItem parent,AttributeList list) {
			super(parent,list,TagFromName.Time);
		}
	}

	/***/
	public class PersonNameContentItem extends StringContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public PersonNameContentItem(ContentItem parent,AttributeList list) {
			super(parent,list,TagFromName.PersonName);
		}
	}

	/***/
	public class UIDContentItem extends StringContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public UIDContentItem(ContentItem parent,AttributeList list) {
			super(parent,list,TagFromName.UID);
		}
	}

	/***/
	public class TextContentItem extends StringContentItem {

		/**
		 * @param	parent
		 * @param	list
		 */
		public TextContentItem(ContentItem parent,AttributeList list) {
			super(parent,list,TagFromName.TextValue);
		}
	}

	/**
	 * <p>Construct a content item of the appropriate class from a list of attributes.</p>
	 *
	 * @param	parent	the parent to add the content item to
	 * @param	list	a list of attributes that constitute the content item as it is encoded in a DICOM data set
	 * @return		a content item
	 */
	public ContentItem getNewContentItem(ContentItem parent,AttributeList list) {
		ContentItem contentItem = null;

		if (list == null) {
			contentItem = new UnrecognizedContentItem(parent);
		}
		else {
			String valueType=Attribute.getSingleStringValueOrNull(list,TagFromName.ValueType);
			if (valueType == null) {
				contentItem = new UnrecognizedContentItem(parent,list);
			}
			else if (valueType.equals("CONTAINER")) {
				contentItem = new ContainerContentItem(parent,list);
			}
			else if (valueType.equals("CODE")) {
				contentItem = new CodeContentItem(parent,list);
			}
			else if (valueType.equals("NUM")) {
				contentItem = new NumericContentItem(parent,list);
			}
			else if (valueType.equals("DATETIME")) {
				contentItem = new DateTimeContentItem(parent,list);
			}
			else if (valueType.equals("DATE")) {
				contentItem = new DateContentItem(parent,list);
			}
			else if (valueType.equals("TIME")) {
				contentItem = new TimeContentItem(parent,list);
			}
			else if (valueType.equals("PNAME")) {
				contentItem = new PersonNameContentItem(parent,list);
			}
			else if (valueType.equals("UIDREF")) {
				contentItem = new UIDContentItem(parent,list);
			}
			else if (valueType.equals("TEXT")) {
				contentItem = new TextContentItem(parent,list);
			}
			else if (valueType.equals("SCOORD")) {
				contentItem = new SpatialCoordinatesContentItem(parent,list);
			}
			else if (valueType.equals("TCOORD")) {
				contentItem = new TemporalCoordinatesContentItem(parent,list);
			}
			else if (valueType.equals("COMPOSITE")) {
				contentItem = new CompositeContentItem(parent,list);
			}
			else if (valueType.equals("IMAGE")) {
				contentItem = new ImageContentItem(parent,list);
			}
			else if (valueType.equals("WAVEFORM")) {
				contentItem = new WaveformContentItem(parent,list);
			}
			else {
				contentItem = new UnrecognizedContentItem(parent,list,valueType);
			}
		}

		return contentItem;
	}
}


