/* Copyright (c) 2001-2005, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dicom;

import java.io.BufferedInputStream;
import java.io.FileInputStream;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * <p>A class to describe a set of DICOM files and their features such as SOP Class, Instance and Transfer Syntax UIDs.</p>
 *
 * @author	dclunie
 */
public class SetOfDicomFiles extends HashSet {

	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/dicom/SetOfDicomFiles.java,v 1.3 2006/10/15 13:08:46 dclunie Exp $";
	
	private HashSet setOfSOPClassUIDs = new HashSet();
	
	public Set getSetOfSOPClassUIDs() {return setOfSOPClassUIDs; }
	
	public class DicomFile implements Comparable {
		private String fileName;
		private String sopClassUID;
		private String sopInstanceUID;
		private String transferSyntaxUID;
		private AttributeList list;
		
		public String getFileName() {return fileName; }
		public String getSOPClassUID() {return sopClassUID; }
		public String getSOPInstanceUID() {return sopInstanceUID; }
		public String getTransferSyntaxUID() {return transferSyntaxUID; }
		public AttributeList getAttributeList() {return list; }
		
		public int compareTo(Object o) {
			return fileName == null ? (((DicomFile)o).fileName == null ? 0 : -1 ) : fileName.compareTo(((DicomFile)o).fileName);
		}

		public int hashCode() {
			return fileName == null ? 0 : fileName.hashCode();
		}
		
		public DicomFile(String fileName,String sopClassUID,String sopInstanceUID,String transferSyntaxUID) {
			this.fileName=fileName;
			this.sopClassUID=sopClassUID;
			this.sopInstanceUID=sopInstanceUID;
			this.transferSyntaxUID=transferSyntaxUID;
			this.list=null;
		}

		public DicomFile(String fileName,AttributeList list) {
			this(fileName,list,false);
		}

		public DicomFile(String fileName,AttributeList list,boolean keepList) {
			this.fileName=fileName;
			sopClassUID=Attribute.getSingleStringValueOrNull(list,TagFromName.ReferencedSOPClassUIDInFile);
			if (sopClassUID == null) {
				sopClassUID=Attribute.getSingleStringValueOrNull(list,TagFromName.SOPClassUID);
			}
			sopInstanceUID=Attribute.getSingleStringValueOrNull(list,TagFromName.ReferencedSOPInstanceUIDInFile);
			if (sopInstanceUID == null) {
				sopInstanceUID=Attribute.getSingleStringValueOrNull(list,TagFromName.SOPInstanceUID);
			}
			transferSyntaxUID=Attribute.getSingleStringValueOrNull(list,TagFromName.TransferSyntaxUID);
			if (keepList) {
				this.list=list;
			}
			else {
				this.list=null;
			}
		}
		
		/**
		 * <p>Store a description a DICOM file by reading its metaheader, +/- entire attribute list, as necessary.</p>
		 *
		 * @param	fileName	a DICOM file
		 */
		public DicomFile(String fileName) {
			this(fileName,false);
		}

		/**
		 * <p>Store a description a DICOM file by reading its metaheader, +/- entire attribute list, as necessary.</p>
		 *
		 * @param	fileName	a DICOM file
		 * @param	keepList	whether or not to keep the entire attribute list (including pixel data) memory resident
		 */
		public DicomFile(String fileName,boolean keepList) {
			this.fileName=fileName;
			try {
				DicomInputStream i = new DicomInputStream(new BufferedInputStream(new FileInputStream(fileName)));
				list = new AttributeList();
				boolean fullListRead = false;
				if (keepList) {
					list.read(i);
					fullListRead=true;
				}
				else {
					list.readOnlyMetaInformationHeader(i);
				}
				sopClassUID=Attribute.getSingleStringValueOrNull(list,TagFromName.ReferencedSOPClassUIDInFile);
				sopInstanceUID=Attribute.getSingleStringValueOrNull(list,TagFromName.ReferencedSOPInstanceUIDInFile);
				transferSyntaxUID=Attribute.getSingleStringValueOrNull(list,TagFromName.TransferSyntaxUID);
				if (sopClassUID == null || sopInstanceUID == null) {
					if (!fullListRead) {
						list.read(i,TagFromName.PixelData);
					}
					sopClassUID=Attribute.getSingleStringValueOrNull(list,TagFromName.SOPClassUID);
					sopInstanceUID=Attribute.getSingleStringValueOrNull(list,TagFromName.SOPInstanceUID);
				}
				if (!keepList) {
					list=null;
				}
				i.close();
			}
			catch (Exception e) {
				e.printStackTrace(System.err);
			}
		}

		/**
		 * <p>Return a String representing this object's value.</p>
		 *
		 * @return	a string representation of the value of this object
		 */
		public String toString() {
			StringBuffer strbuf = new StringBuffer();
			strbuf.append("file=");
			strbuf.append(fileName);
			strbuf.append(", sopClassUID=");
			strbuf.append(sopClassUID);
			strbuf.append(", sopInstanceUID=");
			strbuf.append(sopInstanceUID);
			strbuf.append(", transferSyntaxUID=");
			strbuf.append(transferSyntaxUID);
			return strbuf.toString();
		}
	}
	
	/**
	 * <p>Return a String representing this object's value.</p>
	 *
	 * @return	a string representation of the value of this object
	 */
	public String toString() {
		StringBuffer strbuf = new StringBuffer();
		int j = 0;
		Iterator i =iterator();
		while (i.hasNext()) {
			strbuf.append("DicomFile [");
			strbuf.append(Integer.toString(j));
			strbuf.append("]:\n");
			strbuf.append(((SetOfDicomFiles.DicomFile)i.next()).toString());
			strbuf.append("\n");
			++j;
		}
		return strbuf.toString();
	}

	/**
	 * <p>Add a DICOM file by reading its metaheader, +/- entire attribute list, as necessary.</p>
	 *
	 * <p>Keeps only the minimal descriptive attributes, and not the entire attribute list (including pixel data) memory resident.</p>
	 *
	 * @param	fileName	a DICOM file
	 */
	public void add(String fileName) {
		add(fileName,false);
	}

	/**
	 * <p>Add a DICOM file by reading its metaheader, +/- entire attribute list, as necessary.</p>
	 *
	 * @param	fileName	a DICOM file
	 * @param	keepList	whether or not to keep the entire attribute list memory resident
	 */
	public void add(String fileName,boolean keepList) {
		DicomFile dicomFile = new DicomFile(fileName,keepList);
		add(dicomFile);
		setOfSOPClassUIDs.add(dicomFile.sopClassUID);
	}

	/**
	 * <p>Add a DICOM file with the specified attributes.</p>
	 *
	 * @param	fileName	a DICOM file
	 */
	public void add(String fileName,String sopClassUID,String sopInstanceUID,String transferSyntaxUID) {
		setOfSOPClassUIDs.add(sopClassUID);
		add(new DicomFile(fileName,sopClassUID,sopInstanceUID,transferSyntaxUID));
	}

	/**
	 * <p>Get the attribute lists for all files, if they were kept during creation.</p>
	 *
	 * @return		an array of attribute lists, each of which will be null unless keeplists was true when created
	 */
	public AttributeList[] getAttributeLists() {
		AttributeList lists[] = new AttributeList[size()];
		int j=0;
		Iterator i =iterator();
		while (i.hasNext()) {
			lists[j] = ((SetOfDicomFiles.DicomFile)i.next()).getAttributeList();
			++j;
		}
		return lists;
	}

	/**
	 * <p>For testing, read all DICOM files and partition them.</p>
	 *
	 * @param	arg	the filenames containing the images to partition
	 */
	public static void main(String arg[]) {
		SetOfDicomFiles setOfDicomFiles = new SetOfDicomFiles();
		for (int a=0; a<arg.length; ++a) {
			String dicomFileName = arg[a];
			try {
				setOfDicomFiles.add(dicomFileName);
			}
			catch (Exception e) {
				e.printStackTrace(System.err);
				System.exit(0);
			}
		}
		System.out.println(setOfDicomFiles.toString());
	}
}

