/* Copyright (c) 2001-2008, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dicom;

import com.pixelmed.utils.FloatFormatter;

import javax.xml.parsers.*;
import org.w3c.dom.*;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;

import java.io.*;
import java.util.*;

/**
 * <p>A class to encode a representation of a DICOM Structured Report object in an XML form,
 * suitable for analysis as human-readable text, or for feeding into an
 * XSLT-based validator.</p>
 *
 * <p>Note that XML representations can either contain only the content tree, or also the additional
 * top level DICOM attributes other than those that encode the content tree, as individual
 * DICOM attributes, in the manner of {@link com.pixelmed.dicom.XMLRepresentationOfDicomObjectFactory XMLRepresentationOfDicomObjectFactory}.</p>
 *
 * <p>A typical example of usage to extract just the content tree would be:</p>
 * <pre>
try {
    AttributeList list = new AttributeList();
    list.read("dicomsrfile",null,true,true);
	StructuredReport sr = new StructuredReport(list);
    Document document = new XMLRepresentationOfStructuredReportObjectFactory().getDocument(sr);
    XMLRepresentationOfStructuredReportObjectFactory.write(System.out,document);
} catch (Exception e) {
    e.printStackTrace(System.err);
 }
 * </pre>
 *
 * <p>or to include the top level attributes as well as the content tree, supply the attribute
 * list as well as the parsed SR content to the write() method:</p>
 * <pre>
try {
    AttributeList list = new AttributeList();
    list.read("dicomsrfile",null,true,true);
	StructuredReport sr = new StructuredReport(list);
    Document document = new XMLRepresentationOfStructuredReportObjectFactory().getDocument(sr,list);
    XMLRepresentationOfStructuredReportObjectFactory.write(System.out,document);
} catch (Exception e) {
    e.printStackTrace(System.err);
 }
 * </pre>
 *
 * <p>or even simpler, if there is no further use for the XML document or the SR tree model:</p>
 * <pre>
try {
    AttributeList list = new AttributeList();
    list.read("dicomsrfile",null,true,true);
    XMLRepresentationOfStructuredReportObjectFactory.createDocumentAndWriteIt(list,System.out);
} catch (Exception e) {
    e.printStackTrace(System.err);
 }
 * </pre>
 *
 * @see com.pixelmed.dicom.StructuredReport
 * @see com.pixelmed.dicom.XMLRepresentationOfDicomObjectFactory
 * @see com.pixelmed.utils.XPathQuery
 * @see org.w3c.dom.Document
 *
 * @author	dclunie
 */
public class XMLRepresentationOfStructuredReportObjectFactory {

	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/dicom/XMLRepresentationOfStructuredReportObjectFactory.java,v 1.4 2008/02/18 18:14:52 dclunie Exp $";

	/***/
	private DocumentBuilder db;
	
	private void addCodedConceptAttributesToDocumentNode(org.w3c.dom.Node documentNode,Document document,CodedSequenceItem codedConcept) {
		if (codedConcept != null) {
			String codeMeaning = codedConcept.getCodeMeaning();
			if (codeMeaning != null) {
				Attr attr = document.createAttribute("cm");
				attr.setValue(codeMeaning);
				documentNode.getAttributes().setNamedItem(attr);
			}
			
			String codeValue = codedConcept.getCodeValue();
			if (codeValue != null) {
				Attr attr = document.createAttribute("cv");
				attr.setValue(codeValue);
				documentNode.getAttributes().setNamedItem(attr);
			}
			
			String codingSchemeDesignator = codedConcept.getCodingSchemeDesignator();
			if (codingSchemeDesignator != null) {
				Attr attr = document.createAttribute("csd");
				attr.setValue(codingSchemeDesignator);
				documentNode.getAttributes().setNamedItem(attr);
			}
		}
	}

	/**
	 * @param	contentItem		content item node of the Structured Report
	 * @param	document
	 * @param	documentParent	the node of the document to add to
	 */
	private void addContentItemsFromTreeToNode(ContentItem contentItem,Document document,org.w3c.dom.Node documentParent) {
		if (contentItem != null) {
			String valueType = contentItem.getValueType();
			if (valueType != null) {
				String elementName = valueType.toLowerCase();
				org.w3c.dom.Node documentNode = document.createElement(elementName);
				documentParent.appendChild(documentNode);
				
				String relationshipType = contentItem.getRelationshipType();
				if (relationshipType != null) {
					Attr attr = document.createAttribute("relationship");
					attr.setValue(relationshipType);
					documentNode.getAttributes().setNamedItem(attr);
				}
				
				CodedSequenceItem conceptName = contentItem.getConceptName();
				if (conceptName != null) {
					org.w3c.dom.Node conceptNameNode = document.createElement("concept");
					documentNode.appendChild(conceptNameNode);
					addCodedConceptAttributesToDocumentNode(conceptNameNode,document,conceptName);
				}
				
				if (contentItem instanceof ContentItemFactory.CodeContentItem) {
					CodedSequenceItem conceptCode = ((ContentItemFactory.CodeContentItem)contentItem).getConceptCode();
					if (conceptCode != null) {
						org.w3c.dom.Node valueNode = document.createElement("value");
						documentNode.appendChild(valueNode);
						addCodedConceptAttributesToDocumentNode(valueNode,document,conceptCode);
					}
				}
				else if (contentItem instanceof ContentItemFactory.NumericContentItem) {
					String value = ((ContentItemFactory.NumericContentItem)contentItem).getNumericValue();
					if (value != null) {
						org.w3c.dom.Node valueNode = document.createElement("value");
						documentNode.appendChild(valueNode);
						valueNode.appendChild(document.createTextNode(value));
					}
					CodedSequenceItem unitsCode = ((ContentItemFactory.NumericContentItem)contentItem).getUnits();
					if (unitsCode != null) {
						org.w3c.dom.Node unitsCodeNode = document.createElement("units");
						documentNode.appendChild(unitsCodeNode);
						addCodedConceptAttributesToDocumentNode(unitsCodeNode,document,unitsCode);
					}
				}
				else if (contentItem instanceof ContentItemFactory.StringContentItem) {
					String value = ((ContentItemFactory.StringContentItem)contentItem).getConceptValue();
					if (value != null) {
						org.w3c.dom.Node valueNode = document.createElement("value");
						documentNode.appendChild(valueNode);
						valueNode.appendChild(document.createTextNode(value));
					}
				}
				else if (contentItem instanceof ContentItemFactory.SpatialCoordinatesContentItem) {
					String graphicType = ((ContentItemFactory.SpatialCoordinatesContentItem)contentItem).getGraphicType();
					if (graphicType != null) {
						org.w3c.dom.Node graphicTypeNode = document.createElement(graphicType.toLowerCase());
						documentNode.appendChild(graphicTypeNode);
						float[] graphicData = ((ContentItemFactory.SpatialCoordinatesContentItem)contentItem).getGraphicData();
						if (graphicData != null) {
							for (int i=0; i<graphicData.length; ++i) {
								org.w3c.dom.Node coordinateNode = document.createElement(i%2 == 0 ? "x" : "y");
								graphicTypeNode.appendChild(coordinateNode);
								coordinateNode.appendChild(document.createTextNode(FloatFormatter.toString(graphicData[i])));
							}
						}
					}
				}
				else if (contentItem instanceof ContentItemFactory.TemporalCoordinatesContentItem) {
					ContentItemFactory.TemporalCoordinatesContentItem temporalCoordinatesContentItem = (ContentItemFactory.TemporalCoordinatesContentItem)contentItem;
					String temporalRangeType = temporalCoordinatesContentItem.getTemporalRangeType();
					if (temporalRangeType != null) {
						org.w3c.dom.Node temporalRangeTypeNode = document.createElement(temporalRangeType.toLowerCase());
						documentNode.appendChild(temporalRangeTypeNode);
						{
							int[] referencedSamplePositions = temporalCoordinatesContentItem.getReferencedSamplePositions();
							if (referencedSamplePositions != null) {
								org.w3c.dom.Node referencedSamplePositionsNode = document.createElement("samplepositions");
								temporalRangeTypeNode.appendChild(referencedSamplePositionsNode);
								for (int i=0; i<referencedSamplePositions.length; ++i) {
									org.w3c.dom.Node referencedSamplePositionNode = document.createElement("position");
									referencedSamplePositionsNode.appendChild(referencedSamplePositionNode);
									referencedSamplePositionNode.appendChild(document.createTextNode(Integer.toString(referencedSamplePositions[i])));
								}
							}
						}
						{
							float[] referencedTimeOffsets = temporalCoordinatesContentItem.getReferencedTimeOffsets();
							if (referencedTimeOffsets != null) {
								org.w3c.dom.Node referencedTimeOffsetsNode = document.createElement("timeoffsets");
								temporalRangeTypeNode.appendChild(referencedTimeOffsetsNode);
								for (int i=0; i<referencedTimeOffsets.length; ++i) {
									org.w3c.dom.Node referencedTimeOffsetNode = document.createElement("offset");
									referencedTimeOffsetsNode.appendChild(referencedTimeOffsetNode);
									referencedTimeOffsetNode.appendChild(document.createTextNode(FloatFormatter.toString(referencedTimeOffsets[i])));
								}
							}
						}
						{
							String[] referencedDateTimes = temporalCoordinatesContentItem.getReferencedDateTimes();
							if (referencedDateTimes != null) {
								org.w3c.dom.Node referencedDateTimesNode = document.createElement("datetimes");
								temporalRangeTypeNode.appendChild(referencedDateTimesNode);
								for (int i=0; i<referencedDateTimes.length; ++i) {
									org.w3c.dom.Node referencedDateTimeNode = document.createElement("datetime");
									referencedDateTimesNode.appendChild(referencedDateTimeNode);
									referencedDateTimeNode.appendChild(document.createTextNode(referencedDateTimes[i]));
								}
							}
						}
					}
				}
				else if (contentItem instanceof ContentItemFactory.CompositeContentItem) {
					{
						String referencedSOPClassUID = ((ContentItemFactory.CompositeContentItem)contentItem).getReferencedSOPClassUID();
						if (referencedSOPClassUID != null) {
							org.w3c.dom.Node referencedSOPClassUIDNode = document.createElement("class");
							documentNode.appendChild(referencedSOPClassUIDNode);
							referencedSOPClassUIDNode.appendChild(document.createTextNode(referencedSOPClassUID));
						}
					}
					{
						String referencedSOPInstanceUID = ((ContentItemFactory.CompositeContentItem)contentItem).getReferencedSOPInstanceUID();
						if (referencedSOPInstanceUID != null) {
							org.w3c.dom.Node referencedSOPInstanceUIDNode = document.createElement("instance");
							documentNode.appendChild(referencedSOPInstanceUIDNode);
							referencedSOPInstanceUIDNode.appendChild(document.createTextNode(referencedSOPInstanceUID));
						}
					}
					if (contentItem instanceof ContentItemFactory.ImageContentItem) {
						ContentItemFactory.ImageContentItem imageContentItem = (ContentItemFactory.ImageContentItem)contentItem;
						{
							int referencedFrameNumber = imageContentItem.getReferencedFrameNumber();
							if (referencedFrameNumber != 0) {
								org.w3c.dom.Node referencedFrameNumberNode = document.createElement("frame");
								documentNode.appendChild(referencedFrameNumberNode);
								referencedFrameNumberNode.appendChild(document.createTextNode(Integer.toString(referencedFrameNumber)));
							}
						}
						{
							int referencedSegmentNumber = imageContentItem.getReferencedSegmentNumber();
							if (referencedSegmentNumber != 0) {
								org.w3c.dom.Node referencedSegmentNumberNode = document.createElement("segment");
								documentNode.appendChild(referencedSegmentNumberNode);
								referencedSegmentNumberNode.appendChild(document.createTextNode(Integer.toString(referencedSegmentNumber)));
							}
						}
						{
							String presentationStateSOPClassUID = imageContentItem.getPresentationStateSOPClassUID();
							String presentationStateSOPInstanceUID = imageContentItem.getPresentationStateSOPInstanceUID();
							if (presentationStateSOPClassUID != null && presentationStateSOPClassUID.length() > 0
							 || presentationStateSOPInstanceUID != null && presentationStateSOPInstanceUID.length() > 0) {
								org.w3c.dom.Node presentationStateReference = document.createElement("presentationstate");
								documentNode.appendChild(presentationStateReference);
								
								org.w3c.dom.Node presentationStateSOPClassUIDNode = document.createElement("class");
								presentationStateReference.appendChild(presentationStateSOPClassUIDNode);
								presentationStateSOPClassUIDNode.appendChild(document.createTextNode(presentationStateSOPClassUID));
								
								org.w3c.dom.Node presentationStateSOPInstanceUIDNode = document.createElement("instance");
								presentationStateReference.appendChild(presentationStateSOPInstanceUIDNode);
								presentationStateSOPInstanceUIDNode.appendChild(document.createTextNode(presentationStateSOPInstanceUID));
							}
						}
						{
							String realWorldValueMappingSOPClassUID = imageContentItem.getRealWorldValueMappingSOPClassUID();
							String realWorldValueMappingSOPInstanceUID = imageContentItem.getRealWorldValueMappingSOPInstanceUID();
							if (realWorldValueMappingSOPClassUID != null && realWorldValueMappingSOPClassUID.length() > 0
							 || realWorldValueMappingSOPInstanceUID != null && realWorldValueMappingSOPInstanceUID.length() > 0) {
								org.w3c.dom.Node realWorldValueMappingReference = document.createElement("realworldvaluemapping");
								documentNode.appendChild(realWorldValueMappingReference);
								
								org.w3c.dom.Node realWorldValueMappingSOPClassUIDNode = document.createElement("class");
								realWorldValueMappingReference.appendChild(realWorldValueMappingSOPClassUIDNode);
								realWorldValueMappingSOPClassUIDNode.appendChild(document.createTextNode(realWorldValueMappingSOPClassUID));
								
								org.w3c.dom.Node realWorldValueMappingSOPInstanceUIDNode = document.createElement("instance");
								realWorldValueMappingReference.appendChild(realWorldValueMappingSOPInstanceUIDNode);
								realWorldValueMappingSOPInstanceUIDNode.appendChild(document.createTextNode(realWorldValueMappingSOPInstanceUID));
							}
						}
						// forget about icon image sequence for now :(
					}
					else if (contentItem instanceof ContentItemFactory.WaveformContentItem) {
						int[] referencedWaveformChannels = ((ContentItemFactory.WaveformContentItem)contentItem).getReferencedWaveformChannels();
						if (referencedWaveformChannels != null && referencedWaveformChannels.length > 0) {
							org.w3c.dom.Node referencedWaveformChannelsNode = document.createElement("channels");
							documentNode.appendChild(referencedWaveformChannelsNode);
							for (int i=0; i<referencedWaveformChannels.length; ++i) {
								org.w3c.dom.Node channelNode = document.createElement("channel");
								referencedWaveformChannelsNode.appendChild(channelNode);
								channelNode.appendChild(document.createTextNode(Integer.toString(referencedWaveformChannels[i])));
							}
						}
					}
				}
				
				// now handle any children
				int n = contentItem.getChildCount();
				for (int i=0; i<n; ++i) {
					addContentItemsFromTreeToNode((ContentItem)(contentItem.getChildAt(i)),document,documentNode);
				}
			}
		}
	}

	/**
	 * <p>Construct a factory object, which can be used to get XML documents from DICOM objects.</p>
	 *
	 * @exception	ParserConfigurationException
	 */
	public XMLRepresentationOfStructuredReportObjectFactory() throws ParserConfigurationException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setNamespaceAware(true);
		db = dbf.newDocumentBuilder();
	}
	
	/**
	 * <p>Given a DICOM attribute list encoding a Structured Report, get an XML document as a DOM tree.</p>
	 *
	 * @param	list	the attribute list
	 */
	public Document getDocument(AttributeList list) {
		return getDocument(null,list);
	}
	
	/**
	 * <p>Given a DICOM Structured Report, get an XML document of the content tree only as a DOM tree.</p>
	 *
	 * @param	sr		the Structured Report
	 */
	public Document getDocument(StructuredReport sr) {
		return getDocument(sr,null);
	}
	
	/**
	 * <p>Given a DICOM Structured Report, get an XML document of the content tree and the top level DICOM elements as a DOM tree.</p>
	 *
	 * @param	sr		the Structured Report			may be null if list is not - will build an sr tree model
	 * @param	list	the attribute list				may be null if only the sr content tree is to be added
	 */
	public Document getDocument(StructuredReport sr,AttributeList list) {
		if (sr == null) {
			try {
				sr = new StructuredReport(list);
			}
			catch (DicomException e) {
				e.printStackTrace(System.err);
			}
		}
		Document document = db.newDocument();
		org.w3c.dom.Node rootElement = document.createElement("DicomStructuredReport");
		document.appendChild(rootElement);
		if (list != null) {
			AttributeList clonedList = (AttributeList)(list.clone());
 			clonedList.removePrivateAttributes();
 			clonedList.removeGroupLengthAttributes();
 			clonedList.removeMetaInformationHeaderAttributes();
 			clonedList.remove(TagFromName.ContentSequence);
			clonedList.remove(TagFromName.ValueType);
			clonedList.remove(TagFromName.ContentTemplateSequence);
			clonedList.remove(TagFromName.ContinuityOfContent);
			clonedList.remove(TagFromName.ConceptNameCodeSequence);
			org.w3c.dom.Node headerElement = document.createElement("DicomStructuredReportHeader");
			rootElement.appendChild(headerElement);
			try {
				new XMLRepresentationOfDicomObjectFactory().addAttributesFromListToNode(clonedList,document,headerElement);
			}
			catch (ParserConfigurationException e) {
				e.printStackTrace(System.err);
			}
		}
		if (sr != null) {
			org.w3c.dom.Node contentElement = document.createElement("DicomStructuredReportContent");
			rootElement.appendChild(contentElement);
			addContentItemsFromTreeToNode((ContentItem)(sr.getRoot()),document,contentElement);
		}
		return document;
	}
	
	/**
	 * @param	documentNode
	 * @param	indent
	 */
	public static String toString(org.w3c.dom.Node documentNode,int indent) {
		StringBuffer str = new StringBuffer();
		for (int i=0; i<indent; ++i) str.append("    ");
		str.append(documentNode);
		if (documentNode.hasAttributes()) {
			NamedNodeMap attrs = documentNode.getAttributes();
			for (int j=0; j<attrs.getLength(); ++j) {
				org.w3c.dom.Node attr = attrs.item(j);
				//str.append(toString(attr,indent+2));
				str.append(" ");
				str.append(attr);
			}
		}
		str.append("\n");
		++indent;
		for (org.w3c.dom.Node child = documentNode.getFirstChild(); child != null; child = child.getNextSibling()) {
			str.append(toString(child,indent));
			//str.append("\n");
		}
		return str.toString();
	}
	
	/**
	 * @param	documentNode
	 */
	public static String toString(org.w3c.dom.Node documentNode) {
		return toString(documentNode,0);
	}
	
	/**
	 * <p>Serialize an XML document (DOM tree).</p>
	 *
	 * @param	out		the output stream to write to
	 * @param	document	the XML document
	 * @exception	IOException
	 */
	public static void write(OutputStream out,Document document) throws IOException, TransformerConfigurationException, TransformerException {
		
		DOMSource source = new DOMSource(document);
		StreamResult result = new StreamResult(out);
		Transformer transformer = TransformerFactory.newInstance().newTransformer();
		Properties outputProperties = new Properties();
		outputProperties.setProperty(OutputKeys.METHOD,"xml");
		outputProperties.setProperty(OutputKeys.INDENT,"yes");
		outputProperties.setProperty(OutputKeys.ENCODING,"UTF-8");	// the default anyway
		transformer.setOutputProperties(outputProperties);
		transformer.transform(source, result);
	}
	
	/**
	 * <p>Serialize an XML document (DOM tree) created from a DICOM Structured Report.</p>
	 *
	 * @param	list	the attribute list
	 * @param	out		the output stream to write to
	 * @exception	IOException
	 * @exception	DicomException
	 */
	public static void createDocumentAndWriteIt(AttributeList list,OutputStream out) throws IOException, DicomException {
		createDocumentAndWriteIt(null,list,out);
	}
	
	/**
	 * <p>Serialize an XML document (DOM tree) created from a DICOM Structured Report.</p>
	 *
	 * @param	sr		the Structured Report
	 * @param	out		the output stream to write to
	 * @exception	IOException
	 * @exception	DicomException
	 */
	public static void createDocumentAndWriteIt(StructuredReport sr,OutputStream out) throws IOException, DicomException {
		createDocumentAndWriteIt(sr,null,out);
	}
	
	/**
	 * <p>Serialize an XML document (DOM tree) created from a DICOM Structured Report.</p>
	 *
	 * @param	sr		the Structured Report			may be null if list is not - will build an sr tree model
	 * @param	list	the attribute list				may be null if only the sr content tree is to be written
	 * @param	out		the output stream to write to
	 * @exception	IOException
	 * @exception	DicomException
	 */
	public static void createDocumentAndWriteIt(StructuredReport sr,AttributeList list,OutputStream out) throws IOException, DicomException {
		try {
			Document document = new XMLRepresentationOfStructuredReportObjectFactory().getDocument(sr,list);
			write(out,document);
		}
		catch (ParserConfigurationException e) {
			throw new DicomException("Could not create XML document - problem creating object model from DICOM"+e);
		}
		catch (TransformerConfigurationException e) {
			throw new DicomException("Could not create XML document - could not instantiate transformer"+e);
		}
		catch (TransformerException e) {
			throw new DicomException("Could not create XML document - could not transform to XML"+e);
		}
	}
		
	/**
	 * <p>Read a DICOM dataset and write an XML representation of it to the standard output.</p>
	 *
	 * @param	arg	one filename of the file containing the DICOM dataset
	 */
	public static void main(String arg[]) {
		try {
			AttributeList list = new AttributeList();
			//System.err.println("reading list");
			list.read(arg[0],null,true,true);
			//System.err.println("making sr");
			StructuredReport sr = new StructuredReport(list);
			//System.err.println("making document");
			Document document = new XMLRepresentationOfStructuredReportObjectFactory().getDocument(sr,list);
			//System.err.println(toString(document));
			write(System.out,document);
		} catch (Exception e) {
			e.printStackTrace(System.err);
		}
	}
}

