/* Copyright (c) 2001-2007, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.network;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

//import java.beans.PropertyChangeEvent;
//import java.beans.PropertyChangeListener;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import javax.swing.text.JTextComponent;

/**
 * <p>This class implements a dialog for users to enter DICOM AE network configuration parameters.</p>
 *
 * @author	dclunie
 */
public class ApplicationEntityConfigurationDialog extends ApplicationEntity {

	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/network/ApplicationEntityConfigurationDialog.java,v 1.2 2007/12/19 22:44:15 dclunie Exp $";

	protected JTextField dicomAETitleField;
	protected JTextField hostnameField;
	protected JTextField portField;
	
	// should probably move this method somewhere else, e.g. into com.pixelmed.dicom.ApplicationEntityAttribute
	
	public static boolean isValidAETitle(String aet) {
		// Per PS 3.5: Default Character Repertoire excluding character code 5CH (the BACKSLASH “\” in ISO-IR 6), and control characters LF, FF, CR and ESC. 16 bytes maximum
		boolean good = true;
		if (aet == null) {
			good = false;
		}
		else if (aet.length() == 0) {
			good = false;
		}
		else if (aet.length() > 16) {
			good = false;
		}
		else if (aet.trim().length() == 0) {		// all whitespace is illegal
			good = false;
		}
		else if (aet.contains("\\")) {
			good = false;
		}
		else {
			int l = aet.length();
			for (int i=0; i<l; ++i) {
				int codePoint = aet.codePointAt(i);
				try {
					Character.UnicodeBlock codeBlock = Character.UnicodeBlock.of(codePoint);
					if (codeBlock != Character.UnicodeBlock.BASIC_LATIN) {
						good = false;
					}
					else if (Character.isISOControl(codePoint)) {
						good = false;
					}
				}
				catch (IllegalArgumentException e) {	// if not a valid code point
					good = false;
				}
			}
		}
		return good;
	}

	public ApplicationEntityConfigurationDialog() {
		super(NetworkDefaultValues.getDefaultApplicationEntityTitle(NetworkDefaultValues.StandardDicomReservedPortNumber));
		presentationAddress = new PresentationAddress(NetworkDefaultValues.getUnqualifiedLocalHostName(),NetworkDefaultValues.StandardDicomReservedPortNumber);
		primaryDeviceType = NetworkDefaultValues.getDefaultPrimaryDeviceType();

		final JDialog dialog = new JDialog();		// final so that button action listeners can get access to it to dispose of it
		//dialog.setSize(width,height);
		//dialog.setTitle(titleMessage);
		dialog.setModal(true);

		dicomAETitleField = new JTextField();
		hostnameField = new JTextField();
		portField = new JTextField();
		
		dicomAETitleField.setText(dicomAETitle);
		hostnameField.setText(presentationAddress.getHostname());
		portField.setText(Integer.toString(presentationAddress.getPort()));
		
		dicomAETitleField.addFocusListener(new java.awt.event.FocusAdapter() {
			public void focusGained(java.awt.event.FocusEvent event) {
				JTextComponent textComponent = (JTextComponent)(event.getSource());
				textComponent.selectAll();
//System.err.println("ApplicationEntityConfigurationDialog().dicomAETitleField.focusGained(): getText() = \""+textComponent.getText()+"\", selected = \""+textComponent.getSelectedText()+"\"");
			}
		});
		
		hostnameField.addFocusListener(new java.awt.event.FocusAdapter() {
			public void focusGained(java.awt.event.FocusEvent event) {
				JTextComponent textComponent = (JTextComponent)(event.getSource());
				textComponent.selectAll();
//System.err.println("ApplicationEntityConfigurationDialog().dicomAETitleField.focusGained(): getText() = \""+textComponent.getText()+"\", selected = \""+textComponent.getSelectedText()+"\"");
			}
		});
		
		portField.addFocusListener(new java.awt.event.FocusAdapter() {
			public void focusGained(java.awt.event.FocusEvent event) {
				JTextComponent textComponent = (JTextComponent)(event.getSource());
				textComponent.selectAll();
//System.err.println("ApplicationEntityConfigurationDialog().dicomAETitleField.focusGained(): getText() = \""+textComponent.getText()+"\", selected = \""+textComponent.getSelectedText()+"\"");
			}
		});
		
		JPanel labelsAndFieldsPanel = new JPanel(new GridLayout(3,2));
        labelsAndFieldsPanel.add(new JLabel("AE Title:"));
        labelsAndFieldsPanel.add(dicomAETitleField);
		
        labelsAndFieldsPanel.add(new JLabel("Hostname or IP address:"));
        labelsAndFieldsPanel.add(hostnameField);
		
        labelsAndFieldsPanel.add(new JLabel("Port:"));
        labelsAndFieldsPanel.add(portField);

		JPanel buttonsPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
		JButton okButton = new JButton("OK");
		okButton.setToolTipText("Accept AE configuration");
		buttonsPanel.add(okButton);
		
		okButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				boolean good = true;
				dicomAETitle = dicomAETitleField.getText();
				if (!isValidAETitle(dicomAETitle)) {
					good=false;
					dicomAETitleField.setText("\\\\\\BAD\\\\\\");		// use backslash character here (which is illegal in AE's) to make sure this field is edited
				}
				String hostname = hostnameField.getText();
				// ? should validate host name (e.g., http://www.ops.ietf.org/lists/namedroppers/namedroppers.2002/msg00591.html)
				int port=0;
				try {
					port = Integer.parseInt(portField.getText());
				}
				catch (NumberFormatException e) {
					good=false;
					portField.setText("\\\\\\BAD\\\\\\");
				}
				presentationAddress = new PresentationAddress(hostname,port);
				if (good) {
					dialog.dispose();
				}
			}
		});
		
		JPanel allPanels = new JPanel(new BorderLayout());
		allPanels.add(labelsAndFieldsPanel,BorderLayout.NORTH);
		allPanels.add(buttonsPanel,BorderLayout.SOUTH);

		dialog.getContentPane().add(allPanels);
		dialog.getRootPane().setDefaultButton(okButton);

		dialog.pack();
		dialog.setVisible(true);
	}
	
	/**
	 * <p>Main method for testing.</p>
	 *
	 * @param	arg	array of zero strings - no command line arguments are expected
	 */
	public static void main(String arg[]) {
		try {
			if (arg.length == 0) {
				ApplicationEntity ae = new ApplicationEntityConfigurationDialog();
System.err.println("ApplicationEntityConfigurationDialog.main(): result of dialog "+ae);
			}
			else {
				throw new Exception("Argument list must be empty");
			}
		}
		catch (Exception e) {
			e.printStackTrace(System.err);
			System.exit(0);
		}
	}
}

