/* Copyright (c) 2006-2008, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.display;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;

import java.awt.geom.Rectangle2D;
import java.awt.geom.RectangularShape;

import java.awt.image.BufferedImage;
import java.awt.image.ComponentSampleModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferShort;
import java.awt.image.DataBufferUShort;
import java.awt.image.Raster;
import java.awt.image.SampleModel;

import java.util.Iterator;
import java.util.Vector;

import com.pixelmed.dicom.Attribute;
import com.pixelmed.dicom.AttributeList;
import com.pixelmed.dicom.CodeStringAttribute;
import com.pixelmed.dicom.DicomException;
import com.pixelmed.dicom.IntegerStringAttribute;
import com.pixelmed.dicom.OtherByteAttribute;
import com.pixelmed.dicom.OtherWordAttribute;
import com.pixelmed.dicom.TagFromName;
import com.pixelmed.dicom.TransferSyntax;
import com.pixelmed.dicom.UnsignedShortAttribute;

/**
 * <p>A class of utility methods for editing image pixel data.</p>
 *
 * @author	dclunie
 */

public class ImageEditUtilities {

	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/display/ImageEditUtilities.java,v 1.6 2008/01/20 17:44:38 dclunie Exp $";
	
	private ImageEditUtilities() {}
	
	/**
	 * <p>Clean up the PhotometricInterpretation.</p>
	 *
	 * <p>For non-monochrome images, will convert a PhotometricInterpretation of YBR_FULL_422 or similar to RGB
	 * if Transfer Syntax UID left over from read in meta header was compressed, since normally after read the
	 * PhotometricInterpretation will be as encoded, even though the pixel data stream has been transformed
	 * during decompression.</p>
	 *
	 * @param	list	the attribute list to correct, if necessary
	 * @exception	DicomException	if something bad happens handling the attribute list
	 */
	static public void sanitizePhotometricInterpretation(AttributeList list) throws DicomException {
		int samplesPerPixel = Attribute.getSingleIntegerValueOrDefault(list,TagFromName.SamplesPerPixel,1);
		String transferSyntaxUID = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.TransferSyntaxUID);
		String photometricInterpretation = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.PhotometricInterpretation);
		if (samplesPerPixel > 1) {
			if (TransferSyntax.mayBeTransformedColorSpace(transferSyntaxUID) && !photometricInterpretation.equals("RGB")) {
//System.err.println("ImageEditUtilities.sanitizePhotometricInterpretation(): replacing PhotometricInterpretation of "+photometricInterpretation+" with RGB");
				list.remove(TagFromName.PhotometricInterpretation);
				{ Attribute a = new CodeStringAttribute(TagFromName.PhotometricInterpretation); a.addValue("RGB"); list.put(a); }
			}
		}
	}

//	static protected boolean isInShapes(Vector shapes,int x,int y) {
//		Iterator it = shapes.iterator();
//		while (it.hasNext()) {
//			Shape shape = (Shape)it.next();
//			if (shape.contains(x,y)) {
//System.err.println("ImageEditUtilities.isInShapes(): found ("+x+","+y+")");
//				return true;
//			}
//		}
//		return false;
//	}
	
	/**
	 * <p>Blackout specified regions in an image, for example to remove burned in identification.</p>
	 *
	 * <p>The accompanying attribute list will be updated with new Pixel Data and related Image Pixel Module attributes.</p>
	 *
	 * <p>Note that original PhotometricInterpretation will be retained; care should be taken by the caller
	 * to change this as appropriate, e.g., from YBR_FULL_422 if read as JPEG to RGB if written as uncompressed.
	 * See, for example, {@link #sanitizePhotometricInterpretation(AttributeList) sanitizePhotometricInterpretation()}.</p>
	 *
	 * @param	srcImg	the image
	 * @param	list	the attribute list corresponding image 
	 * @param	shapes	a {@link java.util.Vector java.util.Vector} of {@link java.awt.Shape java.awt.Shape}, specifed in image-relative coordinates
	 * @exception	DicomException	if something bad happens handling the attribute list
	 */
	static public void blackout(SourceImage srcImg,AttributeList list,Vector shapes) throws DicomException {
//System.err.println("ImageEditUtilities.blackout():");

//long elapsedDrawingTime = 0;
//long elapsedCopyingTime = 0;
//long elapsedReconstructionTime = 0;

		int                bitsAllocated = Attribute.getSingleIntegerValueOrDefault(list,TagFromName.BitsAllocated,0);;
		int                   bitsStored = Attribute.getSingleIntegerValueOrDefault(list,TagFromName.BitsStored,0);;
		int                      highBit = Attribute.getSingleIntegerValueOrDefault(list,TagFromName.HighBit,bitsStored - 1);
		int              samplesPerPixel = Attribute.getSingleIntegerValueOrDefault(list,TagFromName.SamplesPerPixel,1);
		int          pixelRepresentation = Attribute.getSingleIntegerValueOrDefault(list,TagFromName.PixelRepresentation,0);
		String photometricInterpretation = Attribute.getSingleStringValueOrNull(list,TagFromName.PhotometricInterpretation);
		int          planarConfiguration = 0;	// 0 is color-by-pixel, 1 is color-by-plane

		int rows = 0;
		int columns = 0;

		byte   byteDstPixels[] = null;
		short shortDstPixels[] = null;
		Attribute pixelData = null;
	
		int dstIndex=0;
		int numberOfFrames = srcImg.getNumberOfBufferedImages();
		boolean needToCopyEachFrame = true;
		for (int frame=0; frame<numberOfFrames; ++frame) {
			BufferedImage src = srcImg.getBufferedImage(frame);
			columns = src.getWidth();
//System.err.println("ImageEditUtilities.blackout(): columns = "+columns);
			rows = src.getHeight();
//System.err.println("ImageEditUtilities.blackout(): rows = "+rows);
			SampleModel srcSampleModel = src.getSampleModel();
//System.err.println("ImageEditUtilities.blackout(): srcSampleModel = "+srcSampleModel);
			int srcDataType = srcSampleModel.getDataType();
//System.err.println("ImageEditUtilities.blackout(): srcDataType = "+srcDataType);
			Raster srcRaster = src.getRaster();
			DataBuffer srcDataBuffer = srcRaster.getDataBuffer();
			int srcDataBufferType = srcDataBuffer.getDataType();
//System.err.println("ImageEditUtilities.blackout(): srcDataBufferType = "+srcDataBufferType);
			int srcNumBands = srcRaster.getNumBands();
//System.err.println("ImageEditUtilities.blackout(): srcNumBands = "+srcNumBands);
			int srcPixelStride = srcNumBands;
			int srcScanlineStride = columns*srcNumBands;
			if (srcNumBands > 1 && srcSampleModel instanceof ComponentSampleModel) {
				ComponentSampleModel srcComponentSampleModel = (ComponentSampleModel)srcSampleModel;
				srcPixelStride = srcComponentSampleModel.getPixelStride();			// should be either srcNumBands if color-by-pixel, or 1 if color-by-plane
				srcScanlineStride = srcComponentSampleModel.getScanlineStride();	// should be either columns*srcNumBands if color-by-pixel, or columns if color-by-plane
				planarConfiguration = srcPixelStride == srcNumBands ? 0 : 1;
			}
			int srcDataBufferOffset = srcDataBuffer.getOffset();
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] srcDataBufferOffset = "+srcDataBufferOffset);
			int srcFrameLength = rows*columns*srcNumBands;
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] srcFrameLength = "+srcFrameLength);
			int srcDataBufferNumBanks = srcDataBuffer.getNumBanks();
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] srcDataBufferNumBanks = "+srcDataBufferNumBanks);
			
			if (srcDataBufferNumBanks > 1) {
				throw new DicomException("Unsupported type of image - DataBuffer number of banks is > 1, is "+srcDataBufferNumBanks);
			}

			int dstPixelStride = planarConfiguration == 0 ? srcNumBands : 1;
			int dstBandStride  = planarConfiguration == 0 ? 1 : rows*columns;

			if (srcDataBufferType == DataBuffer.TYPE_BYTE) {
				byte[][] srcPixelBanks = null;
				if (srcDataBuffer instanceof DataBufferByte) {
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] DataBufferByte");
					srcPixelBanks = ((DataBufferByte)srcDataBuffer).getBankData();
				}
				else {
					throw new DicomException("Unsupported type of image - DataBuffer is TYPE_BYTE but not instance of DataBufferByte, is "+srcDataBuffer.getClass().getName());
				}
				int srcPixelBankLength = srcPixelBanks[0].length;
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] srcPixelBankLength = "+srcPixelBankLength);
				if (byteDstPixels == null) {
					if (bitsAllocated > 8) {
						bitsAllocated = 8;
					}
					if (bitsStored > 8) {
						bitsStored = 8;
					}
					if (highBit > 7) {
						highBit = 7;
					}
					samplesPerPixel=srcNumBands;
					// leave photometricInterpretation alone
					// leave planarConfiguration alone ... already determined from srcPixelStride if srcNumBands > 1
					int dstPixelsLength = srcFrameLength*numberOfFrames;
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] dstPixelsLength = "+dstPixelsLength);
					if (dstPixelsLength == srcPixelBankLength) {
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] optimizing by using entire multi-frame array rather than copying");
						// optimize for special case of entire multi-frame image data in single array, shared by multiple BufferedImages using srcDataBufferOffset for frames
						// assumes that offsets are in same order as frames
						byteDstPixels = srcPixelBanks[0];
						needToCopyEachFrame = false;		// rather than break, since still need to draw regions
					}
					else {
						byteDstPixels = new byte[dstPixelsLength];
					}
					pixelData = new OtherByteAttribute(TagFromName.PixelData);
					pixelData.setValues(byteDstPixels);
				}
//long startCopyingTime = System.currentTimeMillis();
				if (needToCopyEachFrame) {
					System.arraycopy(srcPixelBanks[0],srcDataBufferOffset,byteDstPixels,dstIndex,srcFrameLength);
					dstIndex+=srcFrameLength;
				}
//elapsedCopyingTime+=System.currentTimeMillis()-startCopyingTime;
//long startDrawingTime = System.currentTimeMillis();
				Iterator it = shapes.iterator();
				while (it.hasNext()) {
					Shape shape = (Shape)it.next();
					if (shape instanceof RectangularShape) {	// this includes Rectangle and Rectangle2D (but also some other things that would need special handling :( )
						RectangularShape rect = (RectangularShape)shape;
//System.err.println("ImageEditUtilities.blackout(): shape is RectangularShape "+rect);
						int startX = (int)rect.getX();
						int startY = (int)rect.getY();
						int stopX = (int)(startX + rect.getWidth());
						int stopY = (int)(startY + rect.getHeight());
						for (int y=startY; y<stopY; ++y) {
							int index = ((rows*frame + y)*columns + startX);
							for (int x=startX; x<stopX; ++x) {
								for (int bandIndex=0; bandIndex<srcNumBands; ++ bandIndex) {
									byteDstPixels[index*dstPixelStride+bandIndex*dstBandStride]=0;
								}
								++index;
							}
						}
					}
				}
//elapsedDrawingTime+=System.currentTimeMillis()-startDrawingTime;
			}
			else if (srcDataBufferType == DataBuffer.TYPE_USHORT || srcDataBufferType == DataBuffer.TYPE_SHORT) {
				short[][] srcPixelBanks = null;
				if (srcDataBuffer instanceof DataBufferShort) {
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] DataBufferShort");
					srcPixelBanks = ((DataBufferShort)srcDataBuffer).getBankData();
				}
				else if (srcDataBuffer instanceof DataBufferUShort) {
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] DataBufferUShort");
					srcPixelBanks =  ((DataBufferUShort)srcDataBuffer).getBankData();
				}
				else {
					throw new DicomException("Unsupported type of image - DataBuffer is TYPE_USHORT or TYPE_SHORT but not instance of DataBufferShort, is "+srcDataBuffer.getClass().getName());
				}
				int srcPixelBankLength = srcPixelBanks[0].length;
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] srcPixelBankLength = "+srcPixelBankLength);
				if (shortDstPixels == null) {
					if (bitsAllocated > 16) {
						bitsAllocated = 16;
					}
					if (bitsStored > 16) {
						bitsStored = 16;
					}
					if (highBit > 15) {
						highBit = 15;
					}
					samplesPerPixel=srcNumBands;
					// leave photometricInterpretation alone
					// leave planarConfiguration alone ... already determined from srcPixelStride if srcNumBands > 1
					int dstPixelsLength = srcFrameLength*numberOfFrames;
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] dstPixelsLength = "+dstPixelsLength);
					if (dstPixelsLength == srcPixelBankLength) {
//System.err.println("ImageEditUtilities.blackout(): Frame ["+frame+"] optimizing by using entire multi-frame array rather than copying");
						// optimize for special case of entire multi-frame image data in single array, shared by multiple BufferedImages using srcDataBufferOffset for frames
						// assumes that offsets are in same order as frames
						shortDstPixels = srcPixelBanks[0];
						needToCopyEachFrame = false;		// rather than break, since still need to draw regions
					}
					else {
						shortDstPixels = new short[dstPixelsLength];
					}
					pixelData = new OtherWordAttribute(TagFromName.PixelData);
					pixelData.setValues(shortDstPixels);
				}
//long startCopyingTime = System.currentTimeMillis();
				if (needToCopyEachFrame) {
					System.arraycopy(srcPixelBanks[0],srcDataBufferOffset,shortDstPixels,dstIndex,srcFrameLength);
					dstIndex+=srcFrameLength;
				}
//elapsedCopyingTime+=System.currentTimeMillis()-startCopyingTime;
//long startDrawingTime = System.currentTimeMillis();
				Iterator it = shapes.iterator();
				while (it.hasNext()) {
					Shape shape = (Shape)it.next();
					if (shape instanceof RectangularShape) {	// this includes Rectangle and Rectangle2D (but also some other things that would need special handling :( )
						RectangularShape rect = (RectangularShape)shape;
//System.err.println("ImageEditUtilities.blackout(): shape is RectangularShape "+rect);
						int startX = (int)rect.getX();
						int startY = (int)rect.getY();
						int stopX = (int)(startX + rect.getWidth());
						int stopY = (int)(startY + rect.getHeight());
						for (int y=startY; y<stopY; ++y) {
							int index = ((rows*frame + y)*columns + startX);
							for (int x=startX; x<stopX; ++x) {
								for (int bandIndex=0; bandIndex<srcNumBands; ++ bandIndex) {
									shortDstPixels[index*dstPixelStride+bandIndex*dstBandStride]=0;
								}
								++index;
							}
						}
					}
				}
//elapsedDrawingTime+=System.currentTimeMillis()-startDrawingTime;
			}
			else {
				throw new DicomException("Unsupported pixel data form - DataBufferType = "+srcDataBufferType);
			}
		}

		list.remove(TagFromName.PixelData);
		list.remove(TagFromName.BitsAllocated);
		list.remove(TagFromName.BitsStored);
		list.remove(TagFromName.HighBit);
		list.remove(TagFromName.SamplesPerPixel);
		list.remove(TagFromName.PixelRepresentation);
		list.remove(TagFromName.PhotometricInterpretation);
		list.remove(TagFromName.PlanarConfiguration);

		list.put(pixelData);
		{ Attribute a = new UnsignedShortAttribute(TagFromName.BitsAllocated); a.addValue(bitsAllocated); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.BitsStored); a.addValue(bitsStored); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.HighBit); a.addValue(highBit); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.Rows); a.addValue(rows); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.Columns); a.addValue(columns); list.put(a); }
		{ Attribute a = new IntegerStringAttribute(TagFromName.NumberOfFrames); a.addValue(numberOfFrames); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.SamplesPerPixel); a.addValue(samplesPerPixel); list.put(a); }
		{ Attribute a = new UnsignedShortAttribute(TagFromName.PixelRepresentation); a.addValue(pixelRepresentation); list.put(a); }
		{ Attribute a = new CodeStringAttribute(TagFromName.PhotometricInterpretation); a.addValue(photometricInterpretation); list.put(a); }
		if (samplesPerPixel > 1) {
			Attribute a = new UnsignedShortAttribute(TagFromName.PlanarConfiguration); a.addValue(planarConfiguration); list.put(a);
		}
		
//long startReconstructionTime = System.currentTimeMillis();
		srcImg.constructSourceImage(list);
//elapsedReconstructionTime+=System.currentTimeMillis()-startReconstructionTime;

//System.err.println("ImageEditUtilities.blackout(): elapsedDrawingTime = "+elapsedDrawingTime);
//System.err.println("ImageEditUtilities.blackout(): elapsedCopyingTime = "+elapsedCopyingTime);
//System.err.println("ImageEditUtilities.blackout(): elapsedReconstructionTime = "+elapsedReconstructionTime);
//System.err.println("ImageEditUtilities.blackout(): done");
	}

}
