/* A simple integer arithmetic logic unit.

 Copyright (c) 1997-2007 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

 PT_COPYRIGHT_VERSION_2
 COPYRIGHTENDKEY

 */
package ptolemy.actor.lib.tutorial;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.IntToken;
import ptolemy.data.Token;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.Workspace;

//////////////////////////////////////////////////////////////////////////
//// SimpleALU

/**
 <p>
 A simple integer arithmetic logic unit.
 </p>

 @author Man-Kit Leung
 @version $Id: SimpleALU.java,v 1.3 2007/12/07 06:25:50 cxh Exp $
 @since Ptolemy II 6.0.2
 @Pt.ProposedRating Red (mankit)
 @Pt.AcceptedRating Red (mankit)
 */
public class SimpleALU extends TypedAtomicActor {
    /** Construct an actor in the specified container with the specified
     *  name.
     *  @param container The container.
     *  @param name The name of this adder within the container.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the name coincides with
     *   an actor already in the container.
     */
    public SimpleALU(CompositeEntity container, String name)
            throws IllegalActionException, NameDuplicationException {
        super(container, name);
        A = new TypedIOPort(this, "A", true, false);
        A.setTypeEquals(BaseType.INT);
        B = new TypedIOPort(this, "B", true, false);
        B.setTypeEquals(BaseType.INT);

        operation = new TypedIOPort(this, "operation", true, false);
        operation.setTypeEquals(BaseType.INT);

        output = new TypedIOPort(this, "output", false, true);
        output.setTypeEquals(BaseType.INT);

    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** Input port for the first operand. The port type is integer.
     */
    public TypedIOPort A;

    /** Input port for the second operand. The port type is integer.
     */
    public TypedIOPort B;

    /** Input port for the operation code. The port type is integer.
     * The value of the input tokens should not exceed the number of
     * operations supported.
     */
    public TypedIOPort operation;

    /** Output port.  The type is inferred from the connections.
     */
    public TypedIOPort output;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Override the base class to set type constraints.
     *  @param workspace The workspace for the new object.
     *  @return A new instance of SimpleALU.
     *  @exception CloneNotSupportedException If a derived class contains
     *   an attribute that cannot be cloned.
     */
    public Object clone(Workspace workspace) throws CloneNotSupportedException {
        SimpleALU newObject = (SimpleALU) super.clone(workspace);
        newObject.A.setTypeEquals(BaseType.INT);
        newObject.B.setTypeEquals(BaseType.INT);
        newObject.operation.setTypeEquals(BaseType.INT);
        return newObject;
    }

    /** If there is at least one token on each of the input ports, the
     *  first token from each port is consumed. The value of the token
     *  from the operation input port determines which operation will be
     *  perform on the tokens from input port A and B. The opcode is
     *  defined as follow:
     *      0 - NOP
     *      1 - Addition (A + B)
     *      2 - Subtraction (A - B)
     *      3 - Multiplication (A * B)
     *      4 - Division (A / B)
     *  @throw IllegalActionException If the input operation code
     *  is not supported.
     */
    public void fire() throws IllegalActionException {
        super.fire();
        Token result = null;

        if (A.hasToken(0) && B.hasToken(0) && operation.hasToken(0)) {
            int opcode = ((IntToken) operation.get(0)).intValue();
            IntToken tokenA = (IntToken) A.get(0);
            IntToken tokenB = (IntToken) B.get(0);

            switch (opcode) {
            case 0: // NOP
                result = IntToken.ZERO;
                break;
            case 1: // addition
                result = tokenA.add(tokenB);
                break;
            case 2: // subtraction
                result = tokenA.subtract(tokenB);
                break;
            case 3: // multiplication
                result = tokenA.multiply(tokenB);
                break;
            case 4: // division
                result = tokenA.divide(tokenB);
                break;
            default:
                throw new IllegalActionException(this,
                        "Unsupported operation code: " + opcode + ".\n");
            }
        }
        output.send(0, result);
    }
}
