package apps;

import tools.*;

import java.text.*;
import java.util.*;
import java.util.logging.*;


 /**
  * Converts FSL b-vectors and b-values to Camino scheme file format.
  * 
  * @author Philip Cook
  * @version $Id: FSL_ToScheme.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
  */
public class FSL_ToScheme {


    /**
     * logging object
     */
    private static final Logger logger = Logger.getLogger("apps.FSL_ToScheme");


    public static void main(String[] args) {

	Locale.setDefault(Locale.UK);

	String bvecfile = null;
	String bvalfile = null;

	String outputFile = null;

	double diffusionTime = 1.0;

	double bScale = 1.0E6;

	boolean flipX = false;
	boolean flipY = false;
	boolean flipZ = false;

	int version = 2;


	// Some scanners define b-value as |g|^2 * \beta
	// where \beta is what they CLAIM the b-value is.
	// If you use normalized gradient directions, you need to increase b accordingly
	// to make the same ADC calculation.
	boolean useGradMod = false;


	CL_Initializer.CL_init(args);

	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals("-bvecfile") || args[i].equals("-bvecs")) {
		bvecfile = args[i+1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-bvalfile") || args[i].equals("-bvals")) {
		bvalfile = args[i+1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-outputfile")) {
		outputFile = args[i+1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-diffusiontime")) {
		diffusionTime = Double.parseDouble(args[i+1]);
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-bscale")) {
		bScale = Double.parseDouble(args[i+1]);
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-usegradmod")) {
		useGradMod = true;
		logger.info("Gradient direction magnitude will be incorporated into b-value");	
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-flipx")) {
		flipX = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-flipy")) {
		flipY = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-flipz")) {
		flipZ = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-version")) {
		version = Integer.parseInt(args[i+1]);
		CL_Initializer.markAsParsed(i,2);
	    }
	}


	String scheme = "";

	CL_Initializer.checkParsing(args);
	    
	FileInput vecIn = new FileInput(bvecfile);

	String[] x = vecIn.readString().trim().split("\\s+");
	
	String[] y = vecIn.readString().trim().split("\\s+");

	String[] z = vecIn.readString().trim().split("\\s+");
	
	vecIn.close();

	FileInput valIn = new FileInput(bvalfile);

	String[] b = valIn.readString().trim().split("\\s+");
       
	if (b.length != x.length) {
	    throw new misc.LoggedException("Number of b-vectors and b-values do not match");
	}

	DecimalFormat df = new DecimalFormat("   0.000000;  -0.000000");

	DecimalFormat v0DF = new DecimalFormat("0.000000");

	DecimalFormat bValDF = new DecimalFormat("   0.000E00");

	boolean warnAboutGradDirMod = false;

	if (version == 0) {

	    scheme += diffusionTime + "\n";
	    scheme += x.length + "\n";
	    	
	    for (int i = 0; i < x.length; i++) {
		double modQ = Math.sqrt( Double.parseDouble(b[i]) * bScale / diffusionTime);

		double vx = Double.parseDouble(x[i]);
		double vy = Double.parseDouble(y[i]);
		double vz = Double.parseDouble(z[i]);
		
		double modV = Math.sqrt(vx*vx + vy*vy + vz*vz);

		// don't worry about small errors (output is formatted to fixed precision anyhow)
		if (modV == 0.0 || Math.abs(1.0 - modV) < 1E-5)
		    modV = 1.0;

		if (useGradMod) {
		    modQ = modQ * modV;
		}
		else if (modV != 1.0) {
		    warnAboutGradDirMod = true;
		}
		
		double qx = (vx / modV * modQ);
		
		if (flipX && qx != 0.0) {
		    qx = -qx;
		}
		
		double qy = (vy / modV * modQ);
		
		if (flipY && qy != 0.0) {
		    qy = -qy;
		}
		
		double qz = (vz / modV * modQ);
		
		if (flipZ && qz != 0.0) {
		    qz = -qz;
		}
		
		
		scheme += v0DF.format(qx) + "\n";
		scheme += v0DF.format(qy) + "\n";
		scheme += v0DF.format(qz) + "\n";
		
	    }
	    
	}
	else if (version == 2) {
  
	    scheme += "VERSION: 2\n";

	    for (int i = 0; i < x.length; i++) {
	
		double vx = Double.parseDouble(x[i]);
		double vy = Double.parseDouble(y[i]);
		double vz = Double.parseDouble(z[i]);
		
		double modV = Math.sqrt(vx*vx + vy*vy + vz*vz);

		if (modV == 0.0 || Math.abs(1.0 - modV) < 1E-5)
		    modV = 1.0;

		if (!useGradMod && modV != 1.0) {
		    warnAboutGradDirMod = true;
		}
		
		vx = (vx / modV);
		
		if (flipX && vx != 0.0) {
		    vx = -vx;
		}
		
		vy = (vy / modV);
		
		if (flipY && vy != 0.0) {
		    vy = -vy;
		}
		
		vz = (vz / modV);
		
		if (flipZ && vz != 0.0) {
		    vz = -vz;
		}

		// get rid of -0.0
		if (vx == 0.0) { 
		    vx = Math.abs(vx);
		}
		if (vy == 0.0) { 
		    vy = Math.abs(vy);
		}
		if (vz == 0.0) { 
		    vz = Math.abs(vz);
		}
				
		scheme += df.format(vx);
		scheme += df.format(vy);
		scheme += df.format(vz);
		
		if (useGradMod) {
		    scheme += bValDF.format(Double.parseDouble(b[i]) * bScale * modV * modV) + "\n";
		}
		else { 
		    scheme += bValDF.format(Double.parseDouble(b[i]) * bScale) + "\n";
		}

	    }

	}
	
	if (warnAboutGradDirMod) {
	    logger.warning("Some measurements have non-unit gradient directions. Directions have been " + 
			   "normalized to unit length");
	}
	

	if (outputFile != null) {
	    FileOutput out = new FileOutput(outputFile);
	    
	    out.writeString(scheme);
		
	    out.close();
	}
	else {
	    System.out.println(scheme);
	    
	}


    } 




}
