package apps;

import java.io.*;
import java.util.logging.Logger;

import tools.*;
import imaging.*;
import inverters.*;
import data.*;
import misc.LoggedException;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Diffusion data inversion application using model fitting.
 * 
 * <dt>Description:
 * 
 * <dd>The program fits simple models, specified by the user, to input data and
 * outputs the model parameters in each voxel.
 * 
 * </dl>
 * 
 * @author Danny Alexander
 * @version $Id: ModelFit.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 *  
 */
public class ModelFit {


    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.apps.ModelFit");


    /**
     * Output manager
     */
    private static OutputManager om;


    public static void main(String[] args) {

        // Parse the command line arguments
        CL_Initializer.CL_init(args);
        CL_Initializer.checkParsing(args);
        CL_Initializer.initImagingScheme();
        CL_Initializer.initDataSynthesizer();

        om = new OutputManager();

        // Choose the inversion to run.
        DiffusionInversion inv = getIndexedInversion(CL_Initializer.inversionIndices,
                CL_Initializer.imPars);

        // Loop over the voxels.
        processVoxels(inv, om);
    }


    /**
     * Runs the inversion on each voxel in the data.
     * 
     * @param inv
     *            The inverter.
     */
    public static void processVoxels(DiffusionInversion inv, OutputManager om) {

        // Loop over the data
        int voxelNumber = 0;
        while (CL_Initializer.data.more())
            try {

                double[] nextVoxel = CL_Initializer.data.nextVoxel();

                // Fit or output background default.
                double backgroundB0 = CL_Initializer.imPars.geoMeanZeroMeas(nextVoxel);
                if (isBG(backgroundB0)) {
                    double[] voxOut = new double[inv.itemsPerVoxel()];

                    // Set the exitcode to -1 to indicate background.
                    voxOut[0] = -1;

                    if (backgroundB0 > 0.0) {
                        voxOut[1] = Math.log(backgroundB0);
                    }
                    else {
                        voxOut[1] = 0.0;  
                    }

                    om.output(voxOut);

                    // The inverter may have other operations to perform
                    // in background voxels.
                    inv.background();

                }
                else {
                    // Fit the model and output the result.
                    double[] fittedData = inv.invert(nextVoxel);
                    om.output(fittedData);
                }
                voxelNumber++;
                logger.fine("Completed voxel: " + voxelNumber);

            }
            catch (DataSourceException e) {
                throw new LoggedException("The data file does not contain a whole number of voxels." +
                                          "Check the scheme file. Got Exception " + e);
            }

        // Output statistics compiled by the inverter.
        inv.close();

        // Close the output streams.
        om.close();

    }

    /**
     * Creates and returns the indexed diffusion inversion.
     * 
     * @param indices
     *            The indices for the inversion(s). Either {index} or {two tensor index, DT index}, 
     *            or {three tensor index, DT index}.
     * 
     * @return The inverter.
     */
    public static DiffusionInversion getIndexedInversion(ModelIndex[] indices,
            Scheme imPars) {

	// test for non-DT inversions first, then go through the 1,2,3 tensor possibilities

        if (indices[0] == ModelIndex.BALL_STICK) {
	    return new BallStickInversion(imPars);
        }
        if (indices[0] == ModelIndex.RESTORE) {
	    
            // Use the RESTORE algorithm
            if (CL_Initializer.sigma == -1) {
                logger
		    .severe("Noise level must be specified (-sigma <std>) for RESTORE.");
                System.exit(1);
            }
            return new RestoreDT_Inversion(imPars, CL_Initializer.sigma);
        }
        if (indices[0] == ModelIndex.ADC) {

            // ADC only inversion.
            return new LinearADC_Inversion(imPars);
        }
        if (indices[0].numDTs == 1) {
	    
            // Single DT inversion.
            return DT_Inversion.getIndexedDT_Inversion(indices[0], imPars);
        }
        if (indices[0].numDTs == 2) {
	    
	    if (indices.length == 1) {
		return new TwoTensorInversion(imPars, indices[0], ModelIndex.LDT);
	    }
	    else {
		return new TwoTensorInversion(imPars, indices[0], indices[1]);
	    }
        }
        if (indices[0].numDTs == 3) {

	    if (indices.length == 1) {
		return new ThreeTensorInversion(imPars, indices[0], ModelIndex.LDT);
	    }
	    else {
		return new ThreeTensorInversion(imPars, indices[0], indices[1]);
	    }
        }

        return null;
    }


    /**
     * Check whether the next voxel is background
     *
     * @return boolean indicating background status.
     */
    public static boolean isBG(double backgroundB0) {

        boolean background = false;
        if (CL_Initializer.BACKGROUNDTHRESHOLD > 0.0) {
            background = backgroundB0 < CL_Initializer.BACKGROUNDTHRESHOLD;
        }
        if(CL_Initializer.bgMask != null) try {
            background = (CL_Initializer.bgMask.nextVoxel()[0] == 0.0);
        } catch(DataSourceException e) {
            throw new LoggedException("Error reading background mask." + e);
        }

        return background;
    }


}
