package apps;

import data.*;

import imaging.*;
import inverters.ModelIndex;
import misc.LoggedException;

import numerics.*;
import tools.*;
import tractography.*;

import java.util.Random;
import java.util.zip.*;
import java.util.logging.*;

import java.io.*;

/**
 * 
 * Does tractography. A list of points can be read from a file, or a single
 * point can be specified on the command line.
 *
 * @author Philip Cook
 * @version $Id: StreamlineTractography.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * 
 */
public class StreamlineTractography {


    public enum ImageType {

	PICO(true),
	BOOTSTRAP(true),
	WILD_BOOTSTRAP(true),
	BAYES_DIRAC(true),
	BAYES_MCMC(true),
	
	TENSOR(false),
	PDS(false),
	BALL_STICK(false);

	ImageType(boolean prob) {
	    this.probabilistic = prob;
	}
	
	final boolean probabilistic;
    }

    
    protected static int DEFAULT_PICO_ITERATIONS = 5000;

    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.apps.StreamlineTractography");


    /**
     * Output manager
     */
    private static OutputManager om;


    public static void main(String[] args) {
	
	if (args.length == 0) {
	    System.exit(0);
	}

	// type of image to construct
	ImageType imageType = null;
	
	PICoPDF picoPDF = PICoPDF.BINGHAM;

	String bootstrapModel = "dt";
	
	int iterations = 0;
	
	// default is to output raw binary
	boolean outputTractVoxels = false;
	boolean outputTractsOOGL = false;  

	// true in addition to outputTractsOOGL if we output binary oogl
	boolean binaryOOGL = false; 

	// set to true if we do any interpolation: DT, vector, tend 
	// (doesn't interpolate but uses a fixed step size)
	boolean interpolated = false;

	// extra flag for interpolation of tensor PDs
	boolean vectorInterpolation = false;

	// extra flag for TEND
	// if (tend && interpolated), do tend with fixed step size
	// if (tend && !interpolated) do tend with FACT
	boolean tend = false;

	// path to image containing f for TEND
	String tendF_File = null;

	// constant f used if file is not given
	double tendF = 0.0;

	// use a constant g, since low anisotropy DTs automatically deflect the direction less
	double tendG = 1.0;

	// dimensions of seed space
	int seedXDataDim = 0;
	int seedYDataDim = 0;
	int seedZDataDim = 0;

	double seedXVoxelDim = 0.0;
	double seedYVoxelDim = 0.0;
	double seedZVoxelDim = 0.0;

	int xDataDim = 0;
	int yDataDim = 0;
	int zDataDim = 0;

	double xVoxelDim = 0.0;
	double yVoxelDim = 0.0;
	double zVoxelDim = 0.0;

	double seedPointX = 0.0;
	double seedPointY = 0.0;
	double seedPointZ = 0.0;

	double anisThresh = 0.0;
	double ipThresh = Math.cos(Math.PI * 80.0 / 180.0);

	double stepSize = 0.0;

	// Analyze image containing seed ROI 
	String seedFile = null;

	boolean voxelCoordinates = false; // true if seeds are specified as voxels

	// text file containing mm seed points
	String seedList = null;

	// contains all ROIs in one volume or seed list
	RegionOfInterest allROIs = null;

        // note that seedIndex and regionIndex count from zero, but for I/O purposes we
        // index from 1. So if -seedindex 1 is given, seedIndex == 0 and the first seed is processed,
        // but the output is outputRoot_1_1.{hdr, img}.

        // allows the user to specify a single seed from the seed file. The point of this
        // is to allow parallel processing while maintaining a consistent output format.
        // If this is set, then count through seeds in seed file and process the seedIndex-th seed
        // but the output file will have the same name as it would if you processed the whole seed file.
        int seedIndex = -1;


        // allows the user to specify a single ROI from the seed file. The point of this
        // is to allow parallel processing while maintaining a consistent output format. 
        // If this is set, all seeds in the specified ROI are processed.
        int regionIndex = -1;


	// if no one-DT data is provided, then an ANIS map is needed
	// otherwise no ANIS threshold can be applied.
	String anisMapFile = null;

	long seed = System.currentTimeMillis();

	Random ran = null;

	String outputRoot = null;
	
	
	// Bayesian options 
	double curvePriorK = 0.0;
	double curvePriorG = 0.0;

	DataModel dataModel = DataModel.CYL_SYMM_DT;
	
	// path to a PICo tractography image
	String externalPriorImageFile = null;

	String externalPriorDataType = "double";


        CL_Initializer.inputDataType = "double";

	// for Bayesian tracking
	CL_Initializer.pointSetInd = 1;

	CL_Initializer.CL_init(args);

	// figure out image type

	if (CL_Initializer.inputModel == null) {
	    throw new LoggedException("An input model is required.");
	}
	
	if (CL_Initializer.inputModel.equals("bootstrap")) {

	    if (CL_Initializer.bsDataFiles == null) {
		imageType = ImageType.WILD_BOOTSTRAP;
	    }
	    else if (CL_Initializer.bsDataFiles.length == 1) {
		throw new LoggedException("Can't do repetition bootstrap with a single image");
	    }
	    else {
		imageType = ImageType.BOOTSTRAP;
	    }

 	    CL_Initializer.inputDataType = "float";

	}
        if (CL_Initializer.inputModel.equals("pico")) {
            imageType = ImageType.PICO;
        }
        else if (CL_Initializer.inputModel.equals("dt")) {
            imageType = ImageType.TENSOR;
        }
        else if (CL_Initializer.inputModel.equals("multitensor")) {
            imageType = ImageType.TENSOR;
        }
        else if (CL_Initializer.inputModel.equals("pds")) {
            imageType = ImageType.PDS;
        }
        else if (CL_Initializer.inputModel.equals("ballstick")) {
            imageType = ImageType.BALL_STICK;
        }
        else if (CL_Initializer.inputModel.equals("bayesdirac")) {

	    // uses raw data

 	    // change default data type; parse again below in case it is set on command line
 	    CL_Initializer.inputDataType = "float";
 	    imageType = ImageType.BAYES_DIRAC;
 	}

	// gzip output if true
	boolean gzip = OutputManager.gzipOut;

	// transforms seed space into diffusion space
	RealMatrix trans = null;

	// diffusion space into seed space
	RealMatrix invTrans = null;

	// true if the seed space is aligned with the diffusion space. This does not always
	// mean that the voxel sizes are the same, just that a point in mm space is the same
	// in both images
	boolean seedSpaceIsDiffusion = true;

	// if true, don't print anything to stderr
	boolean silent = false;

	for (int i = 0; i < args.length; i++) {
	    
	    // input data
            if (args[i].equals("-bsmodel")) {
		bootstrapModel = args[i+1];
		CL_Initializer.markAsParsed(i, 2);
	    }
            if (args[i].equals("-wildbsmodel")) {
		bootstrapModel = args[i+1];
		CL_Initializer.markAsParsed(i, 2);
	    }
            if (args[i].equals("-inputdatatype")) {
		// needed because we change default for bootstrap data
		CL_Initializer.inputDataType = args[i+1];
		CL_Initializer.markAsParsed(i, 2);
	    }

	    // tractography args
	    else if (args[i].equals("-seedpointmm")) {
		seedPointX = Double.parseDouble(args[i + 1]);
		seedPointY = Double.parseDouble(args[i + 2]);
		seedPointZ = Double.parseDouble(args[i + 3]);
		CL_Initializer.markAsParsed(i, 4);
	    }
 	    else if (args[i].equals("-seedpointvox")) {
		seedPointX = Double.parseDouble(args[i + 1]);
		seedPointY = Double.parseDouble(args[i + 2]);
		seedPointZ = Double.parseDouble(args[i + 3]);
		voxelCoordinates = true;
		CL_Initializer.markAsParsed(i, 4);
	    } 
 	    else if (args[i].equals("-seedindex")) {
                int tmp = Integer.parseInt(args[i+1]);

                if (tmp <= 0) {
                    throw new LoggedException("Invalid seed index specified. Minimum index is 1");
                }
                
                seedIndex = tmp - 1;
                CL_Initializer.markAsParsed(i, 2);
	    } 
 	    else if (args[i].equals("-regionindex")) {
                regionIndex = Integer.parseInt(args[i+1]) - 1;
                CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-seedfile")) {
		seedFile = args[i + 1];
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-seedlist")) {
		seedList = args[i + 1];
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-randomseed") || args[i].equals("-seed")) { 
                // don't want CL default of 0, so parse here
                // also named differently to avoid confusion with tracking seed
		// but allow CL default 
		seed = Long.parseLong(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-anisthresh")) {
		anisThresh = Double.parseDouble(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-anisfile")) {
		anisMapFile = args[i + 1];
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-ipthresh")) {
		ipThresh = Double.parseDouble(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-curvethresh")) {
		ipThresh = Math.cos(Math.PI * Double.parseDouble(args[i + 1]) / 180.0);
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-stepsize")) {
		stepSize = Double.parseDouble(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-outputtracts")) {
		if (i < args.length - 1) {

                    if (args[i+1].equals("oogl")) {
                        outputTractsOOGL = true;
                        CL_Initializer.markAsParsed(i, 2);
                    }
                    else if (args[i+1].equals("ooglbinary")) {
                        binaryOOGL = true;
                        outputTractsOOGL = true;
                        CL_Initializer.markAsParsed(i, 2);
                    }
                    else if (args[i+1].equals("voxels")) {
                        outputTractVoxels = true;
                        CL_Initializer.markAsParsed(i, 2);
                    }
                    else if (args[i+1].equals("raw")) { 
                        CL_Initializer.markAsParsed(i, 2);
                    }
		    else {
			// raw is default
			CL_Initializer.markAsParsed(i); 
		    }

                }
		else { 
		    CL_Initializer.markAsParsed(i, 1);
		}
		// raw is default

		   
	    }
	    else if (args[i].equals("-iterations")) {
		iterations = Integer.parseInt(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-pdf")) {
		
		CL_Initializer.markAsParsed(i);

		if (i == args.length - 1) {
		    throw new LoggedException("Missing PDF type");
		}

		picoPDF = PICoPDF.getPDF(args[i+1]);

		CL_Initializer.markAsParsed(i+1);
	    }
	    else if (args[i].equals("-datamodel")) {
		// Bayesian tracking option
		dataModel = DataModel.getModel(args[i+1]);
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-curvepriork")) {
		// Bayesian tracking option
		curvePriorK = Double.parseDouble(args[i+1]);
		curvePriorG = 0.0;
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-curvepriorg")) {
		// Bayesian tracking option
		curvePriorG = Double.parseDouble(args[i+1]);
		curvePriorK = 0.0;
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-extpriorfile")) {
		// Bayesian tracking option
		externalPriorImageFile = args[i+1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-extpriordatatype")) {
		// Bayesian tracking option
		externalPriorDataType = args[i+1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-outputroot")) {
		outputRoot = args[i + 1];
		CL_Initializer.markAsParsed(i,2);
	    } 
	    else if (args[i].equals("-interpolated") || args[i].equals("-interpolate")) {
		interpolated = true;
		CL_Initializer.markAsParsed(i);
		if (args.length > i+1 && args[i+1].equals("vectors")) {
		    vectorInterpolation = true;
		    CL_Initializer.markAsParsed(i+1);
		}
	    }
	    else if (args[i].equals("-tendffile")) {
		tendF_File = args[i + 1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-tend")) {
		tend = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-tendf")) {
		tendF = Double.parseDouble(args[i+1]);
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-tendg")) {
		tendG = Double.parseDouble(args[i+1]);
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-transformation")) { 
		trans = readMatrix(args[i+1]);
		invTrans = invertAffine(trans);
		CL_Initializer.markAsParsed(i,2);
		seedSpaceIsDiffusion = false;
	    }
	    else if (args[i].equals("-silent")) {
		silent = true;
		CL_Initializer.markAsParsed(i);
            }

	}
	

	CL_Initializer.checkParsing(args);


	if (outputRoot == null) {
	    om = new OutputManager();
	}

	// begin check args
	// check all the args make sense
	

	xDataDim = CL_Initializer.dataDims[0];
	yDataDim = CL_Initializer.dataDims[1];
	zDataDim = CL_Initializer.dataDims[2];

	xVoxelDim = CL_Initializer.voxelDims[0];
	yVoxelDim = CL_Initializer.voxelDims[1];
	zVoxelDim = CL_Initializer.voxelDims[2];

	if (seedSpaceIsDiffusion) {
	    // if seed file is given, take data dims and voxel dims from seed file
	    if (seedFile != null) {
		try {
		    ImageHeader seedHeader = ImageHeader.readHeader(seedFile);
		    
		    seedXDataDim = seedHeader.xDataDim();
		    seedYDataDim = seedHeader.yDataDim();
		    seedZDataDim = seedHeader.zDataDim();
		    
		    seedXVoxelDim = seedHeader.xVoxelDim();
		    seedYVoxelDim = seedHeader.yVoxelDim();
		    seedZVoxelDim = seedHeader.zVoxelDim();
		    
		    if (seedXVoxelDim < 0.0 || seedYVoxelDim < 0.0 || seedZVoxelDim < 0.0) {
			logger.warning("Negative voxel size specified in seed file. Using absolute values.");

			seedXVoxelDim = Math.abs(seedXVoxelDim);
			seedYVoxelDim = Math.abs(seedYVoxelDim);
			seedZVoxelDim = Math.abs(seedZVoxelDim);
		    }

		}
		catch(IOException e) {
		    throw new LoggedException(e);
		}

		if (xDataDim == 0 && yDataDim == 0 && zDataDim == 0) {
		    logger.info("No diffusion data dimensions given. Using seed space dimensions.");
		    xDataDim = seedXDataDim;
		    yDataDim = seedYDataDim;
		    zDataDim = seedZDataDim;
		}
		else {
		    if (xDataDim != seedXDataDim || yDataDim != seedYDataDim || zDataDim != seedZDataDim) {
			seedSpaceIsDiffusion = false;
		    }
		}

		if (xVoxelDim == 0.0 && yVoxelDim == 0.0 && zVoxelDim == 0.0) {
		    logger.info("No diffusion voxel dimensions given. Using seed space dimensions.");
		    xVoxelDim = seedXVoxelDim;
		    yVoxelDim = seedYVoxelDim;
		    zVoxelDim = seedZVoxelDim;
		}
		else {
		    if (xVoxelDim != seedXVoxelDim || yVoxelDim != seedYVoxelDim || zVoxelDim != seedZVoxelDim) {
			seedSpaceIsDiffusion = false;
		    }
		    
		}

	    }
	    else {
		// seed space is diffusion, no seed file
		seedXDataDim = xDataDim;
		seedYDataDim = yDataDim;
		seedZDataDim = zDataDim;

		seedXVoxelDim = xVoxelDim;
		seedYVoxelDim = yVoxelDim;
		seedZVoxelDim = zVoxelDim;
	    }
	    if (xDataDim == 0 || yDataDim == 0 || zDataDim == 0) { 
		throw new LoggedException("No data dimensions specified");
	    }
	    if (xVoxelDim == 0.0 || yVoxelDim == 0.0 || zVoxelDim == 0.0) { 
		throw new LoggedException("No data dimensions specified");
	    }

	}
	
	// may have set seedSpaceIsDiffusion to false in above block, hence "if" rather than "else if" here
	if (!seedSpaceIsDiffusion) {

	    if (xDataDim == 0 || yDataDim == 0 || zDataDim == 0 || 
		xVoxelDim == 0.0 || yVoxelDim == 0.0 || zVoxelDim == 0.0) {
		throw new LoggedException("Voxel and data dimensions of diffusion space must " + 
					  "be specified when the seed space is not aligned to the " +
					  "diffusion space");
	    }

	    if (seedFile != null) {
		try {
		    
		    ImageHeader seedHeader = ImageHeader.readHeader(seedFile);
		    
		    seedXDataDim = seedHeader.xDataDim();
		    seedYDataDim = seedHeader.yDataDim();
		    seedZDataDim = seedHeader.zDataDim();
		    
		    seedXVoxelDim = Math.abs(seedHeader.xVoxelDim());
		    seedYVoxelDim = Math.abs(seedHeader.yVoxelDim());
		    seedZVoxelDim = Math.abs(seedHeader.zVoxelDim());
		    
		}
		catch(IOException e) {
		    throw new LoggedException(e);
		}
	    }

	}


	if (stepSize == 0.0) {
	    stepSize = zVoxelDim / 10.0;
	}

        if (seedIndex > -1 && regionIndex > -1) {
            throw new LoggedException("Both -seedindex and -regionindex specified. Only one may be used.");
        }
	
	if (imageType != ImageType.TENSOR) {  
	    // check for anisotropy threshold
	    if (anisThresh > 0.0 && anisMapFile == null) {
		throw new LoggedException("Input model is not tensors, anisotropy map (-anisfile) must be " + 
					  "supplied when -anisthresh is used");
	    }
	}

	ran = new MTRandom(seed);

        // don't want multiple iterations unless we are doing probabilistic tracking
        if (!imageType.probabilistic) {
	    if (iterations != 0) {
		logger.warning("Iterations is set to " + iterations + " but input model is not probabilistic. " + 
			       "Proceeding with deterministic tracking.");
	    }
            iterations = 1;
        }
	else {
	    if (iterations == 0) {
		iterations = DEFAULT_PICO_ITERATIONS;
	    }
	}


	// end check args

	// get seeds
	if (seedFile != null) {
	    
            short[][][] seedVol = readShortVolume(seedFile);

            allROIs = new FreeFormROI(seedVol, seedXVoxelDim, seedYVoxelDim, seedZVoxelDim);

	}
	else if (seedList != null) {
	    allROIs = PointListROI.readPoints(seedList, 
					      new int[] {seedXDataDim, seedYDataDim, seedZDataDim},
					      new double[] {seedXVoxelDim, seedYVoxelDim, seedZVoxelDim});
					      
	}
	else {
	    
            short[][][] seedVol = new short[xDataDim][yDataDim][zDataDim];
	    
	    if (voxelCoordinates) {
                seedVol[(int)seedPointX][(int)seedPointY][(int)seedPointZ] = 1;
	    } else {
                seedVol[(int)(seedPointX / xVoxelDim) ][(int)(seedPointY / yVoxelDim)]
                    [(int)(seedPointZ / zVoxelDim)] = 1;
	    }
	    
            allROIs = new FreeFormROI(seedVol, xVoxelDim, yVoxelDim, zVoxelDim);

	}
	
	int numRegions = allROIs.numberOfRegions();


	// get the voxel classification
	// either from a voxel class map or from a brain / background segmentation
	int[][][] vc = new int[xDataDim][yDataDim][zDataDim];

	// number of PDs in each voxel (for Bayesian images)
	int[][][] voxelNumPDs = new int[xDataDim][yDataDim][zDataDim];

	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    vc[i][j][k] = 0;
		    voxelNumPDs[i][j][k] = 1;
		}
	    }
	}

	
	if (CL_Initializer.bgMaskFile != null) {
	    CL_Initializer.initMaskSource();
    
	    for (int k = 0; k < zDataDim; k++) {
                for (int j = 0; j < yDataDim; j++) {
                    for (int i = 0; i < xDataDim; i++) {
			
			double maskValue = CL_Initializer.bgMask.nextVoxel()[0];

                        vc[i][j][k] = maskValue > 0.0 ? 0 : -1;
			voxelNumPDs[i][j][k] = maskValue > 0.0 ? 1 : 0;
		    }
		}
	    }
	    
	}


	// use VC from file if we have it
	if (CL_Initializer.voxelClassMap != null) {
	    vc = new int[xDataDim][yDataDim][zDataDim];
	    
	    VoxelOrderDataSource vcSource = new VoxelOrderDataSource(CL_Initializer.voxelClassMap,
								     1, "int");

	    for (int k = 0; k < zDataDim; k++) {
                for (int j = 0; j < yDataDim; j++) {
                    for (int i = 0; i < xDataDim; i++) {
                        vc[i][j][k] = (int)vcSource.nextVoxel()[0];
		    }
		}
	    }
	}

	// get the anisotropy map, if any
	double[][][] anisMap = null;

	if (anisMapFile != null) {
	    if (ImageHeader.imageExists(anisMapFile)) {
		try {
		    ImageHeader ih = ImageHeader.readHeader(anisMapFile);
		    anisMap = ih.readVolume(0);
		}
		catch (IOException e) {
		    throw new LoggedException(e);
		}
	    }
	    else { 
		anisMap = new double[xDataDim][yDataDim][zDataDim];

		VoxelOrderDataSource anisSource = 
		    new VoxelOrderDataSource(anisMapFile, 1, CL_Initializer.inputDataType);
		
		for (int k = 0; k < zDataDim; k++) {
		    for (int j = 0; j < yDataDim; j++) {
			for (int i = 0; i < xDataDim; i++) {
			    anisMap[i][j][k] = anisSource.nextVoxel()[0];
			}
		    }
		}
		
	    }
	}
	

	// get the image and the tracker
	FibreTracker tracker = null;

	TractographyImage image = null;
	
	
	if (imageType.probabilistic) {
	    switch (imageType) {
		
	    case BOOTSTRAP :
		
		CL_Initializer.initImagingScheme();
		
		
		if (bootstrapModel.equals("dt") || bootstrapModel.equals("multitensor")) {
		    
		    image = 
			RepBS_DT_TractographyImage.getBS_TractographyImage(CL_Initializer.bsDataFiles,
									   CL_Initializer.inputDataType,
									   CL_Initializer.imPars,
									   CL_Initializer.inversionIndices,
									   vc, anisMap, anisThresh,
									   new int[] {xDataDim, yDataDim, 
										      zDataDim},
									   new double[] {xVoxelDim, 
											 yVoxelDim, 
											 zVoxelDim},
									   ran);
		}
		else {
		    throw new LoggedException("Unsupported bootstrap model : " + bootstrapModel);
		}
		
		break;


	    case WILD_BOOTSTRAP :
		CL_Initializer.initImagingScheme();
		
		if (bootstrapModel.equals("dt")) {
		    image = 
			WildBS_DT_TractographyImage.getBS_TractographyImage(CL_Initializer.inputFile,
									    CL_Initializer.inputDataType,
									    CL_Initializer.imPars,
									    vc, anisMap, anisThresh,
									    new int[] {xDataDim, yDataDim, 
										       zDataDim},
									    new double[] {xVoxelDim, 
											  yVoxelDim, 
											  zVoxelDim},
									    ran);
		}
		else {
		    throw new LoggedException("Unsupported bootstrap model : " + bootstrapModel);
		}

		break;

		

	    case PICO : 
		
		image = PICoTractographyImage.getPICoTractographyImage
		    (CL_Initializer.inputFile, CL_Initializer.inputDataType, CL_Initializer.numPDsIO, 
		     picoPDF, anisMap, anisThresh, new int[] {xDataDim, yDataDim, zDataDim},
		     new double[] {xVoxelDim, yVoxelDim, zVoxelDim}, ran);
		
		break;

	    case BAYES_DIRAC : 

		CL_Initializer.initImagingScheme();

 		// Friman Bayesian method
		// add anisotropy map option
		BayesDiracTractographyImage bi = BayesDiracTractographyImage.getBayesDiracTractographyImage
		    (CL_Initializer.inputFile, CL_Initializer.inputDataType,
		     CL_Initializer.imPars, dataModel, voxelNumPDs, anisMap, anisThresh,
		     new int[] {xDataDim, yDataDim, zDataDim},
		     new double[] {xVoxelDim, yVoxelDim, zVoxelDim}, CL_Initializer.pointSetInd, ran);

		if (curvePriorK > 0.0) {
		    bi.setCurvePriorKappa(curvePriorK);
		}
		if (curvePriorG > 0.0) {
		    bi.setCurvePriorGamma(curvePriorG);
		}

// 		bi.setCurvePriorKappa(curvePriorK);

		if (externalPriorImageFile != null) {
		    PICoTractographyImage ePrior = PICoTractographyImage.getPICoTractographyImage
		    (externalPriorImageFile, externalPriorDataType, CL_Initializer.numPDsIO, 
		     picoPDF, anisMap, anisThresh, new int[] {xDataDim, yDataDim, zDataDim},
		     new double[] {xVoxelDim, yVoxelDim, zVoxelDim}, ran);

		    bi.setExternalPriors(ePrior);
		}

		image = bi;
		

		break;

// 	    case BAYESIAN_MCMC :
// 		// Behrens MCMC Bayesian method
		
	    default : throw new LoggedException("Unsupported image type : " + imageType);
		
	    }
	    
	    
	    if (interpolated) {
		
		tracker = new NC_ProbFibreTracker(image, stepSize, ipThresh, ran);
		
	    }
	    else {
		
		tracker = new NonIntProbFibreTracker(image, ipThresh, ran);
	    }


	}
    	else { // deterministic

            switch (imageType) {
		
	    case PDS :
		image = 
                    PD_TractographyImage.
                    getPD_TractographyImage(CL_Initializer.inputFile, CL_Initializer.inputDataType,
                                            CL_Initializer.numPDsIO, anisMap, anisThresh,
                                            new int[] {xDataDim, yDataDim, zDataDim},
                                            new double[] {xVoxelDim, yVoxelDim, zVoxelDim});

                if (interpolated) {
                    tracker = new LinIntVectorEulerFibreTracker(image, stepSize, ipThresh);
                }
                else {
                    tracker = new NonInterpolatedFibreTracker(image, ipThresh);
                }

		break;

	    case BALL_STICK :
		image = 
                    BallStickTractographyImage.
                    getBallStickTractographyImage(CL_Initializer.inputFile, CL_Initializer.inputDataType,
						  anisMap, anisThresh,
						  new int[] {xDataDim, yDataDim, zDataDim},
						  new double[] {xVoxelDim, yVoxelDim, zVoxelDim});

                if (interpolated) {
                    tracker = new LinIntVectorEulerFibreTracker(image, stepSize, ipThresh);
                }
                else {
                    tracker = new NonInterpolatedFibreTracker(image, ipThresh);
                }

		break;

	    case TENSOR :
                image = 
                    DT_TractographyImage.
                    getDT_TractographyImage(CL_Initializer.inputFile, CL_Initializer.inputDataType,
                                            CL_Initializer.numPDsIO, anisMap, anisThresh, 
                                            new int[] {xDataDim, yDataDim, zDataDim},
                                            new double[] {xVoxelDim, yVoxelDim, zVoxelDim});
		
                
                if (interpolated && vectorInterpolation) {
		    tracker = new LinIntVectorEulerFibreTracker(image, stepSize, ipThresh);
		}
		else if (interpolated) {
		    tracker = new LinIntDTEulerFibreTracker((DT_TractographyImage)image, stepSize, ipThresh);
                }
                else if (tend) {

		    if (tendF_File != null) {
			
			try {
			
			    ImageHeader ih = ImageHeader.readHeader(tendF_File);
			    
			    tracker = new FACT_TendFibreTracker((DT_TractographyImage)image, 
							   ipThresh, ih.readVolume(0), tendG);
			}
			catch (IOException e) {
			    throw new LoggedException(e);
			}
		    }
		    else {
			tracker = new FACT_TendFibreTracker((DT_TractographyImage)image, 
							     ipThresh, tendF, tendG);
		    }
		}
                else {
                    tracker = new NonInterpolatedFibreTracker(image, ipThresh);
                }

		break;

             
	    default : throw new LoggedException("Unsupported image type : " + imageType);
   
            }

	    
	}

        
	StreamlineROI_Filter filter = new StreamlineROI_Filter(seedXDataDim, seedYDataDim, seedZDataDim,
							       seedXVoxelDim, seedYVoxelDim, seedZVoxelDim);
	
	if (!outputTractVoxels) {
	    filter.setResampleTracts(false);
	}
	
	if (trans != null) {
	    filter.setTransIntoSeedSpace(invTrans);
	}

        int seedCounter = 0;

        regions : for (int region = 0; region < allROIs.numberOfRegions(); region++) {

            if (regionIndex > -1) {
                if (region != regionIndex) {
                    continue;
                }
            }
	    
	    RegionOfInterest roi = allROIs.getRegion(region);
	    
            // number ROIs from 1 upwards to maintain correspondence to 
            // values in seed file.
            int outputRegionID = region + 1; 
	    
	    
	    // points defined in diffusion space, in mm
            Point3D[] seeds = transformPoints(roi.getSeedPoints(), trans);
	    
	    // Also need seed voxels in diffusion space
	    PointListROI diffROI = new PointListROI(seeds, 
						    new int[] {xDataDim, yDataDim, zDataDim},
						    new double[] {xVoxelDim, yVoxelDim, zVoxelDim});

	    Voxel[] seedVoxels = diffROI.getVoxels();
            
            if (seedIndex == -1) {
                if (!silent) {
			System.err.println("Processing ROI " + outputRegionID + " of " + numRegions);
            	}
	    }
            else {
                if (seeds.length + seedCounter - 1 < seedIndex) {
                    seedCounter += seeds.length;
                    continue;
                }
                else if (seedCounter > seedIndex) {
                    break regions;
                }
                else {
	            if (!silent) {		
                   	 System.err.println("Processing ROI " + outputRegionID + " of " + numRegions);
                    }
		}
                
            }

            FileOutputStream fout = null;
            DataOutputStream dout = null;

            try {

		if (outputRoot == null) {
		    dout = om.getOutputStream();
		}
		else {
		    if (outputTractsOOGL) {
                    
			if (gzip) {
			    fout = new FileOutputStream
				(outputRoot + outputRegionID + ".oogl.gz");
			    
			    dout = new 
				DataOutputStream(new GZIPOutputStream(fout, 1024*1024*16));
			    
			}
			else {
			    
			    fout = new FileOutputStream(outputRoot + outputRegionID + ".oogl");
			    
			    dout = new 
				DataOutputStream(new BufferedOutputStream(fout, 1024*1024*16));	
			}
			
		    } 
		    else if (outputTractVoxels) {
			
			if (gzip) {
			    fout = new FileOutputStream(outputRoot + outputRegionID + ".Bshort.gz");
			    dout = new DataOutputStream
				(new GZIPOutputStream(fout, 1024*1024*16));
			    
			}
			else {
			    fout = new FileOutputStream(outputRoot + outputRegionID + ".Bshort");
			    
			    dout = new DataOutputStream
				(new BufferedOutputStream(fout, 1024*1024*16));
			    
			}
		    }
		    else { // raw
			
			if (gzip) {
			    fout = new FileOutputStream(outputRoot + outputRegionID + ".Bfloat.gz");
			    dout = new DataOutputStream(new GZIPOutputStream(fout, 1024*1024*16));
			    
			}
			else {
			    fout = new FileOutputStream(outputRoot + outputRegionID + ".Bfloat");
			    dout = new DataOutputStream
				(new BufferedOutputStream(fout, 1024*1024*16));
			}
		    }
     
		}
		
		if (outputTractsOOGL) {
		    dout.write(new String("LIST\n").getBytes());
		}


                for (int sp = 0; sp < seeds.length; sp++) {

                    if (seedIndex > -1) {
                        if (seedCounter != seedIndex) {
                            seedCounter++;
                            continue;
                        }
                    }

                    seedCounter++;

		    if (!silent) {	
                        System.err.print("\rProcessing seed " + (sp + 1) + " of " + seeds.length); 
		    }

                    Point3D seedPoint = seeds[sp];
                    
		    int x = seedVoxels[sp].x;
		    int y = seedVoxels[sp].y;
		    int z = seedVoxels[sp].z;

		    int numPDs = image.numberOfPDs(x,y,z);

		    // if number of PDs is zero, track once
		    // tracker will return the seed point
		    numPDs = numPDs == 0 ? 1 : numPDs;

		    for (int p = 0; p < numPDs; p++) {

			for (int i = 0; i < iterations; i++) {
			    
			    TractCollection tc = filter.processTracts(tracker.trackFromSeed(seedPoint, p));

			    Tract t = tc.getTract(0);
			    
			    if (outputTractsOOGL) {
				
				if (binaryOOGL) {
				    t.writeOOGL_BinaryVECT(dout);
				}
				else{
				    dout.write(t.toOOGLVECT().getBytes());
				}
			    
			    }
			    else if (outputTractVoxels) {
			    
				VoxelList voxels = t.toVoxelList(seedXVoxelDim, seedYVoxelDim, seedZVoxelDim);
				voxels.writeVoxelList(dout);
				
			    }
			    else { // raw
				t.writeRaw(dout);
			    }
			    
			}

		    }
		    
                } // end for seedpoints
               
		if (!silent) { 
              	    System.err.println("\n");
                }

                if (outputRoot != null) {
		    // want to close file 
                    dout.close();
                }		    
		
            }
            catch(IOException e) {
                throw new LoggedException(e);
            }
            
            
        } // end for all ROIs


	// close om stream
        if(om != null)
            om.close();

    }

    protected static short[][][] readShortVolume(String file) {

	try {

            short[][][] vol = null;
            
            ImageHeader ih = ImageHeader.readHeader(file);
           
            int xDataDim = ih.xDataDim();
            int yDataDim = ih.yDataDim();
            int zDataDim = ih.zDataDim();
            
            vol = new short[xDataDim][yDataDim][zDataDim];
                        
    	    DataSource vin = ih.getImageDataSource();

            for (int k = 0; k < zDataDim; k++) {
                for (int j = 0; j < yDataDim; j++) {
                    for (int i = 0; i < xDataDim; i++) {
                        
                        double value = vin.nextVoxel()[0];
                 
                        
                        if (value > 0.0) {
                            
                            if (value > Short.MAX_VALUE) {
                                throw new LoggedException("Maximum value allowed for seeds / waypoints / targets is " + 
							  Short.MAX_VALUE);
                            }
                            
                            vol[i][j][k] = (short)Math.round(value);
                        }
                    }
                }
            }
            
            return vol;
            
        } catch (IOException e) {
            throw new LoggedException(e);
        }
        
    }


    private static RealMatrix readMatrix(String file) {
	FileInput in = new FileInput(file);

	double[][] elements = new double[4][4];

	for (int i = 0; i < 4; i++) {

	    String[] line = in.readString().split("\\s+");

	    for (int j = 0; j < 4; j++) {
		
		elements[i][j] = Double.parseDouble(line[j]);
		
	    }
	}

       RealMatrix trans = new RealMatrix(4,4);
       trans.entries = elements;

       return trans;
	
    }
    

    /**
     * Apply transformation matrix to points. 
     *
     *
     */
    private static Point3D[] transformPoints(Point3D[] points, RealMatrix trans) {

	if (trans == null) {
	    return points;
	}
	
	double[][] transformation = trans.entries;

	int numberOfPoints = points.length;

	Point3D[] newPoints = new Point3D[numberOfPoints + 1];

	double[] pVector = new double[4];
	double[] transPoint = new double[3];	

	for (int p = 0; p < numberOfPoints; p++) {
	    pVector[0] = points[p].x;
	    pVector[1] = points[p].y;
	    pVector[2] = points[p].z;
	    pVector[3] = 1.0;
	    
	    transPoint[0] = 
		transformation[0][0] * pVector[1] + 
		transformation[0][1] * pVector[0] + 
		transformation[0][2] * pVector[2] + 
		transformation[0][3] * pVector[3]; 
	    
	    transPoint[1] = 
		transformation[1][0] * pVector[1] + 
		transformation[1][1] * pVector[0] + 
		transformation[1][2] * pVector[2] + 
		transformation[1][3] * pVector[3]; 

	    transPoint[2] = 
		transformation[2][0] * pVector[1] + 
		transformation[2][1] * pVector[0] + 
		transformation[2][2] * pVector[2] + 
		transformation[2][3] * pVector[3]; 

	    newPoints[p] = (new Point3D(transPoint));
	    
	}

	return newPoints;

    }


    private static RealMatrix invertAffine(RealMatrix trans) {
	throw new LoggedException("-transformation option is not yet supported");
    }

}
