package apps;

import data.*;
import imaging.*;
import misc.*;
import numerics.*;
import tools.*;
import tractography.*;


/**
 *
 * Generates statistics of streamline tracts, either of the tracts themselves or of scalar values
 * along the tracts. The program can either output the raw statistics or combine them into an Analyze
 * image.
 *
 *
 * @author Philip Cook
 * @version $Id: TractStatistics.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 *
 * @see tractography.TractStatisticFilter
 * @see tractography.TractStatisticImage
 * 
 *
 */
public class TractStatistics {


    public static void main(String[] args) {
	
	// if true, use SPM scale factor in scalar header
	boolean spmScale = true;

	double resampleStepSize = 0.1;
	
	boolean resample = true;

	// if input is voxels, resampling can be forced with -resamplestepsize
	boolean forceResample = false;
	
	boolean outputImage = false;


	String scalarFile = null;
	

	CL_Initializer.inputModel = "raw";
	
	CL_Initializer.CL_init(args);


	String outputRoot = "tractstats";


	// If false, only compute the statistic in the voxel in which the fibre was seeded,
	// else statistic is placed in all voxels the fibre intersects
	boolean countIntersect = false;

	// if true, interpolate scalar data set
	boolean interpolated = false;

	// The statistic to compute from the tract: none (output raw scalar values), length (of tract), or 
	// [mean | max | min | median | std] (from image)
	String tractStat = "mean";

	// The statistic to compute in the image, ie what scalar to derive in each voxel :
	// mean, max, min, median, std.

	String imageStat = "mean";

	// Example: we compute mean FA of streamlines, tractStat == mean, countIntersect == true.
	// We get a 4D image where each voxel contains the mean FA of all tracts that intersect the voxel.
	// The imageStat defines what to do with those values in order to produce a 3D scalar output
	// image. If the imageStat is "mean" or "var", these are always weighted, so tracts that intersect
	// more of the voxel have higher weight.


	for (int i = 0; i < args.length; i++) {

	    // image args
            if(args[i].equals("-scalarfile")) {
		scalarFile = args[i + 1];
		CL_Initializer.markAsParsed(i, 2);
	    } 
	    else if (args[i].equals("-countintersect")) {
		countIntersect = true;
	    	CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-interpolate")) {
		interpolated = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-tractstat")) {
		tractStat = args[i+1];
	    	CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-imagestat")) {
		imageStat = args[i+1];
	    	CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-outputroot")) {
		outputRoot = args[i+1];
	    	CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-outputimage")) {
		outputImage = true;
	    	CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-resamplestepsize")) { 
		resampleStepSize = Double.parseDouble(args[i + 1]);
		resample = true;
		forceResample = true;
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-noresample")) { 
		resample = false;
		CL_Initializer.markAsParsed(i);
	    }
	}	    

	CL_Initializer.checkParsing(args);

	if (CL_Initializer.inputModel.equals("voxels")) {
	    resample = false;
	}

	if (outputImage && tractStat.equals("none")) {
	    throw new LoggedException("Tract statistic must be specified when an image is to be generated.");
	}


	if (resample || forceResample) {
	    resample = true;
	}


	ImageHeader scalarHeader = null;

	try {

	    ScalarImage scalars = null;

	    if (scalarFile != null) {
		// derive stats from the tracts and the scalars
		scalarHeader = ImageHeader.readHeader(scalarFile);

		ImageHeader.checkDims(scalarFile, CL_Initializer.dataDims, CL_Initializer.voxelDims);

		// may have negative 
		
		double[][][] scalarVol = scalarHeader.readVolume(0);
		
		scalars = new ScalarImage(scalarVol, CL_Initializer.voxelDims);

	    }


	    TractSource source = new TractSource(CL_Initializer.inputFile, CL_Initializer.inputModel, 
						 CL_Initializer.voxelDims[0], CL_Initializer.voxelDims[1], CL_Initializer.voxelDims[2]);

	    if (outputImage) {

		TractStatisticImage statImage = null;

		if (scalars != null) {
		    		    
		    statImage = new TractStatisticImage(scalars);

		}
		else {
		    statImage = new TractStatisticImage(CL_Initializer.dataDims, CL_Initializer.voxelDims);
		}

		statImage.setTractStatistic(tractStat);
		statImage.setImageStatistic(imageStat);
		statImage.setInterpolate(interpolated);
		statImage.setCountIntersect(countIntersect);
		
		while (source.more()) {
		    Tract t = source.nextTract();

		    if (resample) {
			t = t.resample(resampleStepSize);
		    }
		    
		    statImage.processTract(t);
		}
	    
		double[][][] result = statImage.getImageStatistic();
		
		AnalyzeHeader ahOut = AnalyzeHeader.getHeader(result, scalarHeader.getVoxelDims(), AnalyzeHeader.DT_DOUBLE);		
		AnalyzeHeader.writeImage(result, ahOut, outputRoot, OutputManager.gzipOut);

	    }
	    else {

		// output raw values for each tract

		OutputManager om = new OutputManager();

		TractStatisticFilter statFilter = null;
		
		if (scalars != null) {
		    
		    statFilter = new TractStatisticFilter(scalars);
		}
		else {
		    statFilter = new TractStatisticFilter(CL_Initializer.dataDims, CL_Initializer.voxelDims);
		}

		statFilter.setTractStatistic(tractStat);
		statFilter.setInterpolate(interpolated);

		while (source.more()) {
		    Tract t = source.nextTract();

		    if (resample) {
			t = t.resample(resampleStepSize);
		    }
		    
		    om.output(statFilter.processTract(t));
		}

		om.close();
	    }
	    

	}
	catch (java.io.IOException e) {
	    throw new LoggedException(e);
	}

	
    }







}
