package imaging;

import java.io.*;
import java.util.logging.*;

import data.*;
import misc.LoggedException;

/**
 *
 * Superclass for medical image headers, provides an interface for 
 * basic header information and a method to get the underlying data in voxel order.
 *
 * @author Philip Cook
 * @version $Id: ImageHeader.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 *
 */
public abstract class ImageHeader {

	 
    private static Logger logger = Logger.getLogger("camino.imaging.ImageHeader");


    /** 
     * Analyze : width<br>
     * MetaIO  : DimSize[0]<br>
     * NIFTI-1 : XDIM<br>
     */
    public abstract int xDataDim();


    /** 
     * Analyze : height<br>
     * Meta    : DimSize[1]<br>
     * NIFTI-1 : YDIM<br>
     */
    public abstract int yDataDim();

   
    /** 
     * Analyze : depth<br>
     * Meta    : DimSize[2]<br>
     * NIFTI-1 : ZDIM<br>
     */
    public abstract int zDataDim();


    /** 
     * @return {xDataDim(), yDataDim(), zDataDim()}.
     */
    public abstract int[] getDataDims();


    /** 
     * Analyze : pixelwidth<br> (may be negative).
     * Meta    : ElementSpacing[0]<br>
     * NIFTI-1 : pixdim[0]<br>
     */
    public abstract double xVoxelDim();


    /** 
     * Analyze : pixelheight<br> (may be negative).
     * Meta    : ElementSpacing[1]<br>
     * NIFTI-1 : pixdim[1]<br>
     */
    public abstract double yVoxelDim();


    /** 
     * Analyze : pixeldepth<br> (may be negative).
     * Meta    : ElementSpacing[2]<br>
     * NIFTI-1 : pixdim[2]<br>
     */
    public abstract double zVoxelDim();


    /** 
     * Voxel dims may be negative for Analyze, use checkDims to get the absolute values.
     *
     * @return {xVoxelDim(), yVoxelDim(), zVoxelDim()}.
     */
    public abstract double[] getVoxelDims();


    /** 
     * Analyze : nImages<br>
     * Meta    : channels<br>
     * NIFTI-1 : DIM5<br>
     *
     * @return the number of components in the image, or 1 if this field is zero.
     */
    public abstract int components();


    /**
     * Analyze : origin (SPM, called centre in Camino)<br>
     * Meta    : offset | position | origin<br>
     * NIFTI-1 : qoffset<br>
     *
     * @return the origin of the image.
     */
    public abstract double[] getOrigin();
    


    /**
     * Gets the image data source, scaling applied if applicable. This may
     * involve reading the entire data file from disk if the data is in scanner order.
     */
    public abstract DataSource getImageDataSource();


    /**
     * Gets all the data in the image. Images that store data in voxel order
     * will incur an overhead here while they are converted to scanner order.
     *
     * @return an array of image data, in the order [x][y][z][volume].
     */
    public abstract double[][][][] readVolumeData();


    /**
     * Reads a 3D Analyze volume and returns the result as double.
     *
     * @param index the volume to read, indexed from 0.
     *
     * @return a volume of dimension [xDataDim()][yDataDim()][zDataDim()].
     */
    public abstract double[][][] readVolume(int index);

 

    /**
     * @param fileName the full path to a file, including the extension (extension is optional for Analyze).
     *
     * @return true if the file ends with a recognized extension: .hdr, .img[.gz], .nii[.gz], .mha, .mhd, 
     * and an image exists corresponding to that file name. Also returns true if fileName.[hdr, img[.gz]] exists.
     *
     */
    public static boolean imageExists(String fileName) {
	
	// first check if the file ends in a recognized extension, if so check for
	// that image type only
	if (fileName.endsWith(".nii") || fileName.endsWith(".nii.gz") || fileName.endsWith(".mha") ||
	    fileName.endsWith(".mhd")) {
	    
	    File f = new File(fileName);
	    return f.exists();
	    
	}
	
	// if the file is not a NIFTI / ITK file, then check to see if it is part of an Analyze / NIFTI image
	// ie if there exists fileName.[hdr,img] or fileName.[hdr,img.gz]
	return (AnalyzeHeader.getImageRoot(fileName) != null);
	
    }
    


    /**
     * Returns the appropriate header based on the given file name.
     *
     * @return the header object for the image.
     *
     */
    public static ImageHeader readHeader(String hdrFile) throws IOException {
	
	if (hdrFile == null) {
	    throw new LoggedException("File name required to read image header " + 
				      "(format determined by extension)");
	}


	if (AnalyzeHeader.getImageRoot(hdrFile) != null) {

	    String root = AnalyzeHeader.getImageRoot(hdrFile);
	    
	    // check for header size, Analyze == 348, NIFTI > 348
	    File f = new File(root + ".hdr");

	    long length = f.length();

	    if (length == 348) {
		return AnalyzeHeader.readHeader(root + ".hdr");
	    }
	    else if (length > 348) {
		return Nifti1Dataset.readHeader(root + ".hdr");
	    }
	    else {
		throw new LoggedException("Image header " + hdrFile + " is smaller than 348 bytes.");
	    }
	    
	}
	if (hdrFile.endsWith(".nii") || hdrFile.endsWith(".nii.gz")) {
	    // NIFTI
	    return Nifti1Dataset.readHeader(hdrFile);
	}
	if (hdrFile.endsWith(".mhd") || hdrFile.endsWith(".mha")) {
	    // meta
	    return MetaImageHeader.readHeader(hdrFile);
	}


	throw new LoggedException("Can't find image for input file name " + hdrFile);
    }



    /**
     * Checks data and voxel dimensions from a header with those given in the arrays. 
     * If the arrays contain zeros, the method populates them with the values from the header
     * and returns <code>true</code>, otherwise it checks for consistency with the header and 
     * returns <code>false</code> if they are not consistent.
     *
     * @return true if the dimensions are consistent or zero, false otherwise.
     */
    public static boolean checkDims(String hdrFile, int[] dataDims, double[] voxelDims) {
	
	ImageHeader header = null;

	try {
	    header = readHeader(hdrFile);	
	}
	catch (IOException e) {
	    throw new LoggedException(e);
	}
	
	// warn on negative pixel dims, but we don't change them here so that we
	// may check consistency across multiple volumes
	if (header.xVoxelDim() < 0.0 || header.yVoxelDim() < 0.0 || header.zVoxelDim() < 0.0) {
	    logger.warning("Negative voxel size specified in one or more image files. Using absolute values.");
	}
	
	if (dataDims[0] == 0.0 && dataDims[1] == 0.0 && dataDims[2] == 0.0) {
	    dataDims[0] = header.xDataDim();
	    dataDims[1] = header.yDataDim();
	    dataDims[2] = header.zDataDim();
	}
	else {
	    if (dataDims[0] != header.xDataDim() || dataDims[1] != header.yDataDim() ||
		dataDims[2] != header.zDataDim()) {
		logger.warning("Inconsistent data dimensions detected. Data dimensions of " + 
			       hdrFile + " are " + header.xDataDim() + " " + header.yDataDim() +
			       " " + header.zDataDim());
		return false;
	    }
	}

	if (voxelDims[0] == 0.0 && voxelDims[1] == 0.0 && voxelDims[2] == 0.0) {
	    voxelDims[0] = header.xVoxelDim();
	    voxelDims[1] = header.yVoxelDim();
	    voxelDims[2] = header.zVoxelDim();
	}
	else {
	    if (Math.abs(voxelDims[0] - header.xVoxelDim()) > 1E-6 || 
		Math.abs(voxelDims[1] - header.yVoxelDim()) > 1E-6 || 
		Math.abs(voxelDims[2] - header.zVoxelDim()) > 1E-6 ) {
		
		logger.warning("Inconsistent voxel dimensions detected. Voxel dimensions of " + 
			       hdrFile + " are " + header.xVoxelDim() + " " + 
			       header.yVoxelDim() + " " + header.zVoxelDim());


		return false;
		
	    }
	}

	return true;
    
    }


}
