package imaging;

import java.text.DecimalFormat;
import java.util.logging.Logger;
import java.util.Scanner;
import java.util.Vector;
import java.io.File;
import java.io.IOException;

import misc.*;
import simulation.DiffusionSimulation;
import tools.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * Defines an MR imaging sequence.
 * 
 * <dd>
 * 
 * <dt>Description:
 * 
 * Contains all the parameters describing a diffusion imaging sequence.
 * 
 * <dd>
 * 
 * 
 * </dl>
 * 
 * @author Danny Alexander
 * @version $Id: SchemeV1.java,v 1.5 2009/03/27 01:26:41 bennett Exp $
 *  
 */
public class SchemeV1 implements Scheme {


	/**
	 * logging object
	 */
	transient private static final Logger logger = Logger.getLogger("camino.imaging.SchemeV1");

	/**
	 * Gyromagnetic ratio of protons in water.
	 */
	public static double GAMMA = 2.6751987E8;

	/**
	 * The radial wavenumbers.
	 */
	protected double[] modQ;

	/**
	 * The gradient directions.
	 */
	protected double[][] gDir;

	/**
	 * The gradient strengths.
	 */
	protected double[] modG;

	/**
	 * The pulse separations.
	 */
	protected double[] bigDel;

	/**
	 * The pulse widths.
	 */
	protected double[] smallDel;

	/**
	 * The echo times.
	 */
	protected double[] TE;

	/**
	 * Indicates whether each measurement has b=0.
	 */
	protected boolean[] zero;

	/**
	 * Indicates whether each measurement should be ignored.
	 */
	protected boolean[] ignored;

	/**
	 * Total number of measurements.
	 */
	protected int numMeas;

	/**
	 * Number of b=0 measurements
	 */
	protected int numZeroMeas;

	/**
	 * time of first gradient block
	 */
	protected double[] halfP90;


	/**
	 * The number of measurements to ignore in each voxel.  Note the
	 * methods associated with this variable are now deprecated and
	 * should not be used.
	 */
	protected int ignoreMeasurements = 0;

	/**
	 * the total number of ignored measurements
	 */
	private int numIgnoredMeasurements;


	/**
	 * Constructor that reads the imaging parameters from a text file.
	 * 
	 * @param filename The name of the file.
	 */
	public SchemeV1(String filename) {

		Vector<String> lines = new Vector<String>();

		try {

			Scanner fileScanner = new Scanner(new File(filename));

			// Read in the file line by line.
			fileScanner.useDelimiter("\r\n|\n");

			// Get first line.
			String l1 = fileScanner.next();
			//System.out.println(l1);

			// Store all the lines in a vector.
			while(fileScanner.hasNext()) {

				// ignore empty lines - avoids problems with trailing newlines

				String next = fileScanner.next();

				if (next.length() > 0) {
					lines.add(next);
				}

			}
			fileScanner.close();

		} catch(IOException e) {
			throw new LoggedException(e);
		}

		// Create the pulse-sequence parameter arrays.
		numMeas = lines.size();
		gDir = new double[numMeas][3];
		modQ = new double[numMeas];
		modG = new double[numMeas];
		bigDel = new double[numMeas];
		smallDel = new double[numMeas];
		TE = new double[numMeas];
		zero = new boolean[numMeas];
		ignored = new boolean[numMeas];
		numZeroMeas = 0;

		// Parse the lines into the arrays.
		for(int i=0; i<numMeas; i++) {
			Scanner lineScanner = new Scanner(lines.elementAt(i));
			gDir[i][0] = lineScanner.nextDouble();
			gDir[i][1] = lineScanner.nextDouble();
			gDir[i][2] = lineScanner.nextDouble();

			// normalize grad dir if need be
			double modDir = Math.sqrt(gDir[i][0] * gDir[i][0] + gDir[i][1] * gDir[i][1] + 
					gDir[i][2] * gDir[i][2]);


			if (modDir != 0.0 && (Math.abs(1.0 - modDir) > 1E-5)) {
				for (int j = 0; j < 3; j++) {
					gDir[i][j] /= modDir;
				}
			}

			modG[i] = lineScanner.nextDouble();
			bigDel[i] = lineScanner.nextDouble();
			smallDel[i] = lineScanner.nextDouble();
			TE[i] = lineScanner.nextDouble();
			lineScanner.close();

			modQ[i] = GAMMA*smallDel[i]*modG[i];
			zero[i] = (modQ[i] == 0);
			if(zero[i]) numZeroMeas++;
			ignored[i]=false;

			//System.out.println(i + " " + gDir[i][0] + " " + gDir[i][1] + " " + gDir[i][2] + " " + modG[i] + " " + bigDel[i] + " " + smallDel[i] + " " + TE[i] + " " + modQ[i] + " " + zero[i]);
		}

		// Also need to initialize the halfP90 array, although this
		// information is not in the schemefile.  Initialize them all
		// to zero.
		halfP90 = new double[numMeas];
	}


	/**
	 * Initialize from arrays.
	 *
	 * @param gDirInit the gradient directions, each direction is stored in gDirInit[i] and should
	 * be normalized to 1.0. 
	 * @param modGInit the gradient strengths, in T / m.
	 * @param bigDelInit the diffusion times, in seconds. This should be the actual diffusion time
	 * and not the "effective diffusion time".
	 * @param smallDelInit the gradient pulse durations, in seconds.
	 * @param te_Init the echo time, in seconds.
	 */
	public SchemeV1(double[][] gDirInit, double[] modGInit, double[] bigDelInit, 
			double[] smallDelInit, double[] te_Init, double[] halfP90Init) {
		this(gDirInit,modGInit, bigDelInit, 
				smallDelInit, te_Init, halfP90Init,null);
	}
	/**
	 * Initialize from arrays.
	 *
	 * @param gDirInit the gradient directions, each direction is stored in gDirInit[i] and should
	 * be normalized to 1.0. 
	 * @param modGInit the gradient strengths, in T / m.
	 * @param bigDelInit the diffusion times, in seconds. This should be the actual diffusion time
	 * and not the "effective diffusion time".
	 * @param smallDelInit the gradient pulse durations, in seconds.
	 * @param te_Init the echo time, in seconds.
	 */
	public SchemeV1(double[][] gDirInit, double[] modGInit, double[] bigDelInit, 
			double[] smallDelInit, double[] te_Init, double[] halfP90Init, boolean []ignoreMeasurements) {
		numMeas = gDirInit.length;

		// Create the pulse-sequence parameter arrays.
		gDir = new double[numMeas][3];
		modQ = new double[numMeas];
		modG = new double[numMeas];
		bigDel = new double[numMeas];
		smallDel = new double[numMeas];
		TE = new double[numMeas];

		halfP90 = new double[numMeas];

		zero = new boolean[numMeas];
		numZeroMeas = 0;

		ignored = new boolean[numMeas];
		numIgnoredMeasurements=0;
		for(int i = 0; i < numMeas; i++) {

			gDir[i][0] = gDirInit[i][0];
			gDir[i][1] = gDirInit[i][1];
			gDir[i][2] = gDirInit[i][2];

			modG[i] = modGInit[i];
			bigDel[i] = bigDelInit[i];
			smallDel[i] = smallDelInit[i];
			TE[i] = te_Init[i];

			halfP90[i] = halfP90Init[i];

			modQ[i] = GAMMA*smallDel[i]*modG[i];
			zero[i] = (modQ[i] == 0);
//			if(zero[i]) numZeroMeas++;
			if(ignoreMeasurements==null)
				ignored[i]=false;
			else
				ignored[i]=ignoreMeasurements[i];
			
			if(ignored[i]) 
				this.numIgnoredMeasurements++;
			else 
				if(zero[i]) 
					numZeroMeas++;

		}



	}
	public int numDWMeasurements() {
		return numMeas-numZeroMeas-numIgnoredMeasurements;
		};

	public int numMeasurements() {
		return numMeas;
	}


	public int numZeroMeasurements() {
		return numZeroMeas;
	}


	public double[] getQ(int i) {
		double[] q = new double[3];
		for(int j=0; j<3; j++) {
			q[j] = gDir[i][j]*modQ[i];
		}

		return q;
	}

	public double[] getDir(int i) {
		double[] q = new double[3];
		for(int j=0; j<3; j++) {
			q[j] = gDir[i][j];//*modQ[i];
		}

		return q;
	}

	public double getMeanNonZeroModQ() {
		double sumNZMQ = 0.0;
		for(int i=0; i<numMeas; i++) {
			if(!zero[i]) sumNZMQ += modQ[i];
		}

		return sumNZMQ/(numMeas - numZeroMeas);
	}


	/**
	 * Returns the diffusion time for one measurement.  This method
	 * assumes Gaussian displacements and rectangular pulses, so
	 * returns the effective diffusion time \Delta - \delta/3.
	 * 
	 * @param i The index of the measurement.
	 *
	 * @return The diffusion time.
	 */
	public double getDiffusionTime(int i) {
		return bigDel[i] - smallDel[i]/3.0;
	}


	/**
	 * Returns the diffusion times for all measurements.  This method
	 * assumes Gaussian displacements and rectangular pulses, so
	 * returns the effective diffusion times \Delta - \delta/3.
	 * 
	 * @return Array of diffusion times.
	 */
	public double[] getDiffusionTimes() {
		double[] diffTimes = new double[numMeas];
		for(int i=0; i<numMeas; i++) {
			diffTimes[i] = bigDel[i] - smallDel[i]/3.0;
		}
		return diffTimes;
	}


	/**
	 * Returns the diffusion times for all measurements with nonzero q. 
	 * 
	 * 
	 * @return <code>taus</code>, array of diffusion times, where <code>taus[i]</code> corresponds
	 * to element <code>i</code> in the array returned from <code>normalizeData</code>.
	 */
	public double[] getNonZeroQ_DiffusionTimes() {

		double[] diffTimes = new double[numMeas - numZeroMeas - numIgnoredMeasurements];

		int nextInd = 0;

		for(int i=0; i<numMeas; i++) {
			if(!zero[i] && !ignored[i]) {
				diffTimes[nextInd] = bigDel[i] - smallDel[i] / 3.0;
				nextInd += 1;
			}

		}

		return diffTimes;


	}



	/**
	 * Sets the diffusion time for one measurement.  This method
	 * should be deprecated and replaced with sets for delta and
	 * DELTA.  For now it prints an error and does nothing.
	 * 
	 * @param diffusionTime The new diffusion time.
	 * 
	 * @param i The index of the measurement.
	 */
	public void setDiffusionTime(double diffusionTime, int i) {
		logger.warning("Cannot set diffusion time in Scheme version 1.  Nothing changed.");
	}


	public double[][] getNormNonZeroQs() {
		double[][] nzq = new double[numMeas - numZeroMeas - numIgnoredMeasurements][3];
		int nextInd = 0;
		for(int i=0; i<numMeas; i++) {
			if(!zero[i] && !ignored[i]) {
				for(int j=0; j<3; j++) {
					nzq[nextInd][j] = gDir[i][j];
				}
				nextInd += 1;
			}
		}

		return nzq;
	}


	public double[][] getNonZeroQs() {
		double[][] nzq = new double[numMeas - numZeroMeas - numIgnoredMeasurements][3];
		int nextInd = 0;
		for(int i=0; i<numMeas; i++) {
			if(!zero[i] && !ignored[i]) {
				for(int j=0; j<3; j++) {
					nzq[nextInd][j] = modQ[i]*gDir[i][j];
				}
				nextInd += 1;
			}
		}

		return nzq;
	}


	public double[] normalizeData(double[] data) {
		double normFac = geoMeanZeroMeas(data);
		if(normFac == 0) {
			logger.warning("Mean b=0 measurement is zero.  Cannot normalize.");
			normFac = 1.0;
		}

		double[] norm = new double[numMeas - numZeroMeas - numIgnoredMeasurements];
		int nextInd = 0;
		for(int i=0; i<numMeas; i++) {
			if(!zero[i] && !ignored[i]) {
				norm[nextInd] = data[i]/normFac;
				nextInd += 1;
			}
		}

		return norm;
	}


	public double geoMeanZeroMeas(double[] data) {

		if(numZeroMeas == 0)
			return 0.0;

		double geo = 1.0;
		for(int i=0; i<numMeas; i++) {
			if(zero[i] && !ignored[i]) {
				geo = geo*data[i];
			}
		}

		return Math.pow(geo, 1.0/(double)numZeroMeas);
	}   


	public double getDelta(int i) {
		return smallDel[i];
	}


	public void setDelta(double delta, int i) {
		smallDel[i] = delta;
	}


	public double getDELTA(int i) {
		return bigDel[i];
	}


	public void setDELTA(double delta, int i) {
		bigDel[i] = delta;
	}


	public double getModG(int i) {
		return modG[i];
	}


	public void setModG(double newModG, int i) {
		modG[i] = newModG;
	}


	public double getTE(int i) {
		return TE[i];
	}


	public void setTE(double newTE, int i) {
		TE[i] = newTE;
	}


	public double getHalfP90(int i) {
		return halfP90[i];
	}


	public void setHalfP90(double newHalfP90, int i) {
		halfP90[i] = newHalfP90;
	}


	public double getModQ(int i) {
		return modQ[i];
	}


	/** 
	 * Gets the number of measurements at the end of the sequence that
	 * are ignored in every voxel.  This method is deprecated now and
	 * should not be used, although it does still work.
	 * 
	 * @return the number of measurements
	 */
	public int getIgnore() {
		return ignoreMeasurements;
	}


	/** 
	 * Sets the number of measurements at the end of the sequence that
	 * are ignored in every voxel.  This method is deprecated now and
	 * should not be used, although it does still work.
	 * 
	 * @param i The number of measurements to ignore.
	 */
	public void setIgnore(int i) {
		ignoreMeasurements = i;
	}

	/** return a copy of the big deltas array
	 * 
	 * @return new copy of all big deltas
	 */
	public final double[] getNonZeroBigDeltas(){
		double[] retArr= new double[numMeas-numZeroMeas-numIgnoredMeasurements];

		int ind=0;

		for(int i=0; i<bigDel.length; i++){
			if(!zero[i] && !ignored[i]){
				retArr[ind++]=bigDel[i];
			}
		}

		return retArr;
	}

	/** return a copy of the small deltas array
	 * 
	 * @return new copy of all small deltas
	 */
	public final double[] getNonZeroSmallDeltas(){
		double[] retArr= new double[numMeas-numZeroMeas-numIgnoredMeasurements];

		int ind=0;

		for(int i=0; i<smallDel.length; i++){
			if(!zero[i] && !ignored[i]){
				retArr[ind++]=smallDel[i];
			}
		}

		return retArr;
	}


	/** return a copy of the modGs array
	 * 
	 * @return new copy of all modGs
	 */
	public final double[] getNonZeroModGs(){
		double[] retArr= new double[numMeas-numZeroMeas-numIgnoredMeasurements];

		int ind=0;

		for(int i=0; i<modG.length; i++){
			if(!zero[i] && !ignored[i]){
				retArr[ind++]=modG[i];
			}
		}

		return retArr;
	}


	/**
	 * returns the scheme file version number (one)
	 * 
	 * @return 1
	 */
	public final int getVersion(){
		return 1;
	}


	public Scheme getSubsetScheme(int[] indices) {

		int subset = indices.length;

		double[][] gDirSub = new double[subset][3];
		double[] modG_Sub = new double[subset];
		double[] bigDelSub = new double[subset];
		double[] smallDelSub = new double[subset];
		double[] teSub = new double[subset];
		double[] halfP90Sub = new double[subset];
		boolean[] ignoredSub = new boolean[subset];

		for (int i = 0; i < subset; i++) {
			gDirSub[i][0] = gDir[indices[i]][0];
			gDirSub[i][1] = gDir[indices[i]][1];
			gDirSub[i][2] = gDir[indices[i]][2];

			modG_Sub[i] = modG[indices[i]];
			bigDelSub[i] = bigDel[indices[i]];
			smallDelSub[i] = smallDel[indices[i]];
			teSub[i] = TE[indices[i]];
			halfP90Sub[i] = halfP90[i];
			ignoredSub[i] = ignored[indices[i]];

		}

		return new SchemeV1(gDirSub, modG_Sub, bigDelSub, smallDelSub, teSub, halfP90Sub, ignoredSub);

	}


	public String toString() {
		DecimalFormat df = new DecimalFormat("   0.000000;  -0.000000");

		StringBuffer sb = new StringBuffer();

		sb.append("VERSION: 1");
		sb.append("\n");

		for (int i = 0; i < numMeas; i++) {

			sb.append(df.format(gDir[i][0]));
			sb.append(df.format(gDir[i][1]));
			sb.append(df.format(gDir[i][2]));
			sb.append(df.format(modG[i]));
			sb.append(df.format(bigDel[i]));
			sb.append(df.format(smallDel[i]));
			sb.append(df.format(TE[i]));			
			sb.append("\n");
		}

		return sb.toString();

	}


	/**
	 * Negates the X component of the gradient directions
	 */
	public void flipX() {
		for(int i=0;i< gDir.length;i++) {
			gDir[i][0]=-1*gDir[i][0];
		}
	}

	/**
	 * Negates the Y component of the gradient directions
	 */
	public void flipY() {
		for(int i=0;i< gDir.length;i++) {
			gDir[i][1]=-1*gDir[i][1];
		}
	}

	/**
	 * Negates the Z component of the gradient directions
	 */
	public void flipZ() {
		for(int i=0;i< gDir.length;i++) {
			gDir[i][2]=-1*gDir[i][2];
		}
	}

	/**
	 * @param order the new order of the gradients
	 *
	 * Reorders the gradient directions
	 */
	public void gradOrder(int [] order){
		//expects a 3 element array consisting of 0 (x dir), 1 (y dir) and 2 (z dir)

		double tempX=0;
		double tempY=0;
		double tempZ=0;
		for(int i=0;i< gDir.length;i++) {
			tempX = gDir[i][order[0]];
			tempY = gDir[i][order[1]];
			tempZ = gDir[i][order[2]];

			gDir[i][0]=tempX;
			gDir[i][1]=tempY;
			gDir[i][2]=tempZ;
		}
	}

	public static void main(String[] args) {
		SchemeV1 s = new SchemeV1(args[0]);
	}


	public boolean isIgnored(int j) {
		// TODO Auto-generated method stub
		return ignored[j];
	}

	/**
     * Returns the number of ignored measurements in each voxel.
     * 
     * @return The number of ignored measurements in each voxel.
     */
    public int numIgnoredMeasurements() { return numIgnoredMeasurements;}
}
