package imaging;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Scanner;
import java.util.logging.Logger;

import misc.LoggedException;


/**
 * scheme file for twice- refocused pulse sequence. Constructor parses the 
 * file into the scheme object.
 * 
 * File is assumed to be of the following format
 * 	VERSION: 3
 *  g1 g2 g3 modG DELTA t_del1 del1 t_del2 del2 del3 t_del4 TE
 *  ...
 *  ...
 *  
 *  where t_delN is the time of the start of gradient pulse N
 *  
 *  note that the time of the start of the third pulse is NOT specified, 
 *  as it is assumed to start at the end of the second pulse. Also, the 
 *  duration of the fourth pulse is not specified either but calculated 
 *  so as to enforce del1 + del2 = del3 + del4. If del3 is greater than
 *  del1 + del2 an exception will be thrown.
 *  
 *  This implementation also assumes that the gradient strength in each
 *  pulse is the same, and in the same direction.
 *
 * @author matt (m.hall@cs.ucl.ac.uk)
 *
 */
public class SchemeV3 implements Scheme {

	/** logging object */
	private final Logger logger= Logger.getLogger(this.getClass().getName());
	
	/** gyromagnetic ratio for constants */
	private static final double GAMMA= SchemeV1.GAMMA;
	
	/** number of measurements */
	private final int numMeas;
	
	/** number of zero measurements */
	private final int numZeroMeas;
	
	/** normalised G vectors */
    private final double[][] gDir;

    /** mod Q in each dir */
    private final double[] modQ;
    
    /** mod G in each dir */
    private final double[] modG;

    /** big delta in each dir */
    private final double[] bigDel;
    
    /** duration of first grad block in each dir */
    private final double[] del1;
    
    /** duration of first grad block in each dir */
    private final double[] t_del1;
    
    /** duration of second grad block in each dir */
    private final double[] del2;

    /** duration of first grad block in each dir */
    private final double[] t_del2;
    
    /** duration of third grad block in each dir */
    private final double[] del3;

    /** duration of first grad block in each dir */
    private final double[] t_del3;
        
    /** duration of fourth grad block in each dir */
    private final double[] del4;

    /** duration of first grad block in each dir */
    private final double[] t_del4;
    

    
    /** echo time in each dir */
    private final double[] TE;
    
    /** flag indicating if each dir is zero or not */
    private final boolean[] zero;
	
	/** halfwidth of 90 degree pulse */
    private final double[] halfP90;
	
    
    /** number of measurements to ignore */
    private int ignoreMeasurements=0;
    
    /** 
     * initialise from scheme file. gets name of a version 2 format schemefile and
     * reads the data from there.
     * 
     * @param fname
     */
	public SchemeV3(String fname){
		
        ArrayList<String> lines = new ArrayList<String>();

        try {

            Scanner fileScanner = new Scanner(new File(fname));

            // Read in the file line by line.
            fileScanner.useDelimiter("\r\n|\n");
        
            // Get first line.
            String l1 = fileScanner.next();
            //System.out.println(l1);

            // Store all the lines in a vector.
            while(fileScanner.hasNext()) {
                lines.add(fileScanner.next());
            }
            fileScanner.close();

        } catch(IOException e) {
            throw new LoggedException(e);
        }
        
        // Create the pulse-sequence parameter arrays.
        numMeas = lines.size();
        gDir = new double[numMeas][3];
        modQ = new double[numMeas];
        modG = new double[numMeas];
        bigDel = new double[numMeas];
        
        del1 = new double[numMeas];
        t_del1 = new double[numMeas];

        del2 = new double[numMeas];
        t_del2 = new double[numMeas];
        
        del3 = new double[numMeas];
        t_del3 = new double[numMeas];
        
        del4 = new double[numMeas];
        t_del4 = new double[numMeas];
        
        
        TE = new double[numMeas];
        zero = new boolean[numMeas];
        //numZeroMeas = 0;
        int zeroCount=0;
        
        
        // Parse the lines into the arrays.
        for(int i=0; i<numMeas; i++) {
            Scanner lineScanner = new Scanner(lines.get(i));
            gDir[i][0] = lineScanner.nextDouble();
            gDir[i][1] = lineScanner.nextDouble();
            gDir[i][2] = lineScanner.nextDouble();
            modG[i] = lineScanner.nextDouble();
            bigDel[i] = lineScanner.nextDouble();
        
            t_del1[i] = lineScanner.nextDouble();
            del1[i] = lineScanner.nextDouble();
            
            t_del2[i] = lineScanner.nextDouble();
            del2[i] = lineScanner.nextDouble();
            
            t_del3[i] = t_del2[i] + del2[i];
            del3[i]= lineScanner.nextDouble();

            System.err.println(i+": gDir= ("+gDir[i][0]+","+gDir[i][1]+","+gDir[i][2]+") G="+modG[i]+" DEL="+bigDel[i]
                                + "t_del1= "+t_del1[i]+" t_del2= "+t_del2[i]+" t_del3="+t_del3[i]+" del1= "+del1[i]+" del2= "+del2[i]+" del3= "+del3[i]);
            
            
            if(del3[i]>(del1[i]+del2[i])){
            	throw new LoggedException(" third gradient pulse in direction "
            				+i+" of schemefile is longer than sum of first two.");
            }
            
            t_del4[i] = lineScanner.nextDouble();
            del4[i] = (del1[i]+del2[i])-del3[i];
            
            TE[i] = lineScanner.nextDouble();
            lineScanner.close();

            modQ[i] = GAMMA*(del1[i]+del2[i])*modG[i];
            zero[i] = (modQ[i] == 0);
            if(zero[i]) zeroCount++;

            //System.out.println(i + " " + gDir[i][0] + " " + gDir[i][1] + " " + gDir[i][2] + " " + modG[i] + " " + bigDel[i] + " " + smallDel[i] + " " + TE[i] + " " + modQ[i] + " " + zero[i]);
        }

        numZeroMeas= zeroCount;
        
        // Also need to initialize the halfP90 array, although this
        // information is not in the schemefile.  Initialize them all
        // to zero.
        halfP90 = new double[numMeas];
		
	}
	
	
	
	private SchemeV3(double[][] gDir, double[] modG, double[] bigDel, double[] del1, double[] t_del1, 
						double[] del2, double[] t_del2, double[] del3, double[] t_del4, double[] te, double[] halfP90){
		
		numMeas=gDir.length;
		
		this.gDir= gDir;
		this.modG= modG;
		this.bigDel= bigDel;
		this.del1=del1;
		this.t_del1=t_del1;
		this.del2=del2;
		this.t_del2= t_del2;
		this.del3=del3;
		this.t_del4=t_del4;
		this.TE=te;
		this.halfP90=halfP90;
		
		this.zero= new boolean[numMeas];
		
		this.t_del3= new double[numMeas];
		this.del4= new double[numMeas];
		this.modQ= new double[numMeas];
		
		int zeroCount=0;
		for(int i=0; i<numMeas; i++){
			
			this.t_del3[i]= t_del2[i] + del2[i];
			this.del4[i]= (del1[i]+del2[i]) - del3[i];
			
			this.modQ[i] = GAMMA*(del1[i]+del2[i])*modG[i];
			
			if(modG[i]==0.0){
				this.zero[i]= true;
				zeroCount++;
			}
			else{
				this.zero[i]=false;
			}
		}
		
		this.numZeroMeas= zeroCount;
	}
	
	
	
	
	/**
	 * flips the x direction of all gradients
	 */
	public void flipX() {
		for(int i=0; i<gDir.length; i++){
			gDir[i][0]*=-1.0;
		}
	}

	/**
	 * flips the y direction of all gradients
	 */
	public void flipY() {
		for(int i=0; i<gDir.length; i++){
			gDir[i][1]*=-1.0;
		}

	}

	/**
	 * flips the z-direction of all the gradients
	 */
	public void flipZ() {
		for(int i=0; i<gDir.length; i++){
			gDir[i][2]*=-1.0;
		}
	}

	
	
	/**
	 * returns the geometric mean of the unweighted measurements
	 * 
	 * @param data voxel of data
	 */
	public double geoMeanZeroMeas(double[] data) {
        if(numZeroMeas == 0)
            return 0.0;

        double geo = 1.0;
        for(int i=0; i<numMeas; i++) {
            if(zero[i]) {
                geo = geo*data[i];
            }
        }

        return Math.pow(geo, 1.0/(double)numZeroMeas);
	}

	/** 
	 * @return big delta in direction i
	 */
	public double getDELTA(int i) {
		return bigDel[i];
	}

	/**
	 * @return sum of first two block durations in direction i
	 */
	public double getDelta(int i) {
		return del1[i]+del2[i];
	}

	/**
	 * @return DELTA[i] - (del1[i] + del2[i])/3
	 */
	public double getDiffusionTime(int i) {
		return bigDel[i]- (del1[i] + del2[i])/3.0;
	}

	/**
	 * @return a new array containing all diffusion times
	 */
	public double[] getDiffusionTimes() {
		double[] diffTimes = new double[numMeas];
        for(int i=0; i<numMeas; i++) {
            diffTimes[i] = getDiffusionTime(i);
        }
        return diffTimes;
	}

	/**
	 * @return half pulse width in direction i
	 */
	public double getHalfP90(int i) {
		return halfP90[i];
	}

	/** 
     * Gets the number of measurements at the end of the sequence that
     * are ignored in every voxel.  This method is deprecated now and
     * should not be used, although it does still work.
     * 
     * @return the number of measurements
     */
    public int getIgnore() {
        return ignoreMeasurements;
    }
	
	
    public double getMeanNonZeroModQ() {
        double sumNZMQ = 0.0;
        for(int i=0; i<numMeas; i++) {
            if(!zero[i]) sumNZMQ += modQ[i];
        }

        return sumNZMQ/(numMeas - numZeroMeas);
    }
    

	/**
	 * @return gradient strength in direction i
	 */
	public double getModG(int i) {
		return modG[i];
	}

	
	/**
	 * @return mod q in direction i
	 */
	public double getModQ(int i) {
		return modQ[i];
	}

    /**
     * Returns the diffusion times for all measurements with nonzero q. 
     * 
     * 
     * @return <code>taus</code>, array of diffusion times, where <code>taus[i]</code> corresponds
     * to element <code>i</code> in the array returned from <code>normalizeData</code>.
     */
    public double[] getNonZeroQ_DiffusionTimes() {

    	double[] diffTimes = new double[numMeas - numZeroMeas];
	
		int nextInd = 0;
	
	    for(int i=0; i<numMeas; i++) {
	    	if(!zero[i]) {
	    		diffTimes[nextInd] = getDiffusionTime(i);
	    		nextInd += 1;
		    }
		}
	
	    return diffTimes;
	
    }
	

    /**
     * @return a new array containing the normalised non-zero gradient directions
     */
    public double[][] getNormNonZeroQs() {
        double[][] nzq = new double[numMeas - numZeroMeas][3];
        int nextInd = 0;
        for(int i=0; i<numMeas; i++) {
            if(!zero[i]) {
                for(int j=0; j<3; j++) {
                    nzq[nextInd][j] = gDir[i][j];
                }
                nextInd += 1;
            }
        }

        return nzq;
    }

    
    /**
     * @return new array of non-zer q-vectors
     */
    public double[][] getNonZeroQs() {
        double[][] nzq = new double[numMeas - numZeroMeas][3];
        int nextInd = 0;
        for(int i=0; i<numMeas; i++) {
            if(!zero[i]) {
                for(int j=0; j<3; j++) {
                    nzq[nextInd][j] = modQ[i]*gDir[i][j];
                }
                nextInd += 1;
            }
        }

        return nzq;
    }
    
    
    /**
     * @return new instance of i-th q-vector
     */
    public double[] getQ(int i) {
        double[] q = new double[3];
        for(int j=0; j<3; j++) {
            q[j] = gDir[i][j]*modQ[i];
        }

        return q;
    }

    public Scheme getSubsetScheme(int[] indices) {

    	int subset = indices.length;

    	double[][] gDirSub = new double[subset][3];
        double[] modG_Sub = new double[subset];
        double[] bigDelSub = new double[subset];
        double[] del1Sub = new double[subset];
        double[] tDel1Sub= new double[subset];
        double[] del2Sub= new double[subset];
        double[] tDel2Sub= new double[subset];
        double[] del3Sub= new double[subset];
        double[] tDel4Sub= new double[subset];
        double[] teSub = new double[subset];
        double[] halfP90Sub = new double[subset];

    	for (int i = 0; i < subset; i++) {
    	    gDirSub[i][0] = gDir[indices[i]][0];
    	    gDirSub[i][1] = gDir[indices[i]][1];
    	    gDirSub[i][2] = gDir[indices[i]][2];
    	    
    	    modG_Sub[i] = modG[indices[i]];
    	    bigDelSub[i] = bigDel[indices[i]];
    	    
    	    del1Sub[i]= del1[indices[i]];
    	    tDel1Sub[i]= t_del1[indices[i]];
    	    del2Sub[i]= del2[indices[i]];
    	    tDel2Sub[i]= t_del2[indices[i]];
    	    del3Sub[i]= del3[indices[i]];
    	    tDel4Sub[i]= t_del4[indices[i]];
    	    
    	    teSub[i] = TE[indices[i]];
    	    halfP90Sub[i] = halfP90[i];
    	    
    	}

    	//return new SchemeV1(gDirSub, modG_Sub, bigDelSub, smallDelSub, teSub, halfP90Sub);
    	return new SchemeV3(gDirSub, modG_Sub, bigDelSub, del1Sub, tDel1Sub, del2Sub, tDel2Sub, 
    			del3Sub, tDel4Sub, teSub, halfP90Sub);
    	
    }

	/**
	 * @return 2
	 */
	public int getVersion() {
		return 3;
	}

	
    /**
     * Reorders the gradient directions	
     * expects a 3 element array consisting of 0 (x dir), 1 (y dir) and 2 (z dir)
     * 
     * @param order the new order of the gradients
     *
     *
     */
    public void gradOrder(int [] order){

    	double tempX=0;
		double tempY=0;
		double tempZ=0;
		for(int i=0;i< gDir.length;i++) {
		    tempX = gDir[i][order[0]];
		    tempY = gDir[i][order[1]];
		    tempZ = gDir[i][order[2]];
	
		    gDir[i][0]=tempX;
		    gDir[i][1]=tempY;
		    gDir[i][2]=tempZ;
		}
    }
    
	
    public double[] normalizeData(double[] data) {
        double normFac = geoMeanZeroMeas(data);
        if(normFac == 0) {
            logger.warning("Mean b=0 measurement is zero.  Cannot normalize.");
            normFac = 1.0;
        }

        double[] norm = new double[numMeas - numZeroMeas];
        int nextInd = 0;
        for(int i=0; i<numMeas; i++) {
            if(!zero[i]) {
                norm[nextInd] = data[i]/normFac;
                nextInd += 1;
            }
        }
        
        return norm;
    }
    

	/**
	 * @return the total number of measurements (incl. b-zeros)
	 */
	public int numMeasurements() {
		return numMeas;
	}

	/**
	 * @return number of unweighted measurements
	 */
	public int numZeroMeasurements() {
		return numZeroMeas;
	}

	/**
	 * sets the big delta value in dir i
	 */
	public void setDELTA(double delta, int i) {
		bigDel[i]= delta;
	}

	/**
	 * @deprecated
	 * 
	 * sets the small delta value (i.e. del1+del2) in
	 * direction i.
	 * 
	 * THIS SHOULD NOT BE USED
	 * 
	 * @param delta value of delta
	 * @param i index of value to change
	 */
	public void setDelta(double delta, int i) {
		logger.warning("setDelta in scheme V2 called. no action taken.");
	}

	/**
	 * @deprecated
	 */
	public void setDiffusionTime(double diffusionTime, int i) {
		logger.warning("attempt to directly set diffusion time in V2 scheme. no action taken");
	}

	/**
	 * sets the half width of 90 degree pulse in direction i
	 * 
	 * @param halfP90 new value
	 * @param i index of value to change
	 * 
	 */
	public void setHalfP90(double halfP90, int i) {
		this.halfP90[i]=halfP90;

	}

    /** 
     * Sets the number of measurements at the end of the sequence that
     * are ignored in every voxel.  This method is deprecated now and
     * should not be used, although it does still work.
     * 
     * @param i The number of measurements to ignore.
     */
    public void setIgnore(int i) {
        ignoreMeasurements = i;
    }
    
	/**
	 * sets modG value in direction i
	 * 
	 * @param new value
	 * @param index of value to change
	 */
	public void setModG(double modG, int i) {
		this.modG[i]= modG;
	}

	
	/** scheme V2 specifics */
	/**
	 * @return time of start of first grad block in dirn i
	 */
	public final double getTdel1(int i){
		return t_del1[i];
	}

	/**
	 * @return time of start of second grad block in dirn i
	 */
	public final double getTdel2(int i){
		return t_del2[i];
	}

	/**
	 * @return time of start of third grad block in dirn i
	 */
	public final double getTdel3(int i){
		return t_del3[i];
	}

	/**
	 * @return time of start of fourth grad block in dirn i
	 */
	public final double getTdel4(int i){
		return t_del4[i];
	}

	/**
	 * @return duration of start of first grad block in dirn i
	 */
	public final double getDel1(int i){
		return del1[i];
	}

	/**
	 * @return duration of start of first grad block in dirn i
	 */
	public final double getDel2(int i){
		return del2[i];
	}

	/**
	 * @return duration of start of first grad block in dirn i
	 */
	public final double getDel3(int i){
		return del3[i];
	}

	/**
	 * @return duration of start of first grad block in dirn i
	 */
	public final double getDel4(int i){
		return del4[i];
	}

	public int numDWMeasurements() {return numMeasurements()-numZeroMeasurements();};
	public boolean isIgnored(int j) {return false;};
	/**
     * Returns the number of ignored measurements in each voxel.
     * 
     * @return The number of ignored measurements in each voxel.
     */
    public int numIgnoredMeasurements() { return 0;}
}
