package inverters;

import optimizers.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Marquardt fitter especially for fitting models to diffusion data.
 * 
 * <dt>Description:
 * 
 * <dd>Contains methods that initialize the sigmas in the Levenberg-Marquardt
 * fitting routine and method that allow new measurements to be specified to run
 * repeated fits without creating new objects.
 * 
 * </dl>
 * 
 * @version $Id: DiffDataFitter.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author Danny Alexander
 *  
 */
abstract public class DiffDataFitter extends MarquardtChiSqFitter {

    /**
     * The diffusion times.
     */
    protected double[] taus;

    /**
     * The number of q=0 images.
     */
    protected double M;

    /**
     * The number of parameters in the model to fit.
     */
    protected int noParams;

    /**
     * Initializes the instance variables and data structures.
     * 
     * @param indepVals
     *            The matrix of wavenumbers q without the zero wavenumbers.
     * 
     * @param depVals
     *            The normalized measurements.
     * 
     * @param diffusionTimes
     *            The array of diffusion times.
     * 
     * @param nob0s
     *            The number of q=0 measurements.
     */
    protected void initialize(double[][] indepVals, double[] depVals,
            double[] diffusionTimes, int nob0s) throws MarquardtMinimiserException {

	// index diffusion times from 1 like y and q
        taus = new double[diffusionTimes.length + 1];

	for (int i = 0; i < diffusionTimes.length; i++) {
	    taus[i+1] = diffusionTimes[i];
	}

        M = (double) nob0s;

        // Pass sampled points to initialiser, as well as the
        // number of parameters of the model.
        initData(indepVals, depVals, noParams);

        // Initialize the parameters to the starting point and
        // set the standard deviations of the measurements.
        initASigs();
    }

    /**
     * Initialises a new fitting procedure with the same independent variables
     * (wavenumbers), but a new set of measurements (dependent variables).
     * 
     * @param depVals
     *            The new set of measurements.
     */
    public void newDepVals(double[] depVals) throws MarquardtMinimiserException {
        if (depVals.length != y.length - 1) {
            System.err.println(depVals.length + " " + (y.length - 1));
            throw new MarquardtMinimiserException(
                    "New data contains the wrong number of values.");
        }
        for (int i = 0; i < ndata; i++) {
            y[i + 1] = depVals[i];
        }

        // Reinitialize the starting parameters and expected variances
        // of the measurements.
        initASigs();

    }

    /**
     * Initialises the parameter values and their standard deviations.
     */
    protected void initASigs() {

        // Initialize the parameters
        initAs();

        // Initialize the standard deviations.
        initSigs();

    }

    /**
     * Initializes the parameters to a default starting position before
     * optimization.
     */
    abstract protected void initAs();

    /**
     * Initializes the standard deviations of the samples.
     */
    protected void initSigs() {

        //Initialise standard deviations. Assuming constant
        //sd, then var(s/s0) = (sig(s)/s0)^2 (1 + (s/s0)^2/M)
        //Where M is the number of s0 acquisitions, sig(s) is
        //the standard deviation of each MR measurement. Here
        //we can ignore the constant sig(s)/s0 scaling.
        for (int i = 1; i <= ndata; i++) {
            if (M > 0) {
                sig[i] = Math.sqrt(1.0 + y[i] * y[i] / M);
            }
        }
    }

    /**
     * Returns the indexed wavenumber.
     * 
     * @param x
     *            The independent variable array.
     * 
     * @param i
     *            The index of the wavenumber required.
     * 
     * @return The wavenumber of the i-th sample.
     */
    public static double[] getQ(double[][] x, int i) {
        double[] q = new double[3];

        q[0] = x[i][0];
        q[1] = x[i][1];
        q[2] = x[i][2];

        return q;
    }

}
