package inverters;

import optimizers.*;
import misc.*;
import java.util.logging.Logger;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Non-linear least squares fitter of diffusion tensor to DW-MR data.
 * 
 * <dt>Description:
 * 
 * <dd>This class implements the abstract methods of <code>MarquardtChiSqFitter</code> to provide
 * a Levenburg-Marquardt algorithm for fitting a diffusion tensor to DW-MR
 * measurements. The fitter fits the model to the normalized data directly
 * without taking logs, so that the noise statistics are less corrupted. The
 * diffusion tensor is constrained to be positive definite by optimizing the
 * parameters of its Cholesky decomposition.
 * 
 * </dl>
 * 
 * @version $Id: DiffTensorFitter.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author Danny Alexander
 *  
 */
public class DiffTensorFitter extends TensorModelFitter {

    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.inversions.DiffTensorFitter");


    /**
     * Default constructor.
     */
    public DiffTensorFitter() {
    }

    /**
     * The constructor requires a list of independent values (indepVals, the
     * wavenumbers) and associated dependent values (depVals, the data). The
     * number of unweighted acquisitions that are made (nob0s) is required to
     * estimate the noise levels of each data item. The diffusion time used in
     * the imaging sequence is also required.
     * 
     * @param indepVals
     *            The matrix of wavenumbers q without the zero wavenumbers.
     * 
     * @param depVals
     *            The normalized measurements.
     * 
     * @param diffusionTimes
     *            The array of diffusion time.
     * 
     * @param nob0s
     *            The number of q=0 acquisitions.
     */
    public DiffTensorFitter(double[][] indepVals, double[] depVals, double[] diffusionTimes,
            int nob0s) throws MarquardtMinimiserException {
        noParams = 6;
        initialize(indepVals, depVals, diffusionTimes, nob0s);
    }

    /**
     * Sets the parameters to the starting point for the
     * optimization. We choose an isotropic diffusion tensor with
     * typical trace for brain data.
     */
    protected void initAs() {

        //Initial values of the parameters and the standard
        //deviations of each data point can be set here.

        //Initialise the off diagonal tensor elements to zero.
        a[2] = a[3] = a[5] = 0.0;

        //Initialise on diagonals to non-zero.
        double traceD = 21.0E-10;
        a[1] = Math.sqrt(traceD / 3.0);
        a[4] = Math.sqrt(traceD / 3.0);
        a[6] = Math.sqrt(traceD / 3.0);

    }

    /**
     * Sets the parameters to an alternative starting point.
     *
     * @param startDT An array with the format output by all DT
     * inversions: [exitcode lnA(0) Dxx Dxy Dxz Dyy Dyz Dzz].  The
     * optimization begins from the diffusion tensor it specifies.
     */
    public void setStartPoint(double[] dtParams) {

        DT startDT = new DT(dtParams[2], dtParams[3], dtParams[4], dtParams[5], dtParams[6], dtParams[7]);
        double[] u = getCholParams(startDT);

        // Look out for failure of the Cholesky decomposition, which can
        // happen if the linear fit has a non-positive eigenvalue.
        boolean cholFailure = false;
        for(int i=0; i<6; i++) {
            a[i+1] = u[i];
            if(u[i] == 0.0)
                cholFailure = true;
        }

        if(cholFailure) {
            // Resort to a diagonal tensor with the same abs trace.
            double traceD = dtParams[2] + dtParams[5] + dtParams[7];
            a[1] = a[4] = a[6] = Math.sqrt(Math.abs(traceD/3.0));
            a[2] = a[3] = a[5] = 0.0;
            logger.info("Chol failure.  Negative evals in linear fitted DT.  Using identity for starting point. traceD = " + traceD);
        }
    }

    /**
     * Compute the value of the model at the specified sample point with
     * parameter estimates in atry.
     * 
     * @param atry
     *            Parameter values to try.
     * 
     * @param i
     *            The index of the sample point.
     * 
     * @return The value of the model at the sample point.
     */
    protected double yfit(double[] atry, int i) {

        // Compute contribution from tensor.
        DT d = getDT_Chol(atry, 1);
        double[] q = getQ(x, i);
        double yVal = Math.exp(-taus[i] * d.contractBy(q));

        return yVal;
    }

    /**
     * Overrides the default to compute the derivatives analytically.
     */
    protected double[] dydas(double[] atry, int i) {

        double[] derivs = new double[ma + 1];

        double[] q = getQ(x, i);
        double qDotRow1 = q[0] * atry[1] + q[1] * atry[2] + q[2] * atry[3];
        double qDotRow2 = q[1] * atry[4] + q[2] * atry[5];
        double qDotRow3 = q[2] * atry[6];

        // Get the tensor from the parameter values.
        DT d = getDT_Chol(atry, 1);

        // Estimate the measurement from the tensor
        double estim = Math.exp(-taus[i] * d.contractBy(q));

        insertCholDerivs(derivs, taus[i], 1, q, qDotRow1, qDotRow2, qDotRow3, 1.0, estim);

        //Check the derivatives against numerical derivatives.
        //   	double[] numDs2 = dydasNumerical(atry, i);
        //   	System.out.println("i = " + i);
        //   	for(int j=0; j<derivs.length; j++) {
        //   	   System.out.print(derivs[j] + " " + numDs2[j] + " : ");
        //  	}
        //   	System.out.println();

        return derivs;
    }

    /**
     * Returns the diffusion tensor represented by the current
     * parameter values.
     * 
     * @return The diffusion tensor specified by the parameter settings.
     */
    public DT getDiffTensor() {
        return getDT_Chol(a, 1);
    }

}
