package inverters;

import optimizers.*;
import misc.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Non-linear least squares fitter of diffusion tensor to DW-MR data.
 * 
 * <dt>Description:
 * 
 * <dd>Similar to <code>DiffTensorFitter</code>, which uses Levenburg-Marquardt to fit the
 * diffusion tensor, but this version uses the full Hessian in the optimization.
 * 
 * </dl>
 * 
 * @version $Id: DiffTensorFitterHess.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author Danny Alexander
 *  
 */
public class DiffTensorFitterHess extends DiffTensorFitter {

    /**
     * The constructor requires a list of independent values (<code>indepVals</code>, the
     * wavenumbers) and associated dependent values (<code>depVals</code>, the data). The
     * number of unweighted acquisitions that are made (<code>nob0s</code>) is required to
     * estimate the noise levels of each data item. The diffusion time used in
     * the imaging sequence is also required.
     * 
     * @param indepVals
     *            The matrix of wavenumbers q without the zero wavenumbers.
     * 
     * @param depVals
     *            The normalized measurements.
     * 
     * @param diffusionTimes
     *            The array of diffusion time.
     * 
     * @param nob0s
     *            The number of q=0 acquisitions.
     */
    public DiffTensorFitterHess(double[][] indepVals, double[] depVals,
            double[] diffusionTimes, int nob0s) throws MarquardtMinimiserException {
        noParams = 6;
        useHessian = true;
        initialize(indepVals, depVals, diffusionTimes, nob0s);
    }

    /**
     * Overrides the default to compute the second derivatives analytically.
     */
    protected double[][] d2yda2s(double[] atry, int i) {

        double[][] derivs2 = new double[ma + 1][ma + 1];

        double[] q = getQ(x, i);

        // Get the tensor from the parameter values.
        DT d = getDT_Chol(atry, 1);

        // Estimate the measurement from the tensor
        double estim = Math.exp(-taus[i] * d.contractBy(q));

        // Some constants that appear throughout.
        double c1 = (2 * atry[1] * q[0] * atry[1] * q[0] + 2 * atry[2] * q[1] * atry[2]
                * q[1] + 4 * atry[2] * q[1] * atry[3] * q[2] + 2 * atry[3] * q[2]
                * atry[3] * q[2] + 4 * atry[1] * q[0] * (atry[2] * q[1] + atry[3] * q[2]))
                * taus[i] - 1;
        double c2 = (atry[1] * q[0] + atry[2] * q[1] + atry[3] * q[2]) * taus[i] * taus[i];
        double c3 = c2 * (atry[4] * q[1] + atry[5] * q[2]);
        double c4 = (2 * atry[4] * q[1] * atry[4] * q[1] + 4 * atry[4] * q[1] * atry[5]
                * q[2] + 2 * atry[5] * q[2] * atry[5] * q[2])
                * taus[i] - 1;
        double c5 = (atry[4] * q[1] + atry[5] * q[2]) * taus[i] * taus[i];

        derivs2[1][1] = 2 * q[0] * q[0] * taus[i] * c1 * estim;
        derivs2[1][2] = derivs2[2][1] = 2 * q[0] * q[1] * taus[i] * c1 * estim;
        derivs2[1][3] = derivs2[3][1] = 2 * q[0] * q[2] * taus[i] * c1 * estim;
        derivs2[1][4] = derivs2[4][1] = 4 * q[0] * q[1] * c3 * estim;
        derivs2[1][5] = derivs2[5][1] = 4 * q[0] * q[2] * c3 * estim;
        derivs2[1][6] = derivs2[6][1] = 4 * atry[6] * q[0] * q[2] * q[2] * c2 * estim;

        derivs2[2][2] = 2 * q[1] * q[1] * taus[i] * c1 * estim;
        derivs2[2][3] = derivs2[3][2] = 2 * q[1] * q[2] * taus[i] * c1 * estim;
        derivs2[2][4] = derivs2[4][2] = 4 * q[1] * q[1] * c3 * estim;
        derivs2[2][5] = derivs2[5][2] = 4 * q[1] * q[2] * c3 * estim;
        derivs2[2][6] = derivs2[6][2] = 4 * atry[6] * q[1] * q[2] * q[2] * c2 * estim;

        derivs2[3][3] = 2 * q[2] * q[2] * taus[i] * c1 * estim;
        derivs2[3][4] = derivs2[4][3] = 4 * q[1] * q[2] * c3 * estim;
        derivs2[3][5] = derivs2[5][3] = 4 * q[2] * q[2] * c3 * estim;
        derivs2[3][6] = derivs2[6][3] = 4 * atry[6] * q[2] * q[2] * q[2] * c2 * estim;

        derivs2[4][4] = 2 * q[1] * q[1] * taus[i] * c4 * estim;
        derivs2[4][5] = derivs2[5][4] = 2 * q[1] * q[2] * taus[i] * c4 * estim;
        derivs2[4][6] = derivs2[6][4] = 4 * atry[6] * q[1] * q[2] * q[2] * c5 * estim;

        derivs2[5][5] = 2 * q[2] * q[2] * taus[i] * c4 * estim;
        derivs2[5][6] = derivs2[6][5] = 4 * atry[6] * q[2] * q[2] * q[2] * c5 * estim;

        derivs2[6][6] = 2 * q[2] * q[2] * taus[i]
                * (-1 + 2 * atry[6] * q[2] * atry[6] * q[2] * taus[i]) * estim;

        // 	System.out.println(d);
        // 	for(int j=0; j<atry.length; j++) {
        // 	    System.out.print(atry[j] + " ");
        // 	}
        // 	System.out.println();
        // 	for(int j=0; j<q.length; j++) {
        // 	    System.out.print(q[j] + " ");
        // 	}
        // 	System.out.println();

        // 	//Check against numerical derivatives.
        //   	double[][] numDs2 = d2yda2sNumerical(atry, i);
        //    	System.out.println("i = " + i);
        //    	for(int j=1; j<derivs2.length; j++) {
        // 	    for(int k=1; k<derivs2[0].length; k++) {
        // 		System.out.print(derivs2[j][k] + " " + numDs2[j][k] + " : ");
        // 	    }
        // 	    System.out.println();
        // 	}
        // 	System.out.println();

        return derivs2;
    }

}
