package inverters;

import optimizers.*;
import misc.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Non-linear least squares fitter of diffusion tensor to DW-MR data.
 * 
 * <dt>Description:
 * 
 * <dd>Similar to <code>DiffTensorUnConFitter</code>, which uses Levenburg-Marquardt to fit the
 * diffusion tensor, but this version uses the full Hessian in the optimization.
 * 
 * </dl>
 * 
 * @version $Id: DiffTensorUnConFitterHess.java,v 1.2 2005/08/18 11:07:44
 *          ucacmgh Exp $
 * @author Danny Alexander
 *  
 */
public class DiffTensorUnConFitterHess extends DiffTensorUnConFitter {

    /**
     * The constructor requires a list of independent values (indepVals, the
     * wavenumbers) and associated dependent values (depVals, the data). The
     * number of unweighted acquisitions that are made (nob0s) is required to
     * estimate the noise levels of each data item. The diffusion time used in
     * the imaging sequence is also required.
     * 
     * @param indepVals
     *            The matrix of wavenumbers q without the zero wavenumbers.
     * 
     * @param depVals
     *            The normalized measurements.
     * 
     * @param diffusionTimes
     *            The array of diffusion times.
     * 
     * @param nob0s
     *            The number of q=0 acquisitions.
     */
    public DiffTensorUnConFitterHess(double[][] indepVals, double[] depVals,
            double[] diffusionTimes, int nob0s) throws MarquardtMinimiserException {
        noParams = 6;
        useHessian = true;
        initialize(indepVals, depVals, diffusionTimes, nob0s);
    }

    /**
     * Overrides the default to compute the second derivatives analytically.
     */
    protected double[][] d2yda2s(double[] atry, int i) {

        double[][] derivs2 = new double[ma + 1][ma + 1];

        double[] q = getQ(x, i);
        DT d = new DT(atry[1], atry[2], atry[3], atry[4], atry[5], atry[6]);
        double estim = Math.exp(-taus[i] * d.contractBy(q));

        derivs2[1][1] = taus[i] * taus[i] * q[0] * q[0] * q[0] * q[0] * estim;
        derivs2[1][2] = derivs2[2][1] = 2.0 * taus[i] * taus[i] * q[0] * q[0] * q[0] * q[1]
                * estim;
        derivs2[1][3] = derivs2[3][1] = 2.0 * taus[i] * taus[i] * q[0] * q[0] * q[0] * q[2]
                * estim;
        derivs2[1][4] = derivs2[4][1] = taus[i] * taus[i] * q[0] * q[0] * q[1] * q[1] * estim;
        derivs2[1][5] = derivs2[5][1] = 2.0 * taus[i] * taus[i] * q[0] * q[0] * q[1] * q[2]
                * estim;
        derivs2[1][6] = derivs2[6][1] = taus[i] * taus[i] * q[0] * q[0] * q[2] * q[2] * estim;

        derivs2[2][2] = 4.0 * taus[i] * taus[i] * q[0] * q[0] * q[1] * q[1] * estim;
        derivs2[2][3] = derivs2[3][2] = 4.0 * taus[i] * taus[i] * q[0] * q[0] * q[1] * q[2]
                * estim;
        derivs2[2][4] = derivs2[4][2] = 2.0 * taus[i] * taus[i] * q[0] * q[1] * q[1] * q[1]
                * estim;
        derivs2[2][5] = derivs2[5][2] = 4.0 * taus[i] * taus[i] * q[0] * q[1] * q[1] * q[2]
                * estim;
        derivs2[2][6] = derivs2[6][2] = 2.0 * taus[i] * taus[i] * q[0] * q[1] * q[2] * q[2]
                * estim;

        derivs2[3][3] = 4.0 * taus[i] * taus[i] * q[0] * q[0] * q[2] * q[2] * estim;
        derivs2[3][4] = derivs2[4][3] = 2.0 * taus[i] * taus[i] * q[0] * q[1] * q[1] * q[2]
                * estim;
        derivs2[3][5] = derivs2[5][3] = 4.0 * taus[i] * taus[i] * q[0] * q[1] * q[2] * q[2]
                * estim;
        derivs2[3][6] = derivs2[6][3] = 2.0 * taus[i] * taus[i] * q[0] * q[2] * q[2] * q[2]
                * estim;

        derivs2[4][4] = taus[i] * taus[i] * q[1] * q[1] * q[1] * q[1] * estim;
        derivs2[4][5] = derivs2[5][4] = 2.0 * taus[i] * taus[i] * q[1] * q[1] * q[1] * q[2]
                * estim;
        derivs2[4][6] = derivs2[6][4] = 2.0 * taus[i] * taus[i] * q[1] * q[1] * q[2] * q[2]
                * estim;

        derivs2[5][5] = 4.0 * taus[i] * taus[i] * q[1] * q[1] * q[2] * q[2] * estim;
        derivs2[5][6] = derivs2[6][5] = 2.0 * taus[i] * taus[i] * q[1] * q[2] * q[2] * q[2]
                * estim;

        derivs2[6][6] = taus[i] * taus[i] * q[2] * q[2] * q[2] * q[2] * estim;

        //Check against numerical derivatives.
        //    	double[][] numDs2 = d2yda2sNumerical(atry, i);
        //    	System.out.println("i = " + i);
        //    	for(int j=1; j<derivs2.length; j++) {
        // 	    for(int k=1; k<derivs2[0].length; k++) {
        // 		System.out.print(derivs2[j][k] + " " + numDs2[j][k] + " : ");
        // 	    }
        // 	    System.out.println();
        // 	}
        // 	System.out.println();

        return derivs2;
    }

}
