package inverters;

import optimizers.*;
import misc.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Non-linear least squares fitter of a three tensor model with prolate
 * axisymmetric tensors to DW-MR data.
 * 
 * <dt>Description:
 * 
 * <dd>This class implements the abstract methods of MarquardtChiSqFitter to
 * provide a Levenburg-Marquardt algorithm for fitting a three tensor model with
 * prolate axisymmetric tensors to DW-MR measurements.
 * 
 * </dl>
 * 
 * $Id: ThreeTensorAxiSymFitter.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * 
 * @author Danny Alexander
 *  
 */
public class ThreeTensorAxiSymFitter extends ThreeTensorTwoAxiSymFitter {

    /**
     * Default constructor does nothing.
     */
    public ThreeTensorAxiSymFitter() {
    }

    /**
     * The constructor requires a list of independent values (indepVals) and
     * associated dependent values (depVals) - these are the data. Together with
     * the number of unweighted acquisitions that are made (nob0s), which is
     * required to estimate the noise levels of each data item. The b-value used
     * in the imaging sequence is also required.
     * 
     * @param indepVals
     *            The matrix of wavenumbers q without the zero wavenumbers.
     * 
     * @param depVals
     *            The normalized measurements.
     * 
     * @param diffusionTimes
     *            The array of diffusion times.
     * 
     * @param nob0s
     *            The number of q=0 measurements.
     */
    public ThreeTensorAxiSymFitter(double[][] indepVals, double[] depVals,
            double[] diffusionTimes, int nob0s) throws MarquardtMinimiserException {
        noParams = 14;
        dt1StartIndex = 1;
        dt2StartIndex = 5;
        dt3StartIndex = 9;
        mixPar1Index = 13;
        mixPar2Index = 14;
        initialize(indepVals, depVals, diffusionTimes, nob0s);
    }

    /**
     * Overridden as there is no need to compute the axisymmetric diffusion
     * tensor.
     */
    protected double[] initParams3() {
        double traceD3 = 1.8E-9;
        double[] params = new double[4];
        for (int i = 0; i < 3; i++) {
            params[i] = 0.0;
        }
        params[3] = Math.sqrt(traceD3 / 3.0);

        return params;
    }

    /**
     * Overridden as there is no need to compute the axisymmetric diffusion
     * tensor.
     */
    protected double[] fAndBetaToParams3(double[] fBeta) {
        double[] params = new double[4];

        params[0] = fBeta[0];
        params[1] = fBeta[1];
        params[2] = fBeta[2];
        params[3] = Math.sqrt(fBeta[3]);

        return params;
    }

    /**
     * Overridden to replace the second DT by the axisymmetric representation.
     */
    protected void insertTensorParamDerivs(double[] derivs, double[] atry, double[] q,
            double d1Contrib, double d2Contrib, double d3Contrib, double p1, double p2, double tau) {

        double modQSq = q[0] * q[0] + q[1] * q[1] + q[2] * q[2];

        double s1 = atry[dt1StartIndex + 3];
        double qDotF1 = q[0] * atry[dt1StartIndex] + q[1] * atry[dt1StartIndex + 1]
                + q[2] * atry[dt1StartIndex + 2];

        double s2 = atry[dt2StartIndex + 3];
        double qDotF2 = q[0] * atry[dt2StartIndex] + q[1] * atry[dt2StartIndex + 1]
                + q[2] * atry[dt2StartIndex + 2];

        double s3 = atry[dt3StartIndex + 3];
        double qDotF3 = q[0] * atry[dt3StartIndex] + q[1] * atry[dt3StartIndex + 1]
                + q[2] * atry[dt3StartIndex + 2];

        insertAxiSymDerivs(derivs, tau, dt1StartIndex, q, s1 * modQSq, qDotF1, p1,
                d1Contrib);
        insertAxiSymDerivs(derivs, tau, dt2StartIndex, q, s2 * modQSq, qDotF2, p2,
                d2Contrib);
        insertAxiSymDerivs(derivs, tau, dt3StartIndex, q, s3 * modQSq, qDotF3, 1.0 - p1
                - p2, d3Contrib);

    }

    /**
     * Overridden to use the axisymmetric representation.
     */
    protected DT getDT3(double[] a) {
        return getDT_AxiSym(a, dt3StartIndex);
    }

    /**
     * Overridden to use the axisymmetric representation.
     */
    protected double[] dt3ToParams(DT dt3) {
        return getAxiSymParams(dt3);
    }

}
