/* SimulationParams.java created on 28-Nov-2005
 * (simulation)
 * 
 * author: Matt Hall (m.hall@cs.ucl.ac.uk)
 * 
 */
package simulation;

import java.util.logging.Logger;

import simulation.dynamics.StepAmenderFactory;
import simulation.dynamics.StepGeneratorFactory;
import simulation.geometry.SubstrateFactory;
import simulation.geometry.ParallelCylinderSubstrate;
import simulation.measurement.ScanFactory;
import tools.CL_Initializer;

import imaging.Scheme;

/**
 *  Camino fibre reconstruction and tracking toolkit
 * 
 * SimulationParams (simulation)
 * 
 * contains extra parameters for simulations not contained in
 * imaging params.
 * 
 * 
 *
 * @author Matt Hall (m.hall@cs.ucl.ac.uk)
 *
 */
public class SimulationParams {

    private Logger logger = Logger.getLogger(this.getClass().getName());

    
    /**
     * indicates whether to run a separate simulation for
     * each voxel (true) or generate all voxels from the 
     * same simulation
     */
    public static boolean sim_separate_runs = false;
    
    public static int sim_inflamm_increments = 10;
    
    /**
     * number of walkers in the simulation 
     */
    public static int sim_N_walkers= 10000;
    
    /**
     * number of timesteps in the diffusion simulation
     */
    public static int sim_tmax= 100000;
    
    /**
     * membrane transition probability
     */
    public static double sim_p= 0.0;
    
    /**
     * initial conditions flag
     */
    public static int sim_initial= SimulationParams.UNIFORM;
    
    /**
     * the default geometry type
     */
    public static int sim_geomType= SubstrateFactory.CYL_1_FIXED;
    
    /**
     * the substrate size
     */
    public static double sim_L=20.0;
    
    /**
     * cell size for cellular lattice
     */
    public static double sim_l=1.0;
    
    /**
     * stripe width for striped substrate
     */
    public static int sim_stripethickness=3;
    
    /**
     * percolation prob for perc lattic
     */
    public static double sim_p_perc=0.5;
    
    /**
     * fill fraction for fixed frac
     */
    public static double sim_fixedFrac=0.75;
    
    /**
     * size of filled part of the module
     */
    public static int sim_modFill=4;
    
    /**
     * size of unfilled part of module 
     */
    public static int sim_modFree=1;
    
    /**
     * the size of the central voxel
     */
    public static double sim_voxelSize= 10.0;
    
    /**
     * the type of step distibution
     */
    public static int sim_stepType= StepGeneratorFactory.FIXEDLENGTH;
    
 
    /**
     * has the simualtion set delta?
     */
    public static boolean sim_delta_set=false;
    
    /**
     * simulation delta value
     */
    public static double sim_delta;
    
    /**
     * has the simulation set DELTA?
     */
    public static boolean sim_DELTA_set=false;
    
    /**
     * simulation diffusion time
     */
    public static double sim_DELTA;
    
       
    /**
     * has the simulation set gradient strength?
     */
    public static boolean sim_G_set=false;
    
    /**
     * simulation G value
     */
    public static double sim_G;

    /**
     * cylinder packing 
     */
    public static int sim_cyl_pack = ParallelCylinderSubstrate.HEX;

    /**
     * number of cylinders along a side of square distributed 
     * radius cylinder substrate
     */
    public static int sim_cyl_dist_size=20;
    
    /**
     * distributed cylinder min radius (meters)
     */
    public static double sim_cyl_min_r=0.0;
    
    /**
     * distributed cylinder max radius (meters)
     */
    public static double sim_cyl_max_r=2E-5;
    
    /** 
     * fixed cylinder radius (meters)
     */
    public static double sim_cyl_r = 1E-5;
    
    /**
     * fixed cylinder separation (meters)
     */
    public static double sim_cyl_R = 3E-5;
    
    /** 
     * outer radius of myelinated cylinders
     */
    public static double sim_cyl_r1= sim_cyl_r/2.0;
    
    /**
     * core diffusivity of myelinated cylinders
     */
    public static double sim_cyl_D1= CL_Initializer.DIFF_CONST;
    
    /**
     * sheath diffusivity in meylinated cyliners
     */
    public static double sim_cyl_D2= CL_Initializer.DIFF_CONST;
    
    /**
     * angle between crossing fibres on crossing substrate. 
     * 90 degrees by default.
     */
    public static double sim_cAngle= Math.PI/2;
    
    /**
     * number of cylinders on substrate
     */
    public static int sim_num_cylinders=20;
    
    /**
     * number of facets on a facetted cylinder
     */
    public static int sim_num_facets=0;
    
    /**
     * name of PLY file to read
     */
    public static String sim_plyfile=null;
    

    /**
     * density of square grid used for sptial sorting
     */
    public static int sim_spatial_grid_size= 10;
    
    /**
     * type of step amender
     */
    public static int sim_amender_type=StepAmenderFactory.ELESTIC_REFLECTOR;
    
    /**
     * scan type
     */
    public static int scanType= ScanFactory.PGSE_SCAN;
    
    /** flag for delta-peaked initially posiitoned walkers */
    public static final int SPIKE=0;
    
    /** flag for uniformly distributed initial conditions */
    public static final int UNIFORM=1;
    
    /** flag for special (debug) initial conditions. one intra one extra */
    public static final int SPECIAL=2;
    
    /** flag for intracellular walkers only */
    public static final int INTRACELLULAR=3;
    
    /** flag for extracellular walkers only */
    public static final int EXTRACELLULAR=4;
    

    
    /** flag for compartmental signal: default (no compartmental segregation */
    public static final int NOCOMPS= 0;
    
    /** flag for compartmental signal: extracellular only */
    public static final int EXTRAONLY= 1;
    
    /** flag for compartmental signal: intracellular only */
    public static final int INTRAONLY= 2;
    
    /** flag for compartmental signal: output for each component */
    public static final int ALLCOMPS= 3;
    
    
    
    
    
    
    /** number of walkers */
    private final int N_walkers;
    
    /** number of timesteps (exclusing transient) */
    private final int tmax;
    
    /** the increment of time corresponding to a timestep */ 
    private final double dt;   
    
    /** the type of step distribution we'd like */
    private final int stepType;
    
    /** additional step parameters */
    private double[] stepParams;
       
    /** the size of the imaging voxel */
    private final double voxelSize;
    
    /** the membrane transition probability */
    private final double p;
    
    /** initial conditions flag */
    private final int initial;
    
    /** geometry type */
    private final int geometryType;
    
    /** extra geometry parameters */
    private final Object[] geometryParams;
    
    /** name of trajectories file */
    public static String trajFile=null;
    
    /** size of traj file output buffer (one meg by default)*/
    public int buffsize=1048576;
    
    /** duration of simulation in seconds */
    public static double duration;
    
    /** are we generating trajectories? */
    public static boolean trajectories=false;
    
    /** flag to read out cylinder info */
    public static boolean substrateInfo= false;
    
    /** only run a single iteration of the substrate? -1 if no. other number to run */
    public static int sim_onlyRun= -1;
    
    /** compartmental signal output? default no */
    public static int sim_compartmentSignal= NOCOMPS;
    
    
    public SimulationParams(int N_walkers, int tmax, double p, 
            				int initial, int geomType, Object[] geomParams, 
            				int stepType,
            				double voxelSize, Scheme imPars){

        this.N_walkers=N_walkers;
        this.tmax=tmax;
        this.p=p;
        this.geometryType=geomType;
        this.geometryParams=geomParams;
        this.stepType=stepType;
        this.voxelSize=voxelSize;
                
        /** set the echo time and dt increment properly */
        //double TE=sim_DELTA+sim_delta;
        double TE=0.0;
        for(int i=0; i<imPars.numMeasurements(); i++){
        	if(TE<imPars.getDelta(i)+imPars.getDELTA(i)){
        		TE=imPars.getDelta(i)+imPars.getDELTA(i);
        	}
        }
        double halfP90= imPars.getHalfP90(0);
        
        this.dt=(halfP90+TE)/((double)tmax);
        
        if(imPars.getDelta(0)/dt<50){
            logger.warning("WARNING: There are only "+(int)(imPars.getDelta(0)/dt)
                    + "timesteps in the gradient block. Less than 50 not recommended to"
                    + "achieve good averaging");
        }

        trajectories=false;
        
        this.initial=initial;
    }

    
    /** alternate constructor with no scheme object. this instantiates
     * a simulation that generates a trajectories file instead of scan
     * results.
     * 
     * @param N_walkers
     * @param tmax
     * @param p
     * @param initial
     * @param geomType
     * @param geomParams
     * @param stepType
     * @param voxelSize
     * @param duration
     */
    public SimulationParams(int N_walkers, int tmax, double p, 
			int initial, int geomType, Object[] geomParams, 
			int stepType,
			double voxelSize, double duration){

    	this.N_walkers=N_walkers;
		this.tmax=tmax;
		this.p=p;
		this.geometryType=geomType;
		this.geometryParams=geomParams;
		this.stepType=stepType;
		this.voxelSize=voxelSize;
		
		//this.trajFile= new String("N="+N_walkers+"_T="+tmax+"_m"+CL_Initializer.seed+".traj");
		
		/** set the echo time and dt increment properly */
		this.dt=duration/((double)tmax);
		
		this.duration= duration;
		this.trajectories=true;
		
		this.initial=initial;
}

    
    
    
    
    
    
    /**
     * @return Returns the dt.
     */
    public double getDt() {
        return dt;
    }

    /**
     * @return Returns the n_walkers.
     */
    public int getN_walkers() {
        return N_walkers;
    }

    /**
     * @return Returns the tmax.
     */
    public int getTmax() {
        return tmax;
    }
    public static void main(String[] args) {
    }
    
    /**
     * @return Returns the membrane transition prob p.
     */
    public double getP() {
        return p;
    }
    
    /**
     * @return Returns the geometryType.
     */
    public int getGeometryType() {
        return geometryType;
    }
    
    /**
     * @return Returns the geometryParams.
     */
    public Object[] getGeometryParams() {
        return geometryParams;
    }
    /**
     * @return Returns the initial.
     */
    public int getInitialConditions() {
        return initial;
    }
    /**
     * @return Returns the stepParams.
     */
    public double[] getStepParams() {
        return stepParams;
    }
    /** sets the parameters for the step generator
     * 
     * @param steParams array of step generator parameters
     */
    public void setStepParams(double[] stepParams){
        this.stepParams=stepParams;
    }
    /**
     * @return Returns the stepType.
     */
    public int getStepType() {
        return stepType;
    }
    /**
     * @return Returns the voxelSize.
     */
    public double getVoxelSize() {
        return voxelSize;
    }
    
}
