package simulation.geometry;

import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.logging.Logger;

import misc.LoggedException;

import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import simulation.dynamics.StepGeneratorFactory;
import simulation.dynamics.Walker;

public abstract class CylinderSubstrate extends Substrate{

	
	// logging object
	Logger logger = Logger.getLogger(this.getClass().getName());

	/** size of substrate */
	protected double[] L;
	
	// the cylinders 
	protected Cylinder cylinder[];
	
	// dimensionality of space
	protected final int D= DiffusionSimulation.D;

	/** which cylinder was crossed */
	protected int[] cylCrossed;
	
	/** membrane permeability of crossed cylinder */
	protected double[] cyl_p;
	
	/** list of shortest distances to barrier crossings */
	protected double[] shortestDist;
	
	/** list of d[0] values for barrier intersections */
	protected double[] intOriginDist;
	
	/** list of intersection normals for barrier intersections */
	protected double[][] intNormal;
	
	/** were we initially inside or outside the cylinder crossed? */
	protected boolean[] intInside;

	/** which cylinder was the most recent detected intersection with? */
	private int lastCrossed=-1;	

	
		
	public CylinderSubstrate(double[] L, SimulationParams simParams){
		super(simParams, L);

		this.L=L;
	}
	
	
	

	/** 
	 * checks if a walker's step will take it across a membrane or not.
	 * this just involves checking every cylinder in turn. The cylinder
	 * crosses() method should fill in the blanks where necessary.
	 * 
	 * @param walker the walker to check
	 * @param offset offset from current walker position for beginning of step
	 * @param stepVector the step being made
	 * @param normal space to store the normal if a barrier is crossed
	 * @param d space to store barrier distance from origin dotted with normal
	 * @param skipCurrent flag indicating we're sitting on a barrier and should 
	 *        ignore the closest one.
	 * @param originLength the original length of the step vector;
	 * 
	 * @return true or false
	 */
	public boolean crossesMembrane(Walker walker, double[] offset, double[] stepVector,
			double[] normal, double[] d, boolean skipCurrent, double origLength, 
			boolean[] in, double[] p) {
		double len= 0.0;
		double[] walkerPos= new double[D];
		double[] intDist= new double[1];
		
		boolean[] intIn= new boolean[1];
		double[] intP= new double[1];
		
		for(int i=0; i<stepVector.length; i++){
			len += stepVector[i]*stepVector[i];
		}
		getSubstrateCoords(walker.r, offset, walkerPos);
		len=Math.sqrt(len);
		
		if(len/origLength<=1E-14){
			return false;
		}
			
		int numIntersections=0;
		boolean skip=skipCurrent;
		
		for(int i=0; i<cylinder.length; i++){
			
			if(skipCurrent){
				if(i==lastCrossed){
					intIn[0]=in[0];
					skip=true;
				}
				else{
					intIn[0]=cylinder[i].inside(walkerPos);
					skip= false;
				}				
			}
			
			if(cylinder[i].crosses(walkerPos, stepVector, normal, d, skip, origLength, intDist, intIn, intP)){

				shortestDist[numIntersections]=intDist[0];
				intOriginDist[numIntersections]=d[0];
				for(int j=0; j<D; j++){
					intNormal[numIntersections][j]=normal[j];
				}
				intInside[numIntersections]=intIn[0];
				cylCrossed[numIntersections]=i;

				cyl_p[numIntersections]=intP[0];
				intFactor[numIntersections]= 0.9999;
				
				numIntersections++;
				
			}
			
		}
		
		// check if we've intersected the boundaries of the substrate
		if(checkBoundaryIntersection(walkerPos, offset, stepVector, normal, d, skipCurrent, origLength, intDist, intIn, intP)){
            shortestDist[numIntersections]=intDist[0];
		    intOriginDist[numIntersections]=d[0];
		    for(int j=0; j<D; j++){
		        intNormal[numIntersections][j]=normal[j];
		    }
		    intInside[numIntersections]=intIn[0];
		    
		    // in this case we haven't crossed a cylinder, so must check them all next time
		    cylCrossed[numIntersections]=-1; 

		    cyl_p[numIntersections]=intP[0];
		    intFactor[numIntersections]= 1.0;
        
		    numIntersections++;
	    }
		
		if(numIntersections==1){
			// if there's one interesection everything is  
			// as it should be so just copy the distance
			// and normal over and return true
			d[0]=intOriginDist[0];
			for(int i=0; i<D; i++){
				normal[i]=intNormal[0][i];
			}
			in[0]=intInside[0];
			
			lastCrossed=cylCrossed[0];
			
			p[0]=cyl_p[0];
			factor= intFactor[0];
			
			return true;
		}
		
		if(numIntersections>1){
			
			// if there are more than intersections
			// we must pick the one that would happen
			// first, and set up the distance and 
			// normal accordingly
			double closestDist=Double.MAX_VALUE;
			int closest=-1;
			
			// find closest interesection to the walker
			for(int i=0; i<numIntersections; i++){
				if(shortestDist[i]<closestDist){
					closestDist=shortestDist[i];
					closest=i;
					lastCrossed=cylCrossed[i];
				}
			}
			
			if(closest==-1){
				return false;
			}
			
			// set the distance and normal
			d[0]=intOriginDist[closest];
			for(int i=0; i<D; i++){
				normal[i]=intNormal[closest][i];
			}
			in[0]=intInside[closest];
			
			p[0]=cyl_p[closest];
			factor= intFactor[closest];
			
			return true;
		}
		
		// in this case there are no intersections
		return false;
	}


	public abstract double[] getSubstrateSize();

	public abstract double getPeakCoord();

	public void init(){
		
	}

	/**
	 * change the substrate dimensions
	 * 
	 * @param L the new dimensions
	 */
	public void setSubstrateDims(double[] L){
	    this.L=L;
	}
	
	public boolean intracellular(Walker walker){
		
		for(int i=0; i<cylinder.length; i++){
			if(cylinder[i].inside(walker.r)){
				return true;
			}
		}
		
		return false;
	}

	
	final void setCylinders(Cylinder[] cylinder) {
		this.cylinder = cylinder;
		
		cylCrossed= new int[cylinder.length+1];
		cyl_p= new double[cylinder.length+1];
		shortestDist= new double[cylinder.length+1];
		intOriginDist= new double[cylinder.length+1];
		intNormal= new double[cylinder.length+1][D];
		intInside= new boolean[cylinder.length+1];
		
		intFactor= new double[cylinder.length+1];
		
		/*
		 * debug code. outputs cylinder cross sections to csv file.
		 *
		Writer cylWriter;
		int N=20;
		
		try{
			cylWriter= new FileWriter(cylinder.length+"_cyls_sq.csv");
			
			for(int i=0; i<cylinder.length; i++){
				double[] cylP= cylinder[i].getPosition();
				double cylR= cylinder[i].getRadius();
				
				for(int j=0; j<N; j++){
					double t1= (((double)j)/(double)N)*2.0*Math.PI;
					double t2= (((double)(j+1))/(double)N)*2.0*Math.PI;
					
					double x1= cylP[0]+cylR*Math.cos(t1);
					double y1= cylP[1]+cylR*Math.sin(t1);
					
					double x2= cylP[0]+cylR*Math.cos(t2);
					double y2= cylP[1]+cylR*Math.sin(t2);
					
					cylWriter.write(x1+","+y1+"\n"+x2+","+y2+"\n");
				}
			}
			cylWriter.flush();
			cylWriter.close();
		}
		catch(IOException ioe){
			throw new LoggedException(ioe);
		}*/
	}
}
