/**
 * 
 */
package simulation.geometry;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.logging.Logger;

import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import simulation.dynamics.Walker;

import numerics.Vector3D;

/**
 * @author localadmin
 *
 */
public class FacetCylinderSubstrate extends CylinderSubstrate {
	
	/** dimensionality of space */
	private final int D= DiffusionSimulation.D;
	
	/** logging object */
	private final Logger logger = Logger.getLogger(this.getClass().getName());
	
	/** array of cylinders */
	private final FacetCylinder cylinder;
	
	
	/**
	 * constructor. square-packed cylinders of given radius.
	 * 
	 * @param L dimensions of unit cell
	 * @param r radius of circle approximated by facets
	 * @param facets number of facets
	 */
	public FacetCylinderSubstrate(double L, double r, int facets, SimulationParams simParams){
		super(new double[]{L, L, L}, simParams);

		if(r>L/2.0){
			logger.warning("cylinder radius "+r+
					"is greater than separation "+(L/2.0)+
						"in Facetted Cylinder");
		}
		
		
		Vector3D V= new Vector3D(new double[]{0.0, 0.0, 1.0});
		Vector3D P= new Vector3D(new double[]{L/2.0, L/2.0, 0.0});
		
		this.cylinder= new FacetCylinder(V, P, r, facets, simParams.getP());
	}
	
	

	public boolean crossesMembrane(Walker walker, double[] offset, double[] stepVector,
			double[] normal, double[] d, boolean skipCurrent, double origLength, 
			boolean[] in, double[] p) {
		
		// return the intersection coordinates.
		double[] intDist = new double[1];
		
		double[] subsCoords= new double[D];
		
		for(int i=0; i<D; i++){
			subsCoords[i]=walker.r[i]+offset[i];
			double windingNum= Math.floor(subsCoords[i]/L[i]);
			subsCoords[i]=subsCoords[i]-windingNum*L[i];
		}
		
		boolean crosses= cylinder.crosses(subsCoords, stepVector, normal, d, skipCurrent, origLength, intDist, in, p);

		/*if(crosses){
			try{
				Writer intWriter = new BufferedWriter(new FileWriter("intPoint.csv"));
			
				intWriter.write(intDist[0]+","+intDist[1]+"\n");
			
				intWriter.flush();
				intWriter.close();
			}
			catch(IOException ioe){
				throw new RuntimeException(ioe);
			}
		}*/

		
		return crosses;
	}
	
	
	/** 
	 * @see simulation.geometry.CylinderSubstrate#getPeakCoord()
	 */
	public double getPeakCoord() {

		return L[0]/2.0;
	}

	/**
	 * @see simulation.geometry.CylinderSubstrate#getSubstrateCoords(double[], double[])
	 */
	/*protected void getSubstrateCoords(double[] walkerPos, double[] offset, double[] newPos) {
		// square packing has a square cell -- easy.
		for(int i=0; i<D; i++){
			newPos[i]= walkerPos[i]-Math.floor(walkerPos[i]/L)*L;
		}

	}*/

	/**
	 * @see simulation.geometry.CylinderSubstrate#getSubstrateSize()
	 */
	public double[] getSubstrateSize() {
		return L;
	}

	
	public boolean intracellular(Walker walker){
		
		if(cylinder.inside(walker.r)){
			return true;
		}
		
		return false;
	}

	
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// TODO Auto-generated method stub

	}

}
