/* ModCellularLattice.java created on 08-Dec-2005
 * (simulation)
 * 
 * author: Matt Hall (m.hall@cs.ucl.ac.uk)
 * 
 */
package simulation.geometry;

import java.util.logging.Logger;

import simulation.SimulationParams;

/**
 *  Camino fibre reconstruction and tracking toolkit
 * 
 * ModCellularLattice (simulation)
 * 
 * initialises a lattice of "modules" or free and occupied cells.
 * Modules are periodic blocks of cells, some of which are occupied
 * and some of which are not. They are defined by 2 parameters: modFill
 * and modFree. For example, if X denotes an occupied cell and 0 denotes 
 * a free one, then:
 * 
 * XXX00XXX00XXX00
 * XXX00XXX00XXX00
 * XXX00XXX00XXX00
 * 000000000000000
 * 000000000000000
 * XXX00XXX00XXX00
 * XXX00XXX00XXX00
 * XXX00XXX00XXX00
 * 000000000000000
 * 000000000000000
 * 
 * is a lattice made up of modules with modFill=3 and modFree=2 so
 * that each 5x5 module looks like
 * 
 * XXX00
 * XXX00
 * XXX00
 * 00000
 * 00000
 * 
 * by varying modFree and modFree the exact pattern is altered.
 *
 * @author Matt Hall (m.hall@cs.ucl.ac.uk)
 *
 */
public class ModCellularLattice extends CellularLattice {

    /** logging object */
    private Logger logger = Logger.getLogger(this.getClass().getName());
    
    /** size of filled block of cells */
    private final int modFill;
    
    /** size of gaps between blocks of cells */
    private final int modFree;
    
    /**
     * 
     * 
     * @param l cell size
     * @param L lattice size
     * @param modFill size of block of filled cells
     * @param modFree size of gaps between blocks of free cells
     */
    public ModCellularLattice(double l, int L, int modFill, int modFree, SimulationParams simParams) {
        super(l, L, simParams);
        
        this.modFill=modFill;
        this.modFree=modFree;
        
        initLattice();
    }

    /** initialises lattice of module with the specified parameters
     * @see simulation.geometry.CellularLattice#initLattice()
     */
    public void initLattice() {
        int i,j;

        int moduleSize=modFill+modFree;

        for(i=0; i<occupiedLength; i++){
          boolean filled=true;
          int index=i;


          for(j=0; j<D; j++){
            int n_j=index%L;

            if(n_j%moduleSize>=modFill){
                filled=false;
                break;
            }

            index/=L;
          }

          occupied[i]=filled;
        }
    }

    public static void main(String[] args) {
    }
}
