package tools;

import java.util.Random;
import java.util.Scanner;
import java.util.Vector;
import java.util.logging.Logger;
import java.io.*;

import apps.SphFuncPD_Stats;
import numerics.*;
import misc.*;
import imaging.*;
import data.*;
import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import simulation.dynamics.StepGeneratorFactory;
import simulation.geometry.Cylinder;
import simulation.geometry.SubstrateFactory;
import simulation.geometry.ParallelCylinderSubstrate;
import simulation.measurement.ScanFactory;
import sphfunc.*;
import inverters.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Parses command line arguments common to many of the Camino
 * applications and sets up the common objects.
 * 
 * <dt>Description:
 * 
 * <dd>Several Camino applications have common command
 * line options. This class reads them all, parses them and stored
 * them as publicly accessible structure fields.
 * 
 * The class also creates objects that are common to the Camino
 * applications using the command line options. For example, it sets up the
 * input data stream, the output stream, the Scheme object.
 * 
 * </dl>
 * 
 * @author Danny Alexander
 * @version $Id: CL_Initializer.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 *  
 */
public class CL_Initializer {

    /**
     * logging object
     */
    private static Logger logger= Logger.getLogger("camino.tools.CL_Initializer"); 
    
    /**
     * The number of values per voxel for the ball and stick model.
     */
    public static final int BALLSTICKVALSPERVOXEL = 7;

    /**
     * The number of values per voxel in single tensor data.
     */
    public static final int ONETENVALSPERVOXEL = DT_Inversion.ITEMSPERVOX;

    /**
     * The number of values per voxel in two-tensor data.
     */
    public static final int TWOTENVALSPERVOXEL = TwoTensorInversion.ITEMSPERVOX;

    /**
     * The number of values per voxel in three-tensor data.
     */
    public static final int THREETENVALSPERVOXEL = ThreeTensorInversion.ITEMSPERVOX;

    /**
     * Standard test function index.
     */
    public static int testFunction = -1;

    /**
     * Indicates whether or not the user specified a non-standard test function.
     */
    public static boolean nonStandardTestFunction = false;

    /**
     * Indicates whether to synthesise data using simulation
     */
    public static boolean brownianSimulation = false;
    
    /**
     * diffusion simulation parameters object
     */
    public static SimulationParams simParams= null;
    
    /**
     * simulation diffusivity
     */
    public static double DIFF_CONST=2.02E-9; // m^2 s^{-1}
    
    /**
     * volume fraction
     */
    public static double volFrac = 1;

    /**
     * Dimensions of voxel grid
     */
    public static int[] dataDims = {0,0,0};

    /**
     * Voxel size in millimeters
     */
    public static double[] voxelDims = {0,0,0};

    /**
     * distance of centre of circle from centre of voxel grid
     */
    public static double centreDist=0;

    /**
     * If a non-standard test function was specified, the parameters are read
     * from the command line into this array.
     */
    public static double[] testFunctionData = {};

    /**
     * Index specifying the inversion.
     */
    public static ModelIndex[] inversionIndices = new ModelIndex[] {ModelIndex.LDT};

    /**
     * Index of the random rotation index to apply to the test function.
     */
    public static int rotationIndex = 0;

    /**
     * The number of voxels to process.
     */
    public static int numVoxels = 0;

    /**
     * The index of the scanning sequence.
     */
    public static int sequenceIndex = 0;


    /** Flag to see if someone has used -bmx. Forgetting the scheme file can cause silent errors. */
    public static boolean sequenceIndexSet = false;

    /**
     * Alternative setting for the diffusion time. If left negative, defaults
     * are used.
     */
    public static double diffusionTime = -1;

    /**
     * Alternative setting for lambda1 in the standard test functions. This can
     * be used to change the anisotropy. If left negative, defaults are used.
     */
    public static double lambda1 = -1;

    /**
     * Specifies the angle of rotation for StandardTestFunctions.dt2. If left
     * zero, dt2 is not rotated.
     */
    public static double dt2rotangle = 0;

    /**
     * Specifies the mixing parameter for StandardTestFunctions.dt2. If left
     * negative, the mixing parameter stays at 0.5.
     */
    public static double dt2mix = -1;

    /**
     * Alternative setting for scaling the standard test functions. If left
     * negative the default scale is used.
     */
    public static double scale = -1;

    // The parameters of a standard sequence with fixed |q| and
    // points from an electrostatic point set.

    /**
     * number of elements in a voxel
     */
    public static int numElements=-1;
    
    /**
     * The number of zero measurements.
     */
    public static int M = -1;

    /**
     * The number of non-zero measurements.
     */
    public static int N = -1;

    /**
     * The fixed |q|.
     */
    public static double modQ = -1;


    /**
     * Scale |q| by this factor.
     *
     */
    public static double qScale = 1.0;


    /**
     * Scale tau by this factor.
     *
     */
    public static double tauScale = 1.0;

    /**
     * The signal to noise ratio in synthetic data.
     */
    public static double SNR = -1;

    /**
     * The number of bootstraps used in bootstrapping. The
     * bootstrapper is only used if bootstrap > 0. If bootstrap == 1,
     * the wild bootstrapper is used.
     */
    public static int bootstrap = -1;

    
    /**
     * Model used in wild bootstrapping.
     *
     */
    public static String wildBootstrapModel = null;


    /**
     * List of files containing repeated measurements of the same data for use in 
     * bootstrapping.
     *
     */
    public static String[] bsDataFiles = null;

    /**
     * Random number generator seed.
     */
    public static int seed = 36558013;

    /**
     * Name of file containing voxel classification.  Used in classified
     * model fit.
     */
    public static String voxelClassMap = null;

    /**
     * Maximum number of components in classified model fitting.
     */
    public static int maxTensorComponents = 2;

    /**
     * shape parameter for gamma distributed random numbers
     */
    public static double gamma_k=1.84037;
    
    /**
     * scale parameter for gamma distribution
     */
    public static double gamma_beta= 7.8E-7;
    
    /**
     * Array of indices of the models in classified model fitting.  The
     * default expects the output of VoxelClassify and does linear DT
     * fitting where SH order is 0 or 2 and full two-tensor fitting
     * where SH order is 4.
     */
    public static ModelIndex[][] classifiedModelIndices = new ModelIndex[][]
    {{ModelIndex.LDT}, {ModelIndex.LDT}, {ModelIndex.LDT}, 
     {ModelIndex.POSPOS, ModelIndex.LDT}};

    /**
     * Name of the input data file.
     */
    public static String inputFile = null;

    /**
     * Name of the background mask file.
     */
    public static String bgMaskFile = null;


    /**
     * Name of the imaging scheme file.
     */
    public static String schemeFile = null;

    /**
     * The expected standard deviation of the noise in the data.
     */
    public static double sigma = -1;

    /**
     * Specifies the type of noise to add.
     */
    public static String noiseType = "rician";

    /**
     * Name of the outlier map for RESTORE.
     */
    public static String outlierMapFile = null;


    /**
     * Name of the noise variance map file for inversions that compute it.
     *
     */
    public static String noiseVarianceMapFile = null;

    /**
     * Name of the residual vector map file for inversions that compute it.
     *
     */
    public static String residualVectorMapFile = null;


    /**
     * Name of the matrix file for linear reconstructions.
     */
    public static String matrixFile;

    /**
     * Specifies whether or not the data should be normalized before linear
     * reconstruction.
     */
    public static boolean lrNormalize = false;

    /**
     * Specifies whether or not the linear reconstruction should use the log
     * measurements.
     */
    public static boolean lrLog = false;

    /**
     * The type of the input data
     */
    public static String inputDataType = "float";

    /**
     * The type of the background mask
     */
    public static String bgMaskType = "short";

    /**
     * The type of model represented by the input data. Normally this is null,
     * but synthetic data can be generated from model parameters on the input.
     */
    public static String inputModel = null;

    /**
     * The type of basis function used for linear reconstructions.  The default
     * setting is rbf.
     */
    public static int basisType = BasisSumFactory.TUCH_RBF;

    /**
     * Index of the ISCodes point set to use for computing spherical
     * integrals.
     */
    public static int pointSetInd = 3;

    /**
     * Flag indicating that the IS codes point set index was set.
     */
    public static boolean pointSetIndSet = false;

    /**
     * Number of principal directions in the input or output stream.
     */
    public static int numPDsIO = 3;

    /**
     * The maximum order in the fitted spherical harmonic series.
     */
    public static int maxOrder = 4;

    /**
     * The f-test thresholds. If they are left at these default values, the
     * program does not do the thresholding, but outputs the f-statistic
     * probabilities instead.
     */
    public static double f1 = -1;

    public static double f2 = -1;

    public static double f3 = -1;

    /**
     * Specification of the maximum entropy deconvolution filter. The default
     * filter is spike deconvolution with bd = 1.0.
     */
    public static double[] kernelParams = { 0.0, 1.0 };

    /**
     * Specifies the set of reconstruction directions for maximum entropy
     * deconvolution if it is different from the direction set specified in the
     * scheme file.
     */
    public static int mePointSet = -1;

    /**
     * if set to any integer other than -1,
     * this variable causes every nth gradient acquisition direction
     * to be excluded from the reconstruction parameter set
     */
    public static int skipEvery = -1;
    
    /**
     * Threshold on the q=0 measurements to detect background.
     */
    public static double BACKGROUNDTHRESHOLD = 0.0;

    /**
     * Threshold on the q=0 measurements to detect CSF.
     */
    public static double CSFTHRESHOLD = -1.0;

    /**
     * The parameters of the imaging sequence corresponding to the input data.
     */
    public static Scheme imPars = null;

    /**
     * The test function to use for simulations.
     */
    public static ModelPDF p = null;
    
    
    /**
     * The data source to process.
     */
    public static DataSource data = null;

    /**
     * The background mask.
     */
    public static DataSource bgMask = null;

    /**
     * Flags to indicate parsing of arguments
     */
    private static boolean[] parsed = null;


    /**
     * Name of a file containing an affine image transformation.
     */
    public static String transformFile = null;

    /**
     * Names of files containing an image warp.
     */
    public static String transformFileX = null;
    public static String transformFileY = null;
    public static String transformFileZ = null;

    /**
     * Name of the reorientation strategy to use.
     */
    public static String reorientation = "ppd";


    /**
     * Empty constructor.
     */
    private CL_Initializer() {

    }

    /**
     * The constructor requires only the array of command line arguments.
     * 
     * @param args
     *            The array of command line arguments to a DiffusionInversions
     *            application.
     */
    public static void CL_init(String[] args) {

        parsed = new boolean[args.length];

        for (int i = 0; i < parsed.length; i++) {
            parsed[i] = false;
        }

        boolean numPDsSet = false;
        boolean maxTensorComponentsSet = false;

        // Loop over the command line arguments and
        // store the required values in the appropriate variables.
        for (int i = 0; i < args.length; i++) {

            if(args[i].equals("-argfile")) try {

                // Read extra arguments from argfile and augment args array.
                args = augmentArgs(args, args[i+1]);

                // Also need to augment the parsed array.
                boolean[] parsedNew = new boolean[args.length];
                for(int j=0; j<i; j++) {
                    parsedNew[j] = parsed[j];
                }
                for(int j=i; j<parsedNew.length; j++) {
                    parsedNew[j] = false;
                }

                parsed = parsedNew;

                // Mark the argfile arguments as parsed.
                markAsParsed(i);
                markAsParsed(i+1);

            } catch(Exception e) {
                throw new LoggedException(e.toString() + "\nAttempting to continue...");
            }

            if (args[i].equals("-testfunc")) {
                // parse the arguments
                testFunction = Integer.parseInt(args[i + 1]);

                // flag and value both processed
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-gaussmix")) {
                nonStandardTestFunction = true;
                int components = Integer.parseInt(args[i + 1]);
                testFunctionData = new double[1 + components * 7];
                testFunctionData[0] = (double) components;

                // flag parsed
                markAsParsed(i);
                // components flag parsed
                markAsParsed(i + 1);

                for (int j = 0; j < components * 7; j++)
                    try {
                        testFunctionData[j + 1] = Double.parseDouble(args[i + j + 2]);
                        parsed[i + j + 2] = true;
                    }
                    catch (Exception e) {
                       
                        throw new LoggedException(
                                "Enter {Dxx, Dxy, Dxz, Dyy, Dyz, Dzz, mix} for each component diffusion tensor.");
                    }
            }
            if(args[i].equals("-delta")){
                throw new LoggedException("-delta option disabled.  Needs fixing.");
                //SchemeV0.setDelta(Double.parseDouble(args[i+1]));
                //markAsParsed(i, 2);
            }
            if (args[i].equals("-rotation")) {
                rotationIndex = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-inversion")) {
		inversionIndices = ModelIndex.getIndices(new String[] {args[i+1]});
                markAsParsed(i);
                markAsParsed(i + 1);
	    }
            if (args[i].equals("-model")) {
		String[] tmp = new String[1000];
                int t = 0;

                while (t + i + 1 < args.length && !args[i+t+1].startsWith("-")) {
		    tmp[t] = args[i+t+1];
		    t++;
                }

		String[] inversionStrings = new String[t];

		System.arraycopy(tmp, 0, inversionStrings, 0, t);

		inversionIndices = ModelIndex.getIndices(inversionStrings);
		
		markAsParsed(i, t+1);
		

            }
            if (args[i].equals("-bootstrap")) {
                bootstrap = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-seed")) {
                seed = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-voxels")) {
                numVoxels = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-bmx")) {
                sequenceIndex = Integer.parseInt(args[i + 1]);
		sequenceIndexSet = true;
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-voxelelements")){
            	numElements= Integer.parseInt(args[i+1]);
            	markAsParsed(i, 2);
            }
            if (args[i].equals("-fixedmodq")) {
                M = Integer.parseInt(args[i + 1]);
                N = Integer.parseInt(args[i + 2]);
                modQ = Double.parseDouble(args[i + 3]);
                diffusionTime = Double.parseDouble(args[i + 4]);
                markAsParsed(i);
                markAsParsed(i + 1);
                markAsParsed(i + 2);
                markAsParsed(i + 3);
                markAsParsed(i + 4);
            }
            if (args[i].equals("-qscale")) {
                qScale = Double.parseDouble(args[i + 1]);
                markAsParsed(i,2);
            }
            if (args[i].equals("-tau")) {
                diffusionTime = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-tauscale")) {
                tauScale = Double.parseDouble(args[i + 1]);
                markAsParsed(i,2);
            }
            if (args[i].equals("-lambda1")) {
                lambda1 = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-scale")) {
                scale = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-dt2rotangle")) {
                dt2rotangle = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-dt2mix")) {
                dt2mix = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-snr")) {
                SNR = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-outputfile")) {
                OutputManager.outputFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-gzip")) {
                OutputManager.gzipOut = true;
                markAsParsed(i);
            }
            if (args[i].equals("-inputfile")) {
                inputFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);

            }
            if (args[i].equals("-bsdatafiles")) {
		String[] tmp = new String[1000];
                int t = 0;
                while (t + i + 1 < args.length && !args[t+i+1].startsWith("-")) {
                    tmp[t] = args[i+t+1];
                    t++;
                }
		bsDataFiles = new String[t];

		bootstrap = t;
		
		System.arraycopy(tmp, 0, bsDataFiles, 0, t);
		markAsParsed(i, t+1);

            }
	    if (args[i].equals("-wildbsmodel")) {
                wildBootstrapModel = args[i + 1];
		bootstrap = 1;
                markAsParsed(i,2);
            }
            if (args[i].equals("-bgmask")) {
                bgMaskFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);

            }
            if (args[i].equals("-schemefile")) {
                schemeFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-voxclassmap")) {
                voxelClassMap = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-outliermap")) {
                outlierMapFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-noisemap")) {
                noiseVarianceMapFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-residualmap")) {
                residualVectorMapFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-matrix")) {
                matrixFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-normalize")) {
                lrNormalize = true;
                markAsParsed(i);
            }
            if (args[i].equals("-log")) {
                lrLog = true;
                markAsParsed(i);
            }
            if (args[i].equals("-sigma")) {
                sigma = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-noisetype")) {
		// better to do lower case conversion here than at every call to
		// nextVoxel() in the DataSynthesizer
                noiseType = args[i + 1].toLowerCase();
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-inputdatatype")) {
                inputDataType = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-outputdatatype")) {
                OutputManager.outputDataType = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-maskdatatype")) {
                bgMaskType = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-inputmodel")) {
                inputModel = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
		    if (args[i].equals("-basistype")) {
		    	if(args[i + 1].equals("rbf")) {
		    		basisType = BasisSumFactory.TUCH_RBF;
			}
			else if(args[i + 1].equals("sh")) {
			    basisType = BasisSumFactory.SPHERICAL_HARMONICS;
			}
			else {
                    throw new LoggedException("Unrecognized basis type: " + args[i + 1]);
            }
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-order")) {
                maxOrder = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-pointset")) {
                pointSetInd = Integer.parseInt(args[i + 1]);
                pointSetIndSet = true;
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-maxcomponents")) {
                maxTensorComponents = Integer.parseInt(args[i + 1]);
		maxTensorComponentsSet = true;
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-classifiedmodels")) {
                int numModels = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);

                String[] inversionStrings = new String[numModels];
		for(int j=0; j<numModels; j++) {
		    inversionStrings[j] = args[i+2+j];
		    markAsParsed(i+2+j);
		}

		classifiedModelIndices = ModelIndex.getClassifiedModelIndices(inversionStrings);
		
            }
            if (args[i].equals("-rbfpointset")) {
                RBF_Sum.setPoints(SphericalPoints.getElecPointSet(Integer
                        .parseInt(args[i + 1])));
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-rbfsigma")) {
                TuchRBF_Sum.setSigma(Double.parseDouble(args[i + 1]));
                ;
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-filter")) {
                markAsParsed(i);
                if (args[i + 1].equals("SPIKE")) {
                    kernelParams = new double[2];
                    kernelParams[0] = 0.0;
                    kernelParams[1] = Double.parseDouble(args[i + 2]);
                    markAsParsed(i + 1);
                    markAsParsed(i + 2);
                }
                else if (args[i + 1].equals("PAS")) {
                    kernelParams = new double[2];
                    kernelParams[0] = 1.0;
                    kernelParams[1] = Double.parseDouble(args[i + 2]);
                    markAsParsed(i + 1);
                    markAsParsed(i + 2);
                }
                else if (args[i + 1].equals("TENSOR")) {
                    kernelParams[0] = 2.0;
                    kernelParams[1] = Double.parseDouble(args[i+2]);
                    markAsParsed(i + 1);
                    markAsParsed(i + 2);
                }
                else {
                    throw new LoggedException("Unrecognized filter type: " + args[i + 1]);
                }
            }
            if (args[i].equals("-mepointset")) {
                mePointSet = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-ftest")) {
                f1 = Double.parseDouble(args[i + 1]);
                f2 = Double.parseDouble(args[i + 2]);
                f3 = Double.parseDouble(args[i + 3]);
                markAsParsed(i);
                markAsParsed(i + 1);
                markAsParsed(i + 2);
                markAsParsed(i + 3);
            }
            if (args[i].equals("-bgthresh")) {
                BACKGROUNDTHRESHOLD = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-csfthresh")) {
                CSFTHRESHOLD = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }

            // SphFuncPD_stats parsing
            if (args[i].equals("-numpds")) {
                numPDsIO = Integer.parseInt(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
                numPDsSet = true;
            }
            if (args[i].equals("-density")) {
                SphFuncPD_Stats.setDensity(Integer.parseInt(args[i + 1]));
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-searchradius")) {
                SphericalFunction.setSearchRadius(Double.parseDouble(args[i + 1]));
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-paramscale")) {
                SphericalFunction.SFPD_PARAMSCALE = Double.parseDouble(args[i + 1]);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-meanthreshscale")) {
                SphFuncPD_Stats.setPdThresh(Double.parseDouble(args[i + 1]));
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-stdthresh")) {
                SphFuncPD_Stats.setStdsFromMeanF(Double.parseDouble(args[i + 1]));
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-nooutput")) {
                SphFuncPD_Stats.setNoOutput(true);
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-v")) {
                DB.setVerbosity(Integer.parseInt(args[i + 1]));
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-trans")) {
                transformFile = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-xtrans")) {
                transformFileX = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-ytrans")) {
                transformFileY = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-ztrans")) {
                transformFileZ = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }
            if (args[i].equals("-reorientation")) {
                reorientation = args[i + 1];
                markAsParsed(i);
                markAsParsed(i + 1);
            }

            if(args[i].equals("-gamma")){
            	gamma_k=Double.parseDouble(args[i+1]);
            	gamma_beta= Double.parseDouble(args[i+2]);
            	markAsParsed(i,3);
            }
            
           if(args[i].equals("-walkers")){
                SimulationParams.sim_N_walkers= Integer.parseInt(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-tmax")){
                SimulationParams.sim_tmax= Integer.parseInt(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-p")){
                SimulationParams.sim_p=Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-G")){
                SimulationParams.sim_G=Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                SimulationParams.sim_G_set=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-del")){
                SimulationParams.sim_delta=Double.parseDouble(args[i+1]);
                SimulationParams.sim_delta_set=true;
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-DEL")){
                SimulationParams.sim_DELTA=Double.parseDouble(args[i+1]);
                SimulationParams.sim_DELTA_set=true;
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-diffusivity")){
                DIFF_CONST=Double.parseDouble(args[i+1]);
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-D1")){
            	SimulationParams.sim_cyl_D1= Double.parseDouble(args[i+1]);
            	brownianSimulation=true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-D2")){
            	SimulationParams.sim_cyl_D2= Double.parseDouble(args[i+1]);
            	brownianSimulation=true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equalsIgnoreCase("-crossangle")){
                SimulationParams.sim_cAngle =Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
           
            else if(args[i].equalsIgnoreCase("-compartmentsignal")){
                if(args[i+1].equalsIgnoreCase("EXTRAONLY")){
                    SimulationParams.sim_compartmentSignal= SimulationParams.EXTRAONLY;
                    brownianSimulation=true;
                    markAsParsed(i, 2);
                }
                if(args[i+1].equalsIgnoreCase("INTRAONLY")){
                    SimulationParams.sim_compartmentSignal= SimulationParams.INTRAONLY;
                    brownianSimulation=true;
                    markAsParsed(i, 2);
                }
                if(args[i+1].equalsIgnoreCase("ALL")){
                    SimulationParams.sim_compartmentSignal= SimulationParams.ALLCOMPS;
                    brownianSimulation=true;
                    markAsParsed(i, 2);
                }
            }
	   else if(args[i].equals("-volfrac")){
                volFrac=Double.parseDouble(args[i+1]);
                markAsParsed(i, 2);
            }
	   else if(args[i].equals("-gridsize") || args[i].equals("-datadims")){
                dataDims[0]=Integer.parseInt(args[i+1]);
                dataDims[1]=Integer.parseInt(args[i+2]);
                dataDims[2]=Integer.parseInt(args[i+3]);
                markAsParsed(i, 4);
            }
	   else if(args[i].equals("-voxeldims")){
                voxelDims[0]=Double.parseDouble(args[i+1]);
                voxelDims[1]=Double.parseDouble(args[i+2]);
                voxelDims[2]=Double.parseDouble(args[i+3]);
                markAsParsed(i, 4);

            }
	   else if(args[i].equals("-centredist")){
                centreDist=Double.parseDouble(args[i+1]);
                markAsParsed(i, 2);
            }
           else if(args[i].equals("-initial")){
                if(args[i+1].equalsIgnoreCase("uniform")){
                    SimulationParams.sim_initial= SimulationParams.UNIFORM;
                    markAsParsed(i+1);
                }
                else if((args[i+1].equalsIgnoreCase("spike"))||(args[i+1].equalsIgnoreCase("delta"))){
                    SimulationParams.sim_initial= SimulationParams.SPIKE;
                    markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("special")){
                	SimulationParams.sim_initial= SimulationParams.SPECIAL;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("intra")){
                	SimulationParams.sim_initial= SimulationParams.INTRACELLULAR;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("extra")){
                	SimulationParams.sim_initial= SimulationParams.EXTRACELLULAR;
                	markAsParsed(i+1);
                }
                else{
                    logger.warning("simulation initial conditions '"+args[i+1]+
                            "' default value (UNIFORM) will be used");
                }
                brownianSimulation=true;
                markAsParsed(i);
            }
           	else if(args[i].equalsIgnoreCase("-duration")){	
           		SimulationParams.duration=Double.parseDouble(args[i+1]);
           		SimulationParams.trajectories=true;
           		brownianSimulation=true;
           		markAsParsed(i, 2);
           	}
           	else if(args[i].equalsIgnoreCase("-trajfile")){
           		SimulationParams.trajFile=args[i+1];
           		brownianSimulation=true;
           		markAsParsed(i, 2);
           	}
            else if(args[i].equalsIgnoreCase("-separateruns")){
            	SimulationParams.sim_separate_runs=true;
            	brownianSimulation=true;
            	markAsParsed(i);
            }
            else if(args[i].equalsIgnoreCase("-facets")){
            	SimulationParams.sim_num_facets= Integer.parseInt(args[i+1]);
            	brownianSimulation=true;
            	markAsParsed(i, 2);
            }
            else if((args[i].equals("-geometry"))||(args[i].equals("-substrate"))){
                if(args[i+1].equalsIgnoreCase("cell-iso")){
                    SimulationParams.sim_geomType= SubstrateFactory.CELL_ISO;
                    markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("cell-striped")){
                    SimulationParams.sim_geomType= SubstrateFactory.CELL_STRIPED;
                    markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("cell-fixedfrac")){
                    SimulationParams.sim_geomType= SubstrateFactory.CELL_FIXEDFRAC;
                    markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("cell-modular")){
                    SimulationParams.sim_geomType= SubstrateFactory.CELL_MODULAR;
                    markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("cell-perc")){
                    SimulationParams.sim_geomType= SubstrateFactory.CELL_PERC;
                    markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("cylinder")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_1_FIXED;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("inflammation")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_1_INFLAM;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("facetted")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_1_FACET;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("ply")){
                	SimulationParams.sim_geomType=SubstrateFactory.TRI_PLY_MESH;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("percolation")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_1_PERC;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("myelinated")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_1_MYELIN;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("crossing")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_2_FIXED;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("distribcylinder")){
                	SimulationParams.sim_geomType=SubstrateFactory.CYL_1_DISTRIB;
                	markAsParsed(i+1);
                }
                else if(args[i+1].equalsIgnoreCase("empty")){
                    SimulationParams.sim_geomType=SubstrateFactory.EMPTY;
                    markAsParsed(i+1);
                }
                else{
                    logger.warning("unrecognised geometry type '"+args[i+1]+"' will use default");
                }
                brownianSimulation=true;
                markAsParsed(i);
            }
            else if(args[i].equalsIgnoreCase("-substrateinfo")){
                SimulationParams.substrateInfo= true;
                markAsParsed(i);
            }
            else if(args[i].equals("-packing")){
            	if(args[i+1].equalsIgnoreCase("square")){
            		SimulationParams.sim_cyl_pack=ParallelCylinderSubstrate.SQUARE;
            		markAsParsed(i+1);
            	}
            	else if(args[i+1].equalsIgnoreCase("hex")){
            		SimulationParams.sim_cyl_pack=ParallelCylinderSubstrate.HEX;
            		markAsParsed(i+1);
            	}
            	else{
            		logger.warning("cylinder packing '"+args[i+1]+"' not supported. will default to hexagonal packing");
            	}
            	brownianSimulation= true;
            	markAsParsed(i);
            }
            else if(args[i].equals("-numcylinders")){
            	SimulationParams.sim_cyl_dist_size=Integer.parseInt(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-cylinderminrad")){
            	SimulationParams.sim_cyl_min_r=Double.parseDouble(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-cylindermaxrad")){
            	SimulationParams.sim_cyl_max_r=Double.parseDouble(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-cylinderrad")){
            	SimulationParams.sim_cyl_r=Double.parseDouble(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-cylinderinnerrad")){
            	SimulationParams.sim_cyl_r1=Double.parseDouble(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-cylindersep")){
            	SimulationParams.sim_cyl_R=Double.parseDouble(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-increments")){
            	SimulationParams.sim_inflamm_increments= Integer.parseInt(args[i+1]);
            	brownianSimulation= true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equalsIgnoreCase("-onlyrun")){
                SimulationParams.sim_onlyRun= Integer.parseInt(args[i+1]);
                brownianSimulation= true;
                markAsParsed(i,2);
            }
            else if(args[i].equals("-voxelsize")){
                SimulationParams.sim_voxelSize= Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equalsIgnoreCase("-plyfile")){
            	SimulationParams.sim_plyfile= new String(args[i+1]);
            	String suffix=SimulationParams.sim_plyfile.substring(SimulationParams.sim_plyfile.length()-4);
            	if(!suffix.equalsIgnoreCase(".ply")){
            		SimulationParams.sim_plyfile+=".ply";
            	}
            	brownianSimulation=true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-steptype")){
                if(args[i+1].equals("fixedlength")){
                    SimulationParams.sim_stepType= StepGeneratorFactory.FIXEDLENGTH;
                    markAsParsed(i+1);
                }
                else{
                    logger.warning("step type '"+args[i+1]+"' unknown. default " +
                    		"(fixedlegnth) generator will be used");
                }
                brownianSimulation=true;
                markAsParsed(i);
            }
            else if(args[i].equals("-substratesize")){
            	SimulationParams.sim_L=Double.parseDouble(args[i+1]);
            	brownianSimulation=true;
            	markAsParsed(i, 2);
            }
            else if(args[i].equals("-latticesize")){
                SimulationParams.sim_L=Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-cellsize")){
                SimulationParams.sim_l=Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-stripethickness")){
                SimulationParams.sim_stripethickness=Integer.parseInt(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-pperc")){
                SimulationParams.sim_p_perc=Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-fixedfrac")){
                SimulationParams.sim_fixedFrac=Double.parseDouble(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-modfill")){
                SimulationParams.sim_modFill=Integer.parseInt(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
            else if(args[i].equals("-modfree")){
                SimulationParams.sim_modFree=Integer.parseInt(args[i+1]);
                brownianSimulation=true;
                markAsParsed(i, 2);
            }
        }

        // set numPDsIO

        if (!numPDsSet && inputModel != null) {
            
            if (inputModel.equals("dt")) {
                numPDsIO = 1;
            }
            else if (inputModel.equals("twotensor")) {
                numPDsIO = 2;
            }
            else if (inputModel.equals("threetensor")) {
                numPDsIO = 3;
            }
            else if (inputModel.equals("pico")) {
                numPDsIO = 1;
            }
            else if (inputModel.equals("ballstick")) {
                numPDsIO = 1;
            }
            else if (inputModel.equals("bayesdirac")) {
                numPDsIO = 1;
            }
            else if (inputModel.equals("multitensor")) {
                numPDsIO = maxTensorComponents;
            }
        }

        else if (numPDsSet && inputModel != null) {

            boolean wrongNumberOfPDs = false;

            if (inputModel.equals("dt")) {
                if (numPDsIO != 1) {
                    wrongNumberOfPDs = true;  
                }
            }
            else if (inputModel.equals("twotensor")) {
                if (numPDsIO != 2) {
                    wrongNumberOfPDs = true;  
                }
            }
            else if (inputModel.equals("threetensor")) {
                if (numPDsIO != 3) {
                    wrongNumberOfPDs = true;  
                }
            }
            else if (inputModel.equals("ballstick")) {
                if (numPDsIO != 1) {
                    wrongNumberOfPDs = true;  
                }
            }
            

            if (wrongNumberOfPDs) {
                throw new LoggedException("Wrong number of PDs (" + numPDsIO + 
                                          ") specified for input model " + inputModel);
            }
            
        }

        // now do maxTensorComponents
        if (!maxTensorComponentsSet && inputModel != null) {
            
            // (PAC) Better to tell people to use multitensor with -maxcomponents, rather than 
            // twotensor or threetensor. track and dteig no longer recommend twotensor and threetensor.

            if (inputModel.equals("dt")) {
                maxTensorComponents = 1;
            } 
            else if (inputModel.equals("twotensor")) {
                maxTensorComponents = 2; 
            } 
            else if (inputModel.equals("threetensor")) {
                maxTensorComponents = 3;
            }
        }
        else if (maxTensorComponentsSet && inputModel != null) {

            boolean wrongComponents = false;
            
            if (inputModel.equals("dt")) {
                if (maxTensorComponents != 1) {
                    wrongComponents = true;  
                }
            }
            else if (inputModel.equals("twotensor")) {
                if (maxTensorComponents != 2) {
                    wrongComponents = true;  
                }
            }
            else if (inputModel.equals("threetensor")) {
                if (maxTensorComponents != 3) {
                    wrongComponents = true;  
                }
            }
            
            
            if (wrongComponents) {
                throw new LoggedException("Wrong number of tensor components (" + maxTensorComponents + 
                                          ") specified for input model " + inputModel);
            }
            
        }

    }


    /**
     * Checks the arguments and logs any args that have not been marked as parsed. 
     */
    public static final void checkParsing(String[] args) {

        if (parsed == null) {
            RuntimeException e=new RuntimeException(
                    "attempt to check command line argument parsing made"
                    + " before parsing has been attempted!");

            StackTraceElement[] stackTrace=e.getStackTrace();
    	    String stString=new String();
    	    
    	    // log the exceptions message
    	    logger.severe(e.toString());
    	    
    	    // log the stack trace
    	    for(int ii=0; ii<stackTrace.length; ii++){
    	        stString+=stackTrace[ii]+"\n";
    	    }
    	    logger.severe(stString);

            
            throw e; 
        }

        for (int i = 0; i < parsed.length; i++) {
            if (!parsed[i]) {
                logger.warning("WARNING: couldn't parse arg " + i + ": '" + args[i]
                        + "'");
            }
        }
    }


    /**
     * Augments the command-line arguments array with arguments
     * read from a config file.
     *
     * @param args The current args array
     *
     * @param filename The name of the config file.
     *
     * @return The augmented args array.
     */
    public static String[] augmentArgs(String[] args, String filename) throws FileNotFoundException {

        // Read the file.
        Scanner s = new Scanner(new File(filename));
        Vector<String> newArgs = new Vector<String>();
        while(s.hasNext()) {
            newArgs.add(s.next());
        }
        s.close();

        // Create the augmented args array.
        String[] augArgs = new String[args.length + newArgs.size()];
        for(int i=0; i<args.length; i++) {
            augArgs[i] = args[i];
        }
        for(int i=0; i<newArgs.size(); i++) {
            augArgs[args.length + i] = newArgs.elementAt(i);
        }

        return augArgs;
    }


    /**
     * Initializes the Scheme object containing the acqusition
     * information.
     */
    public static void initImagingScheme() {

        // Sort out the parameters of the imaging sequence.
        if (M >= 0) {
            imPars = new SchemeV0(M, N, qScale * modQ, tauScale * diffusionTime);
        }
        else if (schemeFile != null) {
            int schemeVersion = getSchemeVersion(schemeFile);
            if(schemeVersion == 0) {
                imPars = new SchemeV0(schemeFile, qScale, tauScale);
            }
            else if (schemeVersion == 1) {
                imPars = new SchemeV1(schemeFile);
            }                
            else if(schemeVersion == 2){
                imPars = new B_MatrixScheme(schemeFile);
                
            }
            else if(schemeVersion == 3){
                imPars = new SchemeV3(schemeFile);
            }
            else{
            	throw new LoggedException("unsupported scheme file format "+schemeVersion);
            }
        }
        else if (sequenceIndex >= 0) {
	    if (!sequenceIndexSet) {
		logger.warning("Scheme file not specified, defaulting to sequence index 0");
	    }
	    imPars = SchemeV0.getSchemeV0(sequenceIndex, qScale, tauScale);
        }
    }


    /**
     * Determines the version of a schemefile by reading the first line.
     *
     * @param schemefile The name of the schemefile.
     *
     * @return The format version.
     */
    public static int getSchemeVersion(String schemefile) {

        FileInput f = new FileInput(schemefile);

	// trim to avoid whitespace issues
        String line1 = f.readString().trim();
    
	f.close();

        if(line1.equals("VERSION: 1")) {
            return 1;
        }

        if(line1.equals("VERSION: 2")) {
            return 2;
        }
        if(line1.equals("VERSION: 3")){
            SimulationParams.scanType= ScanFactory.TWICE_REFOCUSED;
            return 3;
        }


        return 0;
    }


    /**
     * Initializes the reconstruction direction set for maximum entropy
     * deconvolution.
     */
    public static void initMaxEnt() {

        if (mePointSet > 0) {
            // Set the reconstruction directions from the specified
            // electrostatic point set.
            MaxEntProfile.setReconDirs(SphericalPoints.getElecPointSet(mePointSet));
            
            /*double[][] reconDirs= SphericalPoints.getElecPointSet(mePointSet);
            
            // construct a rotation about the y axis by 20 degrees
            double theta =27.0*Math.PI/180.0;

            double cosTheta=Math.cos(theta);
            double sinTheta=Math.sin(theta);
                        
            // rotate each point
            for(int i=0; i<reconDirs.length; i++){
                double[] newPt=new double[3];

                newPt[0]= cosTheta*reconDirs[i][0] + sinTheta*reconDirs[i][2];
                newPt[1]= reconDirs[i][1];
                newPt[2]= -sinTheta*reconDirs[i][0] + cosTheta*reconDirs[i][2];
                
                reconDirs[i]=newPt;
            }
            
            MaxEntProfile.setReconDirs(reconDirs);*/
        }
        else {
            // Set the central points in the maximum entry profile to the
            // set of gradient directions in the schemefile.
            MaxEntProfile.setReconDirs(imPars.getNormNonZeroQs());
        }

    }

    
    
    /**
     * Specifies a specific data source object.
     *
     * @param newDataSource The data source.
     */
    public static void setDataSource(DataSource newDataSource) {
        data = newDataSource;
    }
    
    
    /**
     * Specifies a specific data source object for the background
     * mask.
     *
     * @param newMaskSource The data source.
     */
    public static void setMaskSource(DataSource newMaskSource) {
        bgMask = newMaskSource;
    }
    
    
    /**
     * Initializes an external data source for spherical function parameters.
     */
    public static void initSphFuncDataSource() {
        if (inputModel == null) {

            // If no model was specified for the input data, the input data
            // is diffusion-weighted MRI measurements.

            data = new VoxelOrderDW_DataSource(inputFile, inputDataType, imPars);
        }
        else if (inputModel.equals("dt")) {
            data = new VoxelOrderDataSource(inputFile, ONETENVALSPERVOXEL, inputDataType);
        }
        else if (inputModel.equals("ballstick")) {
            data = new VoxelOrderDataSource(inputFile, BALLSTICKVALSPERVOXEL, inputDataType);
        }
        else if (inputModel.equals("sh")) {
            data = new VoxelOrderDataSource(inputFile, SphericalHarmonics
                    .evenFuncsUpTo(maxOrder) + 2, inputDataType);
        }
        else if (inputModel.equals("rbf")) {
            data = new VoxelOrderDataSource(inputFile, RBF_Sum.numPoints() + 2, inputDataType);
        }
        else if (inputModel.equals("maxent")) {
            data = new VoxelOrderDataSource(inputFile, MaxEntProfile.numLambdas() + 2,
                    inputDataType);
        }
        else {
            RuntimeException e = new RuntimeException("Unknown input model type: " + inputModel);
            
            StackTraceElement[] stackTrace=e.getStackTrace();
    	    String stString=new String();
    	    
    	    // log the exceptions message
    	    logger.severe(e.toString());
    	    
    	    // log the stack trace
    	    for(int ii=0; ii<stackTrace.length; ii++){
    	        stString+=stackTrace[ii]+"\n";
    	    }
    	    logger.severe(stString);

    	    throw e;
        }

        initMaskSource();
    }


    /**
     * Initializes an external data source for tensor data
     */
    public static void initTensorDataSource() {

        if (inputModel.equals("dt")) {
            data = new VoxelOrderDataSource(inputFile, ONETENVALSPERVOXEL, inputDataType);
        }
        else if (inputModel.equals("twotensor")) {
            data = new VoxelOrderDataSource(inputFile, TWOTENVALSPERVOXEL, inputDataType);
        }
        else if (inputModel.equals("threetensor")) {
            data = new VoxelOrderDataSource(inputFile, THREETENVALSPERVOXEL, inputDataType);
        }
        else if (inputModel.equals("multitensor")) {
            data = new VoxelOrderDataSource(inputFile, 3+7*maxTensorComponents, inputDataType);
        }
        else {
            throw new LoggedException("Cannot process non-tensor data from model: " + inputModel);
        }

        initMaskSource();
    }

    
    /**
     * Initializes the data source for the background mask.
     */
    public static void initMaskSource() {
        if(bgMaskFile != null) {

	    if (ImageHeader.imageExists(bgMaskFile)) {

		ImageHeader ih = null;

		try {
		    ih = ImageHeader.readHeader(bgMaskFile);
		}
		catch (IOException e) {
		    throw new LoggedException("Cannot read mask header " + bgMaskFile);
		}

		bgMask = ih.getImageDataSource();
	    }
	    else { 
		bgMask = new VoxelOrderDataSource(bgMaskFile, 1, bgMaskType);
	    }
        }
    }


    /**
     * Assembles the geometry parameters array for the chosen substrate.
     * 
     */
    public static Object[] getGeometryParams(int geomType){
        if(SimulationParams.sim_geomType==SubstrateFactory.CELL_ISO){
        	// cell size, lattice size
            return new Object[] {new Double(SimulationParams.sim_l), new Integer((int)SimulationParams.sim_L)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CELL_STRIPED){
        	// cell size, lattice size, width of stripes
            return new Object[] {new Double(SimulationParams.sim_l), new Integer((int)SimulationParams.sim_L), new Integer(SimulationParams.sim_stripethickness)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CELL_PERC){
        	// cell size, lattice size, percolation prob
            return new Object[] {new Double(SimulationParams.sim_l), new Integer((int)SimulationParams.sim_L), new Double(SimulationParams.sim_p_perc)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CELL_MODULAR){
        	// cell size, lattice size, width of filled part, width of empty part
            return new Object[] {new Double(SimulationParams.sim_l), new Integer((int)SimulationParams.sim_L), new Integer(SimulationParams.sim_modFill), new Integer(SimulationParams.sim_modFree)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CELL_FIXEDFRAC){
        	// cell size, lattice size, fraction of filled cells
            return new Object[] {new Double(SimulationParams.sim_l), new Integer((int)SimulationParams.sim_L), new Double(SimulationParams.sim_fixedFrac)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_1_FIXED){
        	// cylinder sep, cylinder size, packing flag
        	return new Object[] {new Double(SimulationParams.sim_cyl_R), new Double(SimulationParams.sim_cyl_r), new Integer(SimulationParams.sim_cyl_pack)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_1_DISTRIB){
        	// substrate size, gamma k param, gamma beta param, number of cylinders
        	return new Object[] {new Double(SimulationParams.sim_L), 
        						 new Integer(SimulationParams.sim_cyl_dist_size), 
        						 new Double(gamma_k), 
        						 new Double(gamma_beta)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_1_INFLAM){
        	// substrate size, num cylinders, rmin, rmax, num steps
        	//return new Object[] {new Double(SimulationParams.sim_voxelSize), new Integer(SimulationParams.sim_cyl_dist_size), new Double(SimulationParams.sim_cyl_min_r), new Double(SimulationParams.sim_cyl_max_r), new Integer(numVoxels)};
        	return new Object[]{new Double(gamma_k), new Double(gamma_beta),
        			new Integer(SimulationParams.sim_cyl_dist_size), new Integer(numVoxels), 
        			new Integer(SimulationParams.sim_inflamm_increments), new Double(SimulationParams.sim_L)};
        	
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_1_FACET){
        	// unit cell size, cyl radius and number of facets
        	return new Object[]{new Double(SimulationParams.sim_cyl_R), new Double(SimulationParams.sim_cyl_r), new Integer(SimulationParams.sim_num_facets)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.TRI_PLY_MESH){
        	// read geometry from a PLY file. name, size and sep
        	return new Object[]{new String(SimulationParams.sim_plyfile), new Double(SimulationParams.sim_cyl_r), new Double(SimulationParams.sim_cyl_R)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_1_PERC){
        	// substrate size, num cylinders, rmin, rmax, num steps
        	//return new Object[] {new Double(SimulationParams.sim_voxelSize), new Integer(SimulationParams.sim_cyl_dist_size), new Double(SimulationParams.sim_cyl_min_r), new Double(SimulationParams.sim_cyl_max_r), new Integer(numVoxels)};
        	return new Object[]{new Double(SimulationParams.sim_cyl_min_r), new Double(SimulationParams.sim_cyl_max_r),
        			new Integer(SimulationParams.sim_cyl_dist_size), new Integer(numVoxels), 
        			new Integer(SimulationParams.sim_inflamm_increments), new Double(SimulationParams.sim_L),
        			new Integer(SimulationParams.sim_spatial_grid_size)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_2_FIXED){
        	// cylinder sep, cylinder radius (in future crossing angle)
        	return new Object[]{new Double(SimulationParams.sim_cyl_R),
        						new Double(SimulationParams.sim_cyl_r)};
        	
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.CYL_1_MYELIN){
        	
        	return new Object[]{new Double(SimulationParams.sim_cyl_R), 
        						new Double(SimulationParams.sim_cyl_r1), 
        						new Double(SimulationParams.sim_cyl_r),
        						new Double(SimulationParams.sim_cyl_D1),
        						new Double(SimulationParams.sim_cyl_D2),
        						new Double(SimulationParams.sim_p),
        						new Integer(SimulationParams.sim_cyl_pack)};
        }
        else if(SimulationParams.sim_geomType==SubstrateFactory.EMPTY){
            
            return new Object[]{};
            
        }
        else{
            logger.warning("unrecognised geometry type "+SimulationParams.sim_geomType
                    	+"returning null parameters array");
            return null;
        }
    }
    
    
 
    /**
     * Initializes the synthetic data source. Call initImagingScheme before this method.
     */
    public static void initDataSynthesizer() {

        // Set up the test function if specified.
        if (testFunction >= 0) {

            // Set up the specified options.
            if (rotationIndex > 0) {
                StandardTestFunctions.setTransformation(Rotations
                        .randomRotMat(new Random(rotationIndex)));
            }
            if (scale > 0) {
                StandardTestFunctions.setScale(scale);
            }
            if (dt2rotangle != 0.0) {
                StandardTestFunctions.setDT2RotationAngle(dt2rotangle);
            }
            if (dt2mix > 0) {
                StandardTestFunctions.setMix3(1.0 - dt2mix);
            }
            if (lambda1 > 0) {
                StandardTestFunctions.setLambda1(lambda1);
            }
            p = StandardTestFunctions.getFunction(testFunction);
        }

        // A non-standard test function may be specified on the
        // command line.
        if (nonStandardTestFunction) {
            p = constructTestFunction(testFunctionData);

        }
        
        // Create the data source.
        if (p != null) {
	    // If a test function has been specified. The data source
	    // is a data synthesizer using that function.
	    if (bootstrap > 0 && wildBootstrapModel != null) {
		
		DataSource input = getDataSynthesizer(p, imPars, SNR, 1, seed, -1);
		
		if (wildBootstrapModel.equals("dt")) {
		    data = new WildBS_DT_DataSynth(input, imPars, numVoxels, seed);
		}
		else {
		    throw new LoggedException("Wild bootstrap is not supported for model: " + 
						  wildBootstrapModel);
		}
		
	    }
	    else { 
		data = getDataSynthesizer(p, imPars, SNR, numVoxels, seed, bootstrap);
	    }

		
        }
        else if (inputModel == null) {
	    if(brownianSimulation){
		// if we've set up one or more diffusion simulation
		// parameters, then initialise the data source as a
		// difusion simulation
		double[] stepParams= null;
		Object[] geomParams= getGeometryParams(SimulationParams.sim_geomType);
		
		if(SimulationParams.trajectories){
			simParams = new SimulationParams(SimulationParams.sim_N_walkers, SimulationParams.sim_tmax, 
					 SimulationParams.sim_p, SimulationParams.sim_initial, SimulationParams.sim_geomType, geomParams, 
					 SimulationParams.sim_stepType, SimulationParams.sim_voxelSize, SimulationParams.duration);
			
		}
		else{
			simParams = new SimulationParams(SimulationParams.sim_N_walkers, SimulationParams.sim_tmax, 
						 SimulationParams.sim_p, SimulationParams.sim_initial, SimulationParams.sim_geomType, geomParams, 
						 SimulationParams.sim_stepType, SimulationParams.sim_voxelSize, imPars);
		}
		
		stepParams=StepGeneratorFactory.getStepParamsArray(SimulationParams.sim_stepType,
								   simParams, imPars);
		
		simParams.setStepParams(stepParams);
		
		if (bootstrap > 0) {
			
		    // HACK ALERT
		    // simulation automatically reads CL parameters
		    int tmpVoxels = numVoxels;

		    numVoxels = bootstrap;

		    DiffusionSimulation sim = new DiffusionSimulation(simParams, imPars); 

		    // restore value
		    numVoxels = tmpVoxels;
		
		    if (wildBootstrapModel != null) {
		    	if (wildBootstrapModel.equals("dt")) {
		    		data = new WildBS_DT_DataSynth(data, imPars, numVoxels, seed);
		    	}
		    	else {
		    		throw new LoggedException("Wild bootstrap is not supported for model: " + 
						      wildBootstrapModel);
		    	}
		    }
	
		    // bootstrap the simulation
		    data = new BootstrapDataSynthFromSimulationInput(sim,
								  bootstrap, 
								  imPars.numMeasurements(),
								  numVoxels, seed);
		}
		else {
			if(SimulationParams.trajectories){
				data = new DiffusionSimulation(simParams);
			}
			else{
				data = new DiffusionSimulation(simParams, imPars); 
			}
		}
	    }
	    else if (bootstrap > 0) {
		// bootstrap with no input model and no p
		// this implies bootstrapping of real data
		if (SNR > -1) { 
		    logger.warning("Cannot set SNR when bootstrapping raw data." + "Program will continue but will not add noise");
		}

		if (wildBootstrapModel != null) {
		    
		    data = new VoxelOrderDW_DataSource(inputFile, inputDataType, imPars);

		    if (wildBootstrapModel.equals("dt")) {
			data = new WildBS_DT_DataSynth(data, imPars, numVoxels, seed);
		    }
		    else {
			throw new LoggedException("Wild bootstrap is not supported for model: " + 
						  wildBootstrapModel);
		    }
		}
		else {
		    data = new BootstrapDataSynthesizer(bsDataFiles, imPars.numMeasurements(), 
							numVoxels, inputDataType, seed);
		}

	    }
	    else{
		// If no model was specified for the input data, the input data
		// is diffusion-weighted MRI measurements.
		
		data = new VoxelOrderDW_DataSource(inputFile, inputDataType, imPars);
		initMaskSource();
	    }
        }
        else {
	    
	    // no p, but have an input model

	    // if bootstrap then make a bootstrapper using whatever was specified as input
	    if (bootstrap > 0) {

		DataSynthFromInput input = null;		

		// If repetition bootstrapping, we set snr = -1 in the data source.
		// The bootstrap object will duplicate each voxel's noise-free data
		// and add noise itself.
		double inputSNR = -1.0;
		
		// If wild bootstrapping, we need one voxel of noisy data from the input source
		if (wildBootstrapModel != null) {
		    inputSNR = SNR;
		}
		
		if (inputModel.equals("dt")) {
		    input = new DataSynthFromDT_Input(inputFile, inputDataType, imPars, inputSNR,
						     seed);
		}
		else if (inputModel.equals("twotensor")) {
		    input = new DataSynthFromTwoTensorInput(inputFile, inputDataType, imPars,
							   inputSNR, seed);
		}
		else if (inputModel.equals("threetensor")) {
		    input = new DataSynthFromThreeTensorInput(inputFile, inputDataType,
							     imPars, inputSNR, seed);
		}
		else if (inputModel.equals("multitensor")) {
		    input = new DataSynthFromMultiTensorInput(inputFile, inputDataType,
							     imPars, maxTensorComponents, inputSNR, seed);
		}
		else if (inputModel.equals("ballstick")) {
		    input = new DataSynthFromBallStickInput(inputFile, inputDataType,
							     imPars, inputSNR, seed);
		}
		else {
		    throw new LoggedException("Unrecognized input model: " + inputModel);
		}

		if (wildBootstrapModel != null) {
		    if (wildBootstrapModel.equals("dt")) {
			data = new WildBS_DT_DataSynth(input, imPars, numVoxels, seed);
		    }
		    else {
			throw new LoggedException("Wild bootstrap is not supported for model: " + 
						  wildBootstrapModel);
		    }
		}
		else {
		    data = 
			new BootstrapDataSynthFromInput(input, bootstrap, imPars,
							numVoxels, SNR, seed);
		}
		
	    }
	    else { 
		// The data is the parameters of models that must be used
		// to generate synthetic data.
		if (inputModel.equals("dt")) {
		    data = new DataSynthFromDT_Input(inputFile, inputDataType, imPars, SNR,
						     seed);
		}
		else if (inputModel.equals("twotensor")) {
		    data = new DataSynthFromTwoTensorInput(inputFile, inputDataType, imPars,
							   SNR, seed);
		}
		else if (inputModel.equals("threetensor")) {
		    data = new DataSynthFromThreeTensorInput(inputFile, inputDataType,
							     imPars, SNR, seed);
		}
		else if (inputModel.equals("multitensor")) {
		    data = new DataSynthFromMultiTensorInput(inputFile, inputDataType,
							     imPars, maxTensorComponents, SNR, seed);
		}
		else if (inputModel.equals("ballstick")) {
		    data = new DataSynthFromBallStickInput(inputFile, inputDataType,
							     imPars, SNR, seed);
		}
		else if (inputModel.equals("sh") || inputModel.equals("rbf")) {
		    //data = new DataSynthFromESHS_Input(inputFile, inputDataType,
		    // imPars, SNR, seed);
		}
		else {
		    throw new LoggedException("Unrecognized input model: " + inputModel);
		}
    
	    }

	} // matches else (ie what we do when p == null and inputmodel != null)

    }

    
    /**
     * Constructs a test function from an array of values given on the command
     * line.
     * 
     * @param testFunctionData
     *            The array of parameters given on the command line.
     * 
     * @return The <code>ModelPDF</code> object encapsulating the test function.
     */
    public static ModelPDF constructTestFunction(double[] testFunctionData) {

        // The test function is assumed to be a mixture of Gaussian
        // densities.

        // The first element of the array specified the number of
        // components in the model.
        int components = (int) testFunctionData[0];

        // Construct the arrays of diffusion tensors and mixing
        // parameters
        DT[] dts = new DT[components];
        double[] mixpars = new double[components];
        for (int i = 0; i < components; i++) {
            dts[i] = new DT(testFunctionData[i * 7 + 1], testFunctionData[i * 7 + 2],
                    testFunctionData[i * 7 + 3], testFunctionData[i * 7 + 4],
                    testFunctionData[i * 7 + 5], testFunctionData[i * 7 + 6]);
            mixpars[i] = testFunctionData[i * 7 + 7];

	    if (rotationIndex > 0) {
		dts[i] = dts[i].transform(Rotations.randomRotMat(new Random(rotationIndex)));
	    }

        }


        GaussianMixture p = new GaussianMixture(dts, mixpars);

            return p;
    }

    /**
     * Creates the source of data from options specified on the command line.
     * 
     * @param p
     *            The model for the spin displacement density to use in a
     *            synthetic data source.
     * 
     * @param imPars
     *            The details of the imaging sequence.
     * 
     * @param SNR
     *            The signal to noise ratio for synthetic data.
     * 
     * @param numVoxels
     *            The number of voxels the data source should produce.
     * 
     * @param seed
     *            Random number generator seed.
     * 
     * @param bootstrap
     *            If zero or less, this function returns a <code>DataSynthesizer</code>
     *            object. If greater than zero, it returns a
     *            <code>SyntheticBootstrapper</code> with <code>bootstrap</code> repeats.
     * 
     * @return The source of data.
     */
    public static DataSource getDataSynthesizer(ModelPDF p, Scheme imPars,
            double SNR, int numVoxels, int seed, int bootstrap) {
        DataSource data = null;

        if (bootstrap <= 0) {

	    // create random here because default is to use Java.util.Random
	    
	    MTRandom ran = new MTRandom(seed);

            data = new DataSynthesizer(p, imPars, SNR, numVoxels, ran);
        }
        else {
            data = new SyntheticBootstrapper(p, imPars, SNR, bootstrap, numVoxels, seed);
        }

        return data;
    }

    /**
     * Mark a single argument <code>i</code> as parsed.
     */
    public static final void markAsParsed(int i) {
        markAsParsed(i, 1);
    }

    /**
     * Mark sequential arguments as parsed, from <code>i</code> to
     * <code>i + numArgs - 1</code>.
     * 
     * @param i
     *            the index of the first argument to mark as parsed.
     * @param numArgs
     *            the number of arguments to mark as parsed.
     */
    public static final void markAsParsed(int i, int numArgs) {
        for (int a = 0; a < numArgs; a++) {
            parsed[i + a] = true;
        }
    }

    

    
    /**
     * routine that throws an exeption, to test exception handling.
     * 
     * @throws Exception
     */
    public static void exceptionTest() throws Exception {

        throw new Exception("test exception thrown");
    }

    
    
    
}

