package tractography;

import data.*;
import imaging.*;
import inverters.*;
import misc.DT;
import numerics.*;

import java.util.Random;

/**
 * Superclass to bootstrap images. Each call to #getPDs(int, int, int) returns a new bootstrap 
 * estimate of the PDs. 
 *
 *
 * @version $Id: BootstrapTractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public abstract class BootstrapTractographyImage extends TractographyImage {
 
    protected final int[][][] voxelClassification;

    protected final Scheme imPars;
    
    protected final int numMeasurements; // number of measurements in each voxel

    protected final Random ran;
  

    /**
     *
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param vectorsPerPD the number of vectors the image contains per principal direction. 
     * @param vc the voxel classification.
     * @param imPars the imaging scheme for the data.
     * @param r a source of random numbers. The java.util.Random class is not recommended, use
     * tools.MTRandom instead.
     *
     */
    public BootstrapTractographyImage(int[] dataDims, double[] voxelDims, int vectorsPerPD, 
				      int[][][] vc, Scheme imPars, Random r) {

	super(dataDims, voxelDims, vectorsPerPD);

	ran = r;

	voxelClassification = vc;
 
	numMeasurements = imPars.numMeasurements();
	
        this.imPars = imPars;

        for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
                    if (vc[i][j][k] < 0) {
                        numPDs[i][j][k] = 0;
                    }
                    else if (vc[i][j][k] < 4) {
                        numPDs[i][j][k] = 1;
                    }
                    else if (vc[i][j][k] == 4) {
                        numPDs[i][j][k] = 2;
                    }
                    else {
                        throw new IllegalArgumentException("Unsupported order " + 
                                                           vc[i][j][k] + " in voxel classification");
                    }
                }
            }
        }
        
        computeIsotropicMask();
    }


    /**
     * Gets a new bootstrap estimate at every call.
     * @return the principal directions in this voxel.
     */
    public final Vector3D[] getPDs(int i, int j, int k) {
        
        setVectors(i,j,k);

        return super.getPDs(i,j,k);
    }

  
   

    /**
     *
     * Sets the vectors for position i, j, k.
     *
     */
    protected abstract void setVectors(int i, int j, int k);
   


}
