package tractography;

import numerics.*;

import java.util.Arrays;
import java.util.Random;

/**
 * <dl>
 * <dt>Purpose: To perform Tractography operations.
 * <BR><BR>
 *
 * <dt>Description:
 * <dd> This class does probabilistic streamline tractography. No interpolation is applied.
 *
 * </dl>
 *
 * @version $Id: NonIntProbFibreTracker.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class NonIntProbFibreTracker extends NonInterpolatedFibreTracker {

    protected final Vector3D[][][][] voxelPDs; 
    private final Random ran;


    /** Construct a tracker with additional options set.
     * @param data the dataset within which the tracking will take place.
     * @param ipThresh the minimum dot product between the tract direction over 
     * adjacent voxels. 
     * @param ran the Random for the randomizer.
     */
    public NonIntProbFibreTracker(TractographyImage data, double ipThresh, Random ran) {

	super(data, ipThresh);
	this.ran = ran;
	voxelPDs = new Vector3D[xDataDim][yDataDim][zDataDim][];
    }

 
    /**
     * Does not make a defensive copy of the array.
     */
    protected final Vector3D[] getPDs(int i, int j, int k) {
	if (visitedVoxel[i][j][k] == 0) {
	    voxelPDs[i][j][k] = image.getPDs(i,j,k);
	}
	
	return voxelPDs[i][j][k];
    }


    /**
     * Does not make a defensive copy of the array.
     */
    protected final Vector3D[] getPDs(int i, int j, int k, Vector3D previousDir) {
	if (visitedVoxel[i][j][k] == 0) {
	    voxelPDs[i][j][k] = image.getPDs(i,j,k, previousDir);
	}
	
	return voxelPDs[i][j][k];
    }



 
    protected Vector3D getTrackingDirection(int i, int j, int k, Vector3D previousDirection) {

	if (visitedVoxel[i][j][k] == 0) {
	    getPDs(i,j,k, previousDirection);
	}

	Vector3D trackingDir = null;

	if (voxelPDs[i][j][k].length == 1) {
	    trackingDir = voxelPDs[i][j][k][0];   
	}
	else {

	    double[] dotProd = new double[voxelPDs[i][j][k].length];
	
	    for (int p = 0; p < voxelPDs[i][j][k].length; p++) {
		dotProd[p] = Math.abs(voxelPDs[i][j][k][p].dot(previousDirection));
	    }

	    double maxDot = -1.0;
            int index = -1;

            for (int p = 0; p < voxelPDs[i][j][k].length; p++) {
                if (dotProd[p] > maxDot) {
                    maxDot = dotProd[p];
                    index = p;
                }
            }

            trackingDir = voxelPDs[i][j][k][index];
            
	}
	
	
	if (previousDirection.dot(trackingDir) > 0.0) {
	    return trackingDir;
	}
	else {
	    return trackingDir.negated();
	}
	
    }
   
    
}
