package edu.jhu.ece.iacl.doc;

import java.io.File;
import java.util.LinkedList;
import java.util.Vector;

import javax.swing.ProgressMonitor;
import javax.swing.tree.TreeNode;

import edu.jhu.ece.iacl.io.FileReaderWriter;
import edu.jhu.ece.iacl.io.MipavController;
import edu.jhu.ece.iacl.pipeline.PipeAlgorithm;
import edu.jhu.ece.iacl.pipeline.PipeLibrary;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.pipeline.graph.PipeAlgorithmFactory;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.pipeline.tree.PackageNode;
import edu.jhu.ece.iacl.plugins.MedicAlgorithmExecutableAdapter;
import edu.jhu.ece.iacl.plugins.MedicAlgorithmMipavAdapter;
import edu.jhu.ece.iacl.io.StringReaderWriter;

/**
 * Class automatically document module definitions
 * 
 * @author John Bogovic (bogovic@jhu.edu)
 */

public class GenerateModuleDoc {
	
	private static final String RootPackageName="MAPS";
	
	/** Location of library directory. */
	private File libraryPath = null;
	/** List of algorithms discovered from library. */
	private Vector<PipeAlgorithmFactory> algos;
	/** Tree node to represent directory. */
	private PackageNode algoRoot = null;
	/** Type of documentation to be written */
	private String type;
	
	/**
	 * Default constructor.
	 */
	public GenerateModuleDoc(){
		algos = new Vector<PipeAlgorithmFactory>();
	}
	/**
	 * Constructor with libraryPath input
	 * 	 @param libraryPath
	 *            library path
	 */
	public GenerateModuleDoc(File librarydir){
		libraryPath=librarydir;
		algos = new Vector<PipeAlgorithmFactory>();
		algoRoot = new PackageNode(libraryPath, "Algorithms");
	}
	
	public void setType(String type){
		
		if(type.contains("wiki")){
			this.type="wiki";
		}else{
			this.type="text";
		}
	}
	
	/**
	 * Set library directory.
	 * 
	 * @param libraryPath
	 *            library path
	 */
	public void setLibraryPath(File f){
		libraryPath=f;
		algoRoot = new PackageNode(libraryPath, "Algorithms");
	}
	
	/**
	 * Searches for modules and writes documentation to file.
	 */
	public void discoverInternalMapsModules() {
		Vector<Class> classes = GenerateModuleDoc.getAllClasses("edu");
		File f;
		
		File discoveredPath = new File(libraryPath, File.separatorChar+RootPackageName);
		if (!discoveredPath.exists()) {
			discoveredPath.mkdir();
		} else {
			for (int i = 0; i < algoRoot.getChildCount(); i++) {
				TreeNode n = algoRoot.getChildAt(i);
				if (n instanceof PackageNode) {
					Object obj = ((PackageNode) n).getUserObject();
				}
			}
		}
		
		for (Class c : classes) {
//			System.out.println(c.getSimpleName());
			try {
				if (ProcessingAlgorithm.class.equals(c.getSuperclass())) {
					PipeAlgorithm algo = new PipeAlgorithm((ProcessingAlgorithm) c.newInstance());
					if (!containsAlgorithm(algo)) {
						File path=createPath(discoveredPath, "DOC"+File.separatorChar+algo.getCategory());
						f = new File(path+"."+algo.getAlgorithm().getClass().getSimpleName() + ".txt");
//						algo.write(f); 
						
						/*
						 * Write the documentation to a file!
						 */
						 String doc = algo.getLabel()+" - DOCUMENTATION";
						 if(type.equals("wiki")){
							 DocMediaWikiWriter mww = new DocMediaWikiWriter(algo.getAlgorithm());
							 GenerateModuleDoc.writeDoc(mww.genDoc(), f);
						 }
//						 GenerateModuleDoc.writeDoc(, f);
						 
						 System.out.println("WROTE TO: " + f);
						 
						 
					}
				}
			} catch (InstantiationException e) {
				System.err.println(e.getMessage());
			} catch (IllegalAccessException e) {
				System.err.println(e.getMessage());
			} catch(Exception e){
				e.printStackTrace();
			}
		}
	}
	

	
	public static Vector<Class> getAllClasses(String rootPckg) {
		return getAllClasses(rootPckg, null);
	}
	
	public static Vector<Class> getAllClasses(String rootPckg,String simpleName) {
		String[] strs = System.getProperty("java.class.path").split(File.pathSeparator);
		LinkedList<File> dirs = new LinkedList<File>();
		LinkedList<String> packs = new LinkedList<String>();
		Vector<Class> classes = new Vector<Class>();
		String pckg;
		for (String str : strs) {
			File f = new File(str, rootPckg.replace(".", File.separator));
			if (f.exists() && f.isDirectory()) {
				dirs.add(f);
				packs.add(rootPckg);
			}
		}
		while (dirs.size() > 0) {
			File dir = dirs.removeFirst();
			String pack = packs.removeFirst();
			File[] files = dir.listFiles();
			// Test all files in directory
			for (File f : files) {
				
				if (f.isDirectory()) {
					dirs.add(f);
					if (pack.length() > 0) {
						packs.add(pack + "." + FileReaderWriter.getFileName(f));
					} else {
						packs.add(FileReaderWriter.getFileName(f));
					}
				} else if (FileReaderWriter.getFileExtension(f).equals("class")) {
					if (pack.length() > 0) {
						pckg = pack + "." + FileReaderWriter.getFileName(f);
					} else {
						pckg = FileReaderWriter.getFileName(f);
					}
					try {
						Class c = Class.forName(pckg);
						if (c != null) {
							if(simpleName==null||c.getSimpleName().equals(simpleName)){
								classes.add(c);
							}
						}
					} catch (ClassNotFoundException e) {
					} catch (UnsatisfiedLinkError e) {
					} catch (VerifyError e) {
					} catch (Error e) {
					}
				}
			}
		}
		return classes;
	}
	
	private boolean containsAlgorithm(PipeAlgorithm algo) {
		Class algoClass = algo.getAlgorithmClass();
		if (MedicAlgorithmExecutableAdapter.class.equals(algoClass)
				|| MedicAlgorithmMipavAdapter.class.equals(algoClass)) {
			return false;
		}
		for (PipeAlgorithmFactory child : algos) {
			Class childClass = child.getModuleClass();
			if ((childClass != null) && (algoClass != null) && childClass.equals(algoClass)) {
				return true;
			}
		}
		return false;
	}
	
	public static boolean writeDoc(String doc, File f){
		try{
			MipavController.setQuiet(true);
			StringReaderWriter.getInstance().write(doc, f);
		}catch(Exception e){
			e.printStackTrace();
			return false;
		}
		return true;
	}
	
	public File createPath(File discoveredPath,String category){
		if(category==null){
			return discoveredPath;
		}
		File f=new File(discoveredPath,category.replace('_',' '));
		if(!f.exists()){
			f.mkdirs();
		}
		return f;
	}
	
	/**
	 * The main function call.
	 * Generates the documentation by calling a number of subroutines.
	 * @param alg The module
	 * @return 
	 */
	public static String genDoc(ProcessingAlgorithm alg){
		String doc = "";
		doc += (alg.getAlgorithmLabel()+"\n\n"); 
		doc += getStatus(alg);
		doc += getDescription(alg);
		doc += getCitations(alg);
		doc += getInputs(alg);
		doc += getOutputs(alg);
		return doc;
	}
	
	public static String getStatus(ProcessingAlgorithm alg){
		return "";
	}
	/**
	 * Returns the description string in the module's "Algorithm Information"
	 * @param The Processsing Algorithm
	 * @return 
	 */
	public static String getDescription(ProcessingAlgorithm alg){
		return alg.getAlgorithmInformation().getDescription()+"\n\n";
	}
	/**
	 * Returns the Citation string in the module's "Algorithm Information"
	 * @param The Processsing Algorithm
	 * @return 
	 */
	public static String getCitations(ProcessingAlgorithm alg){
		String cit =  "PREFERRED CITATIONS:\n";
		for(Citation c : alg.getAlgorithmInformation().getCitations()){
			if(c.getText().length()>0){
				cit += c.getText()+"\n";
			}
		}
		cit+="\n";
		return cit;
	}
	/**
	 * Creates a media wiki table of the module's input parameters
	 * @param alg The Processing Algorithm
	 * @return A wikified string describing the input parameters
	 */
	public static String getInputs(ProcessingAlgorithm alg){
		String input =  "INPUTS:\n";
		input += "Name\tType\tMandatory\tDescription\n";
		int i=0;
		for(ParamModel c : alg.getInput().getValue()){
			if(i>0){
				input += c.getName()+"\t";
				
//				if(c.getClass().getSimpleName()=="ParamVolume"){
//					
//				}else if(c.getClass().getSimpleName()=="ParamFile"){
//					
//				}else if(c.getClass().getSimpleName()=="ParamOption"){
//					
//				}
				input += c.getClass().getSimpleName().replace("Param", "") +"\t";
				
				
				input += c.isMandatory()+"\t";
				if(c.getDescription()==null){
					input += ""+"\t";
				}else{
					input += c.getDescription()+"\t";
				}
				input+="\n";
			}
			i++;
		}
		input+="\n";
		return input;
	}
	/**
	 * Creates a media wiki table of the module's output parameters
	 * @param alg The Processing Algorithm
	 * @return A wikified string describing the output parameters
	 */
	public static String getOutputs(ProcessingAlgorithm alg){
		String output =  "OUTPUT:\n";
		output += "Name\tType\tMandatory\tDescription\n";
		int i=0;
		for(ParamModel c : alg.getOutput().getValue()){
			if(i>0){
				output += c.getName()+"\t";
				
//				if(c.getClass().getSimpleName()=="ParamVolume"){
//					
//				}else if(c.getClass().getSimpleName()=="ParamFile"){
//					
//				}else if(c.getClass().getSimpleName()=="ParamOption"){
//					
//				}
				output += c.getClass().getSimpleName().replace("Param", "") +"\t";
				
				output += c.isMandatory()+"\t";
				if(c.getDescription()==null){
					output += ""+"\t";
				}else{
					output += c.getDescription()+"\t";
				}
				output+="\n";
			}
			i++;
		}
		output+="\n";
		return output;
	}
	
	
	public static String getSeeAlsos(ProcessingAlgorithm alg){
		return "";
	}

	/**
	 * Run me.
	 * @param args
	 */
	public static void main(String[] args){
		System.out.println("Searching for modules...");
		
		
//		File libpath = new File("C:\\Documents and Settings\\John\\My Documents\\Java\\mapslib");
		File libpath = new File("/home/john/MAPS_libs/doctest");
//		File libpath = new File(args[0]);
		GenerateModuleDoc gmd = new GenerateModuleDoc();
		gmd.setLibraryPath(libpath);
		gmd.setType("wiki");
		gmd.discoverInternalMapsModules();
		System.out.println("Finished!");
	}
	
}
