package edu.jhu.ece.iacl.plugins.dti;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import edu.jhmi.rad.medic.utilities.MedicUtil;
import edu.jhu.bme.smile.commons.textfiles.TextFileReader;
import edu.jhu.ece.iacl.algorithms.dti.EstimateTensorLLMSE;
import edu.jhu.ece.iacl.io.CubicVolumeReaderWriter;
import edu.jhu.ece.iacl.io.FileExtensionFilter;
import edu.jhu.ece.iacl.io.ModelImageReaderWriter;
import edu.jhu.ece.iacl.io.StringReaderWriter;
import edu.jhu.ece.iacl.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.*;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.structures.image.ImageData;
import edu.jhu.ece.iacl.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.structures.image.ImageDataUByte;
import edu.jhu.ece.iacl.utility.FileUtil;
import gov.nih.mipav.model.structures.ModelImage;

public class DWITensorEstLLMSE extends ProcessingAlgorithm { 

	/****************************************************
	 * Input Parameters 
	 ****************************************************/
	private ParamFileCollection DWdata4D; 		// SLAB-enabled Imaging Data
	private ParamFileCollection Mask3D;			// SLAB-enabled Binary mask to indicate computation volume
	private ParamOption estOptions;		// Option to attempt to estimate with missing data
	private ParamFile bvaluesTable;		// .b file with a list of b-values
	private ParamFile gradsTable;		// .grad or .dpf file with a list of gradient directions

	/****************************************************
	 * Output Parameters
	 ****************************************************/
	private ParamFileCollection tensorVolume;	// SLAB-enabled A 4D volume with one tensor estimated per pixel

	private static final String rcsid =
		"$Id: DWITensorEstLLMSE.java,v 1.9 2009/04/21 15:28:12 bogovic Exp $";
	private static final String cvsversion =
		"$Revision: 1.9 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");


	protected void createInputParameters(ParamCollection inputParams) {

		/****************************************************
		 * Step 1. Set Plugin Information 
		 ****************************************************/
		inputParams.setName("Tensor Estimation: LLMSE");
		inputParams.setLabel("TensorEst:LLMSE");	
		inputParams.setCategory("Modeling.Diffusion");
		inputParams.setPackage("IACL");
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("http://sites.google.com/site/jhupami/");
		info.add(new AlgorithmAuthor("Bennett Landman","landman@jhu.edu","http://sites.google.com/site/bennettlandman/"));
		info.setDescription("Log-linear minium mean squared error tensor estimation.");
		info.setAffiliation("Johns Hopkins University, Department of Biomedical Engineering");
		info.add(new Citation("Landman BA, Farrell JA, Jones CK, Smith SA, Prince JL, Mori S. Effects of diffusion weighting schemes on the reproducibility of DTI-derived fractional anisotropy, mean diffusivity, and principal eigenvector measurements at 1.5T. Neuroimage 2007;36(4):1123-1138."));
		info.add(new Citation("Basser, PJ, Jones, DK. \"Diffusion-tensor MRI: Theory, experimental design and data analysis - a technical review.\" NMR Biomed 2002; 15(7-8):456-67"));				
		info.setVersion(revnum);	



		/****************************************************
		 * Step 2. Add input parameters to control system 
		 ****************************************************/
		inputParams.add(DWdata4D=new ParamFileCollection("DWI and Reference Image(s) Data (4D)",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions)));
		inputParams.add(gradsTable=new ParamFile("Table of diffusion weighting directions",new FileExtensionFilter(new String[]{"grad","dpf"})));
		inputParams.add(bvaluesTable=new ParamFile("Table of b-values",new FileExtensionFilter(new String[]{"b"})));
		inputParams.add(Mask3D=new ParamFileCollection("Mask Volume to Determine Region of Tensor Estimation (3D)",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions)));
		Mask3D.setMandatory(false); // Not required. A null mask will estimate all voxels.	
		inputParams.add(estOptions=new ParamOption("Attempt to estimate tensors for voxels with missing data?",new String[]{"Yes","No"}));
	}
	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Step 1. Add output parameters to control system 
		 ****************************************************/
		tensorVolume = new ParamFileCollection("Tensor Estimate",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		tensorVolume.setName("Tensor (xx,xy,xz,yy,yz,zz)");
		outputParams.add(tensorVolume);			

	}

	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {		
		TensorEstimationWrapper wrapper=new TensorEstimationWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}
	
	protected class TensorEstimationWrapper extends AbstractCalculation {

		protected void execute() {
			/****************************************************
			 * Step 1. Indicate that the plugin has started.
			 * 		 	Tip: Use limited System.out.println statements
			 * 			to allow end users to monitor the status of
			 * 			your program and report potential problems/bugs
			 * 			along with information that will allow you to 
			 * 			know when the bug happened.  
			 ****************************************************/
			System.out.println("DWITensorEstLLMSE: Start");
//			MedicUtil.setQuiet(false);

			/****************************************************
			 * Step 2. Loop over input slabs
			 ****************************************************/
			List<File> dwList = DWdata4D.getValue();
			List<File> maskList = Mask3D.getValue();
			CubicVolumeReaderWriter rw  = CubicVolumeReaderWriter.getInstance();
			ArrayList<File> outVols = new ArrayList<File>();
			this.addTotalUnits(dwList.size());
			for(int jSlab=0;jSlab<dwList.size();jSlab++) {
				this.setLabel("Load");
				/****************************************************
				 * Step 2. Parse the input data 
				 ****************************************************/
				ImageData dwd=rw.read(dwList.get(jSlab));

				String sourceName = dwd.getName();
				ImageDataFloat DWFloat=new ImageDataFloat(dwd);
				dwd.dispose();
				dwd=null;
		
				/* Read the brain mask */
				ImageData maskVol=null;
				if(maskList!=null)
					if(maskList.size()>jSlab)
						if(maskList.get(jSlab)!=null)
							maskVol=rw.read(maskList.get(jSlab));
				
				
				byte [][][]mask=null;
				if(maskVol!=null) {
					ImageDataUByte maskByte = new ImageDataUByte (maskVol);
					mask = maskByte.toArray3d();
					maskByte.dispose();
					maskByte=null;
					maskVol.dispose();
					maskVol=null;
				}

				/* Read the b values */
				float [][]bs=null;		
				TextFileReader text = new TextFileReader(bvaluesTable.getValue());
				try {
					bs = text.parseFloatFile();
				} catch (IOException e) 
				{
					throw new RuntimeException("LLMSE: Unable to parse b-file");
				}

				/* Read the gradient table  */
				float [][]grads=null;
				text = new TextFileReader(gradsTable.getValue());
				try {
					grads  = text.parseFloatFile();
				} catch (IOException e) { 

					throw new RuntimeException("LLMSE: Unable to parse grad-file");
				}

				/****************************************************
				 * Step 3. Perform limited error checking 
				 ****************************************************/
				// If there are 4 columns in the gradient table, remove the 1st column (indecies)
				if(grads[0].length==4) {
					float [][]g2 = new float[grads.length][3];
					for(int i=0;i<grads.length;i++) 
						for(int j=0;j<3;j++)
							g2[i][j]=grads[i][j+1];
					grads=g2;
				}

				if(grads[0].length!=3)
					throw new RuntimeException("LLMSE: Invalid gradient table. Must have 3 or 4 columns.");
				if(bs[0].length!=1)
					throw new RuntimeException("LLMSE: Invalid b-value table. Must have 1 column.");
				float []bval = new float[bs.length];
				for(int i=0;i<bval.length;i++)
					bval[i]=bs[i][0];

				/****************************************************
				 * Step 4. Run the core algorithm. Note that this program 
				 * 		   has NO knowledge of the MIPAV data structure and 
				 * 		   uses NO MIPAV specific components. This dramatic 
				 * 		   separation is a bit inefficient, but it dramatically 
				 * 		   lower the barriers to code re-use in other applications.  		  
				 ****************************************************/
				this.setLabel("Estimate");
				float[][][][] tensors = EstimateTensorLLMSE.estimate(DWFloat.toArray4d(),bval,grads,mask,estOptions.getValue().compareToIgnoreCase("Yes")==0);
				
				
				/****************************************************
				 * Step 5. Retrieve the image data and put it into a new
				 * 			data structure. Be sure to update the file information
				 * 			so that the resulting image has the correct
				 * 		 	field of view, resolution, etc.  
				 ****************************************************/
				this.setLabel("Save");
//				int []ext=dwd.getModelImage().getExtents();
				//DWFloat=null;DWdata4D.dispose();

				ModelImage img=null;
				ImageData  out= (new ImageDataFloat(tensors));
				out.setHeader(DWFloat.getHeader());
				DWFloat.dispose();
				DWFloat=null;
				tensors = null; //no longer needed
				/*
				 img=out.getModelImage();
//				ext[3]=6;
//				img.setExtents(ext);
				FileUtil.updateFileInfo(dwd.getModelImage(),img);
				img.calcMinMax();					
	*/
				out.setName(sourceName+"_TensorLLMSE");		
				//			FileUtilInfo info = FileUtil.getFileInfo(dwd.getModelImage());			
				File outputSlab = rw.write(out, getOutputDirectory());
				outVols.add(outputSlab);
				out.dispose();
				out=null;			
				//tensorVolume.setName(DWdata4D.getName()+"_TensorLLMSE"+".xml");
				this.incrementCompletedUnits();
			}
			tensorVolume.setValue(outVols);
			
			/****************************************************
			 * Step 6. Let the user know that your code is finished.  
			 ****************************************************/
			System.out.println("DWITensorEstLLMSE: FINISHED");
		}
	}
}
