package edu.jhu.ece.iacl.plugins.hardi;

import imaging.Scheme;
import imaging.SchemeV1;
import inverters.AlgebraicDT_Inversion;
import inverters.DT_Inversion;
import inverters.DiffusionInversion;
import inverters.LinearDT_Inversion;
import inverters.NonLinearDT_Inversion;
import inverters.RestoreDT_Inversion;
import inverters.TensorModelFitter;
import inverters.WeightedLinearDT_Inversion;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Date;

import com.thoughtworks.xstream.XStream;

import edu.jhu.bme.smile.commons.textfiles.TextFileReader;
import edu.jhu.ece.iacl.algorithms.dti.EstimateTensorLLMSE;
import edu.jhu.ece.iacl.algorithms.hardi.CFARIBasisSet;
import edu.jhu.ece.iacl.algorithms.hardi.CFARIEstimator;
import edu.jhu.ece.iacl.io.FileExtensionFilter;
import edu.jhu.ece.iacl.io.StringReaderWriter;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.*;
import edu.jhu.ece.iacl.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.structures.image.ImageDataInt;
import edu.jhu.ece.iacl.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.structures.image.ImageDataUByte;
import edu.jhu.ece.iacl.utility.FileUtil;
import gov.nih.mipav.model.structures.ModelImage;

public class CFARIEstimation extends ProcessingAlgorithm{ 

	/****************************************************
	 * Input Parameters 
	 ****************************************************/
	private ParamVolume DWdata4D; 		// Imaging Data
	private ParamVolume Mask3D;			// Binary mask to indicate computation volume
	private ParamFile SchemeFile;
	/****************************************************
	 * Output Parameters
	 ****************************************************/
	private ParamBoolean optionPD;
	private ParamInteger optionNumBasis;
	private ParamVolume basisIndecies;
	private ParamVolume basisMixture;
	private ParamFloat optionLambda;

	private static final String rcsid =
		"$Id: CFARIEstimation.java,v 1.4 2009/06/27 14:41:37 jhuuser Exp $";
	private static final String cvsversion =
		"$Revision: 1.4 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");
		
	protected void createInputParameters(ParamCollection inputParams) {

		/****************************************************
		 * Step 1. Set Plugin Information 
		 ****************************************************/
		inputParams.setName("CFARI Estimation");
		inputParams.setLabel("CFARI");
		inputParams.setCategory("Modeling.Diffusion");
		inputParams.setPackage("IACL");
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("http://sites.google.com/site/jhupami/");
		info.add(new AlgorithmAuthor("Bennett Landman","landman@jhu.edu","http://sites.google.com/site/bennettlandman/"));
		info.setDescription("");
		info.setLongDescription("");
		info.setAffiliation("Johns Hopkins University");
		info.add(new Citation("B. A. Landman, J. Bogovic, and J. L. Prince. Compressed Sensing of Multiple Intra-Voxel Orientations with Traditional DTI, In Proceedings of the Workshop on Computational Diffusion MRI at the 11th International Conference on Medical Image Computing and Computer Assisted Intervention, New York, NY, September 2008."));				
		info.setVersion(revnum);	


		/****************************************************
		 * Step 2. Add input parameters to control system 
		 ****************************************************/
		inputParams.add(DWdata4D=new ParamVolume("DWI and Reference Image(s) Data (4D)",null,-1,-1,-1,-1));
		inputParams.add(SchemeFile=new ParamFile("CFARI Scheme",new FileExtensionFilter(new String[]{"CFARI"})));
		inputParams.add(Mask3D=new ParamVolume("Mask Volume to Determine Region of Tensor Estimation (3D)",null,-1,-1,-1,1));
		Mask3D.setMandatory(false); // Not required. A null mask will estimate all voxels.	
		inputParams.add(optionPD=new ParamBoolean("Constrain to Positive Mixture?",true));
		inputParams.add(optionLambda=new ParamFloat("Select Lambda",1));
		inputParams.add(optionNumBasis=new ParamInteger("Number of Basis Functions to Export",1,Integer.MAX_VALUE,5));				
	}

	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Step 1. Add output parameters to control system 
		 ****************************************************/
		basisIndecies = new ParamVolume("Basis Function Indices Estimate",null,-1,-1,-1,-1);
		basisIndecies.setName("Basis Indices");
		outputParams.add(basisIndecies);
		basisMixture = new ParamVolume("Basis Mixture Fraction",null,-1,-1,-1,-1);
		basisMixture .setName("Mixture Fraction");
		outputParams.add(basisMixture );			

	}
	protected void execute(CalculationMonitor monitor) {
		/****************************************************
		 * Step 1. Indicate that the plugin has started.
		 * 		 	Tip: Use limited System.out.println statements
		 * 			to allow end users to monitor the status of
		 * 			your program and report potential problems/bugs
		 * 			along with information that will allow you to 
		 * 			know when the bug happened.  
		 ****************************************************/
		System.out.println("CFARIEstimation: Start");

		/****************************************************
		 * Step 2. Parse the input data 
		 ****************************************************/
		System.out.println("Load data.");System.out.flush();
		ImageDataMipav dwd=DWdata4D.getImageData();
		dwd.getHeader().setSliceThickness(0);		//used to suppress printing warning message
		ImageDataFloat DWFloat=new ImageDataFloat(dwd);
		
		ImageDataMipav maskVol=Mask3D.getImageData();
		byte [][][]mask=null;
		if(maskVol!=null) {
			maskVol.getHeader().setSliceThickness(0);		//used to suppress printing warning message
			ImageDataUByte maskByte = new ImageDataUByte (maskVol);
			mask = maskByte.toArray3d();
		}
	
		System.out.println("Load scheme.");System.out.flush();
		CFARIBasisSet basisSet = null;
		
		XStream xstream = new XStream();		
		try {
			ObjectInputStream in = xstream.createObjectInputStream(new FileReader(SchemeFile.getValue()));
			basisSet=(CFARIBasisSet)in.readObject();
			in.close();
		} catch (IOException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
		

		/****************************************************
		 * Step 3. Perform limited error checking 
		 ****************************************************/
		System.out.println("Error checking."); System.out.flush();
		int nBasisReturn = optionNumBasis.getValue().intValue();
		if(nBasisReturn<0 || nBasisReturn>basisSet.getNumberOfBasisFunctions()) {
			throw new RuntimeException("Invalid number of return basis functions requested");
		}
		
		boolean requirePos = optionPD.getValue();

		float lambda= optionLambda.getFloat();
		/****************************************************
		 * Step 4. Run the core algorithm. Note that this program 
		 * 		   has NO knowledge of the MIPAV data structure and 
		 * 		   uses NO MIPAV specific components. This dramatic 
		 * 		   separation is a bit inefficient, but it dramatically 
		 * 		   lower the barriers to code re-use in other applications.  		  
		 ****************************************************/
		System.out.println("Allocate memory."); System.out.flush();
		float [][][][]data=DWFloat.toArray4d();
		int rows = data.length;
		int cols= data[0].length;
		int slices= data[0][0].length;
		int components= data[0][0][0].length;
		int [][][][]basisIndecies = new int[rows][cols][slices][nBasisReturn];
		float [][][][]basisMixtures= new float[rows][cols][slices][nBasisReturn];		
		
		System.out.println("Run CFARI estimate."); System.out.flush();
		CFARIEstimator.estimateCFARI(data, mask, basisSet, basisIndecies, basisMixtures,
				requirePos,lambda);
		
		/****************************************************
		 * Step 5. Retrieve the image data and put it into a new
		 * 			data structure. Be sure to update the file information
		 * 			so that the resulting image has the correct
		 * 		 	field of view, resolution, etc.  
		 ****************************************************/
		System.out.println("Data export."); System.out.flush();
		int []ext=DWFloat.getModelImage().getExtents();
		ModelImage img=null;
		ImageDataInt out=new ImageDataInt(basisIndecies);
		out.getHeader().setSliceThickness(0);		//used to suppress printing warning message
		img=(out).getModelImage();		
		ext[3]=basisIndecies[0][0][0].length;
		img.setExtents(ext);
		FileUtil.updateFileInfo(DWdata4D.getModelImage(),img);		
		img.calcMinMax();		
		out.setName(DWdata4D.getModelImage().getImageName()+"_basisIndecies");
		this.basisIndecies.setValue(out);		
		
		ImageDataFloat out2=new ImageDataFloat(basisMixtures);
		out2.getHeader().setSliceThickness(0);		//used to suppress printing warning message
		img=(out2).getModelImage();		
		ext[3]=basisMixtures[0][0][0].length;
		img.setExtents(ext);
		FileUtil.updateFileInfo(DWdata4D.getModelImage(),img);
		img.calcMinMax();		
		out2.setName(DWdata4D.getModelImage().getImageName()+"_basisMixtures");
		this.basisMixture.setValue(out2);
		
			
		/****************************************************
		 * Step 6. Let the user know that your code is finished.  
		 ****************************************************/
		System.out.println("CFARIEstimation: FINISHED");
	}
}
