/**
 * JIST Extensions for Computer-Integrated Surgery
 *
 * Center for Computer-Integrated Surgical Systems and Technology &
 * Johns Hopkins Applied Physics Laboratory &
 * The Johns Hopkins University
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 * @author Blake Lucas
 */
package edu.jhu.cs.cisst.plugins.visualization;

import java.awt.Dimension;
import java.io.File;
import java.util.List;

import processing.core.PImage;

import edu.jhu.cs.cisst.jist.pipeline.io.PImageReaderWriter;
import edu.jhu.cs.cisst.vent.Visualization;
import edu.jhu.cs.cisst.vent.VisualizationApplication;
import edu.jhu.cs.cisst.vent.VisualizationPlugIn;
import edu.jhu.cs.cisst.vent.widgets.VisualizationActiveContour2D;
import edu.jhu.cs.cisst.vent.widgets.VisualizationImage2D;
import edu.jhu.ece.iacl.algorithms.CommonAuthors;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

// TODO: Auto-generated Javadoc
/**
 * The Class PlugInVisualizeImage.
 */
public class PlugInVisualizeImage2D extends ProcessingAlgorithm implements
		VisualizationPlugIn {

	/** The orig image param. */
	protected ParamVolumeCollection origImageParam;
	protected ParamFile cacheParam;
	protected ParamBoolean captureParam;
	protected ParamObject<PImage> screenshotParam;
	ParamInteger imageWidthParam, imageHeightParam;

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm#createInputParameters
	 * (edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection)
	 */
	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(origImageParam = new ParamVolumeCollection("Images"));
		inputParams.add(imageWidthParam = new ParamInteger("Image Width", 768));
		inputParams
				.add(imageHeightParam = new ParamInteger("Image Height", 768));
		inputParams.add(captureParam = new ParamBoolean(
				"Only Capture Screenshot", false));
		inputParams.add(cacheParam = new ParamFile("Visualization Parameters",
				new FileExtensionFilter(new String[] { "cache" })));
		cacheParam.setMandatory(false);
		inputParams.setName("vis_images_2d");
		inputParams.setLabel("Visualize Image 2D");
		inputParams.setPackage("CISST");
		inputParams.setCategory("Visualization");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(CommonAuthors.blakeLucas);
		info.setAffiliation("Johns Hopkins University, Department of Computer Science");
		info.setDescription("Visualizes images in 2d.");
		info.setVersion(VisualizationImage2D.getVersion());
		info.setStatus(DevelopmentStatus.RC);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm#createOutputParameters
	 * (edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection)
	 */
	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(screenshotParam = new ParamObject<PImage>(
				"Screenshot", new PImageReaderWriter()));
		screenshotParam.setMandatory(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm#execute(edu.jhu.ece
	 * .iacl.jist.pipeline.CalculationMonitor)
	 */
	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		Visualization visual = createVisualization();
		VisualizationApplication app = new VisualizationApplication(this,
				visual);
		app.setMinimumSize(new Dimension(imageWidthParam.getInt() + 400,
				imageHeightParam.getInt() + 50));
		File outputDir = getOutputDirectory();
		screenshotParam.setFileName("screenshot-"
				+ outputDir.getName().replace("exp-", ""));
		if (captureParam.getValue()) {
			File cacheFile = cacheParam.getValue();

			screenshotParam.setObject(app.capture(null, cacheFile));
		} else {
			app.runAndWait();
			PImage img = new PImage(visual.getScreenshot());
			System.out.println("IMAGE " + img);
			screenshotParam.setObject(img);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see edu.jhu.cs.cisst.vent.VisualizationPlugIn#createVisualization()
	 */
	@Override
	public Visualization createVisualization() {
		List<ImageData> images = origImageParam.getImageDataList();
		int width = 0;
		int height = 0;
		for (ImageData img : images) {
			width = Math.max(width, img.getRows());
			height = Math.max(height, img.getCols());
		}
		VisualizationImage2D visual = new VisualizationImage2D();
		for (ImageData img : images) {
			visual.addImage(img);
		}
		return visual;
	}

}
