/**
 * JIST Extensions for Computer-Integrated Surgery
 *
 * Center for Computer-Integrated Surgical Systems and Technology &
 * Johns Hopkins Applied Physics Laboratory &
 * The Johns Hopkins University
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 * @author Blake Lucas
 */
package edu.jhu.cs.cisst.vent.renderer.processing;

import java.awt.Color;
import java.io.File;
import java.net.URISyntaxException;

import javax.media.j3d.BoundingBox;
import javax.vecmath.Point3d;

import edu.jhu.cs.cisst.algorithms.geometry.surface.IsoContourGenerator;
import edu.jhu.cs.cisst.jist.parameter.ParamColor;
import edu.jhu.cs.cisst.jist.pipeline.view.input.ParamDoubleSliderInputView;
import edu.jhu.cs.cisst.jist.pipeline.view.input.ParamIntegerSliderInputView;
import edu.jhu.cs.cisst.vent.VisualizationProcessing;
import edu.jhu.cs.cisst.vent.VisualizationProcessing3D;
import edu.jhu.cs.cisst.vent.widgets.*;
import edu.jhu.cs.cisst.vent.converter.processing.ConvertImageDataToPImage;
import edu.jhu.cs.cisst.vent.renderer.processing.VolumeIsoContourRenderer.IsoContourWorker;
import edu.jhu.cs.cisst.vent.resources.PlaceHolder;
import edu.jhu.cs.cisst.vent.structures.processing.PCurveCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import processing.core.*;

// TODO: Auto-generated Javadoc
/**
 * The Class VolumeSliceRenderer3D renders volume by slice in 3D.
 */
public class ImageRenderer3D extends RendererProcessing3D {

	/** The images. */
	protected PImage image;
	/** The contrast. */
	protected float contrast = 1;

	/** The brightness. */
	protected float brightness = 0;

	/** The transparency. */
	protected float transparency = 1;

	/** The scale. */
	protected float scale = 1;

	protected float depth = 0;

	/**
	 * Sets the transparency.
	 * 
	 * @param transparency
	 *            the new transparency
	 */
	public void setTransparency(float transparency) {
		this.transparency = transparency;
	}

	/** The max. */
	protected double min, max;

	/** The vol to image transform. */
	protected PMatrix3D volToImageTransform;

	/** The component. */
	protected int component = 0;

	/** The slices. */
	protected int rows, cols;

	/** The applet. */
	protected VisualizationProcessing3D applet;

	/** The contrast param. */
	protected ParamFloat contrastParam;

	/** The brightness param. */
	protected ParamFloat brightnessParam;

	/** The transparency param. */
	protected ParamFloat transparencyParam;

	protected ParamFloat depthParam;
	protected ParamFloat strokeWeightParam;
	protected ParamColor strokeColorParam;

	/**
	 * Instantiates a new volume slice renderer.
	 * 
	 * @param img
	 *            the img
	 * @param volToImageTransform
	 *            the vol to image transform
	 * @param applet
	 *            the applet
	 */
	public ImageRenderer3D(PImage img, PMatrix3D volToImageTransform,
			VisualizationProcessing3D applet) {
		this.image = img;
		rows = image.width;
		cols = image.height;
		min = 1E10f;
		max = -1E10f;
		float[] hsv = new float[3];
		this.volToImageTransform = volToImageTransform;
		for (int i = 0; i < img.pixels.length; i++) {
			Color val = new Color(img.pixels[i]);
			val.RGBtoHSB(val.getRed(), val.getGreen(), val.getBlue(), hsv);
			min = Math.min(min, 255.0f * hsv[0]);
			max = Math.max(max, 255.0f * hsv[0]);
		}
		this.applet = applet;
	}

	protected PCurveCollection curve;

	public void setLevelSetImage(ImageDataFloat img) {
		IsoContourGenerator isogen = new IsoContourGenerator();
		curve = new PCurveCollection(isogen.solve(img, 0, -1, -1), false,false);
	}

	/**
	 * Setup.
	 * 
	 * @see edu.jhu.cs.cisst.vent.renderer.processing.RendererProcessing#setup()
	 */
	@Override
	public void setup() {

		bbox.combine(new Point3d(0, 0, -cols));
		bbox.combine(new Point3d(0, 0, cols));
		bbox.combine(new Point3d(0, cols, 0));
		bbox.combine(new Point3d(rows, cols, 0));
		bbox.combine(new Point3d(rows, 0, 0));

		applet.textureMode(applet.NORMALIZED);
	}

	@Override
	public void draw() {
		applet.pushStyle();
		applet.pushMatrix();
		applet.applyMatrix(volToImageTransform);
		applet.scale(scale);
		applet.tint(255, 255, 255, transparency * 255);
		applet.fill(255, 255, 255);
		applet.stroke(255, 153, 0, transparency * 255);
		float w = image.width;
		float h = image.height;
		applet.pushMatrix();
		applet.translate(0, 0, depthParam.getFloat());
		
				
		applet.beginShape(PApplet.QUADS);

		applet.texture(image);
		applet.vertex(0, 0, 0, 0, 0);
		applet.vertex(w, 0, 0, 1, 0);
		applet.vertex(w, h, 0, 1, 1);
		applet.vertex(0, h, 0, 0, 1);
		applet.endShape();
	
		applet.strokeWeight(strokeWeightParam.getFloat());
		applet.stroke(strokeColorParam.getValue().getRGB());

		curve.draw3D(applet);
		applet.popMatrix();
		applet.popMatrix();
		applet.popStyle();
	}

	/**
	 * Sets the contrast.
	 * 
	 * @param contrast
	 *            the new contrast
	 */
	public void setContrast(float contrast) {
		this.contrast = contrast;

	}

	/**
	 * Sets the brightness.
	 * 
	 * @param brightness
	 *            the new brightness
	 */
	public void setBrightness(float brightness) {
		this.brightness = brightness;
	}

	/**
	 * Creates the visualization parameters.
	 * 
	 * @param visualizationParameters
	 *            the visualization parameters
	 * 
	 * @see edu.jhu.cs.cisst.vent.VisualizationParameters#createVisualizationParameters(edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection)
	 */
	@Override
	public void createVisualizationParameters(
			ParamCollection visualizationParameters) {
		visualizationParameters.setName("Image 3D");
		if (curve != null) {
			visualizationParameters.add(strokeColorParam = new ParamColor(
					"Contour Color", new Color(0, 128, 0)));
			visualizationParameters.add(strokeWeightParam = new ParamFloat(
					"Stroke Weight", 0, 10, 2));
		}
		visualizationParameters.add(depthParam = new ParamFloat("Depth", -cols,
				cols, -3*cols/4));
		depthParam.setInputView(new ParamDoubleSliderInputView(depthParam, 4,
				false));

		visualizationParameters.add(contrastParam = new ParamFloat("Contrast",
				-5, 5, contrast));
		contrastParam.setInputView(new ParamDoubleSliderInputView(
				contrastParam, 4, false));
		visualizationParameters.add(brightnessParam = new ParamFloat(
				"Brightness", -5, 5, brightness));
		brightnessParam.setInputView(new ParamDoubleSliderInputView(
				brightnessParam, 4, false));
		visualizationParameters.add(transparencyParam = new ParamFloat(
				"Transparency", 0, 1, 1));
		transparencyParam.setInputView(new ParamDoubleSliderInputView(
				transparencyParam, 4, false));

	}

	/**
	 * Update.
	 * 
	 * @param model
	 *            the model
	 * @param view
	 *            the view
	 * 
	 * @see edu.jhu.ece.iacl.jist.pipeline.view.input.ParamViewObserver#update(edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel,
	 *      edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView)
	 */
	@Override
	public void update(ParamModel model, ParamInputView view) {
		if (model == contrastParam) {
			setContrast(contrastParam.getFloat());
		} else if (model == brightnessParam) {
			setBrightness(brightnessParam.getFloat());
		} else if (model == transparencyParam) {
			setTransparency(transparencyParam.getFloat());
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * edu.jhu.cs.cisst.vent.VisualizationParameters#updateVisualizationParameters
	 * ()
	 */
	@Override
	public void updateVisualizationParameters() {

		setContrast(contrastParam.getFloat());
		setBrightness(brightnessParam.getFloat());
		setTransparency(transparencyParam.getFloat());

	}

	/**
	 * Sets the scale.
	 * 
	 * @param scale
	 *            the new scale
	 */
	public void setScale(float scale) {
		this.scale = scale;
	}

	public void setScale(float scaleX, float scaleY, float scaleZ) {
		this.scale = scaleX;

	}

}
