/**
 * JIST Extensions for Computer-Integrated Surgery
 *
 * Center for Computer-Integrated Surgical Systems and Technology &
 * Johns Hopkins Applied Physics Laboratory &
 * The Johns Hopkins University
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 * @author Blake Lucas
 */
package edu.jhu.cs.cisst.vent.renderer.processing;

import java.awt.Color;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.UUID;

import javax.media.j3d.BoundingBox;
import javax.media.opengl.GL;
import javax.media.opengl.GL2;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import com.jogamp.opengl.util.GLBuffers;

import processing.core.PMatrix3D;
import processing.opengl2.PGraphicsOpenGL2;
import edu.jhu.cs.cisst.jist.pipeline.view.input.*;
import edu.jhu.cs.cisst.jist.parameter.ParamColor;
import edu.jhu.cs.cisst.vent.VisualizationProcessing3D;
import edu.jhu.cs.cisst.vent.converter.processing.ConvertEmbeddedSurfaceToPTriangleMesh;
import edu.jhu.cs.cisst.vent.structures.processing.PTriangleMesh;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamNumberSliderInputView;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.NormalGenerator;

// TODO: Auto-generated Javadoc
/**
 * The Class SurfaceRenderer.
 */
public class TriangleMeshRenderer extends RendererProcessing3D {

	/** The applet. */
	protected VisualizationProcessing3D applet = null;

	/** The visible. */
	protected boolean visible = true;
	protected int vertexComponentLength = 3;
	/** The gouraud. */
	protected boolean gouraud = false;
	protected FloatBuffer vertexBuffer;
	protected FloatBuffer normalBuffer;
	protected IntBuffer indexBuffer;
	protected FloatBuffer texMapBuffer;
	protected FloatBuffer colorBuffer;
	/** The flip normals. */
	protected boolean wireframe = false;
	protected int textureIndex = -1;

	/** The visible param. */
	protected ParamBoolean visibleParam, gouraudParam, wireframeParam,
			showTextureParam, showColorParam;

	/** The diffuse color param. */
	protected ParamColor diffuseColorParam;

	/** The diffuse color. */
	protected Color diffuseColor = new Color(153, 153, 255);
	/** The transparency. */
	float transparency = 1.0f;
	/** The transparency param. */
	protected ParamDouble transparencyParam;

	/**
	 * Sets the visible.
	 * 
	 * @param visible
	 *            the new visible
	 */
	public void setVisible(boolean visible) {
		this.visible = visible;
	}

	public void setTextureIndex(int index) {

		this.textureIndex = index;
	}

	/**
	 * Instantiates a new surface renderer.
	 * 
	 * @param applet
	 *            the applet
	 * @param surf
	 *            the surf
	 */
	public TriangleMeshRenderer(VisualizationProcessing3D applet,
			EmbeddedSurface surf) {
		this.applet = applet;
		this.name = surf.getName();
		Point3f[] vertCopy = surf.getVertexCopy();
		int[] indexes = surf.getIndexCopy();
		Vector3f[] normCopy = NormalGenerator.generate(vertCopy, indexes);
		bbox = new BoundingBox(new Point3d(1E10, 1E10, 1E10), new Point3d(
				-1E10, -1E10, -1E10));
		vertexBuffer = GLBuffers.newDirectFloatBuffer(vertCopy.length * 3);
		normalBuffer = GLBuffers.newDirectFloatBuffer(normCopy.length * 3);
		for (Point3f pt : vertCopy) {
			bbox.combine(new Point3d(pt));
			vertexBuffer.put(pt.x);
			vertexBuffer.put(pt.y);
			vertexBuffer.put(pt.z);
		}
		vertexBuffer.rewind();
		for (Vector3f pt : normCopy) {
			normalBuffer.put(pt.x);
			normalBuffer.put(pt.y);
			normalBuffer.put(pt.z);
		}
		normalBuffer.rewind();
		indexBuffer = GLBuffers.newDirectIntBuffer(indexes);
	}

	public TriangleMeshRenderer(VisualizationProcessing3D applet,
			FloatBuffer vertexBuffer, FloatBuffer colorBuffer,
			IntBuffer indexBuffer, FloatBuffer normalBuffer, int len) {
		this.applet = applet;
		this.indexBuffer = indexBuffer;
		this.vertexBuffer = vertexBuffer;
		this.colorBuffer = colorBuffer;
		this.normalBuffer = normalBuffer;
		this.vertexComponentLength = len;
		bbox = new BoundingBox(new Point3d(1E10, 1E10, 1E10), new Point3d(
				-1E10, -1E10, -1E10));
		while (vertexBuffer.hasRemaining()) {
			bbox.combine(new Point3d(vertexBuffer.get(), vertexBuffer.get(),
					vertexBuffer.get()));

		}
		vertexBuffer.rewind();
	}

	public void setBuffers(FloatBuffer vertexBuffer, FloatBuffer colorBuffer,
			IntBuffer indexBuffer, FloatBuffer normalBuffer,
			FloatBuffer texMapBuffer, int len) {
		this.indexBuffer = indexBuffer;
		this.vertexBuffer = vertexBuffer;
		this.colorBuffer = colorBuffer;
		this.normalBuffer = normalBuffer;
		this.texMapBuffer = texMapBuffer;
		this.vertexComponentLength = len;
	}

	public TriangleMeshRenderer(VisualizationProcessing3D applet,
			FloatBuffer vertexBuffer, FloatBuffer colorBuffer,
			IntBuffer indexBuffer, FloatBuffer normalBuffer,
			FloatBuffer texMapBuffer, int len) {
		this.applet = applet;
		this.indexBuffer = indexBuffer;
		this.vertexBuffer = vertexBuffer;
		this.colorBuffer = colorBuffer;
		this.normalBuffer = normalBuffer;
		this.vertexComponentLength = len;
		this.texMapBuffer = texMapBuffer;
		bbox = new BoundingBox(new Point3d(1E10, 1E10, 1E10), new Point3d(
				-1E10, -1E10, -1E10));
		while (vertexBuffer.hasRemaining()) {
			bbox.combine(new Point3d(vertexBuffer.get(), vertexBuffer.get(),
					vertexBuffer.get()));

		}
		vertexBuffer.rewind();
	}

	protected String name = "Triangle Mesh";

	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Draw.
	 * 
	 * @see edu.jhu.cs.cisst.vent.renderer.processing.RendererProcessing#draw()
	 */
	@Override
	public void draw() {
		applet.pushMatrix();
		if (transform != null)
			applet.applyMatrix(transform);
		applet.pushStyle();
		final float scale = 0.0039215f;
		GL2 gl = (GL2) (((PGraphicsOpenGL2) applet.g).beginGL());
		gl.glEnable(GL2.GL_LINE_SMOOTH);
		if (gouraud)
			gl.glShadeModel(GL2.GL_SMOOTH);
		else
			gl.glShadeModel(GL2.GL_FLAT);
		gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
		gl.glEnableClientState(GL2.GL_NORMAL_ARRAY);
		gl.glVertexPointer(vertexComponentLength, GL.GL_FLOAT, 0, vertexBuffer);
		gl.glNormalPointer(GL.GL_FLOAT, 0, normalBuffer);
		if (visible) {
			if (transparency < 0.99) {
				gl.glDisable(GL2.GL_LIGHTING);
				gl.glDisable(GL2.GL_DEPTH_TEST);
				if (texMapBuffer != null && textureIndex >= 0
						&& showTextureParam.getValue()) {
					gl.glEnable(GL.GL_TEXTURE_2D);

					gl.glEnableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
					gl.glBindTexture(GL.GL_TEXTURE_2D, textureIndex);
					gl.glTexCoordPointer(2, GL.GL_FLOAT, 0, texMapBuffer);
				} else {
					gl.glColor4f(scale * diffuseColor.getRed(), scale
							* diffuseColor.getGreen(),
							scale * diffuseColor.getBlue(), transparency);
				}
				if (colorBuffer != null && showColorParam.getValue()) {
					gl.glEnableClientState(GL2.GL_COLOR_ARRAY);
					gl.glColorPointer(4, GL.GL_FLOAT, 0, colorBuffer);
				}
				gl.glPolygonMode(GL2.GL_FRONT_AND_BACK, GL2.GL_FILL);
				if (indexBuffer != null) {
					gl.glDrawElements(GL.GL_TRIANGLES, indexBuffer.capacity(),
							GL.GL_UNSIGNED_INT, indexBuffer);
				} else {
					gl.glDrawArrays(GL.GL_TRIANGLES, 0, vertexBuffer.capacity()
							/ vertexComponentLength);
				}
				if (colorBuffer != null && showColorParam.getValue()) {

					gl.glDisableClientState(GL2.GL_COLOR_ARRAY);
				}
				if (texMapBuffer != null) {
					gl.glDisableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
					gl.glDisable(GL.GL_TEXTURE_2D);
				}

				gl.glEnable(GL2.GL_DEPTH_TEST);
			} else {
				gl.glEnable(GL2.GL_LIGHTING);
				if (texMapBuffer != null && textureIndex >= 0
						&& showTextureParam.getValue()) {
					gl.glEnable(GL.GL_TEXTURE_2D);

					gl.glEnableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
					gl.glBindTexture(GL.GL_TEXTURE_2D, textureIndex);
					gl.glTexCoordPointer(2, GL.GL_FLOAT, 0, texMapBuffer);
				} else {
					gl.glColor4f(scale * diffuseColor.getRed(), scale
							* diffuseColor.getGreen(),
							scale * diffuseColor.getBlue(), transparency);
				}
				if (colorBuffer != null && showColorParam.getValue()) {
					gl.glEnableClientState(GL2.GL_COLOR_ARRAY);
					gl.glColorPointer(4, GL.GL_FLOAT, 0, colorBuffer);
				}
				gl.glPolygonMode(GL2.GL_FRONT_AND_BACK, GL2.GL_FILL);
				if (indexBuffer != null) {
					gl.glDrawElements(GL.GL_TRIANGLES, indexBuffer.capacity(),
							GL.GL_UNSIGNED_INT, indexBuffer);
				} else {
					gl.glDrawArrays(GL.GL_TRIANGLES, 0, vertexBuffer.capacity()
							/ vertexComponentLength);
				}
				if (colorBuffer != null && showColorParam.getValue()) {

					gl.glDisableClientState(GL2.GL_COLOR_ARRAY);
				}
				if (texMapBuffer != null) {
					gl.glDisableClientState(GL2.GL_TEXTURE_COORD_ARRAY);
					gl.glDisable(GL.GL_TEXTURE_2D);
				}
			}
		}
		if (wireframe) {

			gl.glDisable(GL2.GL_LIGHTING);
			gl.glShadeModel(GL2.GL_FLAT);
			gl.glLineWidth(1.5f);
			gl.glColor4f(0, 0, 0, transparency);
			gl.glPolygonMode(GL2.GL_FRONT_AND_BACK, GL2.GL_LINE);
			if (indexBuffer != null) {
				gl.glDrawElements(GL.GL_TRIANGLES, indexBuffer.capacity(),
						GL.GL_UNSIGNED_INT, indexBuffer);
			} else {
				gl.glDrawArrays(GL.GL_TRIANGLES, 0, vertexBuffer.capacity()
						/ vertexComponentLength);
			}

			gl.glEnable(GL2.GL_LIGHTING);
		}

		gl.glDisableClientState(GL2.GL_NORMAL_ARRAY);
		gl.glDisableClientState(GL2.GL_VERTEX_ARRAY);

		gl.glPolygonMode(GL2.GL_FRONT_AND_BACK, GL2.GL_FILL);
		((PGraphicsOpenGL2) applet.g).endGL();

		((GL2) ((PGraphicsOpenGL2) applet.g).gl).glEnable(GL2.GL_LIGHTING);
		applet.popStyle();
		applet.popMatrix();
	}

	/**
	 * Creates the visualization parameters.
	 * 
	 * @param visualizationParameters
	 *            the visualization parameters
	 * @see edu.jhu.cs.cisst.vent.VisualizationParameters#createVisualizationParameters(edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection)
	 */
	public void createVisualizationParameters(
			ParamCollection visualizationParameters) {
		// TODO Auto-generated method stub
		visualizationParameters.setName("Triangle Mesh");
		visualizationParameters.add(diffuseColorParam = new ParamColor(
				"Diffuse Color", diffuseColor));
		visualizationParameters.add(transparencyParam = new ParamDouble(
				"Transparency", 0, 1, transparency));
		transparencyParam.setInputView(new ParamDoubleSliderInputView(
				transparencyParam, 4, false));
		visualizationParameters.add(gouraudParam = new ParamBoolean(
				"Gouraud Shading", gouraud));
		visualizationParameters.add(wireframeParam = new ParamBoolean(
				"Wireframe", wireframe));
		visualizationParameters.add(visibleParam = new ParamBoolean(
				"Triangles", visible));
		visualizationParameters.add(showColorParam = new ParamBoolean(
				"Color Map", false));
		if (texMapBuffer != null) {
			visualizationParameters.add(showTextureParam = new ParamBoolean(
					"Texture Map", true));
		}
	}

	/**
	 * Update visualization parameters.
	 * 
	 * @see edu.jhu.cs.cisst.vent.VisualizationParameters#updateVisualizationParameters()
	 */
	@Override
	public void updateVisualizationParameters() {
		setVisible(visibleParam.getValue());
		diffuseColor = diffuseColorParam.getValue();
		wireframe = wireframeParam.getValue();
		transparency = transparencyParam.getFloat();
		gouraud = gouraudParam.getValue();
	}

	/**
	 * Update.
	 * 
	 * @param model
	 *            the model
	 * @param view
	 *            the view
	 * @see edu.jhu.ece.iacl.jist.pipeline.view.input.ParamViewObserver#update(edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel,
	 *      edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView)
	 */
	@Override
	public void update(ParamModel model, ParamInputView view) {
		if (model == visibleParam) {
			setVisible(visibleParam.getValue());
		} else if (model == diffuseColorParam) {
			diffuseColor = diffuseColorParam.getValue();
		} else if (model == wireframeParam) {
			wireframe = wireframeParam.getValue();
		} else if (model == gouraudParam) {
			gouraud = gouraudParam.getValue();
		} else if (model == transparencyParam) {
			transparency = transparencyParam.getFloat();
		}
	}

	/**
	 * Setup.
	 * 
	 * @see edu.jhu.cs.cisst.vent.renderer.processing.RendererProcessing#setup()
	 */
	@Override
	public void setup() {
	}

	protected PMatrix3D transform = null;

	public void setTransform(PMatrix3D mat) {
		transform = mat;
	}

}
