package edu.jhu.ece.iacl.algorithms.registration;

import edu.jhu.ece.iacl.jist.structures.image.ImageData;


//N-Dimensional Histogram - Single Array Implementation
//If it is a Joint Histogram between subjects and targets, by convention subjects are placed first
public class NDimHistogramArray{
	private int[] data;
	private int numOfDimensions;
	private int numOfTerms;
	private int numOfBins;
	private int totalVol;
	
	public NDimHistogramArray(int numOfDimensions, int numOfBins) {		
		this.numOfDimensions = numOfDimensions;
		this.numOfBins = numOfBins;
		this.numOfTerms = (int)Math.pow(numOfBins,numOfDimensions);
		data = new int[numOfTerms];
		totalVol=0;
	}
	
	public NDimHistogramArray(int numOfSub, int numOfTar, int numOfBins) {
		this(numOfSub + numOfTar, numOfBins);
	}
	
	public int convertIndex(int[] index){
		int index1D = 0;
		for(int i = 0; i < numOfDimensions; i++){
			index1D += index[i]*(int)Math.pow(numOfBins,i);
		}
		return index1D;
	}
	
	//get bin
	public int get(int[] index){
		return data[convertIndex(index)]; 
	}
	public int get(int[] subIndexes, int[] tarIndexes){
		return get(combineIndexes(subIndexes, tarIndexes));
	}

	//set bin
	public void set(int[] index, int val){
		int index1D = convertIndex(index);
		totalVol -= data[index1D];
		data[index1D] = val;
		totalVol += val;
		
	}
	public void set(int[] subIndexes, int[] tarIndexes, int val){
		set(combineIndexes(subIndexes, tarIndexes), val);
	}

	//raise bin by one
	public void increment(int[] index){
		data[convertIndex(index)]++;
		totalVol++;
	}
	public void increment(int[] subIndexes, int[] tarIndexes){
		increment(combineIndexes(subIndexes, tarIndexes));
	}
	
	//drop bin by one	
	public void decrement(int[] index){
		data[convertIndex(index)]--;
		totalVol--;
	}
	public void decrement(int[] subIndexes, int[] tarIndexes){
		decrement(combineIndexes(subIndexes, tarIndexes));
	}
	
	
	
	public int[] combineIndexes(int[] subIndexes, int[] tarIndexes){
		int[] combIndexes = new int[subIndexes.length + tarIndexes.length];
		for (int i = 0; i < combIndexes.length; i++){
			if(i < subIndexes.length) combIndexes[i] = subIndexes[i];
			else combIndexes[i] = tarIndexes[i-subIndexes.length];
		}
		return combIndexes;
		
	}
	
	//given subject and target images, fill up the joint histogram
	public void fillJointHistogram(ImageData[] subjects, ImageData[] targets, int[] boundingBox){
		int totalLength = subjects.length + targets.length; 
		if((totalLength) != numOfDimensions){
			System.out.format("Subjects/Target Length Does Not Match Histogram Size\n");
			return;
		}
		int[] index = new int[totalLength];
		int bin; 
		clearHist();
		
		for (int i = boundingBox[0]; i <= boundingBox[1]; i++) 
			for (int j = boundingBox[2]; j <= boundingBox[3]; j++) 
				for (int k = boundingBox[4]; k <= boundingBox[5]; k++) {

					//get bin index for all subjects and targets
					for(int ch = 0; ch < totalLength; ch++ ){
						if (ch < subjects.length) bin = subjects[ch].getUByte(i, j, k);
						else bin = targets[ch - subjects.length].getUByte(i, j, k);
						
						//Make sure is in bounds
						if (bin >= numOfBins){
							bin = numOfBins - 1;
							System.out.format("Value outside of bin range\n");
						}
						if (bin < 0) {
							bin = 0;
							System.out.format("Value outside of bin range\n");
						}
						index[ch] = bin;
					}
					//increment bin
					increment(index);
		}		
	}
	
	public double calcEntropy(){
		double entropy = 0;
		double tmp;
		for(int i = 0; i < numOfTerms; i++){
			if (data[i] > 0) {
					tmp = ((double) data[i]) / totalVol;
					entropy -= tmp * Math.log(tmp);
			}
		}
		
		return entropy;
	}
	
	public NDimHistogramArray clone(){
		NDimHistogramArray clone = new NDimHistogramArray(numOfDimensions, numOfBins);
		clone.copy(this);
		return clone;
	}

	public void copy(NDimHistogramArray copyFrom){
		if((numOfDimensions != copyFrom.numOfDimensions) || (numOfBins != copyFrom.numOfBins)){
			System.out.format("Invalid Copy - Objects Don't Match\n");
			return;
		}
		for(int i = 0; i < numOfTerms; i++)	data[i] = copyFrom.data[i];
		totalVol = copyFrom.totalVol;
	}
	
	public int getDimensionNum() {return numOfDimensions;}
	public int getNumOfBins() {return numOfBins;}
	public int getTotalVol() {return totalVol;}
	
	public void clearHist(){
		totalVol = 0;
		data = new int[numOfTerms];
	}

	
}
